/*
 * Copyright (c) 2023 Texas Instruments Incorporated
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 *   Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the
 *   distribution.
 *
 *   Neither the name of Texas Instruments Incorporated nor the names of
 *   its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "Pwm_Platform.h"

/*****************************************************************************

 EPWM_setEmulationMode

*****************************************************************************/
FUNC(void, PWM_CODE) EPWM_setEmulationMode(uint32 base, EPWM_EmulationMode emulationMode)
{
    /*
     Write to FREE_SOFT bits
    */
    HW_WR_REG16(base + PWM_EPWM_TBCTL,
    ((HW_RD_REG16(base + PWM_EPWM_TBCTL) & (~PWM_EPWM_TBCTL_FREE_SOFT_MASK)) |
            ((uint16)emulationMode << PWM_EPWM_TBCTL_FREE_SOFT_SHIFT)));
}

/*****************************************************************************

 EPWM_configureSignal

******************************************************************************/
FUNC(void, PWM_CODE) EPWM_configureSignal(uint32 base, const EPWM_SignalParams *signalParams)
{
    uint16 tbPrdVal = 0U, cmpAVal = 0U, cmpBVal = 0U;

    /*
     Configure Time Base counter Clock
    */
	EPWM_setClockPrescaler(base, signalParams->tbClkDiv,
                           signalParams->tbHSClkDiv);

    /*
     Configure Time Base Counter Mode
    */
    EPWM_setTimeBaseCounterMode(base, signalParams->tbCtrMode);

   
	
	tbPrdVal = signalParams->freqInHz;
    cmpAVal = signalParams->dutyValA;
    cmpBVal = signalParams->dutyValB;

    /*
     Configure TBPRD value
    */
    EPWM_setTimeBasePeriod(base, tbPrdVal);

    /*
     Default Configurations.
    */
    EPWM_disablePhaseShiftLoad(base);
    EPWM_setPhaseShift(base, 0U);
    EPWM_setTimeBaseCounter(base, 0U);

    /*
     Setup shadow register load on ZERO
    */
    EPWM_setCounterCompareShadowLoadMode(base,
                                         EPWM_COUNTER_COMPARE_A,
                                         EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareShadowLoadMode(base,
                                         EPWM_COUNTER_COMPARE_B,
                                         EPWM_COMP_LOAD_ON_CNTR_ZERO);
    /*
     Set Compare values
    */
    EPWM_setCounterCompareValue(base, EPWM_COUNTER_COMPARE_A, cmpAVal);
    EPWM_setCounterCompareValue(base, EPWM_COUNTER_COMPARE_B, cmpBVal);

    /*
     Set actions for ePWMxA & ePWMxB
    */
    if(signalParams->tbCtrMode == EPWM_COUNTER_MODE_UP)
    {
        /*
        Set PWMxA on Zero
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_HIGH,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

        /*
         Clear PWMxA on event A, up count
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_LOW,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

        if(signalParams->invertSignalB == TRUE)
        {
            /*
             Clear PWMxB on Zero
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
            /*
             Set PWMxB on event B, up count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
        }
        else
        {
            /*
            Set PWMxB on Zero
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
            /*
             Clear PWMxB on event B, up count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);

        }
    }
    else if((signalParams->tbCtrMode == EPWM_COUNTER_MODE_DOWN))
    {
        /*
         Set PWMxA on Zero
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_HIGH,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

        /*
         Clear PWMxA on event A, down count
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_LOW,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

        if(signalParams->invertSignalB == TRUE)
        {
            /*
             Clear PWMxB on Zero
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
            /*
             Set PWMxB on event B, down count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
        }
        else
        {
            /*
             Set PWMxB on Zero
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
            /*
             Clear PWMxB on event B, down count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
        }
    }
    else if(signalParams->tbCtrMode == EPWM_COUNTER_MODE_UP_DOWN)
    {
        /*
         Clear PWMxA on Zero
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_LOW,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

        /*
         Set PWMxA on event A, up count
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_HIGH,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);

        /*
         Clear PWMxA on event A, down count
        */
        EPWM_setActionQualifierAction(base,
                                      EPWM_AQ_OUTPUT_A,
                                      EPWM_AQ_OUTPUT_LOW,
                                      EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);

        if(signalParams->invertSignalB == TRUE)
        {
            /*
             Set PWMxB on Zero
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

            /*
             Clear PWMxB on event B, up count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
            /*
             Set PWMxB on event B, down count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
        }
        else
        {
            /*
             Clear PWMxB on Zero
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);

            /*
             Set PWMxB on event B, up count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_HIGH,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
            /*
             Clear PWMxB on event B, down count
            */
            EPWM_setActionQualifierAction(base,
                                          EPWM_AQ_OUTPUT_B,
                                          EPWM_AQ_OUTPUT_LOW,
                                          EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
        }
    }
}

FUNC(void, PWM_CODE) Pwm_ConfigHR_epwm(uint32 baseAddr,uint32 sysClk, float32 peroid, float32 Duty, boolean enable, uint32 outputCh)
{    
    uint32 compCount = 0;
    uint32 Ctrladdr = 0;
    uint32 intperiod = 0;
    float32 Count = 0;
    float32 abduty = 0;
    uint32 intduty = 0;
    uint32 delay;
	float32 temp_val;
	float32 highRes_regVal;
	uint32 MEP_step = 150U; /*150ps choosen, for 100MHZ*/
    float32 dutyFine = 0.0;
	
    /* TBCLK/MEP_step_size_am263x = 5ns/150ps*/
    uint16 MEP_ScaleFactor = (1U/sysClk)/(MEP_step*0.000000000001); 
    
    float32 periodFine = PWM_HR_MIN_HRPWM_PRD_PERCENT;
	
	intduty = (uint32)(Duty);
	abduty = (float32)Duty - (float32)(intduty);
	intperiod = (uint32)(peroid);
	temp_val = (float32)((abduty * MEP_ScaleFactor)) ;
    highRes_regVal = (temp_val * (float32)256U) + (float32)0.5 ;
    
    /* Enable HR PWM signal. */
	if(enable)
	{
        if(baseAddr == 0x50000000UL)
        {
            Ctrladdr = 0x502E0042UL;
        }
        else if(baseAddr == 0x50040000UL)
        {
            Ctrladdr = 0x502E1042UL;
        }
        else if(baseAddr == 0x50080000UL)
        {
            Ctrladdr = 0x502E2042UL;
        }
        else if(baseAddr == 0x500C0000UL)
        {
            Ctrladdr = 0x502E3042UL;
        }
        
        /* Enable MEP Scale Step. */
        HRPWM_setMEPStep(Ctrladdr, MEP_ScaleFactor);
    
        /* Enable Period Control. */
        HRPWM_enablePeriodControl(baseAddr);
		HRPWM_enablePhaseShiftLoad(baseAddr);
        
        /* Set the Channel for HR PWM.  */
        if (((uint32)EPWM_HR_PWMA == outputCh) ||
            ((uint32)EPWM_HR_PWMA_B == outputCh))
        {
            HRPWM_setCounterCompareValue(baseAddr, HRPWM_COUNTER_COMPARE_A, (uint32)(Duty));
            HRPWM_setHiResCounterCompareValue(baseAddr, HRPWM_COUNTER_COMPARE_A, (uint32)(highRes_regVal));
            HRPWM_setMEPEdgeSelect(baseAddr, HRPWM_CHANNEL_A, HRPWM_MEP_CTRL_RISING_AND_FALLING_EDGE);
            HRPWM_setMEPControlMode(baseAddr, HRPWM_CHANNEL_A, HRPWM_MEP_DUTY_PERIOD_CTRL);
            HRPWM_setCounterCompareShadowLoadEvent(baseAddr, HRPWM_CHANNEL_A, HRPWM_LOAD_ON_CNTR_ZERO_PERIOD);
        }

        if (((uint32)EPWM_HR_PWMB == outputCh) ||
            ((uint32)EPWM_HR_PWMA_B == outputCh))
        {
            HRPWM_setCounterCompareValue(baseAddr, HRPWM_COUNTER_COMPARE_B, (uint32)(Duty));
            HRPWM_setHiResCounterCompareValue(baseAddr, HRPWM_COUNTER_COMPARE_B, (uint32)(highRes_regVal));
            HRPWM_setMEPEdgeSelect(baseAddr, HRPWM_CHANNEL_B, HRPWM_MEP_CTRL_RISING_AND_FALLING_EDGE);
            HRPWM_setMEPControlMode(baseAddr, HRPWM_CHANNEL_B, HRPWM_MEP_DUTY_PERIOD_CTRL);
            HRPWM_setCounterCompareShadowLoadEvent(baseAddr, HRPWM_CHANNEL_B, HRPWM_LOAD_ON_CNTR_ZERO_PERIOD);
        }
        
        /* Enable Auto conversion. */
        HRPWM_enableAutoConversion(baseAddr);
        
        for(periodFine = PWM_HR_MIN_HRPWM_PRD_PERCENT; periodFine < 0.9; periodFine += 0.01)
        {		   
		   /* Stablize the HR PWM. */
           Count = ((intperiod - 1) << 8UL) + (float32)(periodFine * 256);
           compCount = (uint32)Count;
           
           /* Set Period. */
           HRPWM_setHiResTimeBasePeriod(baseAddr, compCount);
        }

        for(dutyFine = 0.0; dutyFine < 99.9; dutyFine += 0.01)
        {

            Count = (dutyFine * (float32)(intduty << 8))/100;
            compCount = (Count);
            HRPWM_setCounterCompareValue(baseAddr, HRPWM_COUNTER_COMPARE_A, compCount);
            HRPWM_setCounterCompareValue(baseAddr, HRPWM_COUNTER_COMPARE_B, compCount);
        }
	}
    else
    {
        /* Disable Autoconversion. */
        HRPWM_disableAutoConversion(baseAddr);
        HRPWM_disablePeriodControl(baseAddr);
    }
}
FUNC(void, PWM_CODE) Pwm_HwSetDefReg_epwm(uint32 baseAddr)
{
    /*
      Init all registers to default value
     */                                        
    HW_WR_REG16((baseAddr+ PWM_EPWM_TBCTL)     ,(uint16)PWM_EPWM_TBCTL_RESETVAL)     ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_TBCTL2)    ,(uint16)PWM_EPWM_TBCTL2_RESETVAL)    ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_CMPCTL)    ,(uint16)PWM_EPWM_CMPCTL_RESETVAL)    ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_CMPA)      ,(uint16)PWM_EPWM_CMPA_RESETVAL)      ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_CMPB)      ,(uint16)PWM_EPWM_CMPB_RESETVAL)      ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_AQCTLA)    ,(uint16)PWM_EPWM_AQCTLA_RESETVAL)    ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_AQCTLB)    ,(uint16)PWM_EPWM_AQCTLB_RESETVAL)    ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_AQSFRC)    ,(uint16)PWM_EPWM_AQSFRC_RESETVAL)    ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_AQCSFRC)   ,(uint16)PWM_EPWM_AQCSFRC_RESETVAL)   ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_TZSEL)     ,(uint16)PWM_EPWM_TZSEL_RESETVAL)     ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_ETPS)      ,(uint16)PWM_EPWM_ETPS_RESETVAL)      ;
    HW_WR_REG16((baseAddr+ PWM_EPWM_DECLR)     ,(uint16)PWM_EPWM_DECLR_RESETVAL)     ;
    HW_WR_REG16((baseAddr+ PWM_OTTOCAL_HRPWR)  ,(uint16)PWM_OTTOCAL_HRPWR_RESETVAL)  ;
    HW_WR_REG16((baseAddr+ PWM_OTTOCAL_HRCAL)  ,(uint16)PWM_OTTOCAL_HRCAL_RESETVAL)  ;
    HW_WR_REG16((baseAddr+ PWM_OTTOCAL_HRPRD)  ,(uint16)PWM_OTTOCAL_HRPRD_RESETVAL)  ;
    HW_WR_REG16((baseAddr+ PWM_OTTOCAL_HRMSTEP),(uint16)PWM_OTTOCAL_HRMSTEP_RESETVAL);

    return;
}

void EPWM_Initialize(uint32 epwmbaseadrr)
{
	/* Time Base */
	EPWM_setEmulationMode(epwmbaseadrr, EPWM_EMULATION_FREE_RUN);
	EPWM_setClockPrescaler(epwmbaseadrr, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_1);
	EPWM_setTimeBasePeriod(epwmbaseadrr, 0);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_TBPRD_TBPRDHR);
	EPWM_setPeriodLoadMode(epwmbaseadrr, EPWM_PERIOD_SHADOW_LOAD);
	EPWM_setTimeBaseCounter(epwmbaseadrr, 0);
	EPWM_setTimeBaseCounterMode(epwmbaseadrr, EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setCountModeAfterSync(epwmbaseadrr, EPWM_COUNT_MODE_DOWN_AFTER_SYNC);
	EPWM_disablePhaseShiftLoad(epwmbaseadrr);
	EPWM_setPhaseShift(epwmbaseadrr, 0);
	EPWM_enableSyncOutPulseSource(epwmbaseadrr, 0);
	EPWM_setSyncInPulseSource(epwmbaseadrr, EPWM_SYNC_IN_PULSE_SRC_DISABLE);
	EPWM_setOneShotSyncOutTrigger(epwmbaseadrr, EPWM_OSHT_SYNC_OUT_TRIG_SYNC);
	EPWM_forceSyncPulse(epwmbaseadrr);

	/* Counter Compare */
	EPWM_setCounterCompareValue(epwmbaseadrr, EPWM_COUNTER_COMPARE_A, 0);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_CMPA_CMPAHR);
	
	EPWM_setCounterCompareShadowLoadMode(epwmbaseadrr, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);
	EPWM_setCounterCompareValue(epwmbaseadrr, EPWM_COUNTER_COMPARE_B, 0);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_CMPB_CMPBHR);
	
	EPWM_setCounterCompareShadowLoadMode(epwmbaseadrr, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);
	EPWM_setCounterCompareValue(epwmbaseadrr, EPWM_COUNTER_COMPARE_C, 0);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_CMPC);
	
	EPWM_setCounterCompareShadowLoadMode(epwmbaseadrr, EPWM_COUNTER_COMPARE_C, EPWM_COMP_LOAD_ON_CNTR_ZERO);
	EPWM_setCounterCompareValue(epwmbaseadrr, EPWM_COUNTER_COMPARE_D, 0);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_CMPD);
	
	EPWM_setCounterCompareShadowLoadMode(epwmbaseadrr, EPWM_COUNTER_COMPARE_D, EPWM_COMP_LOAD_ON_CNTR_ZERO);

	/* Action Qualifier */
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_AQCSFRC);
	EPWM_setActionQualifierContSWForceShadowMode(epwmbaseadrr, EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_AQCTLA_AQCTLA2);
	EPWM_disableActionQualifierShadowLoadMode(epwmbaseadrr, EPWM_ACTION_QUALIFIER_A);
	EPWM_setActionQualifierShadowLoadMode(epwmbaseadrr, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);
	EPWM_setActionQualifierT1TriggerSource(epwmbaseadrr, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
	EPWM_setActionQualifierT2TriggerSource(epwmbaseadrr, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
	EPWM_setActionQualifierSWAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE);
	EPWM_setActionQualifierContSWForceAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_SW_DISABLED);
	EPWM_disableGlobalLoadRegisters(epwmbaseadrr, EPWM_GL_REGISTER_AQCTLB_AQCTLB2);
	EPWM_disableActionQualifierShadowLoadMode(epwmbaseadrr, EPWM_ACTION_QUALIFIER_B);
	EPWM_setActionQualifierShadowLoadMode(epwmbaseadrr, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);
	EPWM_setActionQualifierT1TriggerSource(epwmbaseadrr, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
	EPWM_setActionQualifierT2TriggerSource(epwmbaseadrr, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
	EPWM_setActionQualifierSWAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE);
	EPWM_setActionQualifierContSWForceAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_SW_DISABLED);

	/* Events */
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_UP);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_UP);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_UP);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_UP);
	EPWM_setActionQualifierAction(epwmbaseadrr, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN);

	/* Trip Zone */
	EPWM_setTripZoneAction(epwmbaseadrr, EPWM_TZ_ACTION_EVENT_TZA, EPWM_TZ_ACTION_HIGH_Z);
	EPWM_setTripZoneAction(epwmbaseadrr, EPWM_TZ_ACTION_EVENT_TZB, EPWM_TZ_ACTION_HIGH_Z);
	EPWM_setTripZoneAction(epwmbaseadrr, EPWM_TZ_ACTION_EVENT_DCAEVT1, EPWM_TZ_ACTION_HIGH_Z);
	EPWM_setTripZoneAction(epwmbaseadrr, EPWM_TZ_ACTION_EVENT_DCAEVT2, EPWM_TZ_ACTION_HIGH_Z);
	EPWM_setTripZoneAction(epwmbaseadrr, EPWM_TZ_ACTION_EVENT_DCBEVT1, EPWM_TZ_ACTION_HIGH_Z);
	EPWM_setTripZoneAction(epwmbaseadrr, EPWM_TZ_ACTION_EVENT_DCBEVT2, EPWM_TZ_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvAction(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_TZB_D, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvAction(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_TZB_U, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvAction(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_TZA_D, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvAction(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_TZA_U, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionA(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_U, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionA(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_D, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionA(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_U, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionA(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_D, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionB(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_U, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionB(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_D, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionB(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_U, EPWM_TZ_ADV_ACTION_HIGH_Z);
	EPWM_setTripZoneAdvDigitalCompareActionB(epwmbaseadrr, EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_D, EPWM_TZ_ADV_ACTION_HIGH_Z);
    EPWM_disableTripZoneAdvAction(epwmbaseadrr);
	EPWM_enableTripZoneSignals(epwmbaseadrr, 0);
	EPWM_enableTripZoneSignals(epwmbaseadrr, 0);
	EPWM_selectCycleByCycleTripZoneClearEvent(epwmbaseadrr, EPWM_TZ_CBC_PULSE_CLR_CNTR_ZERO);
	EPWM_enableTripZoneInterrupt(epwmbaseadrr, 0);

	/* Digital Compare */
	EPWM_setDigitalCompareFilterInput(epwmbaseadrr, EPWM_DC_WINDOW_SOURCE_DCAEVT1);
	EPWM_disableDigitalCompareBlankingWindow(epwmbaseadrr);
	EPWM_setDigitalCompareBlankingEvent(epwmbaseadrr, EPWM_DC_WINDOW_START_TBCTR_PERIOD, EPWM_DC_WINDOW_START_TBCTR_PERIOD);
	EPWM_setDigitalCompareWindowOffset(epwmbaseadrr, 0);
	EPWM_setDigitalCompareWindowLength(epwmbaseadrr, 0);
	EPWM_disableDigitalCompareWindowInverseMode(epwmbaseadrr);
	EPWM_disableDigitalCompareCounterCapture(epwmbaseadrr);
	EPWM_setDigitalCompareCounterShadowMode(epwmbaseadrr, FALSE);
	EPWM_disableDigitalCompareEdgeFilter(epwmbaseadrr);
	EPWM_setDigitalCompareEdgeFilterMode(epwmbaseadrr, EPWM_DC_EDGEFILT_MODE_RISING);
	EPWM_setDigitalCompareEdgeFilterEdgeCount(epwmbaseadrr, EPWM_DC_EDGEFILT_EDGECNT_0);
	EPWM_disableValleyCapture(epwmbaseadrr);
	EPWM_setValleyTriggerSource(epwmbaseadrr, EPWM_VALLEY_TRIGGER_EVENT_SOFTWARE);
	
	EPWM_setValleyTriggerEdgeCounts(epwmbaseadrr, 0, 0);
	EPWM_disableValleyHWDelay(epwmbaseadrr);
	EPWM_setValleySWDelayValue(epwmbaseadrr, 0);
	EPWM_setValleyDelayDivider(epwmbaseadrr, EPWM_VALLEY_DELAY_MODE_SW_DELAY);
	EPWM_enableDigitalCompareTripCombinationInput(epwmbaseadrr, 0, EPWM_DC_TYPE_DCAH);
    EPWM_selectDigitalCompareTripInput(epwmbaseadrr, EPWM_DC_TRIP_TRIPIN1, EPWM_DC_TYPE_DCAH);
	EPWM_enableDigitalCompareTripCombinationInput(epwmbaseadrr, 0, EPWM_DC_TYPE_DCAL);
    EPWM_selectDigitalCompareTripInput(epwmbaseadrr, EPWM_DC_TRIP_TRIPIN1, EPWM_DC_TYPE_DCAL);
	EPWM_setTripZoneDigitalCompareEventCondition(epwmbaseadrr, EPWM_TZ_DC_OUTPUT_A1, EPWM_TZ_EVENT_DC_DISABLED);
	EPWM_setTripZoneDigitalCompareEventCondition(epwmbaseadrr, EPWM_TZ_DC_OUTPUT_A2, EPWM_TZ_EVENT_DC_DISABLED);
	EPWM_disableDigitalCompareADCTrigger(epwmbaseadrr, EPWM_DC_MODULE_A);
	
	EPWM_setDigitalCompareEventSyncMode(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_1, EPWM_DC_EVENT_INPUT_SYNCED);
	EPWM_setDigitalCompareEventSource(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_1, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
	EPWM_setDigitalCompareEventSyncMode(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_2, EPWM_DC_EVENT_INPUT_SYNCED);
	EPWM_setDigitalCompareEventSource(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_2, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
	EPWM_enableDigitalCompareTripCombinationInput(epwmbaseadrr, 0, EPWM_DC_TYPE_DCBH);
    EPWM_selectDigitalCompareTripInput(epwmbaseadrr, EPWM_DC_TRIP_TRIPIN1, EPWM_DC_TYPE_DCBH);
	EPWM_enableDigitalCompareTripCombinationInput(epwmbaseadrr, 0, EPWM_DC_TYPE_DCBL);
    EPWM_selectDigitalCompareTripInput(epwmbaseadrr, EPWM_DC_TRIP_TRIPIN1, EPWM_DC_TYPE_DCBL);
	EPWM_setTripZoneDigitalCompareEventCondition(epwmbaseadrr, EPWM_TZ_DC_OUTPUT_B1, EPWM_TZ_EVENT_DC_DISABLED);
	EPWM_setTripZoneDigitalCompareEventCondition(epwmbaseadrr, EPWM_TZ_DC_OUTPUT_B2, EPWM_TZ_EVENT_DC_DISABLED);

	EPWM_disableDigitalCompareADCTrigger(epwmbaseadrr, EPWM_DC_MODULE_B);
	
	EPWM_setDigitalCompareEventSyncMode(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_1, EPWM_DC_EVENT_INPUT_SYNCED);
	EPWM_setDigitalCompareEventSource(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_1, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
	EPWM_setDigitalCompareEventSyncMode(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_2, EPWM_DC_EVENT_INPUT_SYNCED);
	EPWM_setDigitalCompareEventSource(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_2, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);

	EPWM_setDigitalCompareCBCLatchMode(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_1, EPWM_DC_CBC_LATCH_DISABLED);
	EPWM_selectDigitalCompareCBCLatchClearEvent(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_1, EPWM_DC_CBC_LATCH_CLR_CNTR_ZERO);
	EPWM_setDigitalCompareCBCLatchMode(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_2, EPWM_DC_CBC_LATCH_DISABLED);
	EPWM_selectDigitalCompareCBCLatchClearEvent(epwmbaseadrr, EPWM_DC_MODULE_A, EPWM_DC_EVENT_2, EPWM_DC_CBC_LATCH_CLR_CNTR_ZERO);
	EPWM_setDigitalCompareCBCLatchMode(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_1, EPWM_DC_CBC_LATCH_DISABLED);
	EPWM_selectDigitalCompareCBCLatchClearEvent(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_1, EPWM_DC_CBC_LATCH_CLR_CNTR_ZERO);
	EPWM_setDigitalCompareCBCLatchMode(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_2, EPWM_DC_CBC_LATCH_DISABLED);
	EPWM_selectDigitalCompareCBCLatchClearEvent(epwmbaseadrr, EPWM_DC_MODULE_B, EPWM_DC_EVENT_2, EPWM_DC_CBC_LATCH_CLR_CNTR_ZERO);

	/* Deadband */
    EPWM_disableDeadBandControlShadowLoadMode(epwmbaseadrr);
    EPWM_setDeadBandControlShadowLoadMode(epwmbaseadrr, EPWM_DB_LOAD_ON_CNTR_ZERO);
	EPWM_setRisingEdgeDeadBandDelayInput(epwmbaseadrr, EPWM_DB_INPUT_EPWMA);
	EPWM_setFallingEdgeDeadBandDelayInput(epwmbaseadrr, EPWM_DB_INPUT_EPWMA);
	EPWM_setDeadBandDelayPolarity(epwmbaseadrr, EPWM_DB_RED, EPWM_DB_POLARITY_ACTIVE_HIGH);
	EPWM_setDeadBandDelayPolarity(epwmbaseadrr, EPWM_DB_FED, EPWM_DB_POLARITY_ACTIVE_HIGH);
	EPWM_setDeadBandDelayMode(epwmbaseadrr, EPWM_DB_RED, FALSE);
	EPWM_setDeadBandDelayMode(epwmbaseadrr, EPWM_DB_FED, FALSE);
	EPWM_setDeadBandOutputSwapMode(epwmbaseadrr, EPWM_DB_OUTPUT_A, FALSE);
	EPWM_setDeadBandOutputSwapMode(epwmbaseadrr, EPWM_DB_OUTPUT_B, FALSE);
	
	
	EPWM_disableRisingEdgeDelayCountShadowLoadMode(epwmbaseadrr);
	EPWM_setRisingEdgeDelayCountShadowLoadMode(epwmbaseadrr, EPWM_RED_LOAD_ON_CNTR_ZERO);
    EPWM_setRisingEdgeDelayCount(epwmbaseadrr, 0);
	
	EPWM_disableFallingEdgeDelayCountShadowLoadMode(epwmbaseadrr);
	EPWM_setFallingEdgeDelayCountShadowLoadMode(epwmbaseadrr, EPWM_FED_LOAD_ON_CNTR_ZERO);
    EPWM_setFallingEdgeDelayCount(epwmbaseadrr, 0);
	EPWM_setDeadBandCounterClock(epwmbaseadrr, EPWM_DB_COUNTER_CLOCK_FULL_CYCLE);

	/* Chopper */
	EPWM_disableChopper(epwmbaseadrr);
	EPWM_setChopperDutyCycle(epwmbaseadrr, 0);
	EPWM_setChopperFreq(epwmbaseadrr, 0);
	EPWM_setChopperFirstPulseWidth(epwmbaseadrr, 0);

	/* Event Trigger */
	EPWM_disableInterrupt(epwmbaseadrr);
	EPWM_setInterruptSource(epwmbaseadrr, EPWM_INT_TBCTR_ZERO, EPWM_INT_TBCTR_ZERO);
	EPWM_setInterruptEventCount(epwmbaseadrr, 0);
	EPWM_disableInterruptEventCountInit(epwmbaseadrr);
	EPWM_setInterruptEventCountInitValue(epwmbaseadrr, 0);
	
	EPWM_disableADCTrigger(epwmbaseadrr, EPWM_SOC_A);
	EPWM_setADCTriggerSource(epwmbaseadrr, EPWM_SOC_A, EPWM_SOC_DCxEVT1, EPWM_SOC_DCxEVT1);
	EPWM_setADCTriggerEventPrescale(epwmbaseadrr, EPWM_SOC_A, 0);
	EPWM_disableADCTriggerEventCountInit(epwmbaseadrr, EPWM_SOC_A);
	EPWM_setADCTriggerEventCountInitValue(epwmbaseadrr, EPWM_SOC_A, 0);
	
	EPWM_disableADCTrigger(epwmbaseadrr, EPWM_SOC_B);
	EPWM_setADCTriggerSource(epwmbaseadrr, EPWM_SOC_B, EPWM_SOC_DCxEVT1, EPWM_SOC_DCxEVT1);
	EPWM_setADCTriggerEventPrescale(epwmbaseadrr, EPWM_SOC_B, 0);
	EPWM_disableADCTriggerEventCountInit(epwmbaseadrr, EPWM_SOC_B);
	EPWM_setADCTriggerEventCountInitValue(epwmbaseadrr, EPWM_SOC_B, 0);
	

    /* XCMP Mode */
    EPWM_disableXCMPMode(epwmbaseadrr);
    EPWM_disableSplitXCMP(epwmbaseadrr);
	EPWM_allocAXCMP(epwmbaseadrr, EPWM_XCMP_NONE_CMPA);
    EPWM_setXCMPLoadMode(epwmbaseadrr, EPWM_XCMP_XLOADCTL_LOADMODE_LOADONCE);
    EPWM_setXCMPShadowLevel(epwmbaseadrr, EPWM_XCMP_XLOADCTL_SHDWLEVEL_0);
    EPWM_setXCMPShadowBufPtrLoadOnce(epwmbaseadrr, EPWM_XCMP_XLOADCTL_SHDWBUFPTR_NULL);
    EPWM_setXCMPShadowRepeatBufxCount(epwmbaseadrr, EPWM_XCMP_SHADOW2, 0);
    EPWM_setXCMPShadowRepeatBufxCount(epwmbaseadrr, EPWM_XCMP_SHADOW3, 0);

    EPWM_setXCMPLoadMode(epwmbaseadrr, EPWM_XCMP_XLOADCTL_LOADMODE_LOADONCE);
    EPWM_setXCMPShadowLevel(epwmbaseadrr, EPWM_XCMP_XLOADCTL_SHDWLEVEL_0);
    EPWM_setXCMPShadowBufPtrLoadOnce(epwmbaseadrr, EPWM_XCMP_XLOADCTL_SHDWBUFPTR_NULL);
    EPWM_setXCMPShadowRepeatBufxCount(epwmbaseadrr, EPWM_XCMP_SHADOW2, 0);
    EPWM_setXCMPShadowRepeatBufxCount(epwmbaseadrr, EPWM_XCMP_SHADOW3, 0);

    /* Write values to Reg */
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP1_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP2_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP3_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP4_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP5_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP6_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP7_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP8_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP1_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP2_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP3_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP4_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP5_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP6_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP7_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP8_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP1_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP2_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP3_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP4_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP5_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP6_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP7_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP8_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP1_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP2_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP3_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP4_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP5_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP6_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP7_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XCMP8_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XTBPRD_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XTBPRD_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XTBPRD_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XTBPRD_SHADOW3, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XMINMAX_ACTIVE, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XMINMAX_SHADOW1, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XMINMAX_SHADOW2, 0);
    EPWM_setXCMPRegValue(epwmbaseadrr, EPWM_XMINMAX_SHADOW3, 0);

    /* Events */

    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP1);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP2);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP3);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP4);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_ACTIVE,  EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP1);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP2);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP3);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP4);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW1, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP1);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP2);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP3);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP4);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW2, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP1);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP2);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP3);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP4);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP5);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP6);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP7);
    EPWM_setXCMPActionQualifierAction(epwmbaseadrr, EPWM_XCMP_SHADOW3, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_XCMP8);

    /* Diode Emulation */
    EPWM_disableDiodeEmulationMode(epwmbaseadrr);
    EPWM_setDiodeEmulationMode(epwmbaseadrr, EPWM_DIODE_EMULATION_CBC);
    EPWM_setDiodeEmulationReentryDelay(epwmbaseadrr, 0);
    EPWM_configureDiodeEmulationTripSources(epwmbaseadrr, EPWM_DE_TRIPL, EPWM_DE_TRIP_SRC_INPUTXBAR_OUT0);
    EPWM_configureDiodeEmulationTripSources(epwmbaseadrr, EPWM_DE_TRIPH, EPWM_DE_TRIP_SRC_INPUTXBAR_OUT0);
    EPWM_selectDiodeEmulationPWMsignal(epwmbaseadrr, EPWM_DE_CHANNEL_A, EPWM_DE_SYNC_TRIPHorL);
    EPWM_selectDiodeEmulationTripSignal(epwmbaseadrr, EPWM_DE_CHANNEL_A, EPWM_DE_TRIPL);
    EPWM_selectDiodeEmulationPWMsignal(epwmbaseadrr, EPWM_DE_CHANNEL_B, EPWM_DE_SYNC_TRIPHorL);
    EPWM_selectDiodeEmulationTripSignal(epwmbaseadrr, EPWM_DE_CHANNEL_B, EPWM_DE_TRIPL);
    EPWM_nobypassDiodeEmulationLogic(epwmbaseadrr);
    
    EPWM_disableDiodeEmulationMonitorModeControl(epwmbaseadrr);
    EPWM_setDiodeEmulationMonitorCounterThreshold(epwmbaseadrr, 0);
    EPWM_setDiodeEmulationMonitorModeStep(epwmbaseadrr, EPWM_DE_COUNT_DOWN, 0);
    EPWM_setDiodeEmulationMonitorModeStep(epwmbaseadrr, EPWM_DE_COUNT_UP, 0);

    /* HRPWM */
    HRPWM_disableAutoConversion(epwmbaseadrr);
    HRPWM_setMEPControlMode(epwmbaseadrr, HRPWM_CHANNEL_A, HRPWM_MEP_DUTY_PERIOD_CTRL);
    HRPWM_setMEPControlMode(epwmbaseadrr, HRPWM_CHANNEL_B, HRPWM_MEP_DUTY_PERIOD_CTRL);
    HRPWM_setHiResPhaseShift(epwmbaseadrr, 0);
    HRPWM_setSyncPulseSource(epwmbaseadrr, HRPWM_PWMSYNC_SOURCE_PERIOD);
    HRPWM_disablePhaseShiftLoad(epwmbaseadrr);
    HRPWM_setMEPEdgeSelect(epwmbaseadrr, HRPWM_CHANNEL_A, HRPWM_MEP_CTRL_DISABLE);
    HRPWM_setMEPEdgeSelect(epwmbaseadrr, HRPWM_CHANNEL_B, HRPWM_MEP_CTRL_DISABLE);
    HRPWM_setHiResCounterCompareValue(epwmbaseadrr, HRPWM_COUNTER_COMPARE_A, 1);
    HRPWM_setHiResCounterCompareValue(epwmbaseadrr, HRPWM_COUNTER_COMPARE_B, 1);
    HRPWM_setCounterCompareShadowLoadEvent(epwmbaseadrr, HRPWM_CHANNEL_A, HRPWM_LOAD_ON_CNTR_ZERO);
    HRPWM_setCounterCompareShadowLoadEvent(epwmbaseadrr, HRPWM_CHANNEL_B, HRPWM_LOAD_ON_CNTR_ZERO);
    HRPWM_disablePeriodControl(epwmbaseadrr);
    HRPWM_setHiResTimeBasePeriod(epwmbaseadrr, 0);
    HRPWM_setDeadbandMEPEdgeSelect(epwmbaseadrr, HRPWM_DB_MEP_CTRL_DISABLE);
    HRPWM_setHiResRisingEdgeDelay(epwmbaseadrr, 0);
    HRPWM_setRisingEdgeDelayLoadMode(epwmbaseadrr, HRPWM_LOAD_ON_CNTR_ZERO);
    HRPWM_setHiResFallingEdgeDelayOnly(epwmbaseadrr, 0);
    HRPWM_setFallingEdgeDelayLoadMode(epwmbaseadrr, HRPWM_LOAD_ON_CNTR_ZERO);
    HRPWM_setOutputSwapMode(epwmbaseadrr, FALSE);
    HRPWM_setChannelBOutputPath(epwmbaseadrr, HRPWM_OUTPUT_ON_B_NORMAL);

	/* Global Load */
	EPWM_disableGlobalLoad(epwmbaseadrr);
	EPWM_setGlobalLoadTrigger(epwmbaseadrr, EPWM_GL_LOAD_PULSE_CNTR_ZERO);
	EPWM_setGlobalLoadEventPrescale(epwmbaseadrr, 0);
	EPWM_disableGlobalLoadOneShotMode(epwmbaseadrr);
}