/*
*
* Copyright (c) 2023 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/
/**
 *  \file   Pwm_Priv.c
 *
 *  \brief  This file contains the Internal APIs for PWM.
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "Pwm_Priv.h"
#include "Pwm_Platform.h"

/* ========================================================================== */
/*                                Macros                                      */
/* ========================================================================== */


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */
 uint32 Pwm_GetBaseAddr(Pwm_ChannelType ChannelNumber);
/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define PWM_START_SEC_CODE
#include "Pwm_MemMap.h"

FUNC(void, PWM_CODE) Pwm_ResetChObj(Pwm_ChObjType *chObj)
{
    chObj->chCfg_PC.channelId  = (Pwm_ChannelType)0U;
#ifdef PWM_USE_EPWM
    chObj->chCfg_PC.outputCh      = 0U;
#endif /*PWM_USE_EPWM*/
    chObj->chCfg.instanceClkHz = 0U;
    chObj->chCfg.dutyCycle     = 0U;
    chObj->chCfg.hwPeriod      = 0U;
    chObj->chCfg.polarity      = PWM_LOW;
    chObj->chCfg.idleState     = PWM_LOW;
    chObj->chCfg.channelClass  = PWM_FIXED_PERIOD;
    chObj->chCfg.prescale      = 0U;
#ifdef PWM_USE_EPWM
    chObj->chCfg.hsPrescale    = 0U;
    chObj->chCfg.enableHR      = FALSE;
#endif /*PWM_USE_EPWM*/
#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
    chObj->chCfg.notificationHandler = (Pwm_NotifyFuncType)NULL_PTR;
    chObj->channelNotifyActiveFlag   = (uint32) FALSE;
#endif
    chObj->baseAddr = 0U;
    chObj->channelState = PWM_STATUS_UNINIT;
    return;
}

FUNC(void, PWM_CODE) Pwm_CopyConfig(
    Pwm_ChObjType *chObj, const Pwm_ChannelConfigType *chCfg,
    const Pwm_ChannelConfigType_PC *chCfg_pc)
{
    Pwm_ChannelType channelNumber;
    /* Copying the Config stucture to Channel Object*/
    (FUNC(void, PWM_CODE))memcpy(&chObj->chCfg, &chCfg[0U], sizeof(Pwm_ChannelConfigType));

    /*Copying the pre-compile config structure to Channel Object */
    chObj->chCfg_PC.channelId= chCfg_pc->channelId;
#ifdef PWM_USE_EPWM
    chObj->chCfg_PC.outputCh= chCfg_pc->outputCh;
#endif /*PWM_USE_EPWM*/

    channelNumber = chObj->chCfg_PC.channelId;
    /* Store base address of all the configured channels */
    chObj->baseAddr = chCfg[0U].baseaddr;
    /* Init channelForcedIdle var default to FALSE */
    chObj->channelForcedIdle = (boolean) FALSE;
    chObj->channelState = PWM_STATUS_INIT;
    return;
}

FUNC(uint32, PWM_CODE) Pwm_GetBaseAddr(
    Pwm_ChannelType ChannelNumber)
{
    uint32 baseAddr;

#if (PWM_DEV_ERROR_DETECT == STD_ON)
    if (ChannelNumber >= (uint32) PWM_MAX_NUM_CHANNELS)
    {
        (FUNC(void, PWM_CODE)) Pwm_reportDetError(PWM_SID_INIT, PWM_E_PARAM_CHANNEL);
        baseAddr = 0U;
    }
    else
#endif
    {
        baseAddr = Pwm_ChObj[ChannelNumber].baseAddr;
    }
    return (baseAddr);
}

FUNC(void, PWM_CODE) Pwm_SetIdleState(const Pwm_ChObjType *chObj)
{
    const Pwm_ChannelConfigType *pChannelConfig = &chObj->chCfg;
    const Pwm_ChannelConfigType_PC *channelConfig_PC = &chObj->chCfg_PC;

    uint32 baseAddr;
    EPWM_ActionQualifierSWOutput swTrigAction;
    baseAddr = chObj->baseAddr;

    if (PWM_HIGH == pChannelConfig->idleState)
    {
        swTrigAction = EPWM_AQ_SW_OUTPUT_HIGH;
    }
    else
    {
        swTrigAction = EPWM_AQ_SW_OUTPUT_LOW;
    }

	/* Program AQCSFRC Active Register Reload From Shadow Options */
	EPWM_setActionQualifierContSWForceShadowMode(baseAddr, EPWM_AQ_SW_IMMEDIATE_LOAD);

    /* Enable both outputs epwmxA and epwmxB which is same configuration
     * if channelConfig_PC->outputCh == PWM_OUTPUT_CH_BOTH_A_AND_B */
    if (((uint32)PWM_OUTPUT_CH_A == channelConfig_PC->outputCh) ||
        ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh))
    {
        /* Continuous software forced output on A */
        EPWM_setActionQualifierContSWForceAction(baseAddr, EPWM_AQ_OUTPUT_A, swTrigAction);
    }

    if (((uint32)PWM_OUTPUT_CH_B == channelConfig_PC->outputCh) ||
        ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == channelConfig_PC->outputCh))
    {
        /* Continuous software forced output on B */
        EPWM_setActionQualifierContSWForceAction(baseAddr, EPWM_AQ_OUTPUT_B, swTrigAction);
    }
}


FUNC(void, PWM_CODE) Pwm_HwUnitInit(Pwm_ChObjType *chObj)
{
    uint32 baseAddr;
    Pwm_ChannelConfigType *pChannelConfig = &chObj->chCfg;
    EPWM_SignalParams signalParams;
    uint32 absolute_dutyCycle;
    float64 duty_cycle_percent;
    uint32 duty_cycle_percent_int;
    float64 duty_cycle_percent_dec;
    float64 duty_cycle_period;
    uint32 phwPeriod = 0U;
	
    baseAddr = chObj->baseAddr;
    phwPeriod = pChannelConfig->hwPeriod;
    
	/* Initialize the EPWM module. */
    EPWM_Initialize(baseAddr);
	
	/* Explicit PWM clock enable must not be performed by driver.*/
    Pwm_ChannelConfigType_PC *channelConfig_PC = &chObj->chCfg_PC;
	
    if(pChannelConfig->enableHR == TRUE)
    {
        signalParams.tbCtrMode = EPWM_COUNTER_MODE_UP;

        phwPeriod = phwPeriod * 2;

        duty_cycle_percent = (float32) (((float32)(pChannelConfig->dutyCycle)) /
                ((float32) 32768U));

        /*0x8000 is 32768, calculating percentage from hex*/
        duty_cycle_period = (float64)((float64) duty_cycle_percent * (float64) ((phwPeriod)));
        duty_cycle_percent_int = (uint32) ((phwPeriod));
        duty_cycle_percent_dec = duty_cycle_percent_int - duty_cycle_period;

        /* Set the parameters of PWM. */
        signalParams.freqInHz =  duty_cycle_percent_int;
        signalParams.dutyValA =  (uint16)duty_cycle_period;
        signalParams.dutyValA =  (uint16)duty_cycle_percent_int - duty_cycle_percent_dec;
        signalParams.dutyValB =  (uint16)duty_cycle_period;
        signalParams.dutyValB =  (uint16)duty_cycle_percent_int - duty_cycle_percent_dec;

        /* Set the Polarity. */
        if(pChannelConfig->polarity == PWM_HIGH)
        {
            signalParams.invertSignalB = (boolean)0;
        }
        else
        {
            signalParams.invertSignalB = (boolean)1;
        }

        signalParams.sysClkInHz = pChannelConfig->instanceClkHz;

        signalParams.tbClkDiv = pChannelConfig->prescale;
        signalParams.tbHSClkDiv = pChannelConfig->hsPrescale;

        /* Configure PWM Signal.  */
        EPWM_configureSignal(baseAddr, &signalParams);

        /* Configure HR PWM.  */
        Pwm_ConfigHR_epwm( baseAddr,
                           signalParams.sysClkInHz,
                           (float32) (phwPeriod),
                           duty_cycle_period,
                           pChannelConfig->enableHR,
                           (uint32)channelConfig_PC->outputCh);
    }
    else
    {
        signalParams.tbCtrMode = EPWM_COUNTER_MODE_UP_DOWN;

        /* Calculation for high resolution */
        duty_cycle_percent = (float32) (((float32)(pChannelConfig->dutyCycle)) /
                ((float32) 32768U));

        /*0x8000 is 32768, calculating percentage from hex*/
        duty_cycle_period = (float64)((float64) duty_cycle_percent * (float64) (pChannelConfig->hwPeriod));
        duty_cycle_percent_int = (uint32) (pChannelConfig->hwPeriod);
        duty_cycle_percent_dec = duty_cycle_percent_int - duty_cycle_period;

        /* Set the parameters of PWM. */
        signalParams.freqInHz =  duty_cycle_percent_int;
        signalParams.dutyValA =  (uint16)duty_cycle_period;
        signalParams.dutyValA =  (uint16)duty_cycle_percent_dec;
        signalParams.dutyValB =  (uint16)duty_cycle_period;
        signalParams.dutyValB =  (uint16)duty_cycle_percent_dec;

        /* Set the Polarity. */
        if(pChannelConfig->polarity == PWM_HIGH)
        {
            signalParams.invertSignalB = (boolean)0;
        }
        else
        {
            signalParams.invertSignalB = (boolean)1;
        }

        signalParams.sysClkInHz = pChannelConfig->instanceClkHz;

        signalParams.tbClkDiv = pChannelConfig->prescale;
        signalParams.tbHSClkDiv = pChannelConfig->hsPrescale;

        /* Configure PWM Signal.  */
        EPWM_configureSignal(baseAddr, &signalParams);
    }
 
    return ;
}

#if (PWM_SET_PERIOD_AND_DUTY_API==STD_ON) || \
    (PWM_SET_DUTY_CYCLE_API == STD_ON)
FUNC(void, PWM_CODE) Pwm_SetDutyCycle_Internal(
    Pwm_ChObjType *chObj, uint16 DutyCycle)
{

    uint32 period;
    uint32 baseAddr, outputCh, sysClk;
    boolean enableHR;
    Pwm_OutputStateType polarity;
    EPWM_ActionQualifierSWOutput swTrigAction;
    EPWM_SignalParams signalParams;
    uint32 absolute_dutyCycle;
    float64 duty_cycle_percent;
    uint32 duty_cycle_percent_int;
    float64 duty_cycle_percent_dec;
    float64 duty_cycle_period;
	
    baseAddr = chObj->baseAddr;
    outputCh = chObj->chCfg_PC.outputCh;
    polarity = chObj->chCfg.polarity;
    enableHR = chObj->chCfg.enableHR;
    sysClk   = chObj->chCfg.instanceClkHz;

    /* Reactivate channel if output was forced to idle */
    /*TI_INSPECTED 331 S: MISRAC_2012_7.2
     * "Reason - External Package errors like Compiler which are
     *  not part of MCAL cannot be fixed." */
    /*TI_INSPECTED 434 S: MISRAC_2012_10.3
             * "Reason - Tool Issue Not able to analyse boolean type" */

			 /* Check for IDLE state. */
    if ((boolean)TRUE == chObj->channelForcedIdle)
    {
        /* Disable SW Forced Action qualifiers */
        baseAddr = chObj->baseAddr;
        if (PWM_HIGH == chObj->chCfg.idleState)
        {
            swTrigAction = EPWM_AQ_SW_OUTPUT_HIGH;
        }
        else
        {
            swTrigAction = EPWM_AQ_SW_OUTPUT_LOW;
        }

	    /* Program AQCSFRC Active Register Reload From Shadow Options */
	    EPWM_setActionQualifierContSWForceShadowMode(baseAddr, EPWM_AQ_SW_IMMEDIATE_LOAD);
		
		/* Enable both outputs epwmxA and epwmxB which is same configuration
         * if channelConfig_PC->outputCh == PWM_OUTPUT_CH_BOTH_A_AND_B */
        if (((uint32)PWM_OUTPUT_CH_A == outputCh) ||
            ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == outputCh))
        {
            /* Continuous software forced output on A */
            EPWM_setActionQualifierContSWForceAction(baseAddr, EPWM_AQ_OUTPUT_A, EPWM_AQ_SW_DISABLED);
        }

        if (((uint32)PWM_OUTPUT_CH_B == outputCh) ||
            ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == outputCh))
        {
            /* Continuous software forced output on B */
            EPWM_setActionQualifierContSWForceAction(baseAddr, EPWM_AQ_OUTPUT_B, EPWM_AQ_SW_DISABLED);
        }

        /* Enable both outputs epwmxA and epwmxB which is same configuration
         * if channelConfig_PC->outputCh == PWM_OUTPUT_CH_BOTH_A_AND_B */
        if (((uint32)PWM_OUTPUT_CH_A == outputCh) ||
            ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == outputCh))
        {
            /* Continuous software forced output on A */
            EPWM_forceActionQualifierSWAction(baseAddr, EPWM_AQ_OUTPUT_A);
        }

        if (((uint32)PWM_OUTPUT_CH_B == outputCh) ||
            ((uint32)PWM_OUTPUT_CH_BOTH_A_AND_B == outputCh))
        {
            /* Continuous software forced output on B */
            EPWM_forceActionQualifierSWAction(baseAddr, EPWM_AQ_OUTPUT_B);
        }
        chObj->channelForcedIdle = (boolean)FALSE;
    }

    /* Updating Init time dutyCycle which is used to enable notifications,
     * if the duty cycle is not 0% or 100% */
    chObj->chCfg.dutyCycle = DutyCycle;
    period   = (uint32) EPWM_getTimeBasePeriod(baseAddr);
	
    if(chObj->chCfg.enableHR == TRUE)
    {
        signalParams.tbCtrMode = EPWM_COUNTER_MODE_UP;

        /* Calculation for high resolution */
        duty_cycle_percent = (float32) (((float32)(chObj->chCfg.dutyCycle)) /
                ((float32) 32768U));

        /*0x8000 is 32768, calculating percentage from hex*/
        duty_cycle_period = (float64)((float64) duty_cycle_percent * (float64) (period));
        duty_cycle_percent_int = (uint32) (period);
        duty_cycle_percent_dec = duty_cycle_percent_int - duty_cycle_period;

        /* Update DutyCycle */
        /* Check the Output outputCh */
        /* Set the parameters of PWM. */
        signalParams.freqInHz =  period;
        signalParams.dutyValA =  (uint16)duty_cycle_period;
        signalParams.dutyValA =  (uint16)duty_cycle_percent_int - duty_cycle_percent_dec;
        signalParams.dutyValB =  (uint16)duty_cycle_period;
        signalParams.dutyValB =  (uint16)duty_cycle_percent_int - duty_cycle_percent_dec;
        /* Set the Polarity. */
        if(chObj->chCfg.polarity == PWM_HIGH)
        {
            signalParams.invertSignalB = (boolean)0;
        }
        else
        {
            signalParams.invertSignalB = (boolean)1;
        }

        signalParams.sysClkInHz = sysClk;
        signalParams.tbClkDiv = chObj->chCfg.prescale;
        signalParams.tbHSClkDiv = chObj->chCfg.hsPrescale;

        /* Genearte the PWM signal. */
        EPWM_configureSignal(baseAddr, &signalParams);

        /* Configure HR PWM. */
        Pwm_ConfigHR_epwm( baseAddr,
                       signalParams.sysClkInHz,
                       signalParams.freqInHz,
                       duty_cycle_period,
                       chObj->chCfg.enableHR,
                       (uint32)outputCh);
    }
    else
    {
        signalParams.tbCtrMode = EPWM_COUNTER_MODE_UP_DOWN;

        /* Calculation for high resolution */
        duty_cycle_percent = (float32) (((float32)(chObj->chCfg.dutyCycle)) /
                ((float32) 32768U));

        /*0x8000 is 32768, calculating percentage from hex*/
        duty_cycle_period = (float64)((float64) duty_cycle_percent * (float64) (period));
        duty_cycle_percent_int = (uint32) (period);
        duty_cycle_percent_dec = duty_cycle_percent_int - duty_cycle_period;

        /* Update DutyCycle */
        /* Check the Output outputCh */
        /* Set the parameters of PWM. */
        signalParams.freqInHz =  period;
        signalParams.dutyValA =  (uint16)duty_cycle_period;
        signalParams.dutyValA =  (uint16)duty_cycle_percent_dec;
        signalParams.dutyValB =  (uint16)duty_cycle_period;
        signalParams.dutyValB =  (uint16)duty_cycle_percent_dec;
        /* Set the Polarity. */
        if(chObj->chCfg.polarity == PWM_HIGH)
        {
            signalParams.invertSignalB = (boolean)0;
        }
        else
        {
            signalParams.invertSignalB = (boolean)1;
        }

        signalParams.sysClkInHz = sysClk;
        signalParams.tbClkDiv = chObj->chCfg.prescale;
        signalParams.tbHSClkDiv = chObj->chCfg.hsPrescale;

        /* Genearte the PWM signal. */
        EPWM_configureSignal(baseAddr, &signalParams);

    }
}
#endif


#if (STD_ON == PWM_DEINIT_API)
FUNC(void, PWM_CODE) Pwm_IpDeInit_epwm(
    const Pwm_ChObjType *chObj)
{
    uint32 baseAddr = chObj->baseAddr;
	
    /* Set registers to default value */
    Pwm_HwSetDefReg_epwm(baseAddr);

    Pwm_SetIdleState(chObj);

    /* Disable all Notifications */
    EPWM_disableInterrupt(chObj->baseAddr);
	EPWM_clearEventTriggerInterruptFlag(chObj->baseAddr);

    /* Explicit PWM clock disable must not be performed by driver.
     * Refer TRM for more details */
    return;
}
#endif

#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
/*******************************************************************************
 *   Function Name : Pwm_ChannelNotificationISR
 ******************************************************************************/
/*! \Description: Interrupt Subroutine for Notification
 *
 *  \context    Function is called from interrupt level
 ******************************************************************************/
FUNC(void, PWM_CODE) Pwm_ChannelNotificationISR_epwm(
    Pwm_ChannelType ChannelNumber)
{
    /*TI_INSPECTED 93 S : MISRAC_2012_R.11.1
     * "Reason - Pointer typecast required here to match
     *           destination data type" */
    /*TI_INSPECTED 95 S : MISRAC_2012_R.11.4
     * "Reason - Pointer typecast required here since same element is
     * required to hold different types" */
    /*TI_INSPECTED 606 S : MISRAC_2012_R.11.1
     * "Reason - Pointer typecast required here to match
     *           destination data type" */
    Pwm_NotifyFuncType pwmNotification = (Pwm_NotifyFuncType)NULL;
    Pwm_ChObjType *chObj = &Pwm_ChObj[ChannelNumber];
    EPWM_clearEventTriggerInterruptFlag(chObj->baseAddr);
    if (NULL != (FUNC(void, PWM_CODE) *)chObj)
    {
        if ((chObj->chCfg.dutyCycle != PWM_DUTY_0_PERCENT) &&
                (chObj->chCfg.dutyCycle != PWM_DUTY_100_PERCENT))
        {
            /* Call Notification */
            if (NULL != (FUNC(void, PWM_CODE) *)chObj->chCfg.notificationHandler)
            {
                pwmNotification = chObj->chCfg.notificationHandler;
                pwmNotification();
            }
        }
    }
}

#endif /*(STD_ON == PWM_NOTIFICATION_SUPPORTED)*/

#if (STD_ON == PWM_REGISTER_READBACK_API)
FUNC(void, PWM_CODE) Pwm_IpRegisterReadback_epwm(uint32 baseAddr,
            Pwm_RegisterReadbackType *RegRbPtr)
{
    RegRbPtr->pwmTbCtl1 = HW_RD_REG16(baseAddr + PWM_EPWM_TBCTL);
	RegRbPtr->pwmTbCtl2 = HW_RD_REG16(baseAddr + PWM_EPWM_TBCTL2);
    RegRbPtr->pwmTbPhs  = HW_RD_REG16(baseAddr + PWM_EPWM_TBPHS);
    RegRbPtr->pwmTbCnt  = HW_RD_REG16(baseAddr + PWM_EPWM_TBCTR);
}
#endif /*PWM_REGISTER_READBACK_API*/

#if (STD_ON == PWM_DEV_ERROR_DETECT)
FUNC(void, PWM_CODE) Pwm_reportDetError(uint8 apiId, uint8 errorId)
{
    (void) Det_ReportError(PWM_MODULE_ID, PWM_INSTANCE_ID, apiId, errorId);
    return;
}
#endif  /* #if (STD_ON == PWM_DEV_ERROR_DETECT) */

#define PWM_STOP_SEC_CODE
#include "Pwm_MemMap.h"
