/* ======================================================================
*   Copyright (C) 2022 Texas Instruments Incorporated
*
*   All rights reserved. Property of Texas Instruments Incorporated.
*   Restricted rights to use, duplicate or disclose this code are
*   granted through contract.
*
*   The program may not be used without the written permission
*   of Texas Instruments Incorporated or against the terms and conditions
*   stipulated in the agreement under which this program has been
*   supplied.
* ==================================================================== */

/**
 * \file       mcspi.c
 *
 * \brief      This file contains the function definitions for the device
 *             abstraction layer for MCSPI.
 **/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
/**
 * \brief This is to disable HW_SYNC_BARRIER for AM263 due to performance
 *        requirement
 */
#define MEM_BARRIER_DISABLE

/*LDRA_NOANALYSIS*/
#include <stdint.h>
/*LDRA_ANALYSIS*/
//#include <hw_include/lld_mcspi.h>
#include <include/hw/hw_types.h>
#include "mcspi.h"
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define MCSPI_CLKD_MASK       (0xF)

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
void McSPIClkConfig(uint32 baseAddr,
                    uint32 spiInClk,
                    uint32 spiOutClk,
                    uint32 chNum,
                    uint32 clkMode)
{
    uint32 fRatio;
    uint32 spiClk;
    uint32 clkD;
    /*TI_INSPECTED 105 D: MISRAC_2012_R.2.2
     * "Reason - LDRA tool issue not able to understand HW_WR_FIELD32 macros
     *" */
    uint32 extClk;

    /* Calculate the value of fRatio. */
    fRatio = spiInClk / spiOutClk;

    /* If fRatio is not a power of 2, set granularity of 1 clock cycle */
    if ((uint32) 0U != (fRatio & (fRatio - 1U)))
    {
        /* Set the clock granularity to 1 clock cycle.*/
        HW_WR_FIELD32(
            baseAddr + MCSPI_CHCONF(chNum),
            MCSPI_CH0CONF_CLKG,
            MCSPI_CH0CONF_CLKG_ONECYCLE);

        /* Calculate the ratios clkD and extClk based on fClk */
        extClk = (fRatio - 1U) >> 4U;
        clkD   = (fRatio - 1U) & (uint32) MCSPI_CLKD_MASK;

        /* Set the extClk field of MCSPI_CHCTRL register.*/
        HW_WR_FIELD32(
            baseAddr + MCSPI_CHCTRL(chNum),
            MCSPI_CH0CTRL_EXTCLK,
            extClk);
    }
    else
    {
        clkD   = 0U;
        /* Clock granularity of power of 2.*/
        HW_WR_FIELD32(
            baseAddr + MCSPI_CHCONF(chNum),
            MCSPI_CH0CONF_CLKG,
            MCSPI_CH0CONF_CLKG_POWERTWO);

        while (1U != fRatio)
        {
            fRatio /= 2U;
            clkD++;
        }
    }

    /* Configure the clkD field of MCSPI_CHCONF register.*/
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_CLKD,
        clkD);

    /*Clearing the clkMode field of MCSPI_CHCONF register.*/
    spiClk = HW_RD_REG32(baseAddr + MCSPI_CHCONF(chNum));

    spiClk &= ~((uint32) MCSPI_CH0CONF_PHA_MASK | (uint32) MCSPI_CH3CONF_POL_MASK);

    /* Configure the clkMode of MCSPI_CHCONF register.*/
    spiClk |= (clkMode & (MCSPI_CH0CONF_PHA_MASK | MCSPI_CH3CONF_POL_MASK));

    HW_WR_REG32(baseAddr + MCSPI_CHCONF(chNum), spiClk);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It is setting a WordLength
 *  register field in McSPI register" */
void McSPIWordLengthSet(uint32 baseAddr,
                        uint32 wordLength,
                        uint32 chNum)
{
    /* Setting the wordlength field. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_WL,
        wordLength >> MCSPI_CH0CONF_WL_SHIFT);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It is enabling chipSelect
 *  register field in McSPI register" */
void McSPICSEnable(uint32 baseAddr)
{
    /* CLear PIN34 of MCSPI_MODULCTRL register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_PIN34,
        MCSPI_MODULCTRL_PIN34_4PINMODE);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It is disabling chipSelect
 *  register field in McSPI register" */
void McSPICSDisable(uint32 baseAddr)
{
    /* Set PIN34 of MCSPI_MODULCTRL register.*/
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_PIN34,
        MCSPI_MODULCTRL_PIN34_3PINMODE);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It is setting chipSelect
 *  polarity register field in McSPI register" */
void McSPICSPolarityConfig(uint32 baseAddr,
                           uint32 spiEnPol,
                           uint32 chNum)
{
    /* Set the EPOL field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_EPOL,
        spiEnPol >> MCSPI_CH0CONF_EPOL_SHIFT);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It is setting chipSelect
 *  time control register field in McSPI register" */
void McSPICSTimeControlSet(uint32 baseAddr,
                           uint32 csTimeControl,
                           uint32 chNum)
{
    /* Set the TCS field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_TCS0,
        csTimeControl);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It enables the start bit
 *  register field in McSPI register for McSPI channel configiration" */
 void McSPIStartBitEnable(uint32 baseAddr, uint32 chNum)
{
    /* Set the SBE bit of MCSPI_CHCONF register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_SBE,
        MCSPI_CH0CONF_SBE_ENABLED);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the start bit polarity
 *  register field in McSPI register for McSPI channel configiration" */
void McSPIStartBitPolarityConfig(uint32 baseAddr,
                                 uint32 startBitPol,
                                 uint32 chNum)
{
    /* Set the SBPOL bit of MCSPI_CHCONF register with the user sent value */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_SBPOL,
        startBitPol);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It disables the start bit
 *  register field in McSPI register for McSPI channel configiration" */
void McSPIStartBitDisable(uint32 baseAddr, uint32 chNum)
{
    /* Clear the SBE field of MCSPI_CHCONF register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_SBE,
        MCSPI_CH0CONF_SBE_DISABLED);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It enables the Master Mode
 *  register field in McSPI register" */
void McSPIMasterModeEnable(uint32 baseAddr)
{
    /* Clear the MS field of MCSPI_MODULCTRL register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_MS,
        MCSPI_MODULCTRL_MS_MASTER);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It enables the Slave Mode
 *  register field in McSPI register" */
void McSPISlaveModeEnable(uint32 baseAddr)
{
    /* Set the MS field of MCSPI_MODULCTRL register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_MS,
        MCSPI_MODULCTRL_MS_SLAVE);
}

uint32 McSPIMasterModeConfig(uint32 baseAddr,
                               uint32 channelMode,
                               uint32 trMode,
                               uint32 pinMode,
                               uint32 chNum)
{
    uint32 retVal;
    uint32 regVal;

    /* Set the MS field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_SINGLE,
        channelMode);

    /* Set the TRM field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_TRM,
        trMode >> MCSPI_CH0CONF_TRM_SHIFT);

    if (((MCSPI_TX_RX_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_3 == pinMode)) ||
        ((MCSPI_TX_ONLY_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_3 == pinMode)) ||
        ((MCSPI_TX_RX_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_7 == pinMode)) ||
        ((MCSPI_TX_ONLY_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_7 == pinMode)))
    {
        retVal = (uint32) FALSE;
    }
    else
    {
        /* Clear the IS, DPE0, DPE1 fields of MCSPI_CHCONF register. */
        regVal = HW_RD_REG32(baseAddr + MCSPI_CHCONF(chNum));

        regVal &= ~(MCSPI_CH0CONF_IS_MASK |
                    MCSPI_CH0CONF_DPE1_MASK |
                    MCSPI_CH0CONF_DPE0_MASK);

        /* Set the IS, DPE0, DPE1 fields with the user sent values. */
        regVal |= (pinMode & (MCSPI_CH0CONF_IS_MASK |
                              MCSPI_CH0CONF_DPE1_MASK |
                              MCSPI_CH0CONF_DPE0_MASK));

        HW_WR_REG32(baseAddr + MCSPI_CHCONF(chNum), regVal);

        retVal = (uint32) TRUE;
    }

    return retVal;
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the reset register
 *  field in McSPI register" */
void McSPIReset(uint32 baseAddr)
{
    /* Set the SOFTRESET field of MCSPI_SYSCONFIG register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_SYSCONFIG,
        MCSPI_SYSCONFIG_SOFTRESET,
        MCSPI_SYSCONFIG_SOFTRESET_ON);

    /* Stay in the loop until reset is done. */
    while ((MCSPI_SYSSTATUS_RESETDONE_COMPLETED &
           (HW_RD_REG32(baseAddr + MCSPI_SYSSTATUS))) != 1U)
    {
        /* Do nothing - Busy wait */
    }
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It enables the turbo mode
 *  register field in McSPI register" */
void McSPITurboModeEnable(uint32 baseAddr, uint32 chNum)
{
    /* Set the TURBO field on MCSPI_CHCONF register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_TURBO,
        MCSPI_CH0CONF_TURBO_TURBO);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It disables the turbo mode
 *  register field in McSPI register" */
void McSPITurboModeDisable(uint32 baseAddr, uint32 chNum)
{
    /* Clear the TURBO field of MCSPI_CHCONF register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_TURBO,
        MCSPI_CH0CONF_TURBO_OFF);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the FFEW register
 *  field in McSPI register" */
void McSPITxFIFOConfig(uint32 baseAddr,
                       uint32 txFifo,
                       uint32 chNum)
{
    /* Set the FFEW field with user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_FFEW,
        txFifo >> MCSPI_CH0CONF_FFEW_SHIFT);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the FFER register
 *  field in McSPI register" */
void McSPIRxFIFOConfig(uint32 baseAddr,
                       uint32 rxFifo,
                       uint32 chNum)
{
    /* Set the FFER field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_FFER,
        rxFifo >> MCSPI_CH0CONF_FFER_SHIFT);
}

void McSPIFIFOTrigLvlSet(uint32 baseAddr,
                         uint8  afl,
                         uint8  ael,
                         uint32 trMode)
{
    uint32 regVal;

    if (MCSPI_TX_RX_MODE == trMode)
    {
        /* Clear the AFL and AEL fields of MCSPI_XFERLEVEL register. */
        regVal = HW_RD_REG32(baseAddr + MCSPI_XFERLEVEL);

        regVal &= ~((uint32) MCSPI_XFERLEVEL_AFL_MASK | (uint32) MCSPI_XFERLEVEL_AEL_MASK);

        regVal |= (((((uint32) afl - 1U) << MCSPI_XFERLEVEL_AFL_SHIFT) &
                    MCSPI_XFERLEVEL_AFL_MASK) |
                   (((uint32) ael - 1U) & MCSPI_XFERLEVEL_AEL_MASK));

        /* Set the AFL and AEL fields with the user sent value. */
        HW_WR_REG32(baseAddr + MCSPI_XFERLEVEL, regVal);
    }
    else if (MCSPI_TX_ONLY_MODE == trMode)
    {
        /* Set the AEL field with the user sent value. */
        HW_WR_FIELD32(
            baseAddr + MCSPI_XFERLEVEL,
            MCSPI_XFERLEVEL_AEL,
            (uint32) ael - 1U);
    }
    else if (MCSPI_RX_ONLY_MODE == trMode)
    {
        /* Set the AFL field with the user sent value. */
        HW_WR_FIELD32(
            baseAddr + MCSPI_XFERLEVEL,
            MCSPI_XFERLEVEL_AFL,
            (uint32) afl - 1U);
    }
    else
    {
        /*
         * Do nothing
         */
    }
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the word count
 *  register field in McSPI register" */
void McSPIWordCountSet(uint32 baseAddr, uint16 wCnt)
{
    /* Set the WCNT field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_XFERLEVEL,
        MCSPI_XFERLEVEL_WCNT,
        wCnt);
}

void McSPIIntEnable(uint32 baseAddr, uint32 intFlags)
{
    /* Enable the Interrupts. */
    uint32 intrMask;

    intrMask = HW_RD_REG32(baseAddr + MCSPI_IRQENABLE);

    intrMask |= intFlags;

    HW_WR_REG32(baseAddr + MCSPI_IRQENABLE, intrMask);
}

void McSPIIntDisable(uint32 baseAddr, uint32 intFlags)
{
    /* Disable the interrupts. */
    uint32 intrMask;

    intrMask = HW_RD_REG32(baseAddr + MCSPI_IRQENABLE);

    intrMask &= ~intFlags;

    HW_WR_REG32(baseAddr + MCSPI_IRQENABLE, intrMask);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the Intial delay
 *  register field in McSPI register" */
void McSPIInitDelayConfig(uint32 baseAddr, uint32 initDelay)
{
    /* Set the INITDLY field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_INITDLY,
        initDelay >> MCSPI_MODULCTRL_INITDLY_SHIFT);
}

uint32 McSPIReceiveData(uint32 baseAddr, uint32 chNum)
{
    /* Return the data present in the MCSPI_RX register. */
    return (HW_RD_REG32(baseAddr + MCSPI_CHRX(chNum)));
}

uint32 McSPIIntStatusGet(uint32 baseAddr)
{
    /* Return the interrupt status present in the MCSPI_IRQSTATUS register. */
    return (HW_RD_REG32(baseAddr + MCSPI_IRQSTATUS));
}

void McSPIIntStatusClear(uint32 baseAddr, uint32 intFlags)
{
    /* Clear the SSB bit in the MCSPI_SYST register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_SYST,
        MCSPI_SYST_SSB,
        MCSPI_SYST_SSB_OFF);

    /* Clear the interrupt status. */
    HW_WR_REG32(baseAddr + MCSPI_IRQSTATUS, intFlags);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect. It sets the MOA register field
 *  in McSPI register" */
void McSPIMultipleWordAccessConfig(uint32 baseAddr, uint32 moa)
{
    /* Set the MOA fields with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_MOA,
        moa >> MCSPI_MODULCTRL_MOA_SHIFT);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect.It sets the FDAA register field
 *  in McSPI register" */
void McSPIFIFODatManagementConfig(uint32 baseAddr, uint32 fdaa)
{
    /* Set the FDAA field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_FDAA,
        fdaa >> MCSPI_MODULCTRL_FDAA_SHIFT);
}

/*TI_INSPECTED 1 D 1: MISRA-C:2012_R.2.7
 * "Reason - Ensured procedure parameter is used" */
void MCSPISysConfigSetup(uint32 baseAddr,
                         uint32 clockActivity,
                         uint32 sidleMode,
                         uint32 wakeUp,
                         uint32 autoIdle)
{
    HW_WR_REG32(
        baseAddr + MCSPI_SYSCONFIG,
        (clockActivity | sidleMode | wakeUp | autoIdle) &
        (MCSPI_SYSCONFIG_CLOCKACTIVITY_MASK | MCSPI_SYSCONFIG_SIDLEMODE_MASK |
         MCSPI_SYSCONFIG_ENAWAKEUP_MASK | MCSPI_SYSCONFIG_AUTOIDLE_MASK));
}

uint32 MCSPIPinDirSet(uint32 baseAddr,
                        uint32 trMode,
                        uint32 pinMode,
                        uint32 chNum)
{
    uint32 regVal, retVal;

    /* Set the TRM field with the user sent value. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_TRM,
        trMode >> MCSPI_CH0CONF_TRM_SHIFT);

    if (((MCSPI_TX_RX_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_3 == pinMode)) ||
        ((MCSPI_TX_ONLY_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_3 == pinMode)) ||
        ((MCSPI_TX_RX_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_7 == pinMode)) ||
        ((MCSPI_TX_ONLY_MODE == trMode) &&
         (MCSPI_DATA_LINE_COMM_MODE_7 == pinMode)))
    {
        /* Not Supported */
        retVal = 1U;
    }
    else
    {
        /* Clear the IS, DPE0, DPE1 fields of MCSPI_CHCONF register. */
        regVal = HW_RD_REG32(baseAddr + MCSPI_CHCONF(chNum));

        regVal &= ~(MCSPI_CH0CONF_IS_MASK |
                    MCSPI_CH0CONF_DPE0_MASK |
                    MCSPI_CH0CONF_DPE1_MASK);

        HW_WR_REG32(baseAddr + MCSPI_CHCONF(chNum), regVal);

        /* Set the IS, DPE0, DPE1 fields with the user sent values. */
        regVal = HW_RD_REG32(baseAddr + MCSPI_CHCONF(chNum));

        regVal |= (pinMode & (MCSPI_CH0CONF_IS_MASK |
                              MCSPI_CH0CONF_DPE0_MASK |
                              MCSPI_CH0CONF_DPE1_MASK));

        HW_WR_REG32(baseAddr + MCSPI_CHCONF(chNum), regVal);

        retVal = 0U;
    }
    return retVal;
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect.It enables single channel Mode
 *  register field in McSPI register" */
 void MCSPISingleChModeEnable(uint32 baseAddr)
{
    /* Set the SINGLE field of MCSPI_MODULCTRL register to enable single channel
       Mode */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_SINGLE,
        MCSPI_MODULCTRL_SINGLE_SINGLE);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect.It enables multi channel Mode
 *  register field in McSPI register" */
void MCSPIMultiChModeEnable(uint32 baseAddr)
{
    /* Clear the SINGLE field of MCSPI_MODULCTRL register to enable multi
       channel Mode */
    HW_WR_FIELD32(
        baseAddr + MCSPI_MODULCTRL,
        MCSPI_MODULCTRL_SINGLE,
        MCSPI_MODULCTRL_SINGLE_MULTI);
}

/*TI_INSPECTED 65 D : MISRAC_2012_R.2.2
 * "Reason - Void function does have side effect.It sets the slave chip select
 *  register field in McSPI register" */
void McSPISetSlaveChipSel(uint32 baseAddr,
                          uint32 chNum,
                          uint32 slaveChipSel)
{
    /* Set the SPIENSLV field of MCSPI_CHCONF register. */
    HW_WR_FIELD32(
        baseAddr + MCSPI_CHCONF(chNum),
        MCSPI_CH0CONF_SPIENSLV,
        slaveChipSel);
}

/***************************** END OF FILE ************************************/
