/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Cdd_Uart.c
 *
 *  \brief    This file contains Complex Device Driver for UART
 *
 */
/*  -------------------------------------------------------------------------------------------------------------------
*  FILE DESCRIPTION
*  -------------------------------------------------------------------------------------------------------------------
*         File:  Cdd_Uart.c
*    Component:  AM273X Complex Device Driver
*       Module:  UART
*    Generator:  None
*
*  Description: This component provides services for initStatus and control of the UART between different core in the this
*                     Multicore SoC.
*********************************************************************************************************************/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Cdd_Uart_Priv.h"
#if (CDD_UART_DMA_ENABLE == STD_ON)
#include "Cdd_Dma.h"
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((CDD_UART_AR_RELEASE_MAJOR_VERSION!= (4U)) || \
    (CDD_UART_AR_RELEASE_MINOR_VERSION != (3U)) || \
    (CDD_UART_AR_RELEASE_REVISION_VERSION != (1U)))
    #error "CDD UART: AUTOSAR Version Numbers of CDD UART are different!!"
#endif

/* AUTOSAR version information check has to match definition in header file */
#if ((CDD_UART_SW_MAJOR_VERSION != (8U)) || \
    (CDD_UART_SW_MINOR_VERSION != (6U)) )
    #error "CDD UART: Software Version Numbers are inconsistent!!"
#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define CDD_UART_START_SEC_VAR_INIT_UNSPECIFIED
#include "Cdd_Uart_MemMap.h"

/** \brief CDD UART driver status */
#if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
static VAR(CddUart_StatusType, CDD_VAR_ZERO_INIT) CddUart_DriverStatus = CDD_UART_UNINIT;
#endif

#define CDD_UART_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Cdd_Uart_MemMap.h"

#define CDD_UART_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_Uart_MemMap.h"

CddUart_NotifCallbacks CddUart_NotificationCallbacks[CDD_UART_NUM_CHANNELS];

/** \brief CDD UART Init objects */
CddUart_InitObject CddUart_InitObjects[CDD_UART_NUM_CHANNELS];

#if (CDD_UART_DMA_ENABLE == STD_ON)
/** \brief CDD UART EDMA Channel Configuration */
CddUart_EdmaChConfig CddUart_EdmaChannelConfigurations[CDD_UART_NUM_CHANNELS];
#endif

/** \brief CDD UART channel handles */
CddUart_Object CddUart_ChannelObjects[CDD_UART_NUM_CHANNELS];

#define CDD_UART_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Cdd_Uart_MemMap.h"

/*
 * Requirements : SITARAMCU_MCAL-___
 * Design : SITARAMCU_MCAL-___
 */
/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define CDD_UART_START_SEC_CODE
#include "Cdd_Uart_MemMap.h"

#if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
static inline void CddUart_ReportDetError(uint8 apiId, uint8 errorId)
{
    (void) Det_ReportError(CDD_UART_MODULE_ID, CDD_UART_INSTANCE_ID, apiId, errorId);
}
#endif /* (STD_ON == CDD_UART_DEV_ERROR_DETECT) */

/* Design : CDD_UART_DesignId_0__
 * Requirements :  SITARAMCU_MCAL-___,SITARAMCU_MCAL-___, SITARAMCU_MCAL-___
 * Design : SITARAMCU_MCAL-___,SITARAMCU_MCAL-___, SITARAMCU_MCAL-___
 */
#if (STD_ON == CDD_UART_VERSION_INFO_API)
FUNC(void, CDD_UART_CODE) Cdd_Uart_GetVersionInfo(P2VAR(Std_VersionInfoType, AUTOMATIC, CDD_UART_APPL_DATA) VersionInfoPtr)
{
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if (NULL_PTR == VersionInfoPtr)
    {
        CddUart_ReportDetError(CDD_UART_GETVERSIONINFO_SERVICE_ID, CDD_UART_E_PARAM_POINTER);
    }
    else
    #endif /* (STD_ON == CDD_UART_DEV_ERROR_DETECT) */
    {
        VersionInfoPtr->vendorID         = CDD_UART_VENDOR_ID;
        VersionInfoPtr->moduleID         = CDD_UART_MODULE_ID;
        VersionInfoPtr->sw_major_version = (uint8) CDD_UART_SW_MAJOR_VERSION;
        VersionInfoPtr->sw_minor_version = (uint8) CDD_UART_SW_MINOR_VERSION;
        VersionInfoPtr->sw_patch_version = (uint8) CDD_UART_SW_PATCH_VERSION;
    }
}
#endif /* (STD_ON == CDD_UART_VERSION_INFO_API) */

/*  Design:
 *  Requirement(s): SITARAMCU_MCAL-___    */
FUNC(void, CDD_UART_CODE) Cdd_Uart_Init(P2CONST(CddUart_CfgType, AUTOMATIC, CDD_UART_CFG) ConfigurationPtr)
{
    #if (STD_ON == CDD_UART_PRE_COMPILE_VARIANT)
    if (ConfigurationPtr == NULL_PTR)
    {
        ConfigurationPtr = &CDD_UART_INIT_CONFIG_PC;
    }
    #endif  /* (STD_ON == CDD_UART_PRE_COMPILE_VARIANT) */
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if (CDD_UART_UNINIT != CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_INIT_SERVICE_ID, CDD_UART_E_ALREADY_INITIALIZED);
    }
    else if (ConfigurationPtr == NULL_PTR)
    {
        CddUart_ReportDetError(CDD_UART_INIT_SERVICE_ID, CDD_UART_E_PARAM_POINTER);
    }
    else
    #endif
    {
        Std_ReturnType initStatus;
        for (uint8 channel=0; channel<ConfigurationPtr->netChannels; ++channel) 
        {
            CddUart_ChannelObjects[channel].hUartInit = &CddUart_InitObjects[channel];
            Cdd_Uart_Object_Configuration_Mapping(&CddUart_ChannelObjects[channel], &ConfigurationPtr->channelCfg[channel]);

            if (CddUart_ChannelObjects[channel].hUartInit->transferMode == CDD_UART_MODE_DMA) 
            {
                #if (CDD_UART_DMA_ENABLE == STD_ON)
                /* Crossbar Pins selection */
                CddUart_EdmaChannelConfigurations[channel].edmaRxChId = ConfigurationPtr->channelCfg[channel].edmaXbarRxChannelID;
                CddUart_EdmaChannelConfigurations[channel].edmaTxChId = ConfigurationPtr->channelCfg[channel].edmaXbarTxChannelID;

                CddUart_ChannelObjects[channel].hUartInit->uartDmaHandle = (CddUart_DmaHandle) Cdd_Dma_GetHandle();
                CddUart_ChannelObjects[channel].hUartInit->dmaChCfg = (CddUart_DmaChConfig) &CddUart_EdmaChannelConfigurations[channel];

                initStatus = Uart_Cdd_initDma(&CddUart_ChannelObjects[channel]);
                #endif
            }
            else 
            { 
                initStatus = Uart_Cdd_init(&CddUart_ChannelObjects[channel]); 
            }
            if (initStatus != MCAL_SystemP_SUCCESS) 
            {
                #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
                CddUart_ReportDetError(CDD_UART_INIT_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
                #endif
            }
            else
            {
                CddUart_NotificationCallbacks[channel].uartNotificationReadHandler = ConfigurationPtr->channelCfg[channel].uartNotificationHandlers.uartNotificationReadHandler;
                CddUart_NotificationCallbacks[channel].uartNotificationWriteHandler = ConfigurationPtr->channelCfg[channel].uartNotificationHandlers.uartNotificationWriteHandler;
                CddUart_NotificationCallbacks[channel].uartNotificationErrorHandler = ConfigurationPtr->channelCfg[channel].uartNotificationHandlers.uartNotificationErrorHandler;
                #if (CDD_UART_DEV_ERROR_DETECT == STD_ON)
                /* Setting global UART status to INIT once initialization completes */
                CddUart_DriverStatus = CDD_UART_INIT;
                #endif
            }
        }
    }
}

/*  Design:
 *  Requirement(s): SITARAMCU_MCAL-___    */
#if (STD_ON == CDD_UART_DEINIT_API)
FUNC(void, CDD_UART_CODE) Cdd_Uart_DeInit()
{
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if ((CDD_UART_UNINIT == CddUart_DriverStatus))
    {
        CddUart_ReportDetError(CDD_UART_DEINIT_SERVICE_ID, CDD_UART_E_UNINIT);
    }
    else
    {
    #endif
        Std_ReturnType deInitStatus;
        for (uint8 channel=0; channel<CDD_UART_NUM_CHANNELS; ++channel) 
        {
            if (CddUart_ChannelObjects[channel].hUartInit->transferMode == CDD_UART_MODE_DMA) 
            {
                #if (STD_ON == CDD_UART_DMA_ENABLE)
                deInitStatus = Uart_Cdd_deInitDma(&CddUart_ChannelObjects[channel]);
                #endif
            }
            else
            {
                deInitStatus = Uart_Cdd_deInit(&CddUart_ChannelObjects[channel]);
            }
            if (deInitStatus != MCAL_SystemP_SUCCESS) 
            {
                #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
                CddUart_ReportDetError(CDD_UART_DEINIT_SERVICE_ID, CDD_UART_E_INVALID_EVENT);
                CddUart_DriverStatus = CDD_UART_INIT;
                #endif
                return;
            }
        }

    #if (CDD_UART_DEV_ERROR_DETECT == STD_ON)
        /* Setting global UART status to UNINITialized on success */
        CddUart_DriverStatus  = CDD_UART_UNINIT;
    }
    #endif
}
#endif /* (STD_ON == CDD_UART_DEINIT_API) */

/*  Design:
 *  Requirement(s): SITARAMCU_MCAL-___    */
#if (STD_ON == CDD_UART_WRITE_API)
FUNC(Std_ReturnType, CDD_UART_CODE) Cdd_Uart_Write(uint8 ChannelID, P2VAR(CddUartDataBufferType, AUTOMATIC, CDD_UART_APPL_DATA) SrcBufferPtr, uint32 Count, uint32 Timeout)
{
    Std_ReturnType writeStatus = E_NOT_OK;
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if (CDD_UART_UNINIT == CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_WRITEDATA_SERVICE_ID, CDD_UART_E_UNINIT);
    }
    else if (CDD_UART_BUSY == CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_WRITE_LOCKED);
    }
    else if (NULL_PTR == SrcBufferPtr)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_PARAM_POINTER);
    }
    else if (0 == Count)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
    }
    else if (ChannelID >= CDD_UART_NUM_CHANNELS)
    {
        CddUart_ReportDetError(CDD_UART_WRITEDATA_SERVICE_ID, CDD_UART_E_INVALID_CHANNEL);
    }
    else
    #endif
    {
        CddUart_Transaction Transaction;
        Transaction.buf = SrcBufferPtr;
        Transaction.count = Count;
        Transaction.timeout = Timeout;
        if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode==CDD_UART_MODE_POLLING)
        {
            if (Uart_Cdd_writeEx(&CddUart_ChannelObjects[ChannelID], &Transaction) == MCAL_SystemP_SUCCESS) 
            {
                writeStatus = E_OK;
            }
        }
        else if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode==CDD_UART_MODE_INTERRUPT)
        {
            if (Uart_Cdd_writeIntrEx(&CddUart_ChannelObjects[ChannelID], &Transaction) == MCAL_SystemP_SUCCESS)
            {
                writeStatus = E_OK;
            }
        }
        else if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode==CDD_UART_MODE_DMA)
        {
            #if (CDD_UART_DMA_ENABLE == STD_ON)
            if (Uart_Cdd_writeDmaEx(&CddUart_ChannelObjects[ChannelID], &Transaction) == MCAL_SystemP_SUCCESS)
            #endif
            {
                writeStatus = E_OK;
            }
        }
        else
        {
            #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
            CddUart_ReportDetError(CDD_UART_WRITEDATA_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
            #endif
        }
    }

    #if (CDD_UART_DEV_ERROR_DETECT == STD_ON)
    CddUart_DriverStatus = CDD_UART_INIT;
    #endif

    return writeStatus;
}
#endif /* (STD_ON == CDD_UART_WRITE_API) */

/*  Design:
 *  Requirement(s): SITARAMCU_MCAL-___    */
#if (STD_ON == CDD_UART_READ_API)
FUNC(Std_ReturnType, CDD_UART_CODE) Cdd_Uart_Read(uint8 ChannelID, P2VAR(CddUartDataBufferType, AUTOMATIC, CDD_UART_APPL_DATA) DestBufferPtr, uint32 Count, uint32 Timeout)
{
    Std_ReturnType readStatus = E_NOT_OK;
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if (CDD_UART_UNINIT == CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_UNINIT);
    }
    else if (CDD_UART_BUSY == CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_READ_LOCKED);
    }
    else if (NULL_PTR == DestBufferPtr)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_PARAM_POINTER);
    }
    else if (0 == Count)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
    }
    else if (ChannelID >= CDD_UART_NUM_CHANNELS)
    {
        CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_INVALID_CHANNEL);
    }
    else
    #endif
    {
        CddUart_Transaction Transaction;
        Transaction.buf = DestBufferPtr;
        Transaction.count = Count;
        Transaction.timeout = Timeout;
        if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode==CDD_UART_MODE_POLLING) 
        {
            if (Uart_Cdd_readEx(&CddUart_ChannelObjects[ChannelID], &Transaction) == MCAL_SystemP_SUCCESS)
            {
                readStatus = E_OK;
            }
        }
        else if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode==CDD_UART_MODE_INTERRUPT)
        {
            if (Uart_Cdd_readIntrEx(&CddUart_ChannelObjects[ChannelID], &Transaction) == MCAL_SystemP_SUCCESS)
            {
                readStatus = E_OK;
            }
        }
        else if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode==CDD_UART_MODE_DMA) 
        {
            #if (CDD_UART_DMA_ENABLE == STD_ON)
            if (Uart_Cdd_readDmaEx(&CddUart_ChannelObjects[ChannelID], &Transaction) == MCAL_SystemP_SUCCESS)
            #endif
            {
                readStatus = E_OK;
            }
        }
        else
        {
            #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
            CddUart_ReportDetError(CDD_UART_READDATA_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
            #endif
        }
    }

    #if (CDD_UART_DEV_ERROR_DETECT == STD_ON)
    CddUart_DriverStatus = CDD_UART_INIT;
    #endif
    return readStatus;
}
#endif /* (STD_ON == CDD_UART_READ_API) */

/*  Design:
 *  Requirement(s): SITARAMCU_MCAL-___    */
#if (STD_ON == CDD_UART_CANCEL_API)
FUNC(boolean, CDD_UART_CODE) Cdd_Uart_Cancel(uint8 ChannelID, CddUartDataDirectionType TransferType)
{
    boolean cancelStatus = FALSE;
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if (CDD_UART_UNINIT == CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_CANCEL_SERVICE_ID, CDD_UART_E_UNINIT);
    }
    else if (ChannelID >= CDD_UART_NUM_CHANNELS)
    {
        CddUart_ReportDetError(CDD_UART_CANCEL_SERVICE_ID, CDD_UART_E_INVALID_CHANNEL);
    }
    else if (TransferType==UART_READ_TRANSFER_TYPE && CddUart_ChannelObjects[ChannelID].readTrans == NULL_PTR)
    {
        CddUart_ReportDetError(CDD_UART_CANCEL_SERVICE_ID, CDD_UART_E_PARAM_POINTER);
    }
    else if (TransferType==UART_WRITE_TRANSFER_TYPE && CddUart_ChannelObjects[ChannelID].writeTrans == NULL_PTR)
    {
        CddUart_ReportDetError(CDD_UART_CANCEL_SERVICE_ID, CDD_UART_E_PARAM_POINTER);
    }
    else if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode!=CDD_UART_MODE_DMA && CddUart_ChannelObjects[ChannelID].hUartInit->transferMode!=CDD_UART_MODE_INTERRUPT)
    {
        CddUart_ReportDetError(CDD_UART_CANCEL_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
    }
    else
    #endif
    {
        if (TransferType==UART_WRITE_TRANSFER_TYPE) 
        {
            if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode == CDD_UART_MODE_DMA)
            {
                #if (CDD_UART_DMA_ENABLE == STD_ON)
                if (Uart_Cdd_writeCancelDma(&CddUart_ChannelObjects[ChannelID], CddUart_ChannelObjects[ChannelID].writeTrans) == MCAL_SystemP_SUCCESS)
                {
                    cancelStatus = TRUE;
                }
                else
                {
                    cancelStatus = FALSE;
                }
                #endif
            }
            else
            {
                if (Uart_Cdd_writeCancel(&CddUart_ChannelObjects[ChannelID], CddUart_ChannelObjects[ChannelID].writeTrans) == MCAL_SystemP_SUCCESS)
                {
                    cancelStatus = TRUE;
                }
                else
                {
                    cancelStatus = FALSE;
                }
            }
        }
        else
        {
            if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode == CDD_UART_MODE_DMA)
            {
                #if (CDD_UART_DMA_ENABLE == STD_ON)
                if (Uart_Cdd_readCancelDma(&CddUart_ChannelObjects[ChannelID], CddUart_ChannelObjects[ChannelID].readTrans) == MCAL_SystemP_SUCCESS)
                {
                    cancelStatus = TRUE;
                }
                else
                {
                    cancelStatus = FALSE;
                }
                #endif
            }
            else
            {
                if (Uart_Cdd_readCancel(&CddUart_ChannelObjects[ChannelID], CddUart_ChannelObjects[ChannelID].readTrans) == MCAL_SystemP_SUCCESS)
                {
                    cancelStatus = TRUE;
                }
                else
                {
                    cancelStatus = FALSE;
                }
            }
        }
    }
    return cancelStatus;
}
#endif /* (STD_ON == CDD_UART_CANCEL_API) */

/*  Design:
 *  Requirement(s): SITARAMCU_MCAL-___    */
#if (STD_ON == CDD_UART_GETREMAININGWORDS_API)
FUNC(uint32, CDD_UART_CODE) Cdd_Uart_GetRemainingWords(uint8 ChannelID, CddUartDataDirectionType TransferType)
{
    uint32 remainingWords = 0;
    #if (STD_ON == CDD_UART_DEV_ERROR_DETECT)
    if (CDD_UART_UNINIT == CddUart_DriverStatus)
    {
        CddUart_ReportDetError(CDD_UART_GETREMAININGWORDS_SERVICE_ID, CDD_UART_E_UNINIT);
    }
    else if (ChannelID >= CDD_UART_NUM_CHANNELS)
    {
        CddUart_ReportDetError(CDD_UART_GETREMAININGWORDS_SERVICE_ID, CDD_UART_E_INVALID_CHANNEL);
    }
    else if (TransferType != UART_WRITE_TRANSFER_TYPE && TransferType != UART_READ_TRANSFER_TYPE)
    {
        CddUart_ReportDetError(CDD_UART_GETREMAININGWORDS_SERVICE_ID, CDD_UART_E_PARAM_VALUE);
    }
    else
    #endif
    {
        #if (CDD_UART_DMA_ENABLE == STD_ON)
        if (CddUart_ChannelObjects[ChannelID].hUartInit->transferMode == CDD_UART_MODE_DMA)
        {
            CDD_EDMACCEDMACCPaRAMEntry currentPaRAM;
            if (TransferType==UART_WRITE_TRANSFER_TYPE)
            {
                CDD_EDMA_lld_getPaRAM(Cdd_Dma_GetHandle()->baseAddr, ((CddUart_EdmaChConfig *) CddUart_InitObjects[ChannelID].dmaChCfg)->edmaTxParam, &currentPaRAM);
            }
            else
            {
                CDD_EDMA_lld_getPaRAM(Cdd_Dma_GetHandle()->baseAddr, ((CddUart_EdmaChConfig *) CddUart_InitObjects[ChannelID].dmaChCfg)->edmaRxParam, &currentPaRAM);
            }
            remainingWords = currentPaRAM.aCnt * currentPaRAM.bCnt * currentPaRAM.cCnt;
        }
        else
        #endif
        {
            if (TransferType==UART_WRITE_TRANSFER_TYPE)
            { 
                remainingWords = CddUart_ChannelObjects[ChannelID].writeSizeRemaining; 
            }
            else 
            {
                remainingWords = CddUart_ChannelObjects[ChannelID].readSizeRemaining;
            }
        }
    }
    return remainingWords;
}
#endif /* (STD_ON == CDD_UART_GETREMAININGWORDS_API) */

/* Invoke Callback functions */
void Uart_Cdd_readCompleteCallback(CddUart_Handle hUart, CddUart_Transaction *transaction)
{
    uint8 channel=0;
    for (; channel<CDD_UART_NUM_CHANNELS; ++channel)
    {
        if (hUart->baseAddr == CddUart_ChannelObjects[channel].baseAddr)
        {
            break;
        }
    }
    (*CddUart_NotificationCallbacks[channel].uartNotificationReadHandler)();
}

void Uart_Cdd_writeCompleteCallback(CddUart_Handle hUart, CddUart_Transaction *transaction)
{
    uint8 channel=0;
    for (; channel<CDD_UART_NUM_CHANNELS; ++channel)
    {
        if (hUart->baseAddr == CddUart_ChannelObjects[channel].baseAddr)
        {
            break;
        }
    }
    (*CddUart_NotificationCallbacks[channel].uartNotificationWriteHandler)();
}

void Uart_Cdd_errorCallback(CddUart_Handle hUart, CddUart_Transaction *transaction)
{
    uint8 channel=0;
    for (; channel<CDD_UART_NUM_CHANNELS; ++channel)
    {
        if (hUart->baseAddr == CddUart_ChannelObjects[channel].baseAddr)
        {
            break;
        }
    }
    (*CddUart_NotificationCallbacks[channel].uartNotificationErrorHandler)();
}

uint32 Clock_getTicks(void)
{
    TickType startCount = 0U;
    (void)GetCounterValue(CDD_UART_OS_COUNTER_ID, &startCount);
    return ((uint32)startCount);
}

void Clock_uSleep(uint32 usec) __attribute__((optnone))
{
    volatile int i = 0;
    for(int i =0; i<usec*400;i++){}; /* Here 400 is the number of uSeconds per ticks*/
}

uint32 Clock_ticksToUsec(uint32 ticks)
{
    return (uint32)(ticks * 1000);
}

#define CDD_UART_STOP_SEC_CODE
#include "Cdd_Uart_MemMap.h"
