/* ======================================================================
 *   Copyright (C) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Cdd_Uart_Priv.c
 *
 *  \brief    This file contains some internal functions for UART Complex Device Driver
 *
 */
/* -------------------------------------------------------------------------------------------------------------------
*  FILE DESCRIPTION
*  -------------------------------------------------------------------------------------------------------------------
*         File:  Cdd_Uart_Priv.c
*    Component:  UART Complex Device Driver for AM273X
*       Module:  UART
*    Generator:  None
*
*  Description: This component provides services for initialization and control of the UART between different core in the this
*                     Multicore SoC.
*********************************************************************************************************************/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Cdd_Uart_Priv.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* AUTOSAR version information check has to match definition in header file */
#if ((CDD_UART_AR_RELEASE_MAJOR_VERSION!= (4U)) || \
    (CDD_UART_AR_RELEASE_MINOR_VERSION != (3U)) || \
    (CDD_UART_AR_RELEASE_REVISION_VERSION != (1U)))
    #error "CDD UART: AUTOSAR Version Numbers of CDD UART are different!!"
#endif

/* AUTOSAR version information check has to match definition in header file */
#if ((CDD_UART_SW_MAJOR_VERSION != (8U)) || \
    (CDD_UART_SW_MINOR_VERSION != (6U)) )
    #error "CDD UART: Software Version Numbers are inconsistent!!"
#endif

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define CDD_UART_START_SEC_VAR_INIT_32
#include "Cdd_Uart_MemMap.h"

/** \brief CDD UART Channel base addresses on AM263X */
uint32 baseAddresses[6] = { 0x52300000U, 0x52301000U, 0x52302000U, 0x52303000U, 0x52304000U, 0x52305000U };

#define CDD_UART_STOP_SEC_VAR_INIT_32
#include "Cdd_Uart_MemMap.h"

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define CDD_UART_START_SEC_CODE
#include "Cdd_Uart_MemMap.h"

FUNC(void, CDD_UART_CODE) Cdd_Uart_Object_Configuration_Mapping(CddUart_Handle CddUartHandle, P2CONST(CddUart_ConfigType, AUTOMATIC, CDD_UART_CFG) ChannelCfgPtr) {
    CddUartHandle->baseAddr = baseAddresses[ChannelCfgPtr->uartHwUnitID];
    CddUartHandle->state = MCAL_STATE_RESET;
    CddUartHandle->hUartInit->inputClkFreq = ChannelCfgPtr->uartClockFrequency;
    CddUartHandle->hUartInit->baudRate = ChannelCfgPtr->uartBaudRate;
    CddUartHandle->hUartInit->dataLength = ChannelCfgPtr->uartWordLength;
    CddUartHandle->hUartInit->stopBits = ChannelCfgPtr->uartStopBits;
    CddUartHandle->hUartInit->readReturnMode = ChannelCfgPtr->uartReadReturnMode;
    CddUartHandle->hUartInit->parityType = ChannelCfgPtr->uartParity;
    CddUartHandle->hUartInit->operMode = ChannelCfgPtr->uartOperatingMode;
    CddUartHandle->hUartInit->rxTrigLvl = ChannelCfgPtr->uartRXTriggerLevel;
    CddUartHandle->hUartInit->txTrigLvl = ChannelCfgPtr->uartTXTriggerLevel;
    CddUartHandle->hUartInit->hwFlowControlThr = ChannelCfgPtr->uartHwFlowControlThreshold;
    CddUartHandle->hUartInit->transferMode = ChannelCfgPtr->uartIOMode;
}

#define CDD_UART_STOP_SEC_CODE
#include "Cdd_Uart_MemMap.h"
