/*
 *  Copyright (C) 2022 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  \file uart_dma_edma.c
 *
 *  \brief File containing EDMA Driver APIs implementation for UART.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <uart_dma.h>
#if (CDD_UART_DMA_ENABLE == STD_ON)
#include <Cdd_Dma.h>
#include <Cdd_Dma_edma_lld.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/** \brief Transmit EDMA channel event queue number                           */
#define EDMA_UART_TX_EVT_QUEUE_NO                  (0U)
/** \brief Receive EDMA channel event queue number                            */
#define EDMA_UART_RX_EVT_QUEUE_NO                  (1U)

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */
static sint32 UART_edmaChInit(CddUart_Handle hUart, CddUart_EdmaChConfig *edmaChCfg);
static void UART_edmaIsrTx(Cdd_Edma_IntrHandle intrHandle, void *args);
static void UART_edmaIsrRx(Cdd_Edma_IntrHandle intrHandle, void *args);
static void UART_edmaDoNothing(Cdd_Edma_IntrHandle intrHandle, void *args);

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#define CDD_UART_START_SEC_CODE
#include "Cdd_Uart_MemMap.h"

sint32 Uart_Cdd_dmaInit(CddUart_Handle hUart, CddUart_DmaChConfig dmaChCfg)
{
    sint32 status = MCAL_SystemP_SUCCESS;
    CddUart_EdmaChConfig *edmaChCfg = (CddUart_EdmaChConfig *)dmaChCfg;

    status = UART_edmaChInit(hUart, edmaChCfg);

    if (status == MCAL_SystemP_SUCCESS)
    {
        edmaChCfg->isOpen = TRUE;
    }

    return status;
}

static sint32 UART_edmaChInit(CddUart_Handle hUart, CddUart_EdmaChConfig *edmaChCfg)
{
    sint32             status = MCAL_SystemP_FAILURE;
    uint32            baseAddr, regionId;
    uint32            dmaRxCh,dmaTxCh, tccRx, tccTx, tccDummy, paramRx, paramTx, paramDummy;
    uint32            isEdmaInterruptEnabled;
    Cdd_Edma_IntrObject     *edmaIntrObjectRx;
    Cdd_Edma_IntrObject     *edmaIntrObjectTx;
    Cdd_Edma_IntrObject     *edmaIntrObjectDummy;
    Cdd_EDMALLD_Handle         uartEdmaHandle;

    uartEdmaHandle = (Cdd_EDMALLD_Handle)hUart->hUartInit->uartDmaHandle;
    edmaIntrObjectRx = &edmaChCfg->edmaIntrObjRx;
    edmaIntrObjectTx = &edmaChCfg->edmaIntrObjTx;
    edmaIntrObjectDummy = &edmaChCfg->edmaIntrObjDummy;

    if(uartEdmaHandle != NULL)
    {
        status = MCAL_SystemP_SUCCESS;

        /* Read base address of allocated EDMA instance */
        baseAddr = CDD_EDMA_lld_getBaseAddr(uartEdmaHandle);

        /* Read the region ID of the EDMA instance */
        regionId = CDD_EDMA_lld_getRegionId(uartEdmaHandle);

        /* Store the EDMA parameters */
        edmaChCfg->edmaBaseAddr = baseAddr;
        edmaChCfg->edmaRegionId = regionId;

        /* Check if interrupt is enabled */
        isEdmaInterruptEnabled = CDD_EDMA_lld_isInterruptEnabled(uartEdmaHandle);

        if((baseAddr != 0) && (regionId < MCAL_SOC_EDMA_NUM_REGIONS) && (isEdmaInterruptEnabled == TRUE))
        {
            /* Allocate EDMA channel for UART RX transfer */
            dmaRxCh = edmaChCfg->edmaRxChId;
            status += CDD_EDMA_lld_allocDmaChannel(uartEdmaHandle, &dmaRxCh);

            /* Allocate EDMA TCC for UART RX transfer */
            tccRx = CDD_EDMA_RESOURCE_ALLOC_ANY;
            status += CDD_EDMA_lld_allocTcc(uartEdmaHandle, &tccRx);

            /* Allocate a Param ID for UART RX transfer */
            paramRx = CDD_EDMA_RESOURCE_ALLOC_ANY;
            status += CDD_EDMA_lld_allocParam(uartEdmaHandle, &paramRx);

            if(status == MCAL_SystemP_SUCCESS)
            {
                CDD_EDMA_lld_configureChannelRegion(baseAddr, regionId, CDD_EDMA_CHANNEL_TYPE_DMA,
                dmaRxCh, tccRx, paramRx, EDMA_UART_RX_EVT_QUEUE_NO);

                /* Register RX interrupt */
                edmaIntrObjectRx->tccNum = tccRx;
                edmaIntrObjectRx->cbFxn  = &UART_edmaIsrRx;
                edmaIntrObjectRx->appData = (void *) hUart;
                status += CDD_EDMA_lld_registerIntr(uartEdmaHandle, edmaIntrObjectRx);
            }

            if(status == MCAL_SystemP_SUCCESS)
            {
                /* Store the EDMA parameters for UART RX*/
                edmaChCfg->edmaRxParam = paramRx;
                edmaChCfg->edmaRxChId  = dmaRxCh;
                edmaChCfg->edmaTccRx   = tccRx;
            }

            /* Allocate EDMA channel for UART TX transfer */
            dmaTxCh = edmaChCfg->edmaTxChId;
            status += CDD_EDMA_lld_allocDmaChannel(uartEdmaHandle, &dmaTxCh);

            /* Allocate EDMA TCC for UART TX transfer */
            tccTx = CDD_EDMA_RESOURCE_ALLOC_ANY;
            status += CDD_EDMA_lld_allocTcc(uartEdmaHandle, &tccTx);

            /* Allocate a Param ID for UART TX transfer */
            paramTx = CDD_EDMA_RESOURCE_ALLOC_ANY;
            status += CDD_EDMA_lld_allocParam(uartEdmaHandle, &paramTx);

            if(status == MCAL_SystemP_SUCCESS)
            {
                CDD_EDMA_lld_configureChannelRegion(baseAddr, regionId, CDD_EDMA_CHANNEL_TYPE_DMA,
                dmaTxCh, tccTx, paramTx, EDMA_UART_TX_EVT_QUEUE_NO);

                /* Register TX interrupt */
                edmaIntrObjectTx->tccNum = tccTx;
                edmaIntrObjectTx->cbFxn  = &UART_edmaIsrTx;
                edmaIntrObjectTx->appData = (void *) hUart;
                status += CDD_EDMA_lld_registerIntr(uartEdmaHandle, edmaIntrObjectTx);
            }

            if(status == MCAL_SystemP_SUCCESS)
            {
                /* Store the EDMA parameters for UART TX*/
                edmaChCfg->edmaTxParam = paramTx;
                edmaChCfg->edmaTxChId  = dmaTxCh;
                edmaChCfg->edmaTccTx   = tccTx;
            }

            /* Allocate EDMA TCC for Dummy param set */
            tccDummy = CDD_EDMA_RESOURCE_ALLOC_ANY;
            status += CDD_EDMA_lld_allocTcc(uartEdmaHandle, &tccDummy);

            /* Allocate a Param ID for UART TX transfer */
            paramDummy = CDD_EDMA_RESOURCE_ALLOC_ANY;
            status += CDD_EDMA_lld_allocParam(uartEdmaHandle, &paramDummy);

            if(status == MCAL_SystemP_SUCCESS)
            {

                /* Register Dummy interrupt */
                edmaIntrObjectDummy->tccNum = tccDummy;
                edmaIntrObjectDummy->cbFxn  = &UART_edmaDoNothing;
                edmaIntrObjectDummy->appData = (void *) hUart;
                status += CDD_EDMA_lld_registerIntr(uartEdmaHandle, edmaIntrObjectDummy);
            }

            if(status == MCAL_SystemP_SUCCESS)
            {
                /* Store the EDMA parameters for UART TX*/
                edmaChCfg->edmaDummyParam = paramDummy;
                edmaChCfg->edmaTccDummy   = tccDummy;
                edmaChCfg->isOpen = TRUE;
            }
        }
        else
        {
            status = MCAL_SystemP_FAILURE;
        }
    }
    return status;
}

sint32 Uart_Cdd_dmaWrite(CddUart_Handle hUart, CddUart_Transaction *transaction)
{
    sint32             status = MCAL_SystemP_SUCCESS;
    uint32            baseAddr, regionId;
    uint32            dmaTxCh, tccTx, tccDummy, paramTx, paramDummy;
    uint32            isTxFifoEmpty, isEdmaEventPending;
    CDD_EDMACCEDMACCPaRAMEntry    edmaTxParam, edmaDummyParam;
    CddUart_EdmaChConfig   *edmaChCfg;

    isTxFifoEmpty = FALSE;
    isEdmaEventPending = FALSE;

    edmaChCfg = (CddUart_EdmaChConfig *)hUart->hUartInit->dmaChCfg;

    /* Fetch the EDMA paramters for UART transfer */
    baseAddr               = edmaChCfg->edmaBaseAddr;
    regionId               = edmaChCfg->edmaRegionId;

    /* Fetch the EDMA paramters for UART TX transfer */
    dmaTxCh    = edmaChCfg->edmaTxChId;
    tccTx      = edmaChCfg->edmaTccTx;
    tccDummy   = edmaChCfg->edmaTccDummy;
    paramTx    = edmaChCfg->edmaTxParam;
    paramDummy = edmaChCfg->edmaDummyParam;

    /* Initialize TX Param Set */
    CDD_EDMA_lld_ccPaRAMEntry_init(&edmaTxParam);

    /* Transmit param set configuration */
    edmaTxParam.srcAddr       = (uint32) Cdd_Dma_SOC_virtToPhy((void*) transaction->buf);
    edmaTxParam.destAddr      = (uint32) Cdd_Dma_SOC_virtToPhy((uint8 *) hUart->baseAddr + UART_THR);
    edmaTxParam.aCnt          = (uint16) 1;
    edmaTxParam.bCnt          = (uint16) (transaction->count);
    edmaTxParam.cCnt          = (uint16) 1;
    edmaTxParam.bCntReload    = (uint16) edmaTxParam.bCnt;
    edmaTxParam.srcBIdx       = (sint16) edmaTxParam.aCnt;
    edmaTxParam.destBIdx      = (sint16) 0;
    edmaTxParam.srcCIdx       = (sint16) 0;
    edmaTxParam.destCIdx      = (sint16) 0;
    edmaTxParam.linkAddr      = 0xFFFFU;
    edmaTxParam.opt           = 0;
    edmaTxParam.opt          |=
        (CDD_EDMA_OPT_TCINTEN_MASK | ((tccTx << CDD_EDMA_OPT_TCC_SHIFT) & CDD_EDMA_OPT_TCC_MASK));

    /* Write Tx param set */
    CDD_EDMA_lld_setPaRAM(baseAddr, paramTx, &edmaTxParam);

    /* Initialize TX Param Set */
    CDD_EDMA_lld_ccPaRAMEntry_init(&edmaDummyParam);

    /* Dummy param set configuration */
    edmaDummyParam.aCnt          = (uint16) 1;
    edmaDummyParam.linkAddr      = 0xFFFFU;
    edmaDummyParam.opt          |= (CDD_EDMA_OPT_TCINTEN_MASK | CDD_EDMA_OPT_STATIC_MASK |
                                   ((tccDummy << CDD_EDMA_OPT_TCC_SHIFT) & CDD_EDMA_OPT_TCC_MASK));

    /* Write Tx param set */
    CDD_EDMA_lld_setPaRAM(baseAddr, paramDummy, &edmaDummyParam);

    /* Link  dummy param ID */
    CDD_EDMA_lld_linkChannel(baseAddr, paramTx, paramDummy);

    if(!UART_checkCharsAvailInFifo(hUart->baseAddr))
    {
        isTxFifoEmpty = TRUE;
    }

    if(CDD_EDMA_lld_readEventStatusRegion(baseAddr, dmaTxCh))
    {
        isEdmaEventPending = TRUE;
    }

    /* Set event trigger to start UART TX transfer */
    CDD_EDMA_lld_enableTransferRegion(baseAddr, regionId, dmaTxCh,
        CDD_EDMA_TRIG_MODE_EVENT);

    if (isTxFifoEmpty && !isEdmaEventPending)
    {
        /* Set manual trigger to start UART TX transfer */
        CDD_EDMA_lld_enableTransferRegion(baseAddr, regionId, dmaTxCh,
        CDD_EDMA_TRIG_MODE_MANUAL);
    }

    return status;
}

sint32 Uart_Cdd_dmaRead(CddUart_Handle hUart, CddUart_Transaction *transaction)
{
    sint32             status = MCAL_SystemP_SUCCESS;
    uint32            baseAddr, regionId;
    uint32            dmaRxCh, tccRx, paramRx;
    CDD_EDMACCEDMACCPaRAMEntry    edmaRxParam;
    CddUart_EdmaChConfig   *edmaChCfg;

    edmaChCfg = (CddUart_EdmaChConfig *)hUart->hUartInit->dmaChCfg;

    /* Fetch the EDMA paramters for UART transfer */
    baseAddr               = edmaChCfg->edmaBaseAddr;
    regionId               = edmaChCfg->edmaRegionId;

    /* Fetch the EDMA paramters for UART RX transfer */
    dmaRxCh    = edmaChCfg->edmaRxChId;
    tccRx      = edmaChCfg->edmaTccRx;
    paramRx    = edmaChCfg->edmaRxParam;

    /* Initialize RX Param Set */
    CDD_EDMA_lld_ccPaRAMEntry_init(&edmaRxParam);

    /* Receive param set configuration */
    edmaRxParam.srcAddr       = (uint32) Cdd_Dma_SOC_virtToPhy((uint8 *) hUart->baseAddr + UART_RHR);
    edmaRxParam.destAddr      = (uint32) Cdd_Dma_SOC_virtToPhy((void*) transaction->buf);
    edmaRxParam.aCnt          = (uint16) 1;
    edmaRxParam.bCnt          = (uint16) (transaction->count);
    edmaRxParam.cCnt          = (uint16) 1;
    edmaRxParam.bCntReload    = (uint16) edmaRxParam.bCnt;
    edmaRxParam.srcBIdx       = (sint16) 0;
    edmaRxParam.destBIdx      = (sint16) edmaRxParam.aCnt;
    edmaRxParam.srcCIdx       = (sint16) 0;
    edmaRxParam.destCIdx      = (sint16) 0;
    edmaRxParam.linkAddr      = 0xFFFFU;
    edmaRxParam.opt          |=
    (CDD_EDMA_OPT_TCINTEN_MASK | ((tccRx << CDD_EDMA_OPT_TCC_SHIFT) & CDD_EDMA_OPT_TCC_MASK));

    /* Write Rx param set */
    CDD_EDMA_lld_setPaRAM(baseAddr, paramRx, &edmaRxParam);

    /* Set event trigger to start UART RX transfer */
    CDD_EDMA_lld_enableTransferRegion(baseAddr, regionId, dmaRxCh,
         CDD_EDMA_TRIG_MODE_EVENT);

    return status;
}

sint32 Uart_Cdd_dmaDeInit(CddUart_Handle hUart)
{
    sint32            status = MCAL_SystemP_SUCCESS;
    uint32           baseAddr, regionId;
    CddUart_EdmaChConfig   *edmaChCfg;
    Cdd_EDMALLD_Handle         uartEdmaHandle;
    Cdd_Edma_IntrObject     *edmaIntrObjectRx;
    Cdd_Edma_IntrObject     *edmaIntrObjectTx;
    Cdd_Edma_IntrObject     *edmaIntrObjectDummy;
    uint32            dmaRxCh,dmaTxCh, tccRx, tccTx, tccDummy, paramRx, paramTx, paramDummy;
    CddUart_InitHandle     hUartInit;

    /* Check parameters */
    if(NULL == hUart)
    {
        status = MCAL_SystemP_FAILURE;
    }

    if(MCAL_SystemP_SUCCESS == status)
    {
        hUartInit = hUart->hUartInit;
        edmaChCfg    = (CddUart_EdmaChConfig *)hUartInit->dmaChCfg;
        uartEdmaHandle = (Cdd_EDMALLD_Handle)hUartInit->uartDmaHandle;

        /* Fetch the EDMA paramters */
        baseAddr         = edmaChCfg->edmaBaseAddr;
        regionId         = edmaChCfg->edmaRegionId;

        if (edmaChCfg->isOpen != FALSE)
        {
            /* Fetch the EDMA paramters */
            dmaRxCh    = edmaChCfg->edmaRxChId;
            tccRx      = edmaChCfg->edmaTccRx;
            paramRx    = edmaChCfg->edmaRxParam;
            edmaIntrObjectRx = &edmaChCfg->edmaIntrObjRx;

            /* unregister Rx interrupt */
            status += CDD_EDMA_lld_unregisterIntr(uartEdmaHandle, edmaIntrObjectRx);

            /* Free Rx channel */
            CDD_EDMA_lld_freeChannelRegion(baseAddr, regionId, CDD_EDMA_CHANNEL_TYPE_DMA,
                dmaRxCh, CDD_EDMA_TRIG_MODE_EVENT, tccRx, EDMA_UART_RX_EVT_QUEUE_NO);

            if(dmaRxCh != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeDmaChannel(uartEdmaHandle, &dmaRxCh);
            }
            if(tccRx != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeTcc(uartEdmaHandle, &tccRx);
            }
            if(paramRx != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeParam(uartEdmaHandle, &paramRx);
            }

            /* Fetch the EDMA paramters */
            dmaTxCh    = edmaChCfg->edmaTxChId;
            tccTx      = edmaChCfg->edmaTccTx;
            paramTx    = edmaChCfg->edmaTxParam;
            edmaIntrObjectTx = &edmaChCfg->edmaIntrObjTx;

            /* unregister Tx interrupt */
            status += CDD_EDMA_lld_unregisterIntr(uartEdmaHandle, edmaIntrObjectTx);

            /* Free Tx channel */
            CDD_EDMA_lld_freeChannelRegion(baseAddr, regionId, CDD_EDMA_CHANNEL_TYPE_DMA,
                dmaTxCh, CDD_EDMA_TRIG_MODE_EVENT, tccTx, EDMA_UART_TX_EVT_QUEUE_NO);

            if(dmaTxCh != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeDmaChannel(uartEdmaHandle, &dmaTxCh);
            }
            if(tccTx != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeTcc(uartEdmaHandle, &tccTx);
            }
            if(paramTx != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeParam(uartEdmaHandle, &paramTx);
            }

            tccDummy   = edmaChCfg->edmaTccDummy;
            paramDummy    = edmaChCfg->edmaDummyParam;
            edmaIntrObjectDummy = &edmaChCfg->edmaIntrObjDummy;

            /* unregister Dummy interrupt */
            status += CDD_EDMA_lld_unregisterIntr(uartEdmaHandle, edmaIntrObjectDummy);

            if(tccDummy != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeTcc(uartEdmaHandle, &tccDummy);
            }
            if(paramDummy != CDD_EDMA_RESOURCE_ALLOC_ANY)
            {
                CDD_EDMA_lld_freeParam(uartEdmaHandle, &paramDummy);
            }

            edmaChCfg->isOpen = FALSE;
        }
    }

    return status;
}

sint32 Uart_Cdd_dmaDisableChannel(CddUart_Handle hUart,
                                       uint32 isChannelTx)
{
    sint32       status = MCAL_SystemP_SUCCESS;
    uint32      baseAddr, regionId, dmaCh;
    CddUart_EdmaChConfig *edmaChCfg;

    if(NULL != hUart)
    {
        edmaChCfg = (CddUart_EdmaChConfig *)hUart->hUartInit->dmaChCfg;

        baseAddr               = edmaChCfg->edmaBaseAddr;
        regionId               = edmaChCfg->edmaRegionId;

        /* Disable Channel */
        if (isChannelTx == TRUE)
        {
            dmaCh    = edmaChCfg->edmaTxChId;
        }
        else
        {
            dmaCh    = edmaChCfg->edmaRxChId;
        }

        /* Stop the UART event transfer */
        CDD_EDMA_lld_disableTransferRegion(baseAddr, regionId, dmaCh,
            CDD_EDMA_TRIG_MODE_EVENT);
    }
    else
    {
        status = MCAL_SystemP_FAILURE;
    }

    return status;
}

static void UART_edmaIsrTx(Cdd_Edma_IntrHandle intrHandle, void *args)
{
    CddUart_Handle hUart;

    /* Check parameters */
    if(NULL != args)
    {
        hUart = (CddUart_Handle)args;
        hUart->writeTrans->status = UART_TRANSFER_STATUS_SUCCESS;
        Uart_Cdd_writeCompleteCallback(hUart, hUart->writeTrans);
        hUart->writeTrans = NULL;
    }
    return;
}

static void UART_edmaIsrRx(Cdd_Edma_IntrHandle intrHandle, void *args)
{
    CddUart_Handle hUart;

    /* Check parameters */
    if(NULL != args)
    {
        hUart = (CddUart_Handle)args;
        hUart->readTrans->status = UART_TRANSFER_STATUS_SUCCESS;
        Uart_Cdd_readCompleteCallback(hUart, hUart->readTrans);
        hUart->readTrans = NULL;
    }

    return;
}

static void UART_edmaDoNothing(Cdd_Edma_IntrHandle intrHandle, void *args)
{
    /* No implementation required */
}

#define CDD_UART_STOP_SEC_CODE
#include "Cdd_Uart_MemMap.h"

#endif
