/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Wdg.h
 *
 *  \brief    This file contains interface header for WDG MCAL driver
 *
 */

/**
 *  \defgroup MCAL_WDG_API WDG Driver API
 *
 *  The WDG driver provides services for handling the watch dog timer on the
 *  microcontroller
 *
 *  \verbatim
 *        Digital Windowed Watchdog(DWWD) Overview :
 *        The Digital Watchdog Timer(DWT) generates reset after a programmable
 *        period, if not serviced within that period. In DWT, time-out
 *        boundary is configurable.
 *        In DWWD, along with configurable time-out boundary, the start time
 *        boundary is also configurable. The DWWD can generate Reset or
 *        Interrupt, if not serviced within window(Open Window) defined by
 *        start time and time-out boundary. Also the DWWD can generate Reset or
 *        Interrupt if serviced outside Open Window (within Closed Window).
 *        Generation of Reset or Interrupt depends on the DWWD Reaction
 *        configuration.
 *
 *        DWWD Down Counter Overview :
 *        Upper 12 bit part of the down counter is configurable and
 *        remaining 13 bit are always 1.
 *        Minimum possible time-out value is 2^13 RTI clock cycles.
 *        Maximum possible time-out value is 2^25 RTI clock cycles.
 *        Example :
 *        RTI frequency : 32kHz
 *        12 bit preload value : 0x004
 *        25 bit preload value : 0x0009FFF
 *        time-out value(in RTI clock cycles) : 40959.
 *        time-out value(in seconds) : 40959 / 32000 = 1.26859375 seconds.
 *
 *
 *        DWWD Window Sizes Overview :
 *         time-out value...........................................3 2 1 0
 *   (100%)|___________________________Open Window________________________|
 *    (50%)|         Closed Window         |________Open Window___________|
 *    (25%)|            Closed Window                      |___OW_________|
 *  (12.5%)|                 Closed Window                         |_OW___|
 *  (6.25%)|                     Closed Window                        |OW |
 * (3.125%)|                         Closed Window                     |OW|
 *        For time-out value calculation refer DWWD Down Counter Overview.
 *  \endverbatim
 *
 *  @sa MCAL_WDG_CFG
 *  @{
 */
/* Design : WDG_DesignId_006, WDG_DesignId_002, WDG_DesignId_001,
 */
/*
 * Below are the global requirements which are met by this WDG
 * which can't be mapped to a particular source ID
 */
/*
 * Requirements : SWS_Wdg_00079, SWS_Wdg_00061,SWS_Wdg_00031, SWS_Wdg_00040,
 *                AUTORADAR_MCAL-3
 *                SWS_Wdg_00168
 *                
 */

#ifndef WDG_H_
#define WDG_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/* Requirements : SWS_Wdg_00105 */
#include "Std_Types.h"
#include "WdgIf_Types.h"
/* Requirements:SWS_Wdg_00061 */
#include "Wdg_Cfg.h"

#ifdef __cplusplus
extern "C"
{
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/**
 *  \name WDG Driver Module SW Version Info
 *
 *  Defines for WDG Driver version used for compatibility checks
 *  @{
 */
/** \brief Driver Implementation Major Version */
#define WDG_SW_MAJOR_VERSION            (8U)
/** \brief Driver Implementation Minor Version */
#define WDG_SW_MINOR_VERSION            (6U)
/** \brief Driver Implementation Patch Version */
#define WDG_SW_PATCH_VERSION            (2U)
/* @} */

/**
 *  \name WDG Driver Module AUTOSAR Version Info
 *
 *  Defines for WDG Driver AUTOSAR version used for compatibility checks
 *  @{
 */
/** \brief AUTOSAR Major version specification implemented by WDG Driver */
#define WDG_AR_RELEASE_MAJOR_VERSION    (4U)
/** \brief AUTOSAR Minor  version specification implemented by WDG Driver */
#define WDG_AR_RELEASE_MINOR_VERSION    (3U)
/** \brief AUTOSAR Patch version specification implemented by WDG Driver */
#define WDG_AR_RELEASE_REVISION_VERSION (1U)
/* @} */

/**
 *  \name WDG Driver ID Info
 *  @{
 */
/** \brief Texas Instruments Vendor ID */
#define WDG_VENDOR_ID                   ((uint16) 44U)
/** \brief WDG Driver Module ID */
#define WDG_MODULE_ID                   ((uint16) 101U)
/** \brief WDG Driver Instance ID */
#define WDG_INSTANCE_ID                 ((uint8) 0U)
/* @} */
/* DesignId : WDG_DesignId_003 */
/* Requirements : SWS_Wdg_00010 */
/**
 *  \name WDG Error Codes
 *
 *  Error codes returned by Driver functions
 *  @{
 */
/** \brief ERROR:Invalid driver state */
#define WDG_E_DRIVER_STATE              ((uint8) 0x10U)
/** \brief ERROR:Invalid mode param */
#define WDG_E_PARAM_MODE                ((uint8) 0x11U)
/** \brief ERROR:Invalid config param */
#define WDG_E_PARAM_CONFIG              ((uint8) 0x12U)
/** \brief ERROR:Invalid timeout value */
#define WDG_E_PARAM_TIMEOUT             ((uint8) 0x13U)
/** \brief ERROR:Invalid NULL ptr param */
#define WDG_E_PARAM_POINTER             ((uint8) 0x14U)
/** \brief ERROR:Invalid Configuration set selection */
#define WDG_E_INIT_FAILED               ((uint8) 0x15U)
/* @} */

/**
 *  \name WDG Service Ids
 *
 *  The Service Id is one of the argument to Det_ReportError function and is
 *  used to identify the source of the error
 *  @{
 */
/** \brief Wdg_Init() */
#define WDG_API_INIT                   ((uint8) 0x00U)
/** \brief Wdg_SetMode() */
#define WDG_API_SET_MODE               ((uint8) 0x01U)
/** \brief Wdg_SetTriggerCondition() */
#define WDG_SET_TRIGGER_CONDITION      ((uint8) 0x03U)
/** \brief Wdg_GetVersionInfo() */
#define WDG_API_GET_VERSION_INFO       ((uint8) 0x04U)
/** \brief Wdg_Trigger() */
#define WDG_API_TRIGGER                ((uint8) 0x05U)
/* @} */


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */
/*******************************************************************************
 * Exported Type Declarations
 ******************************************************************************/
typedef enum
{
    WDG_UNINIT,
    /**< Watchdog driver is uninitialized. State on power on before
     *   invoking Wdg_Init() */
    WDG_IDLE,
    /**< Watchdog driver is in idle state. Default state when WDG
     *   timer regs are not being programmed */
    WDG_BUSY
    /**< Watchdog driver is reprogramming the WD to trigger the
     *   watchdog */
} Wdg_StatusType;

/** @enum Wdg_Reaction
*   @brief DWWD Reaction on violation
*/
typedef enum
{
    WDG_GENERATE_RESET = 0x00000005U,
    WDG_GENERATE_NMI   = 0x0000000AU
}Wdg_Reaction;

/** @enum Wdg_WindowSize
*   @brief DWWD Window size
*/
typedef enum
{
    WDG_WINDOW_SIZE_100_PERCENT = 0x00000005U,
    WDG_WINDOW_SIZE_50_PERCENT  = 0x00000050U,
    WDG_WINDOW_SIZE_25_PERCENT  = 0x00000500U,
    WDG_WINDOW_SIZE_12_5_PERCENT    = 0x00005000U,
    WDG_WINDOW_SIZE_6_25_PERCENT    = 0x00050000U,
    WDG_WINDOW_SIZE_3_125_PERCENT   = 0x00500000U
}Wdg_WindowSize;

/** \brief Mode info type */
typedef struct
{
    Wdg_Reaction reaction;
    /**< Reaction type */
    Wdg_WindowSize windowSize;
    /**< Window size */
    uint32 preloadValue;
    /**< Preload value */
} Wdg_ModeInfoType;

/* Requirements : SWS_Wdg_00171 */
/** \brief WDG config structure */
typedef struct Wdg_ConfigType_s
{
    WdgIf_ModeType   defaultMode;
    /**< Default mode */
    uint16           instanceId;
    /**< Instance ID */
    uint32           initialTimeOut;
    /**< Initial timeout */
    Wdg_ModeInfoType fastModeCfg;
    /**< Fast mode config */
    Wdg_ModeInfoType slowModeCfg;
    /**< Slow mode config */
} Wdg_ConfigType;

#if (STD_ON == WDG_REGISTER_READBACK_API)
/** \brief WDG configuration register readback data structure */
typedef struct
{
   uint32 Wdg_RtiDwdCtrl;
   /**< Digital WDG control register*/
   uint32 Wdg_RtiDwdprld;
   /**< WDG Period Load register */
   uint32 Wdg_RtiWdStatus;
   /**< WDG status register */
   uint32 Wdg_RtiWdKey;
   /**< WDG Key register */
   uint32 Wdg_RtiDwdCntr;
   /**< WDG counter register */
   uint32 Wdg_RtiWwdRxnCtrl;
   /**< WDG window reaction register */
   uint32 Wdg_RtiWwdSizeCtrl;
   /**< WDG window size controlregister */
}Wdg_RegisterReadbackType;
#endif /* STD_ON == WDG_REGISTER_READBACK_API */


/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

#if (STD_ON == WDG_GET_VERSION_INFO_API)
/**
 *  \brief This service returns the version information of this module
 *
 *  \verbatim
 *  Service name      : Wdg_GetVersionInfo
 *  Syntax            : void Wdg_GetVersionInfo(Std_VersionInfoType*
 *                                  versioninfo)
 *  Service ID[hex]   : 0x04U
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Reentrant
 *  Parameters (in)   : None
 *  Parameters (inout): None
 *  Parameters (out)  : versioninfo - Pointer to where to store the version
 *                      information of this module.
 *  Return value      : None
 *  Description       : This service returns the version information of this
 *                      module
 *  \endverbatim
 */
FUNC(void, WDG_CODE) Wdg_GetVersionInfo(Std_VersionInfoType * versioninfo);
#endif

/**
 *  \brief  This service initializes the WDG driver
 *
 *  \verbatim
 *  Service name      : Wdg_Init
 *  Syntax            : void Wdg_Init( const Wdg_ConfigType* ConfigPtr )
 *  Service ID[hex]   : 0x0
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : ConfigPtr - Pointer to WDG driver configuration set.
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : This service initializes the WDG driver
 *  \endverbatim
 */
FUNC(void, WDG_CODE) Wdg_Init(
    P2CONST(Wdg_ConfigType, AUTOMATIC, WDG_APPL_CONST) ConfigPtr);

/**
 *  \brief  Switches the watchdog into the mode Mode
 *
 *  \verbatim
 *  Service name      : Wdg_SetMode
 *  Syntax            : Std_ReturnType Wdg_SetMode( WdgIf_ModeType Mode )
 *  Service ID[hex]   : 0x01
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : Mode - One of the following statically configured modes
 *                           - WDGIF_OFF_MODE
 *                           - WDGIF_SLOW_MODE
 *                           - WDGIF_FAST_MODE
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : Std_ReturnType
 *  Description       : Switches the watchdog into the mode Mode
 *  \endverbatim
 */
FUNC(Std_ReturnType, WDG_CODE) Wdg_SetMode(WdgIf_ModeType Mode);

/**
 *  \brief  Sets the timeout value for the trigger counter
 *
 *  \verbatim
 *  Service name      : Wdg_SetTriggerCondition
 *  Syntax            : void Wdg_SetTriggerCondition( uint16 timeout )
 *  Service ID[hex]   : 0x03
 *  Sync/Async        : Synchronous
 *  Reentrancy        : Non Reentrant
 *  Parameters (in)   : timeout - Timeout value (milliseconds) for setting the
 *                      trigger counter
 *  Parameters (inout): None
 *  Parameters (out)  : None
 *  Return value      : None
 *  Description       : Sets the timeout value for the trigger counter
 *  \endverbatim
 */
FUNC(void, WDG_CODE) Wdg_SetTriggerCondition(uint16 timeout);

/**
 *  \brief This function triggers the servicing of the watchdog
 *
 *  \verbatim
 *   Function name     : Wdg_Trigger
 *   Syntax            : void Wdg_Trigger( void )
 *   Service ID[hex]   : N/A
 *   Sync/Async        : Synchronous
 *   Reentrancy        : Non-Reentrant
 *   Parameters (in)   : None
 *   Parameters (inout): None
 *   Parameters (out)  : None
 *   Return value      : None
 *   Description       : This function is the watchdog trigger
 *                       This function is invoked from WdgIsr
 *  \endverbatim
 */
FUNC(void, WDG_CODE) Wdg_Trigger(void);

#if (STD_ON == WDG_REGISTER_READBACK_API)
/**
 *  \brief      This method is Wdg_RegisterReadback Api.
 *  Function name     : Wdg_RegisterReadback
 *   Syntax            : void Wdg_RegisterReadback
        (P2VAR(Wdg_RegisterReadbackType, AUTOMATIC, WDG_APPL_DATA) RegisterReadbackPtr)
 *   Service ID[hex]   : N/A
 *   Sync/Async        : Synchronous
 *   Reentrancy        : Non-Reentrant
 *   Parameters (in)   : RegisterReadbackPtr
 *   Parameters (inout): None
 *   Parameters (out)  : None
 *   Return value      : Std_ReturnType
 *   Description       : Function could be called from from task level
 */
FUNC(Std_ReturnType, WDG_CODE) Wdg_RegisterReadback
(P2VAR(Wdg_RegisterReadbackType, AUTOMATIC, WDG_APPL_DATA) RegisterReadbackPtr);

#endif
#ifdef __cplusplus
}
#endif

#endif  /* #ifndef WDG_H_ */

/* @} */
