/*
*
* Copyright (c) 2020 Texas Instruments Incorporated
*
* All rights reserved not granted herein.
*
* Limited License.
*
* Texas Instruments Incorporated grants a world-wide, royalty-free, non-exclusive
* license under copyrights and patents it now or hereafter owns or controls to make,
* have made, use, import, offer to sell and sell ("Utilize") this software subject to the
* terms herein.  With respect to the foregoing patent license, such license is granted
* solely to the extent that any such patent is necessary to Utilize the software alone.
* The patent license shall not apply to any combinations which include this software,
* other than combinations with devices manufactured by or for TI ("TI Devices").
* No hardware patent is licensed hereunder.
*
* Redistributions must preserve existing copyright notices and reproduce this license
* (including the above copyright notice and the disclaimer and (if applicable) source
* code license limitations below) in the documentation and/or other materials provided
* with the distribution
*
* Redistribution and use in binary form, without modification, are permitted provided
* that the following conditions are met:
*
* *       No reverse engineering, decompilation, or disassembly of this software is
* permitted with respect to any software provided in binary form.
*
* *       any redistribution and use are licensed by TI for use only with TI Devices.
*
* *       Nothing shall obligate TI to provide you with source code for the software
* licensed and provided to you in object code.
*
* If software source code is provided to you, modification and redistribution of the
* source code are permitted provided that the following conditions are met:
*
* *       any redistribution and use of the source code, including any resulting derivative
* works, are licensed by TI for use only with TI Devices.
*
* *       any redistribution and use of any object code compiled from the source code
* and any resulting derivative works, are licensed by TI for use only with TI Devices.
*
* Neither the name of Texas Instruments Incorporated nor the names of its suppliers
*
* may be used to endorse or promote products derived from this software without
* specific prior written permission.
*
* DISCLAIMER.
*
* THIS SOFTWARE IS PROVIDED BY TI AND TI'S LICENSORS "AS IS" AND ANY EXPRESS
* OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
* OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL TI AND TI'S LICENSORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
* OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
* OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/**
 *  \file     Eth_Types.h
 *
 *  \brief    This file contains Eth type declarations.
 */

/**
 *  \ingroup MCAL_ETH_API
 *  @{
 */

#ifndef ETH_TYPES_H_
#define ETH_TYPES_H_

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "Std_Types.h"
#include "Eth_Cfg.h"
#include "Eth_GeneralTypes.h"



#ifdef __cplusplus
extern "C"
{
#endif

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/*
 * Pre-Pad Packet Data Offset
 *
 *   The TCP/IP stack library requires that every packet device
 *   include enough L2 header room for all supported headers. In
 *   order to support PPPoE, this requires a 22 byte L2 header.
 *   Thus, since standard Ethernet is only 14 bytes, we must add
 *   on an additional 8 byte offset, or PPPoE can not function
 *   with our driver.
*/
#define ETH_PKT_PREPAD                  ((uint16)8U)

/** \brief MAC address length in bytes. */
#define ETH_MAC_ADDR_LEN                ((uint8) 6U)
/** \brief Total octets in header.  */
#define ETH_HLEN                        ((uint16) 14U)
/** \brief Min. octets in frame sans FCS */
#define ETH_ZLEN                        ((uint16) 60U)
/** \brief Max. octets in payload   */
#define ETH_DATA_LEN                    ((uint16) 1500U)
/** \brief Max. octets in payload   */
#define ETH_VLAN_TAG_SIZE               ((uint16) 4U)

/*
 * Requirements : SWS_Eth_00218 The Ethernet Driver shall ensure that the base
 * addresses of all reception and transmission buffers fulfill the memory
 * alignment requirements for all AUTOSAR data types of the respective platform.
 */
#define ETH_MAX_FRAME_LEN                   (1536U)

/**
 *  \brief Maximum number of CPSW ports supported by the controller.
 */
#define ETH_CPSW_MAX_PORT               ((uint32) 1U)

/** \brief Typedef for cache flush function pointer */
typedef P2FUNC (void, ETH_APPL_CODE, Eth_CacheFlushType)(
    P2VAR(uint8, AUTOMATIC, ETH_APPL_DAT)BufPtr,
    uint32 LenByte);

/** \brief Typedef for cache invalidate function pointer */
typedef P2FUNC (void, ETH_APPL_CODE, Eth_CacheInvalidateType)(
    P2VAR(uint8, AUTOMATIC, ETH_APPL_DAT)BufPtr,
    uint32 LenByte);


/**
 *  \brief List of macros for default config parameters.
 *         This can be used by application as reference to pass config to driver
 *         functions.
 */

/* CPTS related macros */
/* 266 MHz by default (L3 Frequency) */
#define ETH_CPSW_CPTSCFG_CPTSINPUTFREQ            (125000000U)

/* CPDMA related macros */
#define ETH_CPSW_DEFAULT_PACING_MAINCLKFREQ       (125000000U)

#define ETH_CPSW_DEFAULT_MDIOBUSFREQ              (2200000U)
#define ETH_CPSW_DEFAULT_MDIOCLOCKFREQ            (200000000U)
#define ETH_CPSW_DEFAULT_MAINCLKFREQ              (125000000U)

/* prescalar divides down to 1ms interval*/
#define ETH_CPSW_DEFAULT_ALEPRESCALE       (ETH_CPSW_DEFAULT_MAINCLKFREQ / \
                                              1000U)

/* ALE time out value in ms */
#define ETH_CPSW_ALE_AGE_OUT_TIME_IN_MS                        (300000U)

#define ETH_CPSW_MAC_CONFIG_MODEFLG_TXFLOWCNTL        (uint32) (0x08000U)
/**< Enable tx flow control                                                 */
#define ETH_CPSW_MAC_CONFIG_MODEFLG_RXBUFFERFLOWCNTL  (uint32) (0x10000U)
/**< Enable rx buffer flow control                                          */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 * \brief  CPTS Ethernet Message Type Enumeration
 */
typedef enum
{
    ETH_CPTS_MESSAGE_SYNC                   = (0x1U),
    ETH_CPTS_MESSAGE_DELAY_REQ              = (0x2U),
    ETH_CPTS_MESSAGE_PDELAY_REQ             = (0x4U),
    ETH_CPTS_MESSAGE_PDELAY_RESP            = (0x8U),
    ETH_CPTS_MESSAGE_FOLLOW_UP              = (0x100U),
    ETH_CPTS_MESSAGE_DELAY_RESP             = (0x200U),
    ETH_CPTS_MESSAGE_PDELAY_RESP_FOLLOW_UP  = (0x400U),
    ETH_CPTS_MESSAGE_ANNOUNCE               = (0x800U),
    ETH_CPTS_MESSAGE_SIGNALING              = (0x1000U),
    ETH_CPTS_MESSAGE_MANAGEMENT             = (0x2000U),
    ETH_CPTS_MESSAGE_ALL                    =  0xFFFFU,
} Eth_CptsMessageCfg;

/**
 *  \brief Enum of VLAN usage type for CPTS packets
 */
typedef enum
{
    ETH_CPTS_VLAN_TYPE_NONE                 = 0x00U,
    ETH_CPTS_VLAN_TYPE_SINGLE_TAG           = 0x01U,
    ETH_CPTS_VLAN_TYPE_STACKED_TAGS         = 0x02U,
    ETH_CPTS_VLAN_TYPE_INVALID              = 0xFFFFFFFFU
} Eth_CptsVlanType;

/**
 *  \brief CPTS Module Init Configuration Structure
 *         Configuration information for the CPTS modules provided during
 *         initialization.
 */
typedef struct
{
    uint32 cptsInputFreq;
    /**< CPTS input clock frequency in Hz Ex- 266000000UL for L3 266MHz freq. */
    Eth_CptsVlanType vlanType;
    /**<  VLAN Type */
    uint32 msgTypeCfg;
    /**<  Message types on which time stamping is enabled*/
} Eth_CptsConfigType;

/** \brief Enumerates speed configurations. */
typedef enum
{
    ETH_SPEED_10 = 0x00U,
    /**< 10 Mbps operation. */
    ETH_SPEED_100 = 0x01U,
    /**< 100 Mbps operation. */
    ETH_SPEED_1000 = 0x02U
                     /**< 1000 Mbps operation. */
} Eth_SpeedType;

/**
 *  \brief  Type/Speed/Duplex Connection Config Options
 *
 *  Connection Config Options
 *
 */
typedef enum
{
    ETH_MAC_CONN_TYPE_MII_10 = 0x00U,
    /**< MAC connection type for 10Mbps MII mode */
    ETH_MAC_CONN_TYPE_MII_100 = 0x01U,
    /**< MAC connection type for 100Mbps MII mode */
    ETH_MAC_CONN_TYPE_RMII_10 = 0x02U,
    /**< MAC connection type for 10Mbps RMII mode */
    ETH_MAC_CONN_TYPE_RMII_100 = 0x03U,
    /**< MAC connection type for 100Mbps RMII mode */
    ETH_MAC_CONN_TYPE_RGMII_FORCE_100_HALF = 0x04U,
    /**< MAC connection type for forced half-duplex 100Mbps RGMII mode */
    ETH_MAC_CONN_TYPE_RGMII_FORCE_100_FULL = 0x05U,
    /**< MAC connection type for forced full-duplex 100Mbps RGMII mode */
	ETH_MAC_CONN_TYPE_RGMII_FORCE_1000 = 0x06U,
    /**< MAC connection type for forced 1000Mbps RGMII mode */
    ETH_MAC_CONN_TYPE_RGMII_DETECT_INBAND = 0x07U,
    /**< MAC connection type for RGMII inband detection mode (speed determined
     *   based on received RGMII Rx clock) */
} Eth_MacConnectionType;


/** \brief Enumerates duplicity configuration modes. */
typedef enum
{
    ETH_DUPLEX_MODE_HALF = 0x00U,
    /**< Half duplex mode. */
    ETH_DUPLEX_MODE_FULL = 0x01U
                           /**< Full duplex mode. */
} Eth_DuplexModeType;

/**
 *  \brief   CPDMA Init Configuration
 *
 *  \details Configuration information for the CPDMA provided during
 *  initialization.
 *  Note: Most of CPDMA features are not implemented. Only interrupt
 *  pacing is enabled.
 */
typedef struct
{
    uint32 pacingClkFreq;
    /**< CPDMA ICLK frequency for interrupt pacing  */
    uint32 rxInterruptPacingEnabled;
    /**< Receive interrupt pacing enable flag    */
    uint32 rxInterruptsPerMsec;
    /**< Receive interrupt pacing conf  */
    uint32 txInterruptPacingEnabled;
    /**< Transmit interrupt pacing enable flag    */
    uint32 txInterruptsPerMsec;
    /**< Transmit interrupt pacing conf  */
    /**< Register control fields for DMA Control register      */
    uint32 dmaModeFlags;
    /**< Configuation Mode Flags.  */
    uint32 rxThreshCount;
    /**< Free buffer descriptors count threshold to trigger an RX_THRESH
     *  interrupt.Set to zero for no threshold interrupt enablement
     */
} Eth_CpdmaConfigType;

/**
 *  \brief MDIO Module Init Configuration
 *         Configuration information for the MDIO modules provided during
 *         initialization.
 */
typedef struct
{
    uint32 mdioBusFreq;
    /**< Bus frequency for the MII module */
    uint32 mdioClockFreq;
    /**< Clock frequency for MDIO link */
} Eth_MdioConfigType;

/** \brief Structure holding the PHY configuration .*/
typedef struct
{
    uint32 macModeFlags;
    /**< Configuation Mode Flags */
    Eth_MacConnectionType  macConnectionType;
    /**< MAC Connection Type. */
    uint8  macAddr[ETH_MAC_ADDR_LEN];
    /**< MAC address  */
} Eth_MacConfigType;

/**
 *  \brief Structure for buffer.
 */
typedef struct
{
    void  *payload;
    /**< length of this buffer */
    uint16 len;
    /**< pointer to the actual data in the buffer */
} Eth_BufType;

/**
 *  \brief Structure for port configuration.
 */
typedef struct
{
    Eth_MacConfigType macCfg;
    /**< MAC Init Configurations */
} Eth_PortConfigType;

/**
 * \brief Rate limit configuration for switch priorities on port
 *        Note: Highest priorities only can be rate limited.
 */
typedef enum
{
    ETH_PORT_RATE_LIMIT_PRI1_ENABLE = (0x1U),
    ETH_PORT_RATE_LIMIT_PRI2_ENABLE = (0x2U),
    ETH_PORT_RATE_LIMIT_PRI3_ENABLE = (0x4U),
    ETH_PORT_RATE_LIMIT_DISABLE     = (0x0U),
} Eth_portTxRateLimitPrioCfgType;

/**
 *  \brief Holds CPSW Port Rx Rate Limit Configuration for CPPI Port Ingress
 *         Rate Limitaion Operation.
 */
/* Rate limit configuration data structure */
typedef struct
{
   boolean       enableQoS;
   /**< Flag to set QoS (TRUE/FALSE)*/
   uint32        bandwidth[1];
   /**< desired bandwidth in Mbps for channel*/
   uint32        rateLimitCfg;
    /**<  Port rate limit priorities configuration of type
          Eth_portTxRateLimitPrioCfgType */
   uint32        prioPercArray[3U];
    /**<  percentage of bandwidth for each of rate limited priority
      *   Note: Array takes priority percentage in increasing order of priority
      *         i.e. prioPercArray[0] for ETH_PORT_RATE_LIMIT_PRI1_ENABLE,
      *         prioPercArray[1] for ETH_PORT_RATE_LIMIT_PRI2_ENABLE & likewise.
      */
} Eth_cpswQoSCfg;


/**
 *  \brief EgressFifo Configuration.
 */
typedef struct
{
    uint16         egressBufLenByte; 
    uint16         egressBufTotal;
    uint8          egressFifoIdx;
    uint8          egressFifoPriority;
} Eth_EgressFifoType;

/**
 *  \brief IngressFifo Configuration.
 */
typedef struct
{
    uint16         ingressBufLenByte;
    uint16         ingressBufTotal;
    uint8          ingressFifoIdx;
    uint8          ingressFifoPriority;
} Eth_IngressFifoType;
/**
 *  \brief Implementation specific structure of the Eth configuration.
 */
/* Requirements : SWS_Eth_00156 */
typedef struct Eth_ConfigType_s
{
    uint8              ctrlIdx;
    /**< CPSW controller index [Supported value 0 for port 1 ] */
	uint8               portIdx;
	/**< Port Idx >*/
    Eth_PortConfigType  portCfg;
    /**< Port configuration */
#if (ETH_TX_TRAFFIC_SHAPING_ENABLE == STD_ON)
    Eth_cpswQoSCfg    traffiShapingCfg;
    /**< Traffic shaping Configuration */
#endif /* (STD_ON == ETH_TX_TRAFFIC_SHAPING_ENABLE) */
#if (ETH_ENABLE_MII_API == STD_ON)
    Eth_MdioConfigType  mdioCfg;
    /**< MDIO configuration */
#endif

    Eth_CpdmaConfigType cpdmaCfg;
    /**< CPDMA Init Configurations */
#if (ETH_GLOBALTIMESUPPORT_API == STD_ON)
    Eth_CptsConfigType  cptsCfg;
    /**< Time stamp (CPTS) Configurations */
#endif
    uint32              enableCacheOps;
    /**< Since the Eth driver is updating (writing/reading) the packet header,
     *   if the packets are in cached region, then cache flush should be done
     *   for transmit and cache invalidate should be done for receive.
     *
     *   This flag determines whether the cache operations need to be enabled
     *   or not. If enabled, then the below two callback will be called to
     *   perform the cache operation
     */
    Eth_CacheFlushType      cacheFlushOps;
    /**< Cache flush function pointer */
    Eth_CacheInvalidateType cacheInvalidateOps;
    /**< Cache invalidate function pointer */
    Eth_EgressFifoType  egressFifo[8];
    /**< Egress FIFO configuration */
    Eth_IngressFifoType  ingressFifo[8];
    /**< Egress FIFO configuration */
} Eth_ConfigType;

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

/* None */

#ifdef __cplusplus
}
#endif

#endif  /* ETH_TYPES_H_ */

/* @} */
