/* ======================================================================
 *   Copyright (C) 2022-2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     AdcApp.c
 *
 *  \brief    This file contains the Adc test example
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
/*LDRA_NOANALYSIS*/
#include "string.h"
/*LDRA_ANALYSIS*/
#include "assert.h"
/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
#include "SchM_Port.h"
#include "Det.h"
#include "Dem.h"

#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port_Cfg.h"
#include "Port.h"
#include "Cdd_Dma.h"

#include "Adc_Cfg.h"
#include "Adc.h"
#include "Adc_Irq.h"

#include "app_utils.h"
#include "sys_common.h"
#include "AdcApp.h"
#include "Epwm_Platform.h"
#include "trace.h"

#include "sys_vim.h"

/* For PRCM base addresses */
#include "soc.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define ARRAYSIZE(x)                    (sizeof ((x)) / sizeof ((x)[0]))


/* Macros to Enable ADC Application Test and Performance Test */
#define ADC_APPL_TEST                   STD_ON
#define ADC_PERFORMANCE_TEST            STD_OFF
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef void (*AdcAppFxn_t)(void);
#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif
/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */
void _enable_interrupt_(void);
void StartCyclecount(void);
static void AdcApp_platformInit(void);;
static void AdcApp_interruptConfig(void);
static void AdcApp_interruptInit(void);

static void SOC_setEpwmGroup(uint32 epwmInstance, uint32 group);
static void SOC_setEpwmTbClk(uint32 epwmInstance, uint32 enable);
static void SOC_controlModuleUnlockMMR(uint32 domainId, uint32 partition);
static void SOC_controlModuleLockMMR(uint32 domainId, uint32 partition);
static inline void
SOC_xbarSelectInterruptXBarInputSource
(uint32 base, uint8 out, uint32 group0_mask, uint32 group1_mask, 
 uint32 group2_mask, uint32 group3_mask, uint32 group4_mask, 
 uint32 group5_mask, uint32 group6_mask);
static inline void SOC_enableAdcReference(uint32 adcInstance);

static void AdcApp_mainTest(void);
static void Adc_Test(void);
void AdcTest(uint32 vart);
void Stop_Timer(uint8 Api_Id);
void Start_Timer(void);
static void AdcExample_PerformanceTest(void);
static void AdcApp_Notification(uint32 grpIdx);
//static void Epwm_Enable(void);
static uint32 AppUtils_GetAdcVoltage(uint32 adcValue);
void Adc_Trigger(uint32 baseAddr);
void Epwm_Enable(uint32 baseAddr);
/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif

volatile uint32  NotificationCount[20];

Mcu_ClockConfigType AdcMcu_ClkConfig[] =
{
   [0] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI0,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
   [1] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI1,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
   [2] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI2,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
    [3] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI3,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
    [4] =
    {
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
    .Mcu_ClockSourceId = MCU_CLKSRC_3,
    .Mcu_ClockDiv = 3,
    .Mcu_InitCfg  = TRUE,
    }   
};

static Mcu_ConfigType        gAdcAppMcuConfig =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = (Mcu_RamConfigPtrType) NULL_PTR,
    .Mcu_NumberOfRamSectors  = 0,
    .Mcu_ClockConfig         = (Mcu_ClockConfigPtrType)&AdcMcu_ClkConfig,
    .Mcu_NumberOfClockConfig = ARRAYSIZE(AdcMcu_ClkConfig),
};

AdcAppFxn_t                  AdcAppFxnTbl[] =
{
    &AdcApp_mainTest
};

uint32              gTestPassed = E_OK;
/**< Number of times App runs */
Adc_ValueGroupType                Adc_AppBuffer[ADC_MAX_GROUP][ADC_APP_BUF_SIZE_WORD];
/**< SetUp Result Buffer passed to driver */
Adc_ValueGroupType                Adc_AppReadBuffer[ADC_MAX_GROUP][ADC_APP_BUF_SIZE_WORD];
/**< Output Result Buffer */
volatile uint32                   Adc_AppRdBufCount[ADC_MAX_GROUP];
/**< No of Samples Counter */
volatile uint32                   Adc_AppGroupDone[ADC_MAX_GROUP];
/**< Group Conversion Completion flag */
extern const Adc_ConfigType *Adc_ConfigPtr;

volatile uint32     gAdcAppGroupDone[ADC_MAX_GROUP];
uint32 Adc_AppLoopCount = ADC_APP_LOOP_COUNT;

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
#if defined CLANG

void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void  SchM_Enter_Adc_ADC_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Adc_ADC_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void  SchM_Enter_Cdd_Dma_DMA_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Cdd_Dma_DMA_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}


#endif

int main(void)
{
    uint32 i;
                  
    AppUtils_defaultInit();
	AppUtils_sectionInit();

    /*Enable Domain clocks*/
    AdcApp_platformInit();

    /* Enable time base clock for the selected ePWM */
    //SOC_setEpwmTbClk(0, TRUE);
    //SOC_setEpwmGroup(0, 0);
	
    /* Setup Interrupt. */ 
    AdcApp_interruptInit();

#if (ADC_APPL_TEST == STD_ON)
    /* Sample Application to Test ADC module. */
    AppUtils_printf(ADC_APP_NAME ": Sample Application - Initiated !!!\n\r");
    for (i = 0U; i < ARRAYSIZE(AdcAppFxnTbl); i++)
    {
        AdcAppFxnTbl[i]();
    }    
#endif

    GT_1trace(McalAppTrace, GT_INFO,
              " ADC Stack Usage: %d bytes\n\r", AppUtils_getStackUsage());
    if (AppUtils_checkStackAndSectionCorruption() != E_OK)
    {
        gTestPassed = E_NOT_OK;
        GT_0trace(McalAppTrace, GT_ERR, " ADC Stack/section corruption!!!\n\r");
    }
	AppUtils_TimerDeinit();
	
	AppUtils_printf(ADC_APP_NAME ": Sample Application - Completed. !!!\n\r");

    return (0);
}


static void AdcApp_mainTest(void)
{
    uint32              grpIdx;
    uint32              testPassed = E_OK;
    Adc_StatusType      status;
    Std_ReturnType      retVal;
    char option;
    uint32 isPending;
    uint32 loopCount = 0U;
    uint32 loopCnt = 10U;
    uint16 data;
    const Adc_ConfigType *Adc_ConfigPtr = &AdcConfigSet;
    
#if (ADC_VERSION_INFO_API == STD_ON)
    Std_VersionInfoType versioninfo;
    Adc_GetVersionInfo(&versioninfo);
    AppUtils_printf("ADC MCAL Version Info\n\r");
    AppUtils_printf("---------------------\n\r");
    AppUtils_printf("Vendor ID           : %d\n\r", versioninfo.vendorID);
    AppUtils_printf("Module ID           : %d\n\r", versioninfo.moduleID);
    AppUtils_printf("SW Major Version    : %d\n\r", versioninfo.sw_major_version);
    AppUtils_printf("SW Minor Version    : %d\n\r", versioninfo.sw_minor_version);
    AppUtils_printf("SW Patch Version    : %d\n\r", versioninfo.sw_patch_version);
    AppUtils_printf(" \n\r");
#endif  /* #if (ADC_VERSION_INFO_API == STD_ON) */
    
    /* Initilize the timer. */
	AppUtils_TimerInit();

#if (STD_ON == ADC_DMA_MODE)	
	Cdd_Dma_Init(NULL_PTR);
#endif	
	/* Initialize the ADC module. */
    Adc_Init(Adc_ConfigPtr);

    /* Set the Buffer of ADC conversion. */
    for (grpIdx = 0U; grpIdx < Adc_ConfigPtr->maxGroup; grpIdx++)
    {
        /* Reset read counter and buffer content */
        Adc_AppRdBufCount[grpIdx] = 0U;
        memset(
            &Adc_AppReadBuffer[grpIdx][0],
            0U,
            (sizeof (Adc_ValueGroupType) * ADC_APP_READ_BUF_SIZE_WORD));

        /* Check group status - it should be idle */
        status = Adc_GetGroupStatus(grpIdx);
        if (status != ADC_IDLE)
        {
            testPassed = E_NOT_OK;
            AppUtils_printf(ADC_APP_NAME, " ADC Group %d is not IDLE!!\n\r", grpIdx);
        }

        /* Memset result buffer and give to driver */
        memset(
            &Adc_AppBuffer[grpIdx][0],
            0U,
            (sizeof (Adc_ValueGroupType) * ADC_APP_BUF_SIZE_WORD));
        retVal = Adc_SetupResultBuffer(grpIdx, &Adc_AppBuffer[grpIdx][0]);
        
        if (retVal != E_OK)
        {
            testPassed = E_NOT_OK;
            AppUtils_printf(ADC_APP_NAME, " ADC Group %d setup buffer failed!!\n\r",
                grpIdx);
        }
     
        #if (ADC_GRP_NOTIF_CAPABILITY_API == STD_ON)
        /* Enable notification */
        Adc_EnableGroupNotification(grpIdx);
        #endif      /* #if (ADC_GRP_NOTIF_CAPABILITY_API == STD_ON) */
        
    }
    
    /* ****************************************************************************************** */    
    /* Hardware Triggered Group Testing. */
    AppUtils_printf( " Hardware Triggered Group Conversion started. !!!\r\n");
    AppUtils_printf( ADC_APP_NAME ": Hardware Triggered Group Notifiation count is : %d :)!!!\n\r", 
                              NotificationCount[AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_2]);
    
    /* Enable Hardware Triggered Group for ADC. */
    Epwm_Enable(0x50000000ul);
	Epwm_Enable(0x50001000ul);
	
    Adc_EnableHardwareTrigger(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_2);    
    Adc_EnableHardwareTrigger(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_3);    
    Adc_EnableHardwareTrigger(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_4);   
	
    AppUtils_printf( " Wait for 10 seconds, Trigger is not applied. !!!\r\n");
    AppUtils_delay(10000U);
       
    /* External Trigger for ADC. */
    AppUtils_printf( " PWM trigger is applied. !!!\r\n");
    Adc_Trigger(0x50000000ul);
	Adc_Trigger(0x50001000ul);
	
    /* Wait for Group Status to complete. */
    while(ADC_BUSY == Adc_GetGroupStatus(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_2))
    {
        ;
    }
	
	/* Wait for Group Status to complete. */
    while(ADC_BUSY == Adc_GetGroupStatus(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_3))
    {
        ;
    }
	
	/* Wait for Group Status to complete. */
    while(ADC_BUSY == Adc_GetGroupStatus(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_4))
    {
        ;
    }
		
    EPWM_disableADCTrigger(0x50000000ul, EPWM_SOC_A);
    EPWM_disableADCTrigger(0x50000000ul, EPWM_SOC_B);
    EPWM_disableADCTrigger(0x50001000ul, EPWM_SOC_A);
	
    AppUtils_delay(10000U);
    AppUtils_printf( ADC_APP_NAME ": Hardware Triggered Group Notifiation Group 2: %d :)!!!\n\r", 
                              NotificationCount[AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_2]);    
    
    AppUtils_printf( ADC_APP_NAME ": Hardware Triggered Group Notifiation Group 3: %d :)!!!\n\r", 
                              NotificationCount[AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_3]);
							  
    AppUtils_printf( ADC_APP_NAME ": Hardware Triggered Group Notifiation Group 4: %d :)!!!\n\r", 
                              NotificationCount[AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_4]);    
							  
    /* Clear the counter. */
    NotificationCount[AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_2] = 0;
	
    return;
}

static void AdcApp_platformInit(void)
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;

#if (STD_OFF == MCU_NO_PLL)
    uint32 sys_clk_freq_vclk_const;

    /* MCU PLL Config */
    gAdcAppMcuConfig.Mcu_PllConfig = McuModuleConfiguration.Mcu_PllConfig;
    gAdcAppMcuConfig.Mcu_PllSourceId = McuModuleConfiguration.Mcu_PllSourceId;
#endif

    Mcu_Init(&McuModuleConfiguration);
    Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);
    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);
   
    #if (STD_OFF == MCU_NO_PLL)
    if(McuModuleConfiguration.Mcu_PllSourceId == MCU_CLKSRC_DPLL)
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.
        Mcu_PllConfig[MCU_CLKSRC_DPLL].Mcu_PllClk1.MCU_PLL_HSDIV2 / (2e6)) * 30;
    }
    else
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.
        Mcu_PllConfig[MCU_CLKSRC_APLL].Mcu_PllClk1.MCU_PLL_HSDIV0 / (2e6)) * 30;
    }
    Enable_Uart();
    #else
    Enable_Uart();
    #endif

    return;
}

static void AdcApp_interruptConfig(void)
{
    Vim_IntCfg intCfg;
    intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_PULSE;
    
    intCfg.intNum = ADC_CORE0_CONTROLSS_INTRXBAR0_OUT_2;
    intCfg.handler = Adc_ADCINT1_IrqUnit0;
    intCfg.priority = VIM_PRIORITY_15;
    vimRegisterInterrupt(&intCfg);
	
	intCfg.intNum = ADC_CORE0_CONTROLSS_INTRXBAR0_OUT_3;
    intCfg.handler = Adc_ADCINT2_IrqUnit0;
    intCfg.priority = VIM_PRIORITY_14;
    vimRegisterInterrupt(&intCfg);
	
	intCfg.intNum = ADC_CORE0_CONTROLSS_INTRXBAR0_OUT_4;
    intCfg.handler = Adc_ADCINT3_IrqUnit0;
    intCfg.priority = VIM_PRIORITY_13;
    vimRegisterInterrupt(&intCfg);
	
    /*
    intCfg.intNum = ADC_CORE0_CONTROLSS_INTRXBAR0_OUT_2;
    intCfg.handler = Adc_ADCINT2_IrqUnit1;
    intCfg.priority = VIM_PRIORITY_14;
    vimRegisterInterrupt(&intCfg);
    
    intCfg.intNum = ADC_CORE0_CONTROLSS_INTRXBAR0_OUT_3;
    intCfg.handler = Adc_ADCINT3_IrqUnit1;
    intCfg.priority = VIM_PRIORITY_13;
    vimRegisterInterrupt(&intCfg);
    
    intCfg.intNum = ADC_CORE0_CONTROLSS_INTRXBAR0_OUT_4;
    intCfg.handler = Adc_ADCINT4_IrqUnit1;
    intCfg.priority = VIM_PRIORITY_12;
    vimRegisterInterrupt(&intCfg);
	
#if (STD_ON == ADC_DMA_MODE)
    intCfg.type = VIM_INTTRIGTYPE_PULSE;
    intCfg.intNum = 72;
    intCfg.handler = Adc_DMA_IrqChannel;
    intCfg.priority = VIM_PRIORITY_4;
    vimRegisterInterrupt(&intCfg);
#endif
*/
}

static void AdcApp_interruptInit(void)
{
    vimInit();
    AdcApp_interruptConfig();
}

void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        assert(FALSE);
    }
}


void AdcApp_HwUnit0_Group0EndNotification(void)
{
    NotificationCount[ADC_GROUP_ID_0]++;
    AppUtils_printf( ADC_APP_NAME ": Group 0 Channel 0 Conversion Completed. Notification count is  %d :)!!!\n\r", NotificationCount[ADC_GROUP_ID_0]);
    AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_0][0U]), Adc_AppBuffer[ADC_GROUP_ID_0][0U]);
    AdcApp_Notification(ADC_GROUP_ID_0);
}

void AdcApp_HwUnit0_Group1EndNotification(void)
{
    NotificationCount[ADC_GROUP_ID_1]++;
    AppUtils_printf( ADC_APP_NAME ": Group 1 Channel 0 Conversion Completed. Notification count is  %d :)!!!\n\r", NotificationCount[ADC_GROUP_ID_1]);
    AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_1][0U]), Adc_AppBuffer[ADC_GROUP_ID_1][0U]);
	AdcApp_Notification(ADC_GROUP_ID_1);
    
    /* Check the Notification. */
    if(NotificationCount[ADC_GROUP_ID_1] == 100)
    {
       #if (ADC_ENABLE_START_STOP_GROUP_API == STD_ON)
       /* Start Conversion for ADC module. */
       Adc_StopGroupConversion(AdcConf_AdcHwUnit_AdcHwUnit_0_AdcGroup_AdcGroup_1);
       #endif      /* #if (ADC_ENABLE_START_STOP_GROUP_API == STD_ON) */
    }
}

void AdcApp_HwUnit0_Group2EndNotification(void)
{
    NotificationCount[ADC_GROUP_ID_2]++;
	(void)Adc_ReadGroup(ADC_GROUP_ID_2, &Adc_AppReadBuffer[ADC_GROUP_ID_2][0]);
    AppUtils_printf( ADC_APP_NAME ": Group 2 Channel 0 , 1, 2 Conversion Completed. Notification count is  %d :)!!!\n\r", NotificationCount[ADC_GROUP_ID_2]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_2][0U]), Adc_AppBuffer[ADC_GROUP_ID_2][0U]);
	AdcApp_Notification(ADC_GROUP_ID_2);
}

void AdcApp_HwUnit0_Group3EndNotification(void)
{
    NotificationCount[ADC_GROUP_ID_3]++;
	(void)Adc_ReadGroup(ADC_GROUP_ID_3, &Adc_AppReadBuffer[ADC_GROUP_ID_3][0]);
    AppUtils_printf( ADC_APP_NAME ": Group 3 Channel 0 Conversion Completed. Notification count is  %d :)!!!\n\r", NotificationCount[ADC_GROUP_ID_3]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_3][0U]), Adc_AppBuffer[ADC_GROUP_ID_3][0U]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_3][0U]), Adc_AppBuffer[ADC_GROUP_ID_3][0U]);
    AdcApp_Notification(ADC_GROUP_ID_3);
}

void AdcApp_HwUnit0_Group4EndNotification(void)
{
    NotificationCount[ADC_GROUP_ID_4]++;
	(void)Adc_ReadGroup(ADC_GROUP_ID_4, &Adc_AppReadBuffer[ADC_GROUP_ID_4][0]);
    AppUtils_printf( ADC_APP_NAME ": Group 4 Channel 0 Conversion Completed. Notification count is  %d :)!!!\n\r", NotificationCount[ADC_GROUP_ID_4]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_3][0U]), Adc_AppBuffer[ADC_GROUP_ID_3][0U]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_3][0U]), Adc_AppBuffer[ADC_GROUP_ID_3][0U]);
    AdcApp_Notification(ADC_GROUP_ID_4);
}

void AdcApp_HwUnit1_Group5EndNotification(void)
{
    NotificationCount[ADC_GROUP_ID_5]++;
    AppUtils_printf( ADC_APP_NAME ": Group 1 Channel 0 Conversion Completed. Notification count is  %d :)!!!\n\r", NotificationCount[ADC_GROUP_ID_5]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_5][0U]), Adc_AppBuffer[ADC_GROUP_ID_5][0U]);
    //AppUtils_printf( ADC_APP_NAME ": Digital Value for input volatge %dmv is %d:)!!!\n\r", AppUtils_GetAdcVoltage(Adc_AppBuffer[ADC_GROUP_ID_5][0U]), Adc_AppBuffer[ADC_GROUP_ID_5][0U]);
    AdcApp_Notification(ADC_GROUP_ID_5);
}

static void AdcApp_Notification(uint32 grpIdx)
{   
    /* Indication as all Groups are completed. */
    Adc_AppGroupDone[grpIdx] = TRUE;
    
    return;
}

/* ===============================================================================================*/
/*                          Function Definitions  to Initialize the ADC Module.                   */
/* ===============================================================================================*/
static void SOC_controlModuleLockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;
    
    
    if(partition==TOP_CTRL_PARTITION0)
    {
        /*Lock TOP_CTRL*/
        baseAddr = (uint32) CSL_TOP_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_TOP_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_TOP_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 1 */
    }
    
    if(partition==CONTROLSS_CTRL_PARTITION0)
    {
        /*Lock CONTROLSS_CTRL*/
        baseAddr = (uint32) CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 1 */
    }

    return;

    return;
}

static void SOC_controlModuleUnlockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;

    
    if(partition==TOP_CTRL_PARTITION0)
    {
        /*Unlock TOP_CTRL*/
        baseAddr = (uint32) CSL_TOP_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_TOP_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK0_UNLOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_TOP_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK1_UNLOCK_VAL);      /* KICK 1 */
    }
    
    if(partition==CONTROLSS_CTRL_PARTITION0)
    {
        /*Unlock CONTROLSS_CTRL*/
        baseAddr = (uint32) CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK0_UNLOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK1_UNLOCK_VAL);      /* KICK 1 */
    }


    return;
}

static void SOC_setEpwmTbClk(uint32 epwmInstance, uint32 enable)
{
    if(epwmInstance < CSL_ETPWM_PER_CNT)
    {
        /* Time base clock enable register belongs to partition 1 of the CTRL MMR */

        /* Unlock CONTROLSS_CTRL registers */
        SOC_controlModuleUnlockMMR(0, CONTROLSS_CTRL_PARTITION0);

        if(TRUE == enable)
        {
            /* Enable Time base clock in CTRL MMR */
            HW_WR_REG32(CSL_CONTROLSS_CTRL_U_BASE + CSL_CONTROLSS_CTRL_EPWM_CLKSYNC,
                ((HW_RD_REG32(CSL_CONTROLSS_CTRL_U_BASE +
                  CSL_CONTROLSS_CTRL_EPWM_CLKSYNC) & CSL_CONTROLSS_CTRL_EPWM_CLKSYNC_BIT_MASK) | (1 << epwmInstance)));
        }
        else
        {
            /* Disable Time base clock in CTRL MMR */
            HW_WR_REG32(CSL_CONTROLSS_CTRL_U_BASE + CSL_CONTROLSS_CTRL_EPWM_CLKSYNC,
                ((HW_RD_REG32(CSL_CONTROLSS_CTRL_U_BASE +
                  CSL_CONTROLSS_CTRL_EPWM_CLKSYNC) & CSL_CONTROLSS_CTRL_EPWM_CLKSYNC_BIT_MASK) & ~(1 << epwmInstance)));
        }

        /* Lock CONTROLSS_CTRL registers */
        SOC_controlModuleLockMMR(0, CONTROLSS_CTRL_PARTITION0);
    }
}

static void SOC_setEpwmGroup(uint32 epwmInstance, uint32 group)
{
    uint32 baseAddr = CSL_CONTROLSS_CTRL_U_BASE + CSL_CONTROLSS_CTRL_EPWM_STATICXBAR_SEL0;
    uint32 mask, shift;

    /* Unlock CONTROLSS_CTRL registers */
    SOC_controlModuleUnlockMMR(0, CONTROLSS_CTRL_PARTITION0);

    /* Choose the correct base address depending on which ePWM instance is selected*/
    if(epwmInstance > 15)
    {
        baseAddr = baseAddr + 0x4;
        epwmInstance = epwmInstance - 16;
    }

    shift = (epwmInstance << 1);
    /* Create the mask to be written to register */
    mask = (0x3 << shift);

    /* Configure the group for the ePWM instance */
    HW_WR_REG32(baseAddr, (( HW_RD_REG32(baseAddr) & ~mask) | (group <<shift)));

    /* Lock CONTROLSS_CTRL registers */
    SOC_controlModuleLockMMR(0, CONTROLSS_CTRL_PARTITION0);
}

#if 0
static void Epwm_Enable(void)
{
    /* CONFIG_EPWM0 initialization */
    uint32 CONFIG_EPWM0_BASE_ADDR = 0x50000000ul;    
    
    /* Time Base */
    EPWM_setClockPrescaler(CONFIG_EPWM0_BASE_ADDR, EPWM_CLOCK_DIVIDER_4, EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(CONFIG_EPWM0_BASE_ADDR, 2000);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_TBPRD_TBPRDHR);
    EPWM_setPeriodLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_PERIOD_SHADOW_LOAD);
    EPWM_setTimeBaseCounter(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_setTimeBaseCounterMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_MODE_STOP_FREEZE);
    EPWM_setCountModeAfterSync(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNT_MODE_DOWN_AFTER_SYNC);
    EPWM_disablePhaseShiftLoad(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setPhaseShift(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_enableSyncOutPulseSource(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_setSyncInPulseSource(CONFIG_EPWM0_BASE_ADDR, EPWM_SYNC_IN_PULSE_SRC_DISABLE);
    EPWM_setOneShotSyncOutTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_OSHT_SYNC_OUT_TRIG_SYNC);
    

    /* Counter Compare */
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_A, 1000);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPA_CMPAHR);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_B, 0);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPB_CMPBHR);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_C, 0);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPC);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_C, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_D, 0);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPD);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_D, EPWM_COMP_LOAD_ON_CNTR_ZERO);

    /* Action Qualifier */
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_AQCSFRC);
    EPWM_setActionQualifierContSWForceShadowMode(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_AQCTLA_AQCTLA2);
    EPWM_disableActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_A);
    EPWM_setActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);
    EPWM_setActionQualifierT1TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierT2TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierSWAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE);
    EPWM_setActionQualifierContSWForceAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_SW_DISABLED);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_AQCTLB_AQCTLB2);
    EPWM_disableActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_B);
    EPWM_setActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);
    EPWM_setActionQualifierT1TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierT2TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierSWAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE);
    EPWM_setActionQualifierContSWForceAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_SW_DISABLED);

    /* Events */
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN);

    /* Event Trigger */
    EPWM_disableInterrupt(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setInterruptSource(CONFIG_EPWM0_BASE_ADDR, EPWM_INT_TBCTR_ZERO, EPWM_INT_TBCTR_ZERO);
    EPWM_setInterruptEventCount(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_disableInterruptEventCountInit(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setInterruptEventCountInitValue(CONFIG_EPWM0_BASE_ADDR, 0);
    
    EPWM_enableADCTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A);
    EPWM_setADCTriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A, EPWM_SOC_TBCTR_U_CMPA, EPWM_SOC_TBCTR_U_CMPA);
    EPWM_setADCTriggerEventPrescale(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A, 1);
    EPWM_disableADCTriggerEventCountInit(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A);
    EPWM_setADCTriggerEventCountInitValue(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A, 0);
    
    EPWM_disableADCTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B);
    EPWM_setADCTriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B, EPWM_SOC_DCxEVT1, EPWM_SOC_DCxEVT1);
    EPWM_setADCTriggerEventPrescale(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B, 0);
    EPWM_disableADCTriggerEventCountInit(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B);
    EPWM_setADCTriggerEventCountInitValue(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B, 0);

    /* Global Load */
    EPWM_disableGlobalLoad(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setGlobalLoadTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_LOAD_PULSE_CNTR_ZERO);
    EPWM_setGlobalLoadEventPrescale(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_disableGlobalLoadOneShotMode(CONFIG_EPWM0_BASE_ADDR);
    
    /* EPWM Module */
    EPWM_lockRegisters(CONFIG_EPWM0_BASE_ADDR, 0);
}
#endif

static uint32 AppUtils_GetAdcVoltage(uint32 adcValue)
{
    uint32 refVoltage    = 3300U;
    uint32 maxResolution = 4095U;

    return ((adcValue * refVoltage) / maxResolution);
}

void Adc_Trigger(uint32 baseAddr)
{
    //baseAddr = 0x50000000ul;    
    EPWM_setTimeBaseCounterMode(baseAddr, EPWM_COUNTER_MODE_UP);
}

void Adc_DisableTrigger(uint32 baseAddr)
{
   //baseAddr = 0x50000000ul;
   EPWM_disableADCTrigger(baseAddr, EPWM_SOC_A);
   EPWM_disableADCTrigger(baseAddr, EPWM_SOC_B);
   EPWM_setTimeBaseCounterMode(baseAddr, EPWM_COUNTER_MODE_STOP_FREEZE);
}

void Epwm_Enable(uint32 baseAddr)
{
    /* CONFIG_EPWM0 initialization */
    uint32 CONFIG_EPWM0_BASE_ADDR = 0x50000000ul;    
    CONFIG_EPWM0_BASE_ADDR = baseAddr;    
    
    /* Time Base */
    EPWM_setClockPrescaler(CONFIG_EPWM0_BASE_ADDR, EPWM_CLOCK_DIVIDER_4, EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(CONFIG_EPWM0_BASE_ADDR, 25000);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_TBPRD_TBPRDHR);
    EPWM_setPeriodLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_PERIOD_SHADOW_LOAD);
    EPWM_setTimeBaseCounter(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_setTimeBaseCounterMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_MODE_STOP_FREEZE);
    EPWM_setCountModeAfterSync(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNT_MODE_DOWN_AFTER_SYNC);
    EPWM_disablePhaseShiftLoad(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setPhaseShift(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_enableSyncOutPulseSource(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_setSyncInPulseSource(CONFIG_EPWM0_BASE_ADDR, EPWM_SYNC_IN_PULSE_SRC_DISABLE);
    EPWM_setOneShotSyncOutTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_OSHT_SYNC_OUT_TRIG_SYNC);
    

    /* Counter Compare */
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_A, 12500);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPA_CMPAHR);
	
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_B, 12500);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPB_CMPBHR);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_C, 0);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPC);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_C, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareValue(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_D, 0);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_CMPD);
    
    EPWM_setCounterCompareShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_COUNTER_COMPARE_D, EPWM_COMP_LOAD_ON_CNTR_ZERO);

    /* Action Qualifier */
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_AQCSFRC);
    EPWM_setActionQualifierContSWForceShadowMode(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_AQCTLA_AQCTLA2);
    EPWM_disableActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_A);
    EPWM_setActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_A, EPWM_AQ_LOAD_ON_CNTR_ZERO);
    EPWM_setActionQualifierT1TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierT2TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierSWAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE);
    EPWM_setActionQualifierContSWForceAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_SW_DISABLED);
    EPWM_disableGlobalLoadRegisters(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_REGISTER_AQCTLB_AQCTLB2);
    EPWM_disableActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_B);
    EPWM_setActionQualifierShadowLoadMode(CONFIG_EPWM0_BASE_ADDR, EPWM_ACTION_QUALIFIER_B, EPWM_AQ_LOAD_ON_CNTR_ZERO);
    EPWM_setActionQualifierT1TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierT2TriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1);
    EPWM_setActionQualifierSWAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE);
    EPWM_setActionQualifierContSWForceAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_SW_DISABLED);

    /* Events */
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_UP);
    EPWM_setActionQualifierAction(CONFIG_EPWM0_BASE_ADDR, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN);

    /* Event Trigger */
    EPWM_disableInterrupt(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setInterruptSource(CONFIG_EPWM0_BASE_ADDR, EPWM_INT_TBCTR_ZERO, EPWM_INT_TBCTR_ZERO);
    EPWM_setInterruptEventCount(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_disableInterruptEventCountInit(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setInterruptEventCountInitValue(CONFIG_EPWM0_BASE_ADDR, 0);
    
    EPWM_enableADCTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A);
    EPWM_setADCTriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A, EPWM_SOC_TBCTR_U_CMPA, EPWM_SOC_TBCTR_U_CMPA);
    EPWM_setADCTriggerEventPrescale(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A, 1);
    EPWM_disableADCTriggerEventCountInit(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A);
    EPWM_setADCTriggerEventCountInitValue(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_A, 0);  
	
	EPWM_enableADCTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B);
    EPWM_setADCTriggerSource(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B, EPWM_SOC_TBCTR_U_CMPB, EPWM_SOC_TBCTR_U_CMPB);
    EPWM_setADCTriggerEventPrescale(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B, 1);
    EPWM_disableADCTriggerEventCountInit(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B);
    EPWM_setADCTriggerEventCountInitValue(CONFIG_EPWM0_BASE_ADDR, EPWM_SOC_B, 0);
	
    /* Global Load */
    EPWM_disableGlobalLoad(CONFIG_EPWM0_BASE_ADDR);
    EPWM_setGlobalLoadTrigger(CONFIG_EPWM0_BASE_ADDR, EPWM_GL_LOAD_PULSE_CNTR_ZERO);
    EPWM_setGlobalLoadEventPrescale(CONFIG_EPWM0_BASE_ADDR, 0);
    EPWM_disableGlobalLoadOneShotMode(CONFIG_EPWM0_BASE_ADDR);
    
    /* EPWM Module */
    EPWM_lockRegisters(CONFIG_EPWM0_BASE_ADDR, 0);
}

/*****************************************EoF******************************************************/
