/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     DIOExample.c
 *
 *  \brief    This file contains the DIO test example
 */

/**********************************************************************************************************************
 *  INCLUDES
 *********************************************************************************************************************/

/* Include Files */
#include "assert.h"
/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
/*LDRA_ANALYSIS*/
#include "Det.h"
#include "Mcu.h"
#include "EcuM_Cbk.h"
#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port.h"
#include "Dio.h"
#include "Os.h"

#include "app_utils.h"
#include "sys_vim.h"
#include "hw_ctrl_core.h"
/* Starterware Includes */
//#include "interrupt.h"

#define DIO_APP_PERFORMANCE_MODE   STD_ON

#if (STD_ON == DIO_APP_PERFORMANCE_MODE)
static void Dio_Example_PerformanceTest();
uint32 Dio_Performance_Result[30];
#endif


uint8 Dio_MainFunc_Execution;

#ifdef __cplusplus
extern "C" {
#endif

void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();


}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

#ifdef __cplusplus
}
#endif

#define DIO_EXAMPLE_ARRAYSIZE(x)  (sizeof ((x)) / sizeof (x[0]))

/** @fn void main(void)
 *   @brief Application main function
 *   @note This function is empty by default.
 *
 *   This function is called after startup.
 *   The user can use this function to implement the application.
 */
#if (STD_ON == DIO_VERSION_INFO_API)
Std_VersionInfoType       Dio_VersionInfo;
#endif

sint32 Dio_TestState = TRUE;

#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif

Dio_LevelType Dio_PinLevel[64];
Dio_PortLevelType         Dio_PortLevel[4];
Dio_ChannelGroupType      Dio_ChGroup[2];


#if (STD_ON == DIO_APP_PERFORMANCE_MODE)
static void Dio_Example_PerformanceTest()
{
    Std_VersionInfoType versioninfo;
    TickType StartValue = 0, ElapsedValue = 0;

    GetCounterValue(0,&StartValue);
    Dio_ReadChannel(DioConf_DioChannel_GPIOGH_Ch24);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_READ_CHANNEL] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_WriteChannel(DioConf_DioChannel_GPIOGH_Ch24, STD_LOW);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_WRITE_CHANNEL] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_ReadPort(DioConf_DioPort_GPIOGH);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_READ_PORT] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_WritePort(DioConf_DioPort_GPIOGH, Dio_PortLevel[0]);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_WRITE_PORT] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_PortLevel[0] = Dio_ReadChannelGroup(&Dio_ChGroup[0]);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_READ_CHL_GROUP] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_WriteChannelGroup(&Dio_ChGroup[0], Dio_PortLevel[0]);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_WRITE_CHL_GROUP] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_FlipChannel(DioConf_DioChannel_GPIOGH_Ch24);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_FLIP_CHANNEL] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Dio_GetVersionInfo(&versioninfo);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Dio_Performance_Result[DIO_SID_GET_VERSION_INFO] = ElapsedValue;

}
#endif

static void Dio_Example_PlatformInit()
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;
    uint32 sys_clk_freq_vclk_const;

    Mcu_Init(&McuModuleConfiguration);

#if (STD_ON == PORT_PRE_COMPILE_VARIANT)
    AppUtils_printf("Pre-Compile variant is being used..\n\r");
    Port_Init((const Port_ConfigType *) NULL_PTR);
#else
    AppUtils_printf("Post-Build variant is being used..\n\r");
    Port_Init(&PortConfigSet_0);
#endif

    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;

    /*Set up the pinmux for UART tx */
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);

    /*Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);
    vimInit();
    
#if (STD_OFF == MCU_NO_PLL)
    if(McuModuleConfiguration.Mcu_PllSourceId == MCU_CLKSRC_DPLL)
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_DPLL].Mcu_PllClk1.MCU_PLL_HSDIV2 / (2e6)) * 30;
    }
    else
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_APLL].Mcu_PllClk1.MCU_PLL_HSDIV0 / (2e6)) * 30;
    }
    Enable_Uart();
#else
    Enable_Uart();
#endif

}

int32_t main(void)
{
    uint32_least i;
    Std_ReturnType      retVal = E_OK;
    AppUtils_defaultInit();
    AppUtils_sectionInit();
    AppUtils_TimerInit();

    Dio_Example_PlatformInit();

#if (STD_ON == DIO_APP_PERFORMANCE_MODE)
    Dio_Example_PerformanceTest();
#endif

#if DIAB
    AppUtils_printf("DIAB compiled : DioApp: Sample Application - STARTS !!!\n\r");
#elif CLANG
    AppUtils_printf("CLANG compiled : DioApp: Sample Application - STARTS !!!\n\r");
#else
    AppUtils_printf("DioApp: Sample Application - STARTS !!!\n\r");    
#endif    


#if (STD_ON == DIO_VERSION_INFO_API)
    Dio_GetVersionInfo(&Dio_VersionInfo);
    AppUtils_printf("DIO Driver version info:%d.%d.%d\n\r",
                    Dio_VersionInfo.sw_major_version,
                    Dio_VersionInfo.sw_minor_version,
                    Dio_VersionInfo.sw_patch_version);
    AppUtils_printf("DIO Driver Module/Driver:%d.%d \n\r",
                    Dio_VersionInfo.moduleID,
                    Dio_VersionInfo.vendorID);
#endif


    Dio_PinLevel[2] = Dio_ReadChannel(120);
    AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[2]);

    Port_SetPinDirection(PortConf_PortPin_PortPin_0, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_1, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_2, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_3, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_4, PORT_PIN_OUT);



    Dio_PinLevel[0] = Dio_ReadChannel(122);
    AppUtils_printf("Port Value for GPIOGH channel 2 :%x \n\r",Dio_PinLevel[0]);

	if (Dio_PinLevel[0] == STD_HIGH)
	{
    	Dio_FlipChannel(122);
    	AppUtils_printf("Dio_FlipChannel - GPIOGH\n\r");

    	Dio_PinLevel[0] = Dio_ReadChannel(122);
    	AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[0]);
	}
	else
	{
		retVal = E_NOT_OK;
	}
	if (Dio_PinLevel[0] == STD_LOW && retVal ==E_OK)
	{
		Dio_FlipChannel(122);
    	AppUtils_printf("Dio_FlipChannel - GPIOGH\n\r");

    	Dio_PinLevel[0] = Dio_ReadChannel(122);
    	AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[0]);
	}
	else
	{
		retVal = E_NOT_OK;
	}
    Dio_PinLevel[1] = Dio_ReadChannel(120);
    AppUtils_printf("Port Value for GPIOGH :%x \n\r",Dio_PinLevel[1]);
	if (Dio_PinLevel[1] ==STD_HIGH && retVal ==E_OK)
	{	

    	Dio_FlipChannel(120);
    	AppUtils_printf("Dio_FlipChannel - GPIOGH \n\r");

    	Dio_PinLevel[1] = Dio_ReadChannel(120);
    	AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[1]);

	}
	else
	{
		retVal = E_NOT_OK;
	}

	if (Dio_PinLevel[1] ==STD_LOW && retVal ==E_OK)
	{
    	Dio_FlipChannel(120);
    	AppUtils_printf("Dio_FlipChannel -GPIOGH\n\r");

    	Dio_PinLevel[1] = Dio_ReadChannel(120);
    	AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[1]);
	}
	else
	{
		retVal = E_NOT_OK;
	}
	
    Dio_WriteChannel(120, STD_LOW);
	AppUtils_printf("Dio_WriteChannel -GPIOGH\n\r");

    Dio_PinLevel[2] = Dio_ReadChannel(120);
	AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[2]);
	if (Dio_PinLevel[2] ==STD_LOW && retVal ==E_OK)
	{
		retVal = E_OK;
	}
    else
	{
		retVal = E_NOT_OK;
	}
    
    Dio_WriteChannel(120, STD_HIGH);
	AppUtils_printf("Dio_WriteChannel -GPIOGH\n\r");

    Dio_PinLevel[2] = Dio_ReadChannel(120);
    AppUtils_printf("Port Value:%x \n\r",Dio_PinLevel[2]);
	
	if (Dio_PinLevel[2] ==STD_HIGH && retVal ==E_OK)
	{
		retVal = E_OK;
	}
    else
	{
		retVal = E_NOT_OK;
	}	
	

	Dio_PortLevel[0] = 0x00000000;
	Dio_PortLevel[1] = 0x00000001;

	for (i = 4; i < 5; i++)
	{
	    Dio_WritePort(i, Dio_PortLevel[1]);
	    AppUtils_delay(50U);
	    Dio_PortLevel[1] = Dio_ReadPort(i);

	    AppUtils_printf("Port Value Before Flip:%x \n\r",Dio_PortLevel[1]);
	    AppUtils_delay(50U);
	    Dio_FlipChannel(128);
	    AppUtils_delay(50U);
	    Dio_PortLevel[1] = Dio_ReadPort(i);
	    AppUtils_delay(50U);

	    Dio_PortLevel[1] = Dio_PortLevel[0] & Dio_PortLevel[1];
	    AppUtils_printf("After Flip Port Value:%x \n\r",Dio_PortLevel[1]);

	    if (Dio_PortLevel[1] == 0x0000000 && retVal ==E_OK)
	    {
	        retVal = E_OK;
	    }
	    else
	    {
	        retVal = E_NOT_OK;
	    }
	}


    for (i = 0; i < 1; i++)
    {
        Dio_ChGroup[i].port   = 4; 
        Dio_ChGroup[i].offset = 0;
        Dio_ChGroup[i].mask   = 0x01;
        Dio_PortLevel[i]      = 0xFF;
        Dio_WriteChannelGroup(&Dio_ChGroup[i], Dio_PortLevel[i]);
		AppUtils_delay(20U);
		Dio_PortLevel[i] = Dio_ReadChannelGroup(&Dio_ChGroup[i]);
		AppUtils_delay(20U);
        AppUtils_printf("Read channel grp Port Value:%x \n\r",Dio_PortLevel[i]);
		if ((Dio_PortLevel[i] & 0x1) ==STD_HIGH && retVal ==E_OK)
		{
			retVal = E_OK;
		}
    	else
		{
			retVal = E_NOT_OK;
		}
    }

    if (AppUtils_checkStackAndSectionCorruption() != E_OK)
    {
        retVal = E_NOT_OK;
        AppUtils_printf(" DIO Stack/section corruption!!!\n\r");
    }

	if(retVal == E_OK)
	{
		AppUtils_printf("DIO Test Passed!!!\n\r");
	}
	else
	{
		AppUtils_printf("DIO Test Failed!!!\n\r");
	}

	AppUtils_TimerDeinit();
    return retVal;
}

#ifdef __cplusplus
extern "C" {
#endif

void EcuM_CheckWakeup(EcuM_WakeupSourceType wakeupSource)
{
    Dio_MainFunc_Execution = 1;
}

Std_ReturnType Det_ReportError(uint16 ModuleId, uint8 InstanceId, uint8 ApiId,
                               uint8 ErrorId)
{
    assert(FALSE);
    return E_OK;
}

void  SchM_Enter_Dio_DIO_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Dio_DIO_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}


void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}


void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        assert(FALSE);
    }

    return;
}

#ifdef __cplusplus
}
#endif
