/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     FlsApp.c
 *
 *  \brief    This file contains the FLS test example
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
/*LDRA_ANALYSIS*/
#include "Det.h"
#include "Dem.h"
#include "Fls.h"
#include "Fls_Irq.h"
#include "Fls_Brd_Nor.h"
#include "EcuM_Cbk.h"
#include "Mcu.h"
#include "Port_Cfg.h"
#include "Port.h"
/*LDRA_NOANALYSIS*/
#include "string.h"
/*LDRA_ANALYSIS*/
#include "trace.h"

/* Starterware Includes */
#include "sys_vim.h"
#include "app_utils.h"
#include "soc.h"

#include "hw_ctrl_core.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define ARRAYSIZE(x)                    (sizeof ((x)) / sizeof (x[0]))
#define APP_NAME                        "FLS_APP"

#define SECTOR_ERASE (1)  /* This macro should be (1) if sector erase application is needed to run */
#define BLOCK_ERASE  (1)  /* This macro should be (1) if block erase application is needed to run */
#define CHIP_ERASE   (1)  /* This macro should be (1) if bulk/chip erase application is needed to run */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

void FlsApp_InterruptConfig(void);
void main_handling_intr(void);
void main_handling(void);
void main_handling_noDbgMsg(void);
void main_handling_intr_noDbgMsg(void);
void FlsApp_eraseCheck(void);
void FlsApp_multiSectorWriteCheck(void);
int fls_sampleapp(void);

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
#define FLSEXAMPLE_ARRAYSIZE(x)  (sizeof ((x)) / sizeof (x[0]))
uint32 Fls_TestPassed;
volatile uint32 Fls_JobDoneSuccess;
volatile uint32 Fls_JobDoneError;
uint16 test_forloop = 0;
extern uint32  Data_Size_Test;
extern uint32 Data_Value;
uint32 offset = 0x000000U;
uint32 type_of_erase;
extern uint32 sector_or_blocksize;
#define DATA_SIZE_TEST 64*1024


#if (STD_ON == FLS_VERSION_INFO_API)
Std_VersionInfoType       VersionInfo;
#endif
#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif

Std_ReturnType retVal = E_OK;
  /* Buffer containing the known data that needs to be written to flash */
  uint8 txBuf_test[DATA_SIZE_TEST]={0};
  /* Buffer containing the received data */
  uint8 rxBuf_test[DATA_SIZE_TEST]={0};
/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

int main(void)
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;
    uint32 sys_clk_freq_vclk_const;
    AppUtils_defaultInit();
    Mcu_Init(&McuModuleConfiguration);
    Port_Init(&PortConfigSet_0);
    /* Default is AM263x*/
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;
    
    /* Set up the pinmux for UART tx */
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);

    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);
	AppUtils_TimerInit();
    /* Initialize interrupt */

    #if (STD_OFF == MCU_NO_PLL)
    if(McuModuleConfiguration.Mcu_PllSourceId == MCU_CLKSRC_DPLL)
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_DPLL].Mcu_PllClk1.MCU_PLL_HSDIV2 / (2e6)) * 30;
    }
    else
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_APLL].Mcu_PllClk1.MCU_PLL_HSDIV0 / (2e6)) * 30;
    }
    Enable_Uart();
    #else
    Enable_Uart();
    #endif
     
    if (SECTOR_ERASE == 1)
    {
        offset = 0x000000U;
       type_of_erase = FLS_SECTOR_ERASE;
       sector_or_blocksize = Fls_SetEraseType(FLS_SECTOR_ERASE);
       fls_sampleapp();
    }
    if (BLOCK_ERASE == 1)
    {
        offset = 0x000000U;
       type_of_erase = FLS_BLOCK_ERASE;
       sector_or_blocksize = Fls_SetEraseType(FLS_BLOCK_ERASE);
       fls_sampleapp();
    }
    if (CHIP_ERASE == 1)
    {
        offset = 0x000000U;
       type_of_erase = FLS_CHIP_ERASE;
       sector_or_blocksize = Fls_SetEraseType(FLS_CHIP_ERASE);
       fls_sampleapp();
    }
	AppUtils_TimerDeinit();
    return (0);

}

int fls_sampleapp(void)
{
    Std_ReturnType job_accepted = E_OK;
    MemIf_JobResultType Result_type;

   AppUtils_printf("FlsApp: Sample Application - STARTS !!!\n\r");

    if(type_of_erase == FLS_SECTOR_ERASE)  
    {                                      
      AppUtils_printf(" FLS SECTOR ERASE !!! \n\r");                   
    }   
    else if(type_of_erase == FLS_BLOCK_ERASE)  
    {
      AppUtils_printf(" FLS BLOCK ERASE !!! \n\r");  
    }           
    else
    {
      AppUtils_printf(" FLS CHIP ERASE !!! \n\r");  
    }    
   
	#if (STD_ON == FLS_USE_INTERRUPTS)
		AppUtils_printf(APP_NAME ": Configuring Interrupt.\n\r");
		FlsApp_InterruptConfig();
	#endif /* #if (STD_ON == FLS_USE_INTERRUPTS) */

	#if (STD_ON == FLS_PRE_COMPILE_VARIANT)
		AppUtils_printf(APP_NAME ": Variant - Pre Compile being used !!!\n\r");
		Fls_Init((const Fls_ConfigType *) NULL_PTR);
	#else
		AppUtils_printf(APP_NAME ": Variant - Post Build being used !!!\n\r");
		Fls_Init(&FlsConfigSet);
	#endif /* #if (STD_ON == FLS_PRE_COMPILE_VARIANT) */
	
#if (STD_ON == FLS_VERSION_INFO_API)
    Fls_GetVersionInfo(&VersionInfo);
    AppUtils_printf("FLS MCAL version info:%d.%d.%d\n\r",
                    VersionInfo.sw_major_version,
                    VersionInfo.sw_minor_version,
                    VersionInfo.sw_patch_version);
    AppUtils_printf("FLS MCAL Module/Driver:%d.%d \n\r",
                    VersionInfo.moduleID,
                    VersionInfo.vendorID);
#endif /* #if (STD_ON == FLS_VERSION_INFO_API) */

/* FLS driver should be free now - check */
#if ( STD_ON == FLS_GET_STATUS_API)
   MemIf_StatusType      status;
   status = Fls_GetStatus();
   if (status != MEMIF_IDLE)
   {
       AppUtils_printf(APP_NAME ": FLS driver is not IDLE!!\n\r");
   }
#endif /* #if ( STD_ON == FLS_GET_STATUS_API) */
   AppUtils_printf(APP_NAME ": DATA SIZE TEST is 0x%X \n\r",  Data_Size_Test);
   AppUtils_printf(APP_NAME ": Offset is 0x%X \n\r", offset);
   
   volatile uint32 idx;
   volatile uint8 *txPtr_test = txBuf_test;
   volatile uint8 *rxPtr_test = rxBuf_test;

   for(idx = 0U; idx <  Data_Size_Test; idx++)
   {
      *txPtr_test++ = Data_Value;
      *rxPtr_test++ = (uint8)0U;
   }

#if (STD_OFF == FLS_USE_INTERRUPTS)
	
   AppUtils_printf(APP_NAME ": Erasing \n\r");
   if(type_of_erase == FLS_CHIP_ERASE)
   {
      AppUtils_printf(APP_NAME ": Erasing full chip..please wait.. \n\r");
   }
   while (1U)
   {
       job_accepted = Fls_Erase(offset,  Data_Size_Test);
       if (E_OK == job_accepted)
               break;
   }
   main_handling();
#if ( STD_ON == FLS_GET_JOB_RESULT_API)
   Result_type = Fls_GetJobResult();
#endif

#if ( STD_ON == FLS_BLANK_CHECK_API)
		AppUtils_printf(APP_NAME ": Blank Checking \n\r");
	while (1U)
	{
       job_accepted = Fls_BlankCheck(offset,  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
	}
	main_handling();
#endif /*STD_ON == FLS_BLANK_CHECK_API*/

   AppUtils_printf(APP_NAME ": Writing \n\r");
   while (1U)
   {
       job_accepted = Fls_Write(offset, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling();

   AppUtils_printf(APP_NAME ": Reading \n\r");
   while (1U)
   {
       job_accepted = Fls_Read(offset, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling();
   
 #if ( STD_ON == FLS_COMPARE_API)
   AppUtils_printf(APP_NAME ": Write Compare \n\r");
   while (1U)
   {
        /*
         Check if the write operation was successful by comparing txBuf_test (hardcoded)
         with the actual data in flash
        */
       job_accepted = Fls_Compare(offset, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling();
   
   AppUtils_printf(APP_NAME ": Read Compare \n\r");
   while (1U)
   {
        /*
         Check if the read operation was successful by comparing rxBuf_test (previously read)
         with the actual data in flash
        */
       job_accepted = Fls_Compare(offset, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling();
   
#endif /*STD_ON == FLS_COMPARE_API*/
#if ( STD_ON == FLS_CANCEL_API)
    AppUtils_printf(APP_NAME ": \n\r");
    AppUtils_printf(APP_NAME ": Job Cancel \n\r");
    AppUtils_printf(APP_NAME ": \t 1. Writing \n\r");
    while (1U)
    {
        job_accepted = Fls_Write(offset, &txBuf_test[0],  Data_Size_Test);
        if (E_OK == job_accepted)
            break;
    }
    AppUtils_printf(APP_NAME ": \t 2. Canceling \n\r");
    Fls_Cancel();
    if(Fls_JobDoneError == 1)
    {
        AppUtils_printf(APP_NAME ": Job Canceled (SUCCESS) ! \n\r");
    }
    else
    {
        AppUtils_printf(APP_NAME ": Job completed (FAILED) !! \n\r");
        while(1){}
    }
    Fls_JobDoneError = 0;
    AppUtils_printf(APP_NAME ": \n\r");
#endif /*STD_ON == FLS_CANCEL_API*/


if((type_of_erase == FLS_SECTOR_ERASE) || (type_of_erase == FLS_BLOCK_ERASE))
{
 #if ( STD_ON == FLS_COMPARE_API)
   FlsApp_multiSectorWriteCheck();
   FlsApp_eraseCheck();
 #endif
}
#endif

   #if (STD_ON == FLS_USE_INTERRUPTS)

    AppUtils_printf(APP_NAME ": Testing Interrupt Mode! \n\r");

    AppUtils_printf(APP_NAME ": Erasing \n\r");
    if(type_of_erase == FLS_CHIP_ERASE)
    {
      AppUtils_printf(APP_NAME ": Erasing full chip..please wait.. \n\r");
    }
    while (1U)
    {
        job_accepted = Fls_Erase(offset,  Data_Size_Test);
        if (E_OK == job_accepted)
                break;
    }
    main_handling_intr();

   #if ( STD_ON == FLS_BLANK_CHECK_API)
		AppUtils_printf(APP_NAME ": Blank Checking \n\r");
	while (1U)
	{
       job_accepted = Fls_BlankCheck(offset,  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
	}
	main_handling_intr();
   #endif /*STD_ON == FLS_BLANK_CHECK_API*/

    AppUtils_printf(APP_NAME ": Writing \n\r");
    while (1U)
    {
        job_accepted = Fls_Write(offset, &txBuf_test[0],  Data_Size_Test);
        if (E_OK == job_accepted)
            break;
    }
	main_handling_intr();
    AppUtils_printf(APP_NAME ": Reading \n\r");
    while (1U)
    {
        job_accepted = Fls_Read(offset, &rxBuf_test[0],  Data_Size_Test);
        if (E_OK == job_accepted)
            break;
    }
    main_handling_intr();

   #if ( STD_ON == FLS_COMPARE_API)
   AppUtils_printf(APP_NAME ": Write Compare \n\r");
   while (1U)
   {
        /*
         Check if the write operation was successful by comparing txBuf_test (hardcoded)
         with the actual data in flash
        */
       job_accepted = Fls_Compare(offset, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_intr();
   
   AppUtils_printf(APP_NAME ": Read Compare \n\r");
   while (1U)
   {
        /*
         Check if the read operation was successful by comparing rxBuf_test (previously read)
         with the actual data in flash
        */
       job_accepted = Fls_Compare(offset, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_intr();
   
#endif /*STD_ON == FLS_COMPARE_API*/

#if ( STD_ON == FLS_CANCEL_API)
    AppUtils_printf(APP_NAME ": \n\r");
    AppUtils_printf(APP_NAME ": Job Cancel \n\r");
     AppUtils_printf(APP_NAME ": Erasing \n\r");
    if(type_of_erase == FLS_CHIP_ERASE)
    {
      AppUtils_printf(APP_NAME ": Erasing full chip..please wait.. \n\r");
    }
    while (1U)
    {
        job_accepted = Fls_Erase(offset,  Data_Size_Test);
        if (E_OK == job_accepted)
                break;
    }
    main_handling_intr(); 

    AppUtils_printf(APP_NAME ": \t 1. Writing \n\r");
    while (1U)
    {
        job_accepted = Fls_Write(offset, &txBuf_test[0],  Data_Size_Test);
        if (E_OK == job_accepted)
            break;
    }
    AppUtils_printf(APP_NAME ": \t 2. Canceling \n\r");
    Fls_Cancel();
    if(Fls_JobDoneError == 1)
    AppUtils_printf(APP_NAME ": Job Canceled (SUCCESS) ! \n\r");
    else
    {
        AppUtils_printf(APP_NAME ": Job completed (FAILED) !! \n\r");
        while(1){}
    }

    AppUtils_printf(APP_NAME ": \n\r");
#endif/*STD_ON == FLS_CANCEL_API*/
#endif/*STD_ON == FLS_USE_INTERRUPTS*/

    AppUtils_printf(APP_NAME ": \n\r");
    AppUtils_printf(APP_NAME ": ---------- FLS Sample application Done !! ----------  \n\r");
    AppUtils_printf(APP_NAME ": \n\r");
    return (0);
}

void FlsApp_multiSectorWriteCheck()
{
    AppUtils_printf(APP_NAME ": \n\r");
    AppUtils_printf(APP_NAME ": -- Multi Sector Write Check -- \n\r");
    AppUtils_printf(APP_NAME ": \n\r");
     
   volatile uint32 idx;
   volatile uint8 *txPtr_test = txBuf_test;
   volatile uint8 *rxPtr_test = rxBuf_test;
   Std_ReturnType job_accepted = E_OK;  

   for(idx = 0U; idx <  Data_Size_Test; idx++)
   {
      *txPtr_test++ = 0xFC;
      *rxPtr_test++ = (uint8)0U;
   }
    if(type_of_erase == FLS_SECTOR_ERASE)  /* For sector erase, make sure offset should be in multiples of 4096 because sector erase do 4096 erase at a time */
    {                                      /* For block erase, make sure offset should be in multiples of 65536 because block erase do 65536 erase at a time */
      offset = 0x1C000;                      /* For chip erase, make sure offset should be 0 */
    }   
    else if(type_of_erase == FLS_BLOCK_ERASE)  
    {
      offset = 0x20000; 
    }           
    else
    {
      offset = 0x0; 
    }    
    AppUtils_printf(APP_NAME ": Performing Erase of 3 consecutive sectors/blocks starting at offset:0X%X\n\r", offset);
    while (1U)
    {
         job_accepted = Fls_Erase(offset, 3*sector_or_blocksize);

        if (E_OK == job_accepted)
                break;
    }
    main_handling_noDbgMsg();

#if ( STD_ON == FLS_BLANK_CHECK_API)    
    AppUtils_printf(APP_NAME ": Blank Checking \n\r");
	while (1U)
	{
         job_accepted = Fls_BlankCheck(offset, 3*sector_or_blocksize);

       if (E_OK == job_accepted)
           break;
	}
	main_handling();
#endif
    
   offset = offset + sector_or_blocksize; /* start address of next sector/block  */      
   AppUtils_printf(APP_NAME ":Writing a large size of length (NOR_SECTOR_SIZE/NOR_BLOCK_SIZE+256) at (address 0X%X) to check \
                             if the write operation crosses over to the next sector/block \r\n",offset);

   while (1U)
   {
        /* Writing a large size of length (NOR_SECTOR_SIZE+256) to check 
        if the write operation crosses over to the next sector */
        /*This is data size is aligned with page alignment*/
         job_accepted = Fls_Write(offset,  &txBuf_test[0], sector_or_blocksize+256);

       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

    offset =  offset + sector_or_blocksize; /* start address of next sector/block */
      
   AppUtils_printf(APP_NAME ": Reading beginning 512 bytes of sec at (address 0X%X) \r\n",offset);
   while (1U)
   {
       job_accepted = Fls_Read(offset, &rxBuf_test[0], 512);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

   /* Checking if 256th byte has been written */
   if(rxBuf_test[0x100 - 1] != 0xFF && rxBuf_test[0] != 0xFF && rxBuf_test[0x101] == 0xFF)
   {
        AppUtils_printf(APP_NAME ": \r\n");
        AppUtils_printf(APP_NAME ": Multi Sector Write Check : SUCCESS \r\n");
        AppUtils_printf(APP_NAME ": Write operation has crossed over to"
        " the next sector when the write length is greater than a single sector size.\r\n");
   }
   else
   {
        AppUtils_printf(APP_NAME ": \r\n");
        AppUtils_printf(APP_NAME ": Multi Sector Write Check : FAILED \r\n");
   }
        
}

void FlsApp_eraseCheck()
{
    AppUtils_printf(APP_NAME ": \n\r");
    if(type_of_erase == FLS_SECTOR_ERASE)
    AppUtils_printf(APP_NAME ": -- Sector Erase Test -- \n\r");
    else if(type_of_erase == FLS_BLOCK_ERASE)
    AppUtils_printf(APP_NAME ": -- Block Erase Test -- \n\r");
    else
    AppUtils_printf(APP_NAME ": -- Chip Erase Test -- \n\r");
    AppUtils_printf(APP_NAME ": \n\r");
    
   volatile uint32 idx;
   volatile uint8 *txPtr_test = txBuf_test;
   volatile uint8 *rxPtr_test = rxBuf_test;
   uint32 offset1 = 0;
   uint32 offset2 = 0;
   Std_ReturnType job_accepted = E_OK;

   for(idx = 0U; idx <  Data_Size_Test; idx++)
   {
      *txPtr_test++ = idx;
      *rxPtr_test++ = (uint8)0U;
   }

    if(type_of_erase == FLS_SECTOR_ERASE)  /* For sector erase, make sure offset should be in multiples of 4096 because sector erase do 4096 erase at a time */
    {                                      /* For block erase, make sure offset should be in multiples of 65536 because block erase do 65536 erase at a time */
      offset = 0x1C000;                      /* For chip erase, make sure offset should be 0 */
    }   
    else if(type_of_erase == FLS_BLOCK_ERASE)  
    {
      offset = 0x20000; 
    }           
    else
    {
      offset = 0x0; 
    }       
      
    AppUtils_printf(APP_NAME ": Performing Erase of 3 consecutive sectors/Blocks starting at offset:0X%X\n\r", offset);
    while (1U)
    {
         job_accepted = Fls_Erase(offset, 3*sector_or_blocksize);
        if (E_OK == job_accepted)
        break;
    }
    main_handling_noDbgMsg();

     offset1 =  offset + sector_or_blocksize; /* start address of next sector/block */

     offset2 =  offset1 + sector_or_blocksize; /* start address of next sector/block */

   AppUtils_printf(APP_NAME ": Writing Sector at (address 0X%X) ; ",offset);
   while (1U)
   {
        job_accepted = Fls_Write(offset, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

   AppUtils_printf(" Reading Back ; ");
   while (1U)
   {
        job_accepted = Fls_Read(offset, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();
   
   #if ( STD_ON == FLS_COMPARE_API) 
   AppUtils_printf(" Comparing ; ");
   while (1U)
   {
       job_accepted = Fls_Compare(offset, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();
   AppUtils_printf(" Write confirmed at 0X%X sector \n\r",offset);
   #endif
   
   AppUtils_printf(APP_NAME ": Writing next Sector at (address 0X%X) ; ",offset1);
   while (1U)
   {
       job_accepted = Fls_Write(offset1, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

   AppUtils_printf(" Reading Back ; ");
   while (1U)
   {
       job_accepted = Fls_Read(offset1, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

    #if ( STD_ON == FLS_COMPARE_API)   
   AppUtils_printf(" Comparing ; ");
   while (1U)
   {
       job_accepted = Fls_Compare(offset1, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();
   AppUtils_printf(" Write confirmed at 0X%X sector\n\r",offset1);
   #endif

   AppUtils_printf(APP_NAME ": Writing next Sector at (address 0X%X) ; ",offset2);
   while (1U)
   {
       job_accepted = Fls_Write(offset2, &txBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

   AppUtils_printf(" Reading Back ; ");
   while (1U)
   {
       job_accepted = Fls_Read(offset2, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();

     #if ( STD_ON == FLS_COMPARE_API)  
   AppUtils_printf(" Comparing ; ");
   while (1U)
   {
       job_accepted = Fls_Compare(offset2, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();
   AppUtils_printf(" Write confirmed at 0X%X sector\n\r",offset2);
#endif

    AppUtils_printf(APP_NAME ": Performing Erase of middle sector of three consecutive sectors. Offset: 0X%X \n\r",offset1);
    while (1U)
    {
        job_accepted = Fls_Erase(offset1,  Data_Size_Test);
        if (E_OK == job_accepted)
                break;
    }
    main_handling_noDbgMsg();
    
   txPtr_test = txBuf_test;
   rxPtr_test = rxBuf_test;
   for(idx = 0U; idx <  Data_Size_Test; idx++)
   {
      *txPtr_test++ = idx;
      *rxPtr_test++ = (uint8)0xFFU;
   }

 #if ( STD_ON == FLS_COMPARE_API)  
   AppUtils_printf(APP_NAME ": Comparing to confirm erase of sector at 0X%X\n\r",offset1);
   while (1U)
   {
       job_accepted = Fls_Compare(offset1, &rxBuf_test[0],  Data_Size_Test);
       if (E_OK == job_accepted)
           break;
   }
   main_handling_noDbgMsg();
#endif
    
#if ( STD_ON == FLS_COMPARE_API) 
    AppUtils_printf(APP_NAME ": Comparing to check data in sector at 0X%X \n\r",offset);
    while (1U)
    {
        job_accepted = Fls_Compare(offset, &txBuf_test[0],  Data_Size_Test);
        if (E_OK == job_accepted)
            break;
    }
    main_handling_noDbgMsg();
    AppUtils_printf(APP_NAME": Sector at 0X%X not effected by previous erase. \n\r",offset);
#endif

 #if ( STD_ON == FLS_COMPARE_API)       
    AppUtils_printf(APP_NAME ": Comparing to check data in sector at at 0X%X \n\r",offset2);
    while (1U)
    {
        job_accepted = Fls_Compare(offset2, &txBuf_test[0],  Data_Size_Test);
        if (E_OK == job_accepted)
            break;
    }
    main_handling_noDbgMsg();
    AppUtils_printf(APP_NAME": Sector at 0X%X not effected by previous erase. \n\r",offset2);
#endif

    AppUtils_printf(APP_NAME ": \r\n");
    
    if(type_of_erase == FLS_SECTOR_ERASE)
    AppUtils_printf(APP_NAME ": Sector Erase Check : SUCCESS \r\n");
    else if(type_of_erase == FLS_BLOCK_ERASE)
    AppUtils_printf(APP_NAME ": Block Erase Check : SUCCESS \r\n");
    else
    AppUtils_printf(APP_NAME ": Chip Erase Check : SUCCESS \r\n");

    AppUtils_printf(APP_NAME ": Middle Sector at 0X%X only has been erase \r\n",offset1);
}

void FlsApp_InterruptConfig(void)
{
    Vim_IntCfg intCfg;
    intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_PULSE;

    vimInit();
    intCfg.priority = VIM_PRIORITY_4;
    intCfg.intNum = 54U;
    intCfg.handler = Fls_Hdlr;
    vimRegisterInterrupt(&intCfg);
}
void SchM_Enter_Fls_FLS_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Fls_FLS_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}
Std_ReturnType Det_ReportError(uint16 ModuleId,
                               uint8  InstanceId,
                               uint8  ApiId,
                               uint8  ErrorId)
{
    Fls_TestPassed = E_NOT_OK; 
    GT_assert(McalAppTrace, FALSE);
    return (E_OK);
}
Std_ReturnType Det_ReportRuntimeError(VAR(uint16, AUTOMATIC) ModuleId,
                                               VAR(uint8, AUTOMATIC) InstanceId,
                                               VAR(uint8, AUTOMATIC) ApiId,
                                               VAR(uint8, AUTOMATIC) ErrorId)
{
    Fls_TestPassed = E_NOT_OK; 
    GT_assert(McalAppTrace, FALSE);
    return (E_OK);
}
Std_ReturnType Det_ReportTransientFault(
                                               VAR(uint16, AUTOMATIC) ModuleId,
                                               VAR(uint8, AUTOMATIC) InstanceId,
                                               VAR(uint8, AUTOMATIC) ApiId,
                                               VAR(uint8, AUTOMATIC) FaultId)
{
    Fls_TestPassed = E_NOT_OK; 
    GT_assert(McalAppTrace, FALSE);
    return (E_OK);
}
void Fee_JobEndNotification(void)
{
        Fls_JobDoneSuccess = 1U;
}

void Fee_JobErrorNotification(void)
{
        Fls_JobDoneError = 1U;
}
void main_handling()
{
	AppUtils_printf(APP_NAME ": Job Processing in Progress.\n\r");
    while (1U)
    {
        Fls_MainFunction();
        if ( Fls_JobDoneSuccess == 1U)
        {
            AppUtils_printf(APP_NAME ": Job Ends: SUCCESS\n\r");
			retVal = E_OK;
            break;
        }
        if ( Fls_JobDoneError == 1U)
        {
            AppUtils_printf(APP_NAME ": Job Ends: Error\n\r");
			retVal = E_NOT_OK;
            break;
        }
    }

    Fls_JobDoneSuccess = 0U;
    Fls_JobDoneError = 0U;
    return;
}

void main_handling_noDbgMsg()
{
    while (1U)
    {
        Fls_MainFunction();
        if ( Fls_JobDoneSuccess == 1U)
        {
            break;
        }
        if ( Fls_JobDoneError == 1U)
        {
            break;
        }
    }
    Fls_JobDoneSuccess = 0U;
    Fls_JobDoneError = 0U;
    return;
}

void main_handling_intr_noDbgMsg()
{
    while (1U)
    {
        if ( Fls_JobDoneSuccess == 1U)
        {
            break;
        }
        if ( Fls_JobDoneError == 1U)
        {
            break;
        }
    }
    Fls_JobDoneSuccess = 0U;
    Fls_JobDoneError = 0U;
    return;
}

void main_handling_intr()
{
	AppUtils_printf(APP_NAME ": Job Processing in Progress.\n\r");
    while(1U)
    {
        if ( Fls_JobDoneSuccess == 1U)
        {
			AppUtils_printf(APP_NAME ": Job Ends: SUCCESS\n\r");
			retVal = E_OK;
            break;
        }
        if ( Fls_JobDoneError == 1U)
        {
			AppUtils_printf(APP_NAME ": Job Ends: Error\n\r");
			retVal = E_NOT_OK;
            break;
        }

    }

    Fls_JobDoneSuccess = 0U;
    Fls_JobDoneError = 0U;
    return;
}
