/* ======================================================================
 *   Copyright (C) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     GptApp.c
 *
 *  \brief    This file contains the GPT test example
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/*LDRA_NOANALYSIS*/
#include "string.h"
/*LDRA_ANALYSIS*/
#include "assert.h"
/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
#include "SchM_Gpt.h"
#include "SchM_Port.h"
#include "Det.h"
#include "Dem.h"
/*LDRA_ANALYSIS*/
#include "Gpt_Cfg.h"
#include "Gpt.h"
#include "Gpt_Irq.h"

#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port_Cfg.h"
#include "Port.h"


#include "app_utils.h"
#include "trace.h"
#include "sys_common.h"
#include "GptApp.h"

#include "sys_vim.h"

/* For PRCM base addresses */
#include "soc.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define ARRAYSIZE(x)                    (sizeof ((x)) / sizeof ((x)[0]))
#define GptChannel_0                    (0U)

#ifdef GPTTIMER_QT 
  #define GPTIMER_1_SECOND             0x10U
#else 
/*Value of the GPT period for 1 second : 200MHz*/
  #define GPTIMER_1_SECOND             0x5F5E100U
#endif 

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/*None*/

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

void _enable_interrupt_(void);
void StartCyclecount(void);
static void GptApp_platformInit(void);
static void GptApp_interruptConfig(uint32 channelId);
static void GptApp_interruptInit(void);
static void GptApp_interruptDisable(uint32 channelId);

static void Gpt_App_mainTest(void);
typedef void (*Gpt_AppFxn_t)(void);

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

extern const struct Gpt_ConfigType_s Gpt_ChannelConfigSet_0;
volatile uint32                       Gpt_notifyRecvFlag = 0xFFFF;
uint8                        validWakeupSrcDetectedFlag = (uint8) FALSE;

Mcu_ClockConfigType Gpt_Mcu_ClkConfig[] =
{
   [0] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI0,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
   [1] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI1,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
   [2] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI2,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
	[3] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI3,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
	[4] =
    {
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
    .Mcu_ClockSourceId = MCU_CLKSRC_3,
    .Mcu_ClockDiv = 3,
    .Mcu_InitCfg  = TRUE,
    }   
};

static Mcu_ConfigType        gGptAppMcuConfig =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = (Mcu_RamConfigPtrType) NULL_PTR,
    .Mcu_NumberOfRamSectors  = 0,
    .Mcu_ClockConfig         = (Mcu_ClockConfigPtrType)&Gpt_Mcu_ClkConfig,
    .Mcu_NumberOfClockConfig = ARRAYSIZE(Gpt_Mcu_ClkConfig),
};

Gpt_AppFxn_t                  Gpt_AppFxnTbl[] =
{
    &Gpt_App_mainTest
};

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#if defined CLANG
void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
#endif

int main(void)
{
    uint32 i, gTestPassed = E_OK;
    AppUtils_defaultInit();
    AppUtils_sectionInit();
    /*Enable Domain clocks*/
    GptApp_platformInit();
    GptApp_interruptInit();
	AppUtils_TimerInit();

    AppUtils_printf(APP_NAME ": Sample Application - STARTS !!!\n\r");
    for (i = 0U; i < ARRAYSIZE(Gpt_AppFxnTbl); i++)
    {
        Gpt_AppFxnTbl[i]();
    }
    AppUtils_printf(APP_NAME ": Sample Application - DONE !!!\n\r");
    GT_1trace(McalAppTrace, GT_INFO,
                  " GPT Stack Usage: %d bytes\n\r", AppUtils_getStackUsage());
        if (AppUtils_checkStackAndSectionCorruption() != E_OK)
        {
            gTestPassed = E_NOT_OK;
            GT_0trace(McalAppTrace, GT_ERR, " GPT Stack/section corruption!!!\n\r");
        }

        if (E_OK == gTestPassed)
        {
            GT_0trace(McalAppTrace, GT_INFO, " GPT Test Passed!!!\n\r");
        }
        else
        {
            GT_0trace(McalAppTrace, GT_ERR, " GPT Test Failed!!!\n\r");
        }
	AppUtils_TimerDeinit();
    return (0);
}

static void Gpt_App_mainTest(void)
{
    uint32                  countVal, currCountval;
    uint32                  j;
    Gpt_ChannelConfigType   *pChannelCfg;
    Gpt_ChannelType         ChannelId;
    uint32                  tickFreq;
	
#if (GPT_VERSION_INFO_API == STD_ON)
    Std_VersionInfoType    versioninfo;
#endif  /* #if (GPT_VERSION_INFO_API == STD_ON) */

#if (GPT_VERSION_INFO_API == STD_ON)
    /* Get and print version */
    Gpt_GetVersionInfo(&versioninfo);
    AppUtils_printf(" \n\r");
    AppUtils_printf("GPT MCAL Version Info\n\r");
    AppUtils_printf("---------------------\n\r");
    AppUtils_printf("Vendor ID           : %d\n\r", versioninfo.vendorID);
    AppUtils_printf("Module ID           : %d\n\r", versioninfo.moduleID);
    AppUtils_printf("SW Major Version    : %d\n\r", versioninfo.sw_major_version);
    AppUtils_printf("SW Minor Version    : %d\n\r", versioninfo.sw_minor_version);
    AppUtils_printf("SW Patch Version    : %d\n\r", versioninfo.sw_patch_version);
    AppUtils_printf(" \n\r");
#endif

    AppUtils_printf(
        "----------------------------------------------------------------------------------\n\r");
    AppUtils_printf(APP_NAME ": GPTIMER Test Configuration\n\r");
    AppUtils_printf(
        "----------------------------------------------------------------------------------\n\r");

    AppUtils_printf("Initializing channels\n\r");
#if (STD_ON == GPT_PRE_COMPILE_VARIANT)
	AppUtils_printf("Precompile variant is being used .. \n\r");
    Gpt_Init((const Gpt_ConfigType *)NULL_PTR);
#else
	AppUtils_printf("Post-Build variant is being used .. \n\r");
	Gpt_Init(&Gpt_ChannelConfigSet_0);
#endif

    /* change this once debug is done */
    countVal = 1 * GPTIMER_1_SECOND;

    for (j = 0U; j < Gpt_ChannelConfigSet_0.ChannelCount; j++)
    {
        /* Take each channel configuration in configuration set */
        pChannelCfg = (Gpt_ChannelConfigType *)
                      (Gpt_ChannelConfigSet_0.ChannelConfig_pt + j);
        ChannelId = pChannelCfg->ChannelId;
        tickFreq = pChannelCfg->GptChannelTickFrequency;
        AppUtils_printf("------------------------------------------\n\r");
        AppUtils_printf(APP_NAME ": Running GPT Test for channel %d\n\r",
                        (ChannelId));
        AppUtils_printf("------------------------------------------\n\r");

        GptApp_interruptConfig(ChannelId);
        AppUtils_printf(APP_NAME ": Starting timer for %d s \n\r", ( countVal + tickFreq * countVal)/GPTIMER_1_SECOND);
        Gpt_StartTimer(ChannelId, countVal + tickFreq * countVal);

        #if (STD_ON == GPT_TIME_ELAPSED_API)
        // currCountval = Gpt_GetTimeElapsed(ChannelId);
        currCountval = Gpt_GetTimeElapsed(ChannelId);
        AppUtils_printf(APP_NAME ": Time Elapsed Value = %x\n\r",
                        currCountval);
        #endif
        #if (STD_ON == GPT_ENABLE_DISABLE_NOTIFICATION_API)
        AppUtils_printf(
            APP_NAME ": Enable channel notification for this channel\n\r");
        Gpt_EnableNotification(ChannelId);
        #endif

        AppUtils_printf(
            APP_NAME ": Wait for notification(approx. %d seconds)\n\r",  (countVal + tickFreq * countVal)/GPTIMER_1_SECOND);
        /* Wait till you get timer notifications */
        while (ChannelId != Gpt_notifyRecvFlag)
        {
            #if (STD_ON == GPT_TIME_ELAPSED_API)
            currCountval = Gpt_GetTimeElapsed(ChannelId);
            /*AppUtils_printf(APP_NAME ": Time Elapsed Value = %x\n\r",
                            currCountval);*/
            #endif
        }
        AppUtils_printf("\n\r");
        Gpt_notifyRecvFlag = 0xFFFF;

        #if (STD_ON == GPT_ENABLE_DISABLE_NOTIFICATION_API)
        AppUtils_printf(
            APP_NAME ": Disable channel notification for this channel\n\r");
        Gpt_DisableNotification(ChannelId);
        #endif

        AppUtils_printf(
            APP_NAME
            ": Wait till timer overflows, no notification should be received\n\r");
        /* Wait till timer overflows, you should not receive notification as
         * it is disabled */
#ifdef GPTTIMER_QT 
        AppUtils_delay(1U);
#else         
        AppUtils_delay(10U);
#endif         
        #if (STD_ON == GPT_TIME_ELAPSED_API)
        currCountval = Gpt_GetTimeElapsed(ChannelId);
        #endif
        AppUtils_printf(APP_NAME ": Time Elapsed Value = 0x%x\n\r",
                        currCountval);
#ifdef GPTTIMER_QT 
        AppUtils_delay(2U);
#else         
        AppUtils_delay(20U);
#endif         
        #if (STD_ON == GPT_TIME_REMAINING_API)
        currCountval = Gpt_GetTimeRemaining(ChannelId);
        #endif
        AppUtils_printf(APP_NAME ": Time Remaining Value = 0x%x\n\r",
                        currCountval);
        Gpt_notifyRecvFlag = 0xFFFF;
        /* Wait for 1 minute as countVal is FFF and timer tick is ~1ns(20Mhz
         * clk) it will take ~5ms to overflow */
        AppUtils_printf("Waiting for timer to overflow\n\r");
#ifdef GPTTIMER_QT 
        AppUtils_delay(3U);
#else                 
        AppUtils_delay(1000U);
#endif         
        AppUtils_printf("Overflow happened no notification received\n\r");


        AppUtils_printf(APP_NAME ": Stop timer \n\r");
        Gpt_StopTimer(ChannelId);

        GptApp_interruptDisable(ChannelId);

        AppUtils_printf("GPT Test Passed for channel =%d !!!\n\r",
                        (ChannelId));
    }
    /* Deinit current configuration */
#if (STD_ON == GPT_DEINIT_API)
    Gpt_DeInit();
#endif
    AppUtils_printf("GPT Test Passed for configuration!!\n\r");
    AppUtils_printf("GPT Test Completed, njoy life!!!\n\r");
}

static void GptApp_platformInit(void)
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;
#if (STD_OFF == MCU_NO_PLL)
    /* MCU PLL Config */
    gGptAppMcuConfig.Mcu_PllConfig = McuModuleConfiguration.Mcu_PllConfig;
    gGptAppMcuConfig.Mcu_PllSourceId = McuModuleConfiguration.Mcu_PllSourceId;
#endif
    Mcu_Init(&gGptAppMcuConfig);
    Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);
    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
	Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);
	Enable_Uart();

    return;
}

static void GptApp_interruptConfig(uint32 channelId)
{
    Vim_IntCfg intCfg;
    intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_LEVEL;
    if (channelId == 0){
        intCfg.intNum = RTI0_INT0;
        intCfg.handler = Gpt_Ch0Isr;
        intCfg.priority = VIM_PRIORITY_10;
        vimRegisterInterrupt(&intCfg);
    }else if (channelId == 1){
        intCfg.intNum = RTI0_INT1;
        intCfg.priority = VIM_PRIORITY_11;
        intCfg.handler = Gpt_Ch1Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 2){
        intCfg.intNum = RTI0_INT2;
        intCfg.priority = VIM_PRIORITY_12;
        intCfg.handler = Gpt_Ch2Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 3){
        intCfg.intNum = RTI0_INT3;
        intCfg.priority = VIM_PRIORITY_13;
        intCfg.handler = Gpt_Ch3Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 4){
        intCfg.intNum = RTI1_INT0;
        intCfg.priority = VIM_PRIORITY_14;
        intCfg.handler = Gpt_Ch4Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 5){
        intCfg.intNum = RTI1_INT1;
        intCfg.priority = VIM_PRIORITY_15;
        intCfg.handler = Gpt_Ch5Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 6){
        intCfg.intNum = RTI1_INT2;
        intCfg.priority = VIM_PRIORITY_16;
        intCfg.handler = Gpt_Ch6Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 7){
        intCfg.intNum = RTI1_INT3;
        intCfg.priority = VIM_PRIORITY_17;
        intCfg.handler = Gpt_Ch7Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 8){
        intCfg.intNum = RTI2_INT0;
        intCfg.priority = VIM_PRIORITY_18;
        intCfg.handler = Gpt_Ch8Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 9){
        intCfg.intNum = RTI2_INT1;
        intCfg.priority = VIM_PRIORITY_19;
        intCfg.handler = Gpt_Ch9Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 10){
        intCfg.intNum = RTI2_INT2;
        intCfg.priority = VIM_PRIORITY_20;
        intCfg.handler = Gpt_Ch10Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 11){
        intCfg.intNum = RTI2_INT3;
        intCfg.priority = VIM_PRIORITY_21;
        intCfg.handler = Gpt_Ch11Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 12){
        intCfg.intNum = RTI3_INT0;
        intCfg.priority = VIM_PRIORITY_22;
        intCfg.handler = Gpt_Ch12Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 13){
        intCfg.intNum = RTI3_INT1;
        intCfg.priority = VIM_PRIORITY_23;
        intCfg.handler = Gpt_Ch13Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 14){
        intCfg.intNum = RTI3_INT2;
        intCfg.priority = VIM_PRIORITY_24;
        intCfg.handler = Gpt_Ch14Isr;
        vimRegisterInterrupt(&intCfg);
    }else if(channelId == 15){
        intCfg.intNum = RTI3_INT3;
        intCfg.priority = VIM_PRIORITY_25;
        intCfg.handler = Gpt_Ch15Isr;
        vimRegisterInterrupt(&intCfg);
    }
}

static void GptApp_interruptDisable(uint32 channelId)
{
    switch(channelId)
    {
        case 0: vimDisableInterrupt(84U);
                break; 
        case 1: vimDisableInterrupt(85U);
                break;
        case 2: vimDisableInterrupt(86U);
                break;
        case 3: vimDisableInterrupt(87U);
                break;
        case 4: vimDisableInterrupt(91U);
                break;
        case 5: vimDisableInterrupt(92U);
                break;
        case 6: vimDisableInterrupt(93U);
                break;
        case 7: vimDisableInterrupt(94U);
                break;
        case 8: vimDisableInterrupt(98U);
                break;
        case 9: vimDisableInterrupt(99U);
                break;
        case 10: vimDisableInterrupt(100U);
                break;
        case 11: vimDisableInterrupt(101U);
                break;
		case 12: vimDisableInterrupt(105U);
                break;
        case 13: vimDisableInterrupt(106U);
                break;
        case 14: vimDisableInterrupt(107U);
                break;
        case 15: vimDisableInterrupt(108U);
                break;
    }
#ifdef GPTTIMER_QT 
        AppUtils_delay(2U);
#else         
        AppUtils_delay(20U);
#endif                 
    return;

}

static void GptApp_interruptInit(void)
{
    vimInit();
}
uint8 Intc_status;

void SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        assert(FALSE);
    }
}

void Gpt_Channel_Notify5(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 5 :)!!!\n\r");
    Gpt_notifyRecvFlag = 4U; /* Set it to channel number */
}

void Gpt_Channel_Notify6(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 6 :)!!!\n\r");
    Gpt_notifyRecvFlag = 5U; /* Set it to channel number */
}

void Gpt_Channel_Notify7(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 7 :)!!!\n\r");
    Gpt_notifyRecvFlag = 6U; /* Set it to channel number */
}

void Gpt_Channel_Notify8(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 8 :)!!!\n\r");
    Gpt_notifyRecvFlag = 7U; /* Set it to channel number */
}

void Gpt_Channel_Notify9(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 9 :)!!!\n\r");
    Gpt_notifyRecvFlag = 8U; /* Set it to channel number */
}

void Gpt_Channel_Notify10(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 10 :)!!!\n\r");
    Gpt_notifyRecvFlag = 9U; /* Set it to channel number */
}

void Gpt_Channel_Notify11(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 11 :)!!!\n\r");
    Gpt_notifyRecvFlag = 10U; /* Set it to channel number */
}

void Gpt_Channel_Notify12(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 12 :)!!!\n\r");
    Gpt_notifyRecvFlag = 11U; /* Set it to channel number */
}

void Gpt_Channel_Notify13(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 13 :)!!!\n\r");
    Gpt_notifyRecvFlag = 12U; /* Set it to channel number */
}

void Gpt_Channel_Notify14(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 14 :)!!!\n\r");
    Gpt_notifyRecvFlag = 13U; /* Set it to channel number */
}

void Gpt_Channel_Notify15(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 15 :)!!!\n\r");
    Gpt_notifyRecvFlag = 14U; /* Set it to channel number */
}

void Gpt_Channel_Notify16(void)
{
    AppUtils_printf(
        APP_NAME ": GPT Test Notification Receiver channel 16 :)!!!\n\r");
    Gpt_notifyRecvFlag = 15U; /* Set it to channel number */
}


#if ((STD_ON == GPT_WAKEUP_FUNCTIONALITY_API) && (STD_ON == \
                                                      GPT_REPORT_WAKEUP_SOURCE))
void EcuM_CheckWakeup(EcuM_WakeupSourceType wakeupSource)
{
    AppUtils_printf(
        APP_NAME
        ": GPT Test Wakeup event received for wakeupSource =%d :)!!!\n\r",
        wakeupSource);
    validWakeupSrcDetectedFlag = (uint8) TRUE;
}

void EcuM_SetWakeupEvent(EcuM_WakeupSourceType chlWakeupRef)
{
    /* This function will be called after call to Gpt_CheckWakeup.
     * Gpt_CheckWakeup will call this function if wakeup source passed to it is
     * valid source */
    AppUtils_printf(
        APP_NAME
        ": GPT Test Wakeup event set for wakeupSource =%d :)!!!\n\r",
        chlWakeupRef);
}

#endif

/*EoF*/
