/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     I2cExample.c
 *
 *  \brief    This file contains the I2c test example
 */

/**********************************************************************************************************************
 *  INCLUDES
 *********************************************************************************************************************/

/* Include Files */
/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
#include "Det.h"
#include "Dem.h"
/*LDRA_ANALYSIS*/
#include "Cdd_I2c_Cfg.h"
#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port.h"
#include "Port_Cfg.h"
#include "hw_types.h"
#include "esm.h"
#include "sys_vim.h"
#include "app_utils.h"
#include "trace.h"
#include "Cdd_I2c.h"
#include <string.h>

#if(STD_OFF == CDD_I2C_POLLING_MODE)
#include "Cdd_I2c_Irq.h"
#endif

#define APP_NAME "I2cApp"

#define I2CEXAMPLE_ARRAYSIZE(x)  (sizeof ((x)) / sizeof (x[0U]))

/*Temperature result registers*/
#define TMP10X_RESULT_REG               (0x0000U)

/*EEPROM start address*/
#define EEPROM_START_ADDRESS            (0x00U)

/*EEPROM start message length*/
#define EEPROM_START_MESSAGE_LENGTH     (0x01U)

#define EEPROM_MEM_ADDRESS          (0x00U)

#define EEPROM_ADDR_SIZE            (2U)

Cdd_I2c_ConfigPtrType I2cConfigSet = NULL_PTR;

Std_ReturnType testPassed = E_NOT_OK;

/*Buffer to store transmission data for temperature sensor*/
static uint8 txBufferTemperature[1];
/*Buffer to store reception data for temperature sensor*/
static uint8 rxBufferTemperature[2];

/*Buffer to store transmission data for temperature sensor*/
static uint8 txBufferEeprom[32]; 
/*Buffer to store transmission data for temperature sensor*/
static uint8 txBufferEepromStart[2];
/*Buffer to store reception data for temperature sensor*/
static uint8 rxBufferEeprom[26];

static sint16 temperature = (sint16)0;
volatile boolean rxDoneTemperature  = FALSE;
volatile boolean rxDoneEeprom       = FALSE;

#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif

Mcu_RamSectionConfigType    Mcu_RamConfig[] =
{
    [0U] =
    {
    .Mcu_RamSectionBaseAddress = &txBufferTemperature[0U],
    .Mcu_RamDefaultValue       = 0x00,
    .Mcu_RamSectionBytes       = sizeof (txBufferTemperature)
    },
    [1U] =
    {
    .Mcu_RamSectionBaseAddress = &rxBufferTemperature[0U],
    .Mcu_RamDefaultValue       = 0x00,
    .Mcu_RamSectionBytes       = sizeof (rxBufferTemperature)
    },
    [2U] =
    {
    .Mcu_RamSectionBaseAddress = &txBufferEepromStart[0U],
    .Mcu_RamDefaultValue       = 0x00,
    .Mcu_RamSectionBytes       = sizeof (txBufferEepromStart)
    },
    [3U] =
    {
    .Mcu_RamSectionBaseAddress = &txBufferEeprom[0U],
    .Mcu_RamDefaultValue       = 0x00,
    .Mcu_RamSectionBytes       = sizeof (txBufferEeprom)
    },
    [4U] =
    {
    .Mcu_RamSectionBaseAddress = &rxBufferEeprom[0U],
    .Mcu_RamDefaultValue       = 0x00,
    .Mcu_RamSectionBytes       = sizeof (rxBufferEeprom)
    },
};

static Mcu_ClockConfigType      gI2cAppMcuClockConfig[] =
{
    [0] =
    {
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_I2C,
    .Mcu_ClockSourceId = MCU_CLKSRC_2,
    .Mcu_ClockDiv      = 0U, 
    .Mcu_InitCfg       = TRUE
    },
    [1] =
    {
        .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI0,
        .Mcu_ClockSourceId = MCU_CLKSRC_2,
        .Mcu_ClockDiv = 0,
        .Mcu_InitCfg  = TRUE,
    },
    [2] = 
    {
        .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
        .Mcu_ClockSourceId = MCU_CLKSRC_3,
        .Mcu_ClockDiv = 3,
        .Mcu_InitCfg  = TRUE,
    },
};

Mcu_ConfigType              Mcu_Config =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = (Mcu_RamSectionConfigType *) Mcu_RamConfig,
    .Mcu_NumberOfRamSectors  = I2CEXAMPLE_ARRAYSIZE(Mcu_RamConfig),
    .Mcu_ClockConfig         = (Mcu_ClockConfigPtrType) gI2cAppMcuClockConfig,
    .Mcu_NumberOfClockConfig = I2CEXAMPLE_ARRAYSIZE(gI2cAppMcuClockConfig)
};

static void I2cExample_PlatformInit()
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;
    
  #if (STD_OFF == MCU_NO_PLL)
    uint32 sys_clk_freq_vclk_const;

    /* MCU PLL Config */
    Mcu_Config.Mcu_PllConfig = McuModuleConfiguration.Mcu_PllConfig;
    Mcu_Config.Mcu_PllSourceId = McuModuleConfiguration.Mcu_PllSourceId;
  #endif

    Mcu_Init(&Mcu_Config);
    Port_Init(&PortConfigSet_0);
    Mcu_InitRamSection(0);
    Mcu_InitRamSection(1);
    
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);
    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
	Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);

  #if (STD_OFF == MCU_NO_PLL)
    if(McuModuleConfiguration.Mcu_PllSourceId == MCU_CLKSRC_DPLL)
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_DPLL].Mcu_PllClk1.MCU_PLL_HSDIV2 / (2e6)) * 30;
    }
    else
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_APLL].Mcu_PllClk1.MCU_PLL_HSDIV0 / (2e6)) * 30;
    }
    Enable_Uart();
  #else
    Enable_Uart();
  #endif
    
}

static Std_ReturnType I2c_Buffer_Setup(){
    Std_ReturnType returnValue = E_OK;

    /* Setup I2c channels */
    returnValue  = Cdd_I2c_SetupEB(CddI2cConf_CddI2cChannel_CddI2cChannel_0, &txBufferTemperature[0], NULL_PTR, 1); 
    returnValue |= Cdd_I2c_SetupEB(CddI2cConf_CddI2cChannel_CddI2cChannel_1, &txBufferEeprom[0], NULL_PTR, 28);
    returnValue |= Cdd_I2c_SetupEB(CddI2cConf_CddI2cChannel_CddI2cChannel_2, &txBufferEepromStart[0], NULL_PTR, 2);
    returnValue |= Cdd_I2c_SetupEB(CddI2cConf_CddI2cChannel_CddI2cChannel_3, NULL_PTR, &rxBufferTemperature[0], 2);   
    returnValue |= Cdd_I2c_SetupEB(CddI2cConf_CddI2cChannel_CddI2cChannel_4, NULL_PTR, &rxBufferEeprom[0], 26); 
    
    return returnValue;
}

static Std_ReturnType I2cExample_ExternalTest_Data_Write_To_Slaves(Std_ReturnType BufferSetupReturnValue)
{
    Std_ReturnType returnValue = BufferSetupReturnValue;

    if(E_OK == returnValue)
    {
        txBufferTemperature[0] = TMP10X_RESULT_REG;
        strcpy((char *)&txBufferEeprom[0], "  I2C read/write test passed");
        txBufferEeprom[0] = EEPROM_MEM_ADDRESS;     /* EEPROM internal address byte 1 pointer*/
        txBufferEeprom[1] = EEPROM_MEM_ADDRESS;     /* EEPROM internal address byte 2 pointer*/

        while (CDD_I2C_SEQ_OK != Cdd_I2c_GetSequenceResult(CddI2cConf_CddI2cSequence_CddI2cSequence_0))
        {
            /* Wait for sequence to be ready */
        }
        
        /* Start transmission */
        Cdd_I2c_AsyncTransmit(CddI2cConf_CddI2cSequence_CddI2cSequence_0);
        
        while (CDD_I2C_SEQ_OK != Cdd_I2c_GetSequenceResult(CddI2cConf_CddI2cSequence_CddI2cSequence_1))
        {
            /* Wait for sequence to be ready */
        }
        
        /* Start transmission */
        Cdd_I2c_AsyncTransmit(CddI2cConf_CddI2cSequence_CddI2cSequence_1);
    }
    else
    {
        //Do Nothing
    }
    
    return returnValue;
}

static Std_ReturnType I2cExample_ExternalTest_Eeprom_address_pointer_reset(Std_ReturnType BufferSetupReturnValue)
{
    Std_ReturnType returnValue = BufferSetupReturnValue;
    
    txBufferEepromStart[0] = EEPROM_MEM_ADDRESS;     /* Address Byte 1 */
    txBufferEepromStart[1] = EEPROM_MEM_ADDRESS;     /* Address Byte 2 */

    if(E_OK == returnValue)
    {
        while (CDD_I2C_SEQ_OK != Cdd_I2c_GetSequenceResult(CddI2cConf_CddI2cSequence_CddI2cSequence_2))
        {
            /* Wait for sequence to be ready */
        }
        
        /* Start transmission */
        Cdd_I2c_AsyncTransmit(CddI2cConf_CddI2cSequence_CddI2cSequence_2);
    }
    else
    {
        //Do Nothing
    }

    return returnValue;
}

static Std_ReturnType I2cExample_ExternalTest_Receive_Data()
{
    Std_ReturnType returnValue = E_OK;

    #if(STD_ON == CDD_I2C_GET_STATUS_API)   /*#if(STD_ON == CDD_I2C_GET_STATUS_API)*/
    /* Wait for transmission to be complete */
    while ((CDD_I2C_IDLE != Cdd_I2c_GetStatus()) || ((TRUE != rxDoneTemperature) && (TRUE != rxDoneEeprom)))
    #endif  /*#if(STD_ON == CDD_I2C_GET_STATUS_API)*/
    {
        /* Read I2c sequence in polling mode */
        #if(STD_ON == CDD_I2C_POLLING_MODE) /*#if(STD_ON == CDD_I2C_POLLING_MODE)*/
        Cdd_I2c_PollingModeProcessing();
        #elif(STD_OFF == CDD_I2C_POLLING_MODE)
        /*Read Temperature sensor data*/
        Cdd_I2c_AsyncTransmit(CddI2cConf_CddI2cSequence_CddI2cSequence_3);
        /*2s delay to give time to peripheral to copy data to proper locations*/
        AppUtils_delay(2000);
        /*Read EEPROM data*/
        Cdd_I2c_AsyncTransmit(CddI2cConf_CddI2cSequence_CddI2cSequence_4);
        #endif  /*#if(STD_ON == CDD_I2C_POLLING_MODE)*/
        Cdd_I2c_MainFunction();
    }

    if (CDD_I2C_SEQ_OK == Cdd_I2c_GetSequenceResult(CddI2cConf_CddI2cSequence_CddI2cSequence_3))
    {

        /* Create 16 bit temperature */
        temperature = ((uint16)rxBufferTemperature[0] << 8) | (rxBufferTemperature[1]);

        /*
        * 4 LSBs of temperature are 0 according to datasheet
        * since temperature is stored in 12 bits. Therefore,
        * right shift by 4 places
        */
        temperature = temperature >> 4;
        /*
        * If the 12th bit of temperature is set '1' (equivalent to 8th bit of the first byte read),
        * then we have a 2's complement negative value which needs to be sign extended
        */
        if (rxBufferTemperature[0] & 0x80)
        {
            temperature |= 0xF000;
        }
        else
        {
            // Do Nothing
        }

        AppUtils_printf(APP_NAME ": Temperature sensor reading : %f (celcius)\n\r", (temperature / 16.0f));

        rxDoneTemperature = FALSE;

        returnValue = E_OK;
    }
    else
    {
        returnValue = E_NOT_OK;
    }

    if (CDD_I2C_SEQ_OK == Cdd_I2c_GetSequenceResult(CddI2cConf_CddI2cSequence_CddI2cSequence_4)) 
    {
        AppUtils_printf(APP_NAME ": Message read from EEPROM: %s\n\r", rxBufferEeprom);

        rxDoneEeprom = FALSE;

        returnValue = E_OK;
    }
    else
    {
        returnValue = E_NOT_OK;
    }
    return returnValue;
}

void I2c_Eeprom_Data_Write_Complete(void){
    /*Clear EEPROM TX buffer*/
    memset(&txBufferEeprom[0],0,28);
    strcpy((char *)&txBufferEeprom[0], "I2C read/write test passed");
    AppUtils_printf(APP_NAME ": Message written to EEPROM: %s \n\r", txBufferEeprom);
    memset(&txBufferEeprom[0],0,28);
}

void I2c_Temperature_Data_Write_Complete(void){
    AppUtils_printf(APP_NAME ": Temperature write SuccessFull \n\r");
}

void I2c_Temperature_Read_Callback(void)
{
    rxDoneTemperature = TRUE;
}

void I2c_Eeprom_Read_Callback(void)
{
    rxDoneEeprom = TRUE;
}

void  I2c_Eeprom_Address_Ptr_Reset(void){
    AppUtils_printf(APP_NAME ": EEPROM address pointer reset to start read from location : %u\n\r",EEPROM_MEM_ADDRESS);
}

void I2c_Sequence_Error_Report(uint8 Error_code){
    if(CDD_I2C_E_HW_UNIT_BUSY == Error_code){
        AppUtils_printf(APP_NAME ": Hardware unit busy\n\r");
    }
    else if(CDD_I2C_E_CHANNEL_BUSY == Error_code){
        AppUtils_printf(APP_NAME ": Channels busy\n\r");
    }
    else if(CDD_I2C_E_ARBITRATION_LOSS == Error_code){
        AppUtils_printf(APP_NAME ": Arbitration lost\n\r");
    }
    else if(CDD_I2C_E_NACK == Error_code){
        AppUtils_printf(APP_NAME ": No Acknowledgement\n\r");
    }
    else if(CDD_I2C_E_RECEIVE_SHIFT_REGISTER_FULL == Error_code){
        AppUtils_printf(APP_NAME ": Receive shift register full\n\r");
    }
    else if(CDD_I2C_E_PARAM_QUEUE_FULL == Error_code){
        AppUtils_printf(APP_NAME ": Queue full\n\r");
    }
}

void I2c_Temperature_Data_Write_Fail(uint8 Error_Code){
    AppUtils_printf(APP_NAME ": Sequence failed while trying to write to temperature\n\r");
    I2c_Sequence_Error_Report(Error_Code);
}
void I2c_Eeprom_Data_Write_Fail(uint8 Error_Code){
    AppUtils_printf(APP_NAME ": Sequence failed while trying to write to EEPROM\n\r");
    I2c_Sequence_Error_Report(Error_Code);
}

void I2c_Eeprom_Address_Ptr_Fail(uint8 Error_Code){
    AppUtils_printf(APP_NAME ": Sequence failed while trying to reset EEPROM address pointer\n\r");
    I2c_Sequence_Error_Report(Error_Code);
}

void I2c_Temperature_Read_Callback_Fail(uint8 Error_Code){
    AppUtils_printf(APP_NAME ": Sequence failed while trying to read data from temperature sensor\n\r");
    I2c_Sequence_Error_Report(Error_Code);
}

void I2c_Eeprom_Read_Callback_Fail(uint8 Error_Code){
    AppUtils_printf(APP_NAME ": Sequence failed while trying to read data from EEPROM\n\r");
    I2c_Sequence_Error_Report(Error_Code);
}

#if (STD_ON == CDD_I2C_VERSION_INFO_API)
void Print_I2c_Version_Info(Std_VersionInfoType* versionInfoPtr){
    AppUtils_printf(APP_NAME ": Vendor ID: %u\n\r",versionInfoPtr->vendorID);
    AppUtils_printf(APP_NAME ": Module ID: %u\n\r",versionInfoPtr->moduleID);
    AppUtils_printf(APP_NAME ": SW Major version ID: %u\n\r",versionInfoPtr->sw_major_version);
    AppUtils_printf(APP_NAME ": SW Minor version ID: %u\n\r",versionInfoPtr->sw_minor_version);
    AppUtils_printf(APP_NAME ": SW Patch version ID: %u\n\r",versionInfoPtr->sw_patch_version);
}
#endif

#if(STD_OFF == CDD_I2C_POLLING_MODE)
void Cdd_I2c_InterruptConfig(void)
{
    vimInit();
    Vim_IntCfg interruptCfg;
    interruptCfg.map = VIM_INTTYPE_IRQ;
    interruptCfg.type = VIM_INTTRIGTYPE_PULSE;
    interruptCfg.intNum = 44U;
    interruptCfg.handler = &Cdd_I2c_HwUnit0_ISR;
    interruptCfg.priority = VIM_PRIORITY_2;
    vimRegisterInterrupt(&interruptCfg);
    interruptCfg.map = VIM_INTTYPE_IRQ;
    interruptCfg.type = VIM_INTTRIGTYPE_PULSE;
    interruptCfg.intNum = 46U;
    interruptCfg.handler = &Cdd_I2c_HwUnit2_ISR;
    interruptCfg.priority = VIM_PRIORITY_2;
    vimRegisterInterrupt(&interruptCfg);
}
#endif

int main(void)
{
    Std_ReturnType BufferSetupReturnValue = E_OK;

    #if (STD_ON == CDD_I2C_VERSION_INFO_API)
    Std_VersionInfoType VersionInformation;
    Std_VersionInfoType *pVersionInformation = &VersionInformation; 
    #endif

    AppUtils_defaultInit();

    I2cExample_PlatformInit();

    #if(STD_OFF == CDD_I2C_POLLING_MODE)
    Cdd_I2c_InterruptConfig();
    #endif

    start_timer();

    AppUtils_printf(APP_NAME ": Sample Application - STARTS !!!\n\r");
    AppUtils_printf(APP_NAME ": gI2cAppMcuClockConfig[0].Mcu_ClockSourceId = %d gI2cAppMcuClockConfig[0].Mcu_ClockDiv = %d  :\n\r",
    gI2cAppMcuClockConfig[0].Mcu_ClockSourceId, gI2cAppMcuClockConfig[0].Mcu_ClockDiv);

    #if (STD_ON == CDD_I2C_VERSION_INFO_API)
    /*Get I2c Version information*/
    Cdd_I2c_GetVersionInfo(pVersionInformation);
    Print_I2c_Version_Info(pVersionInformation);
    #endif

    /* Do I2c Init */
    Cdd_I2c_Init(I2cConfigSet);
    /*2s delay*/
    AppUtils_delay(2000);
    /*Set up channels*/
    BufferSetupReturnValue = I2c_Buffer_Setup();
    /*Write data to Temperature sensor and EEPROM*/
    testPassed  = I2cExample_ExternalTest_Data_Write_To_Slaves(BufferSetupReturnValue);
    /*5s delay*/
    AppUtils_delay(5000);
    /*Write dummy data to EEPROM to reset address pointer*/
    testPassed |= I2cExample_ExternalTest_Eeprom_address_pointer_reset(BufferSetupReturnValue);
    /*2s delay*/
    AppUtils_delay(2000);
    /*Receive and process data*/
    testPassed |= I2cExample_ExternalTest_Receive_Data();
    /*2s delay*/
    AppUtils_delay(2000);

    if (E_OK == testPassed)
    {
        GT_0trace(McalAppTrace, GT_INFO, " I2C Test Passed!!!\n\r");
    }
    else
    {
        GT_0trace(McalAppTrace, GT_ERR, " I2C Test Failed!!!\n\r");
    }

    /*Stop the timer*/
    stop_timer();
    /*De-initialize I2c module*/
    Cdd_I2c_DeInit();

    return (0);
}

#ifdef __cplusplus
extern "C" {
#endif

void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void  SchM_Enter_Cdd_I2c_I2C_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Cdd_I2c_I2C_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        testPassed = E_NOT_OK;
        GT_assert(McalAppTrace, FALSE);
    }

    return;
}

Std_ReturnType Det_ReportError(uint16 ModuleId, uint8 InstanceId, uint8 ApiId,
                               uint8 ErrorId)
{
    testPassed = E_NOT_OK;
    GT_assert(McalAppTrace, FALSE);
    return E_OK;
}

#ifdef __cplusplus
}
#endif
