/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     IcuApp.c
 *
 *  \brief    This file contains the Icu test example
 */

/**********************************************************************************************************************
 *  INCLUDES
 *********************************************************************************************************************/

/* Include Files */
#include "assert.h"
/*LDRA_NOANALYSIS*/
#include "string.h"
#include "Det.h"
#include "Dem.h"
#include "Std_Types.h"
#include "Os.h"
/*LDRA_ANALYSIS*/
#include "Mcu.h"
#include "Pwm.h"
#include "Icu.h"
#include "EcuM_Cbk.h"
#include "Pwm_Cfg.h"
#include "Icu_Cfg.h"
#include "app_utils.h"
#include "sys_common.h"
#include "trace.h"
#include "sys_vim.h"
#include "hal_stdtypes.h"
#include "soc.h"
#include "Pwm_Irq.h"
#include "Icu_Irq.h"
#include "IcuApp.h"
#include "Port.h"
#include "hw_icu.h"
#include "hw_types.h"
#include "cslr_controlss_inputxbar.h"

// for inputXbar
#define CSL_CONTROLSS_INPUTXBAR_U_BASE                  (0x502D0000U) 
#define CSL_CONTROLSS_INTXBAR_U_BASE                  (0x502D5000ul)
#define INT_XBAR_ECAP0_INT    (0x00000001)
#define CSL_CONTROLSS_INPUTXBAR_STEP        (CSL_CONTROLSS_INPUTXBAR_INPUTXBAR1_GSEL - CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_GSEL)
#define CSL_CONTROLSS_PWMXBAR_STEP          (CSL_CONTROLSS_PWMXBAR_PWMXBAR1_G0 - CSL_CONTROLSS_PWMXBAR_PWMXBAR0_G0)
#define CSL_CONTROLSS_MDLXBAR_STEP          (CSL_CONTROLSS_MDLXBAR_MDLXBAR1_G0 - CSL_CONTROLSS_MDLXBAR_MDLXBAR0_G0)
#define CSL_CONTROLSS_ICLXBAR_STEP          (CSL_CONTROLSS_ICLXBAR_ICLXBAR1_G0 - CSL_CONTROLSS_ICLXBAR_ICLXBAR0_G0)
#define CSL_CONTROLSS_INTXBAR_STEP          (CSL_CONTROLSS_INTXBAR_INTXBAR1_G0 - CSL_CONTROLSS_INTXBAR_INTXBAR0_G0)
#define CSL_CONTROLSS_DMAXBAR_STEP          (CSL_CONTROLSS_DMAXBAR_DMAXBAR1_GSEL - CSL_CONTROLSS_DMAXBAR_DMAXBAR0_GSEL)
#define CSL_CONTROLSS_OUTPUTXBAR_STEP       (CSL_CONTROLSS_OUTPUTXBAR_OUTPUTXBAR1_G0 - CSL_CONTROLSS_OUTPUTXBAR_OUTPUTXBAR0_G0)
#define CSL_CONTROLSS_PWMSYNCOUTXBAR_STEP   (CSL_CONTROLSS_PWMSYNCOUTXBAR_PWMSYNCOUTXBAR1_G0 - CSL_CONTROLSS_PWMSYNCOUTXBAR_PWMSYNCOUTXBAR0_G0)


#define ICU_CHANNEL        (ICU_CHANNEL0)

#define ARRAYSIZE(x)  (sizeof ((x)) / sizeof (x[0]))
#define PWM_SYS_CLOCK   (200000000)
#define INTERRUPT_XBAR_NUMBER (168U)

volatile boolean Icu_NotifyIntervalReached = FALSE;
volatile boolean Icu_SignalNotificationReached = FALSE;
volatile uint32 Icu_SignalNotificationNumber = 0U;

Std_VersionInfoType      VersionInfo;
sint32 TestState = TRUE;
sint32 notifyRecvFlag  = TRUE;

/*Notification function definitions*/
void Icu_SignalNotification_Channel1(void)
{
    Icu_SignalNotificationNumber++;
}

void Icu_TimeStampNotification_Channel1(void)
{
    Icu_NotifyIntervalReached = TRUE;
}

void  Pwm_Notification_Channel1(void){

}

void  Pwm_Notification_Channel2(void){
    
}
void  Pwm_Notification_Channel3(void){
    
}

void SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void SchM_Enter_Icu_ICU_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void SchM_Exit_Icu_ICU_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
    
}

/* 
 This is required by ecap to hit the ISR 
*/
static  void
SOC_xbarSelectInterruptXBarInputSource(uint32 base, uint8 out, uint32 group0_mask, uint32 group1_mask, uint32 group2_mask, uint32 group3_mask, uint32 group4_mask, uint32 group5_mask, uint32 group6_mask)
{
    //TBD: 32 bit field required?
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G0, group0_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G0_SEL_MASK);
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G1, group1_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G1_SEL_MASK);
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G2, group2_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G2_SEL_MASK);
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G3, group3_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G3_SEL_MASK);
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G4, group4_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G4_SEL_MASK);
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G5, group5_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G5_SEL_MASK);
    HW_WR_REG32(base + out*CSL_CONTROLSS_INTXBAR_STEP + CSL_CONTROLSS_INTXBAR_INTXBAR0_G6, group6_mask & CSL_CONTROLSS_INTXBAR_INTXBAR0_G6_SEL_MASK);
}

static void SOC_controlModuleLockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;
    
    if(partition==CONTROLSS_CTRL_PARTITION0)
    {
        /*Lock CONTROLSS_CTRL*/
        baseAddr = (uint32) CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 1 */
    }

    return;
}

static void SOC_controlModuleUnlockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;

    if(partition==CONTROLSS_CTRL_PARTITION0)
    {
        /*Unlock CONTROLSS_CTRL*/
        baseAddr = (uint32) CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK0_UNLOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK1_UNLOCK_VAL);      /* KICK 1 */
    }

    return;
}

static void SOC_setEpwmTbClk(uint32 epwmInstance, uint32 enable)
{
    if(epwmInstance < CSL_ETPWM_PER_CNT)
    {
        /* Time base clock enable register belongs to partition 1 of the CTRL MMR */

        /* Unlock CONTROLSS_CTRL registers */
        SOC_controlModuleUnlockMMR(0, CONTROLSS_CTRL_PARTITION0);

        if(TRUE == enable)
        {
            /* Enable Time base clock in CTRL MMR */
            HW_WR_REG32(CSL_CONTROLSS_CTRL_U_BASE + CSL_CONTROLSS_CTRL_EPWM_CLKSYNC,
                ((HW_RD_REG32(CSL_CONTROLSS_CTRL_U_BASE +
                  CSL_CONTROLSS_CTRL_EPWM_CLKSYNC) & CSL_CONTROLSS_CTRL_EPWM_CLKSYNC_BIT_MASK) | (1 << epwmInstance)));
        }
        else
        {
            /* Disable Time base clock in CTRL MMR */
            HW_WR_REG32(CSL_CONTROLSS_CTRL_U_BASE + CSL_CONTROLSS_CTRL_EPWM_CLKSYNC,
                ((HW_RD_REG32(CSL_CONTROLSS_CTRL_U_BASE +
                  CSL_CONTROLSS_CTRL_EPWM_CLKSYNC) & CSL_CONTROLSS_CTRL_EPWM_CLKSYNC_BIT_MASK) & ~(1 << epwmInstance)));
        }

        /* Lock CONTROLSS_CTRL registers */
        SOC_controlModuleLockMMR(0, CONTROLSS_CTRL_PARTITION0);
    }
}

static void SOC_setEpwmGroup(uint32 epwmInstance, uint32 group)
{
    uint32 baseAddr = CSL_CONTROLSS_CTRL_U_BASE + CSL_CONTROLSS_CTRL_EPWM_STATICXBAR_SEL0;
    uint32 mask, shift;

    /* Unlock CONTROLSS_CTRL registers */
    SOC_controlModuleUnlockMMR(0, CONTROLSS_CTRL_PARTITION0);

    /* Choose the correct base address depending on which ePWM instance is selected*/
    if(epwmInstance > 15)
    {
        baseAddr = baseAddr + 0x4;
        epwmInstance = epwmInstance - 16;
    }

    shift = (epwmInstance << 1);
    /* Create the mask to be written to register */
    mask = (0x3 << shift);

    /* Configure the group for the ePWM instance */
    HW_WR_REG32(baseAddr, (( HW_RD_REG32(baseAddr) & ~mask) | (group <<shift)));

    /* Lock CONTROLSS_CTRL registers */
    SOC_controlModuleLockMMR(0, CONTROLSS_CTRL_PARTITION0);
}

void Icu_InterruptConfig(void)
{
    vimInit();
    Vim_IntCfg intCfg;
    intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_PULSE;
    intCfg.priority = VIM_PRIORITY_15;
    intCfg.intNum = INTERRUPT_XBAR_NUMBER; // using for intrxbar_22
    intCfg.handler = Icu_ch0Notify;
    vimRegisterInterrupt(&intCfg);
}

void SOC_xbarSelectInputXBarInputSource(uint32 base, uint8 out, uint8 group_select, uint8 group0_muxctl, uint8 group1_muxctl)
{
    HW_WR_REG32(base + CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_GSEL + (out * CSL_CONTROLSS_INPUTXBAR_STEP), group_select & CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_GSEL_GSEL_MASK);
    HW_WR_REG32(base + CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_G0   + (out * CSL_CONTROLSS_INPUTXBAR_STEP), group0_muxctl & CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_G0_SEL_MASK);
    HW_WR_REG32(base + CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_G1   + (out * CSL_CONTROLSS_INPUTXBAR_STEP), group1_muxctl & CSL_CONTROLSS_INPUTXBAR_INPUTXBAR0_G1_SEL_MASK);
    
}



static void IcuApp_mainTest(void)
{
start_timer();
#if (STD_ON == ICU_SIGNAL_MEASUREMENT_API)
    uint16 elapsedTime = 0U;
    Icu_DutyCycleType dutyCycleApp;
    uint16 periodTimeApp, activeTimeApp = 0U;
#endif
#if (STD_ON == ICU_TIMESTAMP_API)
    Icu_ActivationType activation = ICU_RISING_EDGE;
    uint16 bufferSize = 6;
    Icu_ValueType timestampArray [bufferSize];
#endif
#if (STD_ON == ICU_EDGE_COUNT_API)
    Icu_EdgeNumberType edgecount = 0;
#endif
#if (STD_ON == ICU_EDGE_DETECT_API)
    Icu_InputStateType inputstate;
#endif


#if (ICU_GET_VERSION_INFO_API == STD_ON)

    Std_VersionInfoType versioninfo;

    /* Version Info Check*/
    Icu_GetVersionInfo(&versioninfo);
   AppUtils_printf( " \n\r");
   AppUtils_printf( APP_NAME ": ICU MCAL Version Info\n\r");
   AppUtils_printf( "---------------------\n\r");
   AppUtils_printf( APP_NAME ": Vendor ID: %d\n\r",versioninfo.vendorID);
   AppUtils_printf( APP_NAME ": Module ID: %d\n\r",versioninfo.moduleID);
   AppUtils_printf( APP_NAME ": SW Major Version: %d\n\r",versioninfo.sw_major_version);
   AppUtils_printf( APP_NAME ": SW Minor Version: %d\n\r",versioninfo.sw_minor_version);
   AppUtils_printf( APP_NAME ": SW Patch Version: %d\n\r",versioninfo.sw_patch_version);
   AppUtils_printf( " \n\r");
#endif

/* ICU Init*/
#if (STD_ON == ICU_PRE_COMPILE_VARIANT)
   AppUtils_printf(APP_NAME  ": Variant - Pre Compile being used !!!\n");
#else
   AppUtils_printf( APP_NAME  ": Variant - Post Build being used !!!\n\r");
#endif
   AppUtils_printf( " \n\r");

#if (STD_ON == ICU_PRE_COMPILE_VARIANT)
#if (STD_ON == ICU_SIGNAL_MEASUREMENT_API || STD_ON == ICU_EDGE_COUNT_API || STD_ON == ICU_EDGE_DETECT_API || STD_ON == ICU_TIMESTAMP_API)
    const Icu_ConfigType *pCfgPtr = &ICU_INIT_CONFIG_PC;
#endif
    Icu_Init(NULL_PTR);
#else
    const Icu_ConfigType *pCfgPtr = &IcuConfigSet;
    Icu_Init(pCfgPtr);
#endif


#if (STD_ON == ICU_SIGNAL_MEASUREMENT_API || STD_ON == ICU_EDGE_COUNT_API || STD_ON == ICU_EDGE_DETECT_API || STD_ON == ICU_TIMESTAMP_API)
    Icu_MeasurementModeType mode = pCfgPtr->chCfg[0].measurementMode;
#endif

#if (STD_ON == ICU_SIGNAL_MEASUREMENT_API)

    if (mode == ICU_MODE_SIGNAL_MEASUREMENT)
    {
       AppUtils_printf(APP_NAME ": Signal Measurement Mode! \n\r");
        Icu_SignalMeasurementPropertyType type = pCfgPtr->chCfg[0].signalMeasurementProperty;

        Icu_StartSignalMeasurement(ICU_CHANNEL);
        AppUtils_delay(100U);
        Icu_StopSignalMeasurement(ICU_CHANNEL);

        elapsedTime = Icu_GetTimeElapsed(ICU_CHANNEL);
        if (type == ICU_PERIOD_TIME)
           AppUtils_printf(APP_NAME ": elapsed time (Period) is  %d us\n\r", elapsedTime);
        else if (type == ICU_HIGH_TIME)
           AppUtils_printf(APP_NAME ": elapsed time (High Time) is  %d us\n\r", elapsedTime);
        else if (type == ICU_LOW_TIME)
           AppUtils_printf(APP_NAME ": elapsed time (Low Time) is  %d us\n\r", elapsedTime);

        Icu_GetDutyCycleValues(ICU_CHANNEL, &dutyCycleApp);
        activeTimeApp = dutyCycleApp.ActiveTime;
        periodTimeApp = dutyCycleApp.PeriodTime;
       AppUtils_printf(APP_NAME ": Active Time is %d \n\r", activeTimeApp);
       AppUtils_printf( APP_NAME ": Period Time is %d \n\r", periodTimeApp);
       float32 duty_value = activeTimeApp/(periodTimeApp*1.0);
       AppUtils_printf(APP_NAME ": Duty Cycle is %f\n\r",duty_value);

        elapsedTime = Icu_GetTimeElapsed(ICU_CHANNEL);
       AppUtils_printf( APP_NAME ": elapsed time read is  %d us\n\r", elapsedTime);

        Icu_StartSignalMeasurement(ICU_CHANNEL);
        AppUtils_delay(100U);
        Icu_StopSignalMeasurement(ICU_CHANNEL);

        elapsedTime = Icu_GetTimeElapsed(ICU_CHANNEL);
        if (type == ICU_PERIOD_TIME)
           AppUtils_printf( APP_NAME ": elapsed time (Period) is  %d us\n\r", elapsedTime);
        else if (type == ICU_HIGH_TIME)
           AppUtils_printf( APP_NAME ": elapsed time (High Time) is  %d us\n\r", elapsedTime);
        else if (type == ICU_LOW_TIME)
           AppUtils_printf( APP_NAME ": elapsed time (Low Time) is  %d us\n\r", elapsedTime);

        Icu_GetDutyCycleValues(ICU_CHANNEL, &dutyCycleApp);
        activeTimeApp = dutyCycleApp.ActiveTime;
        periodTimeApp = dutyCycleApp.PeriodTime;
       AppUtils_printf( APP_NAME ": Active Time is %d \n\r", activeTimeApp);
       AppUtils_printf( APP_NAME ": Period Time is %d \n\r", periodTimeApp);
        duty_value = activeTimeApp/(periodTimeApp*1.0);
       AppUtils_printf(APP_NAME ": Duty Cycle is %f\n\r",duty_value);

    }

#endif /* ICU_SIGNAL_MEASUREMENT_API*/

#if (STD_ON == ICU_TIMESTAMP_API)

    if (mode == ICU_MODE_TIMESTAMP)
    {
       AppUtils_printf( APP_NAME ": Timestamp Mode! \n\r");

        if (activation == ICU_RISING_EDGE)
           AppUtils_printf( APP_NAME ": Setting activation edge to RISING. \n\r");
        else if (activation == ICU_FALLING_EDGE)
           AppUtils_printf( APP_NAME ": Setting activation edge to FALLING. \n\r");
        else if (activation == ICU_BOTH_EDGES)
           AppUtils_printf( APP_NAME ": Setting activation edge to BOTH. \n\r");

        Icu_SetActivationCondition(ICU_CHANNEL, activation);

        Icu_EnableNotification(ICU_CHANNEL);
        Icu_StartTimestamp(ICU_CHANNEL, &timestampArray[0], bufferSize, 6);
        AppUtils_delay(100);
        
        while (!Icu_NotifyIntervalReached)
        {
           AppUtils_printf( APP_NAME ": still looping in timestamp \n\r");
        }
        Icu_NotifyIntervalReached = FALSE;

       AppUtils_printf( APP_NAME ": Array value 0 is %d \n\r", timestampArray[0]);
       AppUtils_printf( APP_NAME ": Array value 1 is %d \n\r", timestampArray[1]);
       AppUtils_printf( APP_NAME ": Array value 2 is %d \n\r", timestampArray[2]);
       AppUtils_printf( APP_NAME ": Array value 3 is %d \n\r", timestampArray[3]);
       AppUtils_printf( APP_NAME ": Array value 4 is %d \n\r", timestampArray[4]);
       AppUtils_printf( APP_NAME ": Array value 5 is %d \n\r", timestampArray[5]);

        Icu_StopTimestamp(ICU_CHANNEL);

    }

#endif /* ICU_TIMESTAMP_API*/

#if (STD_ON == ICU_EDGE_COUNT_API)

    if (mode == ICU_MODE_EDGE_COUNTER)
    {
       AppUtils_printf( APP_NAME ": Edge Count Mode! \n\r");

        Icu_EnableEdgeCount(ICU_CHANNEL);
        /* Wait 1 seconds */
        AppUtils_delay(1000);
        Icu_DisableEdgeCount(ICU_CHANNEL);

        edgecount = Icu_GetEdgeNumbers(ICU_CHANNEL);
       AppUtils_printf( APP_NAME ": edgecount in 1 sec is  %d \n\r", edgecount);

        Icu_ResetEdgeCount(ICU_CHANNEL);

        edgecount = Icu_GetEdgeNumbers(ICU_CHANNEL);
       AppUtils_printf( APP_NAME ": edgecount after reset is %d \n\r", edgecount);

    }

#endif

#if (STD_ON == ICU_EDGE_DETECT_API)

    if (mode == ICU_MODE_SIGNAL_EDGE_DETECT)
    {
       AppUtils_printf( APP_NAME ": Edge Detect Mode! \n\r");

        inputstate = Icu_GetInputState(ICU_CHANNEL);
        if(inputstate == ICU_ACTIVE)
           AppUtils_printf( APP_NAME ": input state is ICU_ACTIVE\n\r");
        else if (inputstate == ICU_IDLE)
           AppUtils_printf( APP_NAME ": input state is ICU_IDLE \n\r");

        Icu_EnableNotification(ICU_CHANNEL);

        Icu_EnableEdgeDetection(ICU_CHANNEL);

        AppUtils_delay(1000);

       AppUtils_printf( APP_NAME ": SignalNotification for Double Edge Detection Reached in 1 sec: %d \n\r", Icu_SignalNotificationNumber);
        Icu_SignalNotificationNumber = 0U;

        Icu_DisableEdgeDetection(ICU_CHANNEL);

        Icu_SetActivationCondition(ICU_CHANNEL,ICU_RISING_EDGE);

        Icu_EnableEdgeDetection(ICU_CHANNEL);

        AppUtils_delay(1000);

       AppUtils_printf( APP_NAME ": SignalNotification for Single Edge Detection Reached in 1 sec: %d \n\r", Icu_SignalNotificationNumber);
        Icu_SignalNotificationNumber = 0U;

        Icu_DisableEdgeDetection(ICU_CHANNEL);

        inputstate = Icu_GetInputState(ICU_CHANNEL);
        if(inputstate == ICU_ACTIVE)
           AppUtils_printf( APP_NAME ": input state is ICU_ACTIVE\n\r");
        else if (inputstate == ICU_IDLE)
           AppUtils_printf( APP_NAME ": input state is ICU_IDLE\n\r");

    }

#endif


#if (STD_ON == ICU_DE_INIT_API)
   AppUtils_printf( APP_NAME ":All sample test case pass \n\r");

    Icu_DeInit();

#endif
stop_timer();

}



uint8 MainFunc_Execution;
int32_t main(void)
/* In order to run the example */
{  
   
    uint32 testId;
    uint32 inputduty, inputfreq;
    uint32 period;
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;



#if (ICU_GET_VERSION_INFO_API == STD_ON)
    Std_VersionInfoType versioninfo;
#endif 

    AppUtils_defaultInit();

    Mcu_Init(&McuModuleConfiguration);
    Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;
    #if defined AWR29XXETS_PLATFORM
      mss_uart_tx_pin = 12;
      mss_uart_rx_pin = 8; 
    #endif
    #if defined (AM273X_PLATFORM) || defined (AM2732S_PLATFORM)
      mss_uart_tx_pin = 101;
      mss_uart_rx_pin = 80;
    #endif     

    /* Set up the pinmux for UART tx */
    Port_SetPinMode(mss_uart_tx_pin,PORT_PIN_MODE_LIN0);

    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);
    SOC_setEpwmTbClk(9, TRUE);
    SOC_setEpwmGroup(9, 0);
    


    Enable_Uart();
   

// for getting epwm output as input in ecap
// In the second argument select the input_xbar you are selecting , so that with help of this you can configure whichever xbar you want to use for input to get into the icu
    SOC_xbarSelectInputXBarInputSource(CSL_CONTROLSS_INPUTXBAR_U_BASE, 0, 0, INPUT_XBAR_GPIO61, 0);
    SOC_xbarSelectInterruptXBarInputSource(CSL_CONTROLSS_INTXBAR_U_BASE, 22, 0, 0, 0, 0, 0, ( INT_XBAR_ECAP0_INT ), 0);


Icu_InterruptConfig();

Pwm_Init(&PwmChannelConfigSet);

    AppUtils_printf("IcuApp: Sample Application - STARTS !!!\n\r ");
 
    IcuApp_mainTest();
}


#if (defined CLANG) || (defined DIAB)
void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
#endif

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
#if 0
        if (MCU_E_MODE_FAILURE == eventId)
        {
           AppUtils_printf("MCU Clock Error!!\n\r");
        }
#endif
       AppUtils_printf("Dem_ReportErrorStatus Clock error");

    }

    return;
}

