/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     Cdd_Ipc_notify_app.c
 *
 *  \brief    This file contains the IPC R50 test app for IPC Notify
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */


/*LDRA_NOANALYSIS*/
#include "string.h"
#include "Std_Types.h"
#include "Det.h"
#include "Dem.h"
/*LDRA_ANALYSIS*/
#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port.h"
#include "app_utils.h"
#include "sys_vim.h"
#include "Cdd_Ipc_Cfg.h"
#include "Cdd_Ipc.h"
#include "Cdd_Ipc_Irq.h"
#include "Cdd_Ipc_app.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define ARRAYSIZE(x)                    (sizeof ((x)) / sizeof (x[0]))

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */


/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

static void IpcApp_platformInit(void);

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */
#define LOCK 1
#define UNLOCK 0
/**
 * The max number of cores participating in the exchange
 * put it as 4 in case all remote cores are being used
 */
#define MAX 4

uint32 gMsgEchoCount = 1000000;

uint32 msgValue;

uint32 gClientId = 4; 
uint32 i,numRemoteCores = 0;

uint32 gRemoteCoreId[] = {
     CDD_IPC_R5FSS0_1_ID,
     CDD_IPC_R5FSS1_0_ID,
     CDD_IPC_R5FSS1_1_ID,
    MAX /* this value indicates the end of the array */
};

volatile uint32 gIpcTransferMutex[MAX] = {LOCK,LOCK,LOCK,LOCK};

static Mcu_ClockConfigType      gIpcAppMcuClockConfig[] =
{
     [0] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
         .Mcu_ClockSourceId = MCU_CLKSRC_3,
         .Mcu_ClockDiv = 3,
         .Mcu_InitCfg  = TRUE,
     },
     [1] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI0,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [2] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI1,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [3] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI2,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [4] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_WDT0,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
};

static Mcu_RamSectionConfigType gIpcAppMcuRamConfig[] =
{
};

static Mcu_ConfigType           gIpcAppMcuConfig =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = &(gIpcAppMcuRamConfig[0]),
    .Mcu_NumberOfRamSectors  = ARRAYSIZE(gIpcAppMcuRamConfig),
    .Mcu_ClockConfig         = &(gIpcAppMcuClockConfig[0]),
    .Mcu_NumberOfClockConfig = ARRAYSIZE(gIpcAppMcuClockConfig)
};

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

/* IPC Interrupt Configuration */
void Cdd_Ipc_InterruptConfig(void)
{
    vimInit();
    Vim_IntCfg interruptCfg;
    interruptCfg.map = VIM_INTTYPE_IRQ;
    interruptCfg.type = VIM_INTTRIGTYPE_PULSE;
    interruptCfg.intNum = 136U;
    interruptCfg.handler = &Cdd_Ipc_Isr;
    interruptCfg.priority = VIM_PRIORITY_15;
    vimRegisterInterrupt(&interruptCfg);
}

void Cdd_Ipc_notify_msgHandler(void* ipcNotifyHandle, uint32 Cdd_Ipc_remoteCoreId, uint16 Cdd_Ipc_remoteClientId, uint32 Cdd_Ipc_msgValue, uint32 timeout,void *args)
{
    /* increment msgValue and send it back until gMsgEchoCount iterations are done */
    if(Cdd_Ipc_msgValue != (gMsgEchoCount-1))
    {
        /* send new message to remote core, that echod our message */
        Cdd_Ipc_msgValue++;
        Cdd_Ipc_Notify_Write(Cdd_Ipc_remoteCoreId, gClientId, Cdd_Ipc_msgValue, 1);
    }
    else
    {
        /**
         * There is one semaphore for each core ID, so post the semaphore for the remote core that
         * has finished all message exchange iterations. Please note this is just a pseudo implementation
         * of the semaphores/mutexes using global variables
         */
        gIpcTransferMutex[Cdd_Ipc_remoteCoreId] = UNLOCK;
    }
}

/* Entry point of example */
int main(void)
{
        
    AppUtils_defaultInit();

    IpcApp_platformInit();                

    Cdd_Ipc_Init(&CddIpcDriver_0);

    start_timer();

    Cdd_Ipc_Notify_RegisterClient(gClientId, Cdd_Ipc_notify_msgHandler, NULL);

    AppUtils_printf(APP_NAME": Sample IPC Application started!\r\n");

    /* Interrupt Configuration (Registering Interrupt to R5 core, ISR) */
    /* Do this after Cdd_Ipc_Init is done because ISR can get triggerred due to SyncAll from Other core 
    and lead to data abort due to Un-initialized IPC handles*/
    Cdd_Ipc_InterruptConfig(); 

    Cdd_Ipc_Notify_Sync_All();
    AppUtils_printf(APP_NAME": Sync All Completed\r\n");

    AppUtils_printf(APP_NAME": Message exchange started by main core \r\n");

    for(i=0; gRemoteCoreId[i]!=MAX; i++)
    {
        uint32 msgValue = 0;
        /* send message's to all participating core's, wait for message to be put in HW FIFO */
        Cdd_Ipc_Notify_Write(gRemoteCoreId[i], gClientId, msgValue, 1);
    }

    /* wait for all messages to be echo'ed back */
    numRemoteCores = 0;
    for(i=0; gRemoteCoreId[i]!=MAX; i++)
    {

        while(LOCK == gIpcTransferMutex[gRemoteCoreId[i]]);
        numRemoteCores++;
    }

    AppUtils_printf(APP_NAME": All echoed messages received by main core from %d remote cores !!!\r\n", numRemoteCores);
    AppUtils_printf(APP_NAME": Messages sent to each core = %d \r\n", gMsgEchoCount);
    AppUtils_printf(APP_NAME": Number of remote cores = %d \r\n", numRemoteCores);

    Cdd_Ipc_Notify_UnregisterClient(gClientId);

    #if (STD_ON == CDD_IPC_DEINIT_API)
    Cdd_Ipc_DeInit(&CddIpcDriver_0);
    #endif
    
    stop_timer();

    return (0);

}

static void IpcApp_platformInit(void)
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;

    
    Mcu_Init(&gIpcAppMcuConfig);
    Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;    

    /* Set up the pinmux for UART tx */
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);

    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);

    Enable_Uart();


    return;
}

#if (defined CLANG) || (defined DIAB)
void SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
#endif

void SchM_Enter_Cdd_Ipc_IPC_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Cdd_Ipc_IPC_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}