/**
 *  \file     Cdd_Ipc_rpmsg_app.c
 *
 *  \brief    This file contains the IPC R50 test app for RpMsg
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */


/*LDRA_NOANALYSIS*/
#include "string.h"
#include <stdio.h>
#include "Std_Types.h"
#include "Det.h"
#include "Dem.h"
/*LDRA_ANALYSIS*/
#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port.h"
#include "app_utils.h"
#include "sys_vim.h"
#include "Cdd_Ipc_Cfg.h"
#include "Cdd_Ipc.h"
#include "Cdd_Ipc_Irq.h"
#include "Cdd_Ipc_app.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define ARRAYSIZE(x)                    (sizeof ((x)) / sizeof (x[0]))

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */


/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

static void IpcApp_platformInit(void);

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */

/**
 * The max number of cores participating in the exchange
 * put it as 4 in case all remote cores are being used
 */
#define MAX 4

#define MAX_MSG_SIZE (64U)

uint32 msgValue;
uint32 gMsgEchoCount = 100000;
uint16 gRemoteServiceEndPt = 13u;

uint16 remoteCoreId, remoteCoreEndPt, msgSize;
 
uint32 msg,i,numRemoteCores = 0;
char msgBuf[MAX_MSG_SIZE];

uint32 gRemoteCoreId[] = {
    CDD_IPC_R5FSS0_1_ID,
    CDD_IPC_R5FSS1_0_ID,
    CDD_IPC_R5FSS1_1_ID,
    MAX /* this value indicates the end of the array */
};


static Mcu_ClockConfigType      gIpcAppMcuClockConfig[] =
{
     [0] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
         .Mcu_ClockSourceId = MCU_CLKSRC_3,
         .Mcu_ClockDiv = 3,
         .Mcu_InitCfg  = TRUE,
     },
     [1] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI0,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [2] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI1,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [3] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI2,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [4] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_WDT0,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
};

static Mcu_RamSectionConfigType gIpcAppMcuRamConfig[] =
{
};

static Mcu_ConfigType           gIpcAppMcuConfig =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = &(gIpcAppMcuRamConfig[0]),
    .Mcu_NumberOfRamSectors  = ARRAYSIZE(gIpcAppMcuRamConfig),
    .Mcu_ClockConfig         = &(gIpcAppMcuClockConfig[0]),
    .Mcu_NumberOfClockConfig = ARRAYSIZE(gIpcAppMcuClockConfig)
};

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

/* IPC Interrupt Configuration */
void Cdd_Ipc_InterruptConfig(void)
{
    vimInit();
    Vim_IntCfg interruptCfg;
    interruptCfg.map = VIM_INTTYPE_IRQ;
    interruptCfg.type = VIM_INTTRIGTYPE_PULSE;
    interruptCfg.intNum = 136U;
    interruptCfg.handler = &Cdd_Ipc_Isr;
    interruptCfg.priority = VIM_PRIORITY_15;
    vimRegisterInterrupt(&interruptCfg);
}

/* Entry point of example */
int main(void)
{
        
    AppUtils_defaultInit();

    IpcApp_platformInit();         

    start_timer();         

    Cdd_Ipc_Init(&CddIpcDriver_0);

    Cdd_Ipc_Construct(&CddIpcDriver_0);

    numRemoteCores = 0;
    for(i=0; gRemoteCoreId[i]!=MAX; i++)
    {
        numRemoteCores++;
    }

    AppUtils_printf(APP_NAME": Sample IPC Application started!\r\n");

    /* Interrupt Configuration (Registering Interrupt to R5 core, ISR) */
    /* Do this after Cdd_Ipc_Init is done because ISR can get triggerred due to SyncAll from Other core 
    and lead to data abort due to Un-initialized IPC handles*/
    Cdd_Ipc_InterruptConfig(); 

    Cdd_Ipc_Notify_Sync_All();
    AppUtils_printf(APP_NAME": Sync All Completed\r\n");

    AppUtils_printf(APP_NAME": RpMsg Message exchange started by main core \r\n");

    for(msg=0; msg<gMsgEchoCount; msg++)
    {
        snprintf(msgBuf, MAX_MSG_SIZE-1, "%d", msg);
        msgBuf[MAX_MSG_SIZE-1] = 0;
        msgSize = strlen(msgBuf) + 1; /* count the terminating char as well */

        /* send the same messages to all cores */
        for(i=0; gRemoteCoreId[i]!=MAX; i++ )
        {
            Cdd_Ipc_RpMsg_SendMsg(msgBuf ,
                                  msgSize, 
                                  gRemoteCoreId[i], gRemoteServiceEndPt, 
                                  CddIpcConf_CddIpclocalEndPt_CddIpcEndPtConfig_0, CDD_IPC_TIMEOUT);
        }

        for(i=0; gRemoteCoreId[i]!=MAX; i++ )
        {
            msgSize = sizeof(msgBuf);
            Cdd_Ipc_RpMsg_RecvMsg(msgBuf, &msgSize, 
                                  CddIpcConf_CddIpclocalEndPt_CddIpcEndPtConfig_0,
                                  &remoteCoreId, &remoteCoreEndPt, CDD_IPC_TIMEOUT);
        }
    }

    AppUtils_printf(APP_NAME": All echoed messages received by main core from %d remote cores !!!\r\n", numRemoteCores);
    AppUtils_printf(APP_NAME": Messages sent to each core = %d \r\n", gMsgEchoCount);
    AppUtils_printf(APP_NAME": Number of remote cores = %d \r\n", numRemoteCores);

    Cdd_Ipc_Destruct(CddIpcConf_CddIpclocalEndPt_CddIpcEndPtConfig_0);

    #if (STD_ON == CDD_IPC_DEINIT_API)
    Cdd_Ipc_DeInit(&CddIpcDriver_0);
    #endif

    stop_timer();

    return (0);

}

static void IpcApp_platformInit(void)
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;

    
    Mcu_Init(&gIpcAppMcuConfig);
    Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;   

    /* Set up the pinmux for UART tx */
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);

    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);


    Enable_Uart();


    return;
}

#if (defined CLANG) || (defined DIAB)
void SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
#endif

void SchM_Enter_Cdd_Ipc_IPC_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Cdd_Ipc_IPC_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}