/* ======================================================================
 *   Copyright (c) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     PortExample.c
 *
 *  \brief    This file contains the DIO test example
 */

/*******************************************************************************
 *  INCLUDES
 ******************************************************************************/

/* Include Files */
#include "assert.h"
/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
/*LDRA_ANALYSIS*/
#include "Det.h"
#include "Dio.h"
#include "Mcu.h"
#include "EcuM_Cbk.h"
#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port.h"
#include "Os.h"


#include "app_utils.h"
#include "sys_vim.h"
#include "hw_ctrl_core.h"
#include "hw_types.h"

#define PORT_APP_PERFORMANCE_MODE   STD_ON

#if (STD_ON == PORT_APP_PERFORMANCE_MODE)
static void Port_Example_PerformanceTest();
uint32 Port_Performance_Result[30];
#endif



uint8 Port_MainFunc_Execution;

#ifdef __cplusplus
extern "C" {
#endif

void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();


}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

#ifdef __cplusplus
}
#endif

#define PORT_EXAMPLE_ARRAYSIZE(x)  (sizeof ((x)) / sizeof (x[0]))

#if (STD_ON == PORT_APP_PERFORMANCE_MODE)
static void Port_Example_PerformanceTest()
{
    const Port_ConfigType *Port_ConfigPtr;
    Std_VersionInfoType versioninfo;
    TickType StartValue = 0, ElapsedValue = 0;

    Port_ConfigPtr = &PortConfigSet_0;

    GetCounterValue(0,&StartValue);
    Port_Init(Port_ConfigPtr);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Port_Performance_Result[PORT_SID_INIT] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Port_SetPinDirection(PortConf_PortPin_PortPin_0, PORT_PIN_OUT);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Port_Performance_Result[PORT_SID_SET_PIN_DIR] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Port_RefreshPortDirection();
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Port_Performance_Result[PORT_SID_REFRESH_PORT_DIR] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Port_GetVersionInfo(&versioninfo);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Port_Performance_Result[PORT_SID_GET_VERSION_INFO] = ElapsedValue;

    GetCounterValue(0,&StartValue);
    Port_SetPinMode(14,PORT_PIN_MODE_LIN0);
    GetElapsedValue(0, &StartValue, &ElapsedValue);
    Port_Performance_Result[PORT_SID_SET_PIN_MODE] = ElapsedValue;

}
#endif


/** @fn void main(void)
 *   @brief Application main function
 *   @note This function is empty by default.
 *
 *   This function is called after startup.
 *   The user can use this function to implement the application.
 */
#if (STD_ON == PORT_VERSION_INFO_API)
Std_VersionInfoType       Port_VersionInfo;
#endif
sint32 Port_TestState = TRUE;
Dio_LevelType Port_PinLevel[64];

static void Port_Example_PlatformInit()
{
    uint16 uart_tx_pin, uart_rx_pin;
    uint32 sys_clk_freq_vclk_const;

    Mcu_Init(&McuModuleConfiguration);

#if (STD_ON == PORT_PRE_COMPILE_VARIANT)
    AppUtils_printf("Pre-Compile variant is being used..\n\r");
    Port_Init((const Port_ConfigType *) NULL_PTR);
#else
    AppUtils_printf("Post-Build variant is being used..\n\r");
    Port_Init(&PortConfigSet_0);
#endif

    uart_tx_pin = 13;
    uart_rx_pin = 14;

    /* Set up the pinmux for UART tx */
    Port_SetPinMode(uart_tx_pin, PORT_PIN_MODE_LIN0);

    /* Set up the pinmux for UART rx */
    Port_SetPinMode(uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(uart_rx_pin, PORT_PIN_IN);

#if (STD_OFF == MCU_NO_PLL)
    if(McuModuleConfiguration.Mcu_PllSourceId == MCU_CLKSRC_DPLL)
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_DPLL].Mcu_PllClk1.MCU_PLL_HSDIV2 / (2e6)) * 30;
    }
    else
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.Mcu_PllConfig[MCU_CLKSRC_APLL].Mcu_PllClk1.MCU_PLL_HSDIV0 / (2e6)) * 30;
    }
    Enable_Uart();
#else
    Enable_Uart();
#endif

}


int32_t main(void)
{
    uint32_least i;
    uint8 j=0;
    Std_ReturnType      retVal = E_OK;
    AppUtils_defaultInit();
    AppUtils_sectionInit();
    AppUtils_TimerInit();

    vimInit();
    Port_Example_PlatformInit();

    #if (STD_ON == PORT_APP_PERFORMANCE_MODE)
    Port_Example_PerformanceTest();
    #endif
 
    AppUtils_printf("CLANG compiled : portApp: Sample Application - STARTS !!!\n\r");

#if (STD_ON == PORT_VERSION_INFO_API)
    Port_GetVersionInfo(&Port_VersionInfo);
    AppUtils_printf("Port Driver version info:%d.%d.%d\n\r",
                    Port_VersionInfo.sw_major_version,
                    Port_VersionInfo.sw_minor_version,
                    Port_VersionInfo.sw_patch_version);
    AppUtils_printf("Port Driver Module/Driver:%d.%d \n\r",
                    Port_VersionInfo.moduleID,
                    Port_VersionInfo.vendorID);
#endif



    Port_SetPinDirection(PortConf_PortPin_PortPin_0, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_1, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_2, PORT_PIN_OUT);
	
	Port_SetPinDirection(PortConf_PortPin_PortPin_0, PORT_PIN_IN);
    Port_SetPinDirection(PortConf_PortPin_PortPin_1, PORT_PIN_IN);
    Port_SetPinDirection(PortConf_PortPin_PortPin_2, PORT_PIN_IN);
	
	Port_SetPinDirection(PortConf_PortPin_PortPin_0, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_1, PORT_PIN_OUT);
    Port_SetPinDirection(PortConf_PortPin_PortPin_2, PORT_PIN_OUT);
	
	
    Port_PinLevel[0] = Dio_ReadChannel(120);
    AppUtils_printf("Port Value for GPIOGH channel :%x \n\r",Port_PinLevel[0]);
	
	if (Port_PinLevel[0] == STD_HIGH)
	{
		Dio_FlipChannel(120);
		AppUtils_printf("Dio_FlipChannel - GPIOGH\n\r");
		Port_PinLevel[0] = Dio_ReadChannel(120);
		AppUtils_printf("Port Value:%x \n\r",Port_PinLevel[0]);
	}
	else
	{
		retVal = E_NOT_OK;
	}
	if (Port_PinLevel[0] == STD_LOW && retVal ==E_OK)
	{
		Dio_FlipChannel(120);
		AppUtils_printf("Dio_FlipChannel - GPIOGH\n\r");
		Port_PinLevel[0] = Dio_ReadChannel(120);
		AppUtils_printf("Port Value:%x \n\r",Port_PinLevel[0]);
	}
	else
	{
		retVal = E_NOT_OK;
	}
	
	Port_PinLevel[1] = Dio_ReadChannel(122);
	AppUtils_printf("Port Value for GPIOGH :%x \n\r",Port_PinLevel[1]);
	
	if (Port_PinLevel[1] ==STD_HIGH && retVal ==E_OK)
	{	

		Dio_FlipChannel(122);
		AppUtils_printf("Dio_FlipChannel - GPIOGH \n\r");
		Port_PinLevel[1] = Dio_ReadChannel(122);
		AppUtils_printf("Port Value:%x \n\r",Port_PinLevel[1]);
		
	}
	else
	{
		retVal = E_NOT_OK;
	}

	if (Port_PinLevel[1] ==STD_LOW && retVal ==E_OK)
	{
		Dio_FlipChannel(122);
		AppUtils_printf("Dio_FlipChannel -GPIOGH\n\r");

		Port_PinLevel[1] = Dio_ReadChannel(122);
		AppUtils_printf("Port Value:%x \n\r",Port_PinLevel[1]);
	}
	else
	{
		retVal = E_NOT_OK;
	}
	
    Dio_WriteChannel(120, STD_LOW);
	AppUtils_printf("Dio_WriteChannel -GPIOGH\n\r");
	
	Port_PinLevel[2] = Dio_ReadChannel(120);
	AppUtils_printf("Port Value:%x \n\r",Port_PinLevel[2]);
	
	if (Port_PinLevel[2] ==STD_LOW && retVal ==E_OK)
	{
		retVal = E_OK;
	}
    else
	{
		retVal = E_NOT_OK;
	}

    Dio_WriteChannel(120, STD_HIGH);
	AppUtils_printf("Dio_WriteChannel -GPIOGH\n\r");

	Port_PinLevel[2] = Dio_ReadChannel(120);
    AppUtils_printf("Port Value:%x \n\r",Port_PinLevel[2]);
	
	if (Port_PinLevel[2] ==STD_HIGH && retVal ==E_OK)
	{
		retVal = E_OK;
	}
    else
	{
		retVal = E_NOT_OK;
	}
	
    if(retVal ==  E_OK)
    {
        /* Testing Port_InversionSelect (Use SW1 switch on board to toggle the GPIO21)*/
        for(j=0; j<3; j++)
        {
            Port_PinLevel[0] = Dio_ReadChannel(21);
        }
    }

    if (AppUtils_checkStackAndSectionCorruption() != E_OK)
    {
        retVal = E_NOT_OK;
        AppUtils_printf(" PORT Stack/section corruption!!!\n\r");
    }

	if(retVal == E_OK)
	{
		AppUtils_printf("PORT Test Passed!!!\n\r");
	}
	else
	{
		AppUtils_printf("PORT Test Failed!!!\n\r");
	}
	
	AppUtils_TimerDeinit();
    return retVal;
}

#ifdef __cplusplus
extern "C" {
#endif

void EcuM_CheckWakeup(EcuM_WakeupSourceType wakeupSource)
{
    Port_MainFunc_Execution = 1;
}
void  SchM_Enter_Dio_DIO_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Dio_DIO_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}


void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        assert(FALSE);
    }

    return;
}

#ifdef __cplusplus
}
#endif
