/* ======================================================================
 *   Copyright (C) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     PwmApp.c
 *
 *  \brief    This file contains the PWM test example
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/*LDRA_NOANALYSIS*/
#include "string.h"
/*LDRA_ANALYSIS*/
#include "assert.h"
/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
#include "SchM_Pwm.h"
#include "SchM_Port.h"
#include "Det.h"
#include "Dem.h"
/*LDRA_ANALYSIS*/
#include "PwmApp.h"
#include "Pwm.h"
#include "Pwm_Irq.h"
#include "Pwm_Platform.h"

#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port_Cfg.h"
#include "Port.h"
#include "Os.h"
#include "trace.h"

#include "app_utils.h"
#include "sys_common.h"

#include "sys_vim.h"

/* For PRCM base addresses */
#include "soc.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define PWM_ARRAYSIZE(x)                (sizeof ((x)) / sizeof ((x)[0]))
#define PWM_CHANNEL_0                   (0U)
#define PWM_CHANNEL_1                   (1U)
#define PWM_CHANNEL                     (PWM_CHANNEL_0)
#define PWM_APP_PERFORM_TEST            (STD_ON)
#define PWM_APP_TEST                    (STD_ON)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/*None*/

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */
void Pwm_Channel_Notify1(void);
void Pwm_Channel_Notify2(void);
void Pwm_Notify_Test(uint32 ch);
void Pwm_enable_interrupt_(void);
void Pwm_Start_Cyclecount(void);
void Pwm_Notification_Channel1(void);
void Pwm_Start_Timer(void);
void Pwm_Stop_Timer(uint8 Api_Id);
static void Pwm_App_PlatformInit(void);
static void Pwm_App_InterruptConfig(uint32 channelId);
static void Pwm_App_InterruptInit(void);
static void Pwm_App_InterruptDisable(uint32 channelId);
static inline void EPWM_PWMconfigureSignal1(uint32 base);
static void SOC_setEpwmGroup(uint32 epwmInstance, uint32 group);
static void SOC_setEpwmTbClk(uint32 epwmInstance, uint32 enable);
static void SOC_controlModuleUnlockMMR(uint32 domainId, uint32 partition);
static void SOC_controlModuleLockMMR(uint32 domainId, uint32 partition);
static void Pwm_Example_PerformanceTest(void);
static inline void
SOC_xbarSelectInterruptXBarInputSource(uint32 base, uint8 out, uint32 group0_mask, uint32 group1_mask, uint32 group2_mask, uint32 group3_mask, uint32 group4_mask, uint32 group5_mask, uint32 group6_mask);
static void Pwm_App_MainTest(void);
typedef void (*PwmAppFxn_t)(void);

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */
/**< Flag used for Demo status */
#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
volatile static uint32  Pwm_App_NotifyCnt = 0U;
#endif

extern const struct Pwm_ConfigType_s PwmChannelConfigSet_0;
#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif

volatile uint32 Pwm_NotifyRecvFlag = 0xFFFF;
uint8 Pwm_Valid_WakeupSrcDetectedFlag = (uint8) FALSE;

Mcu_ClockConfigType Pwm_Mcu_ClkConfig[] =
{
   [0] = 
   { .Mcu_InitCfg       = TRUE,
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SYSCLK,   
    .Mcu_ClockSourceId = MCU_CLKSRC_2,    
    .Mcu_ClockDiv = 0U // Source = Sysclk ie 200MHz, so (200/(0+1) = 200MHz with 200MHz clk)
   },
   [1] =
    {
    .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
    .Mcu_ClockSourceId = MCU_CLKSRC_3,
    .Mcu_ClockDiv = 3,
    .Mcu_InitCfg  = TRUE,
    }   
};

static Mcu_ConfigType        Pwm_App_McuConfig =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = (Mcu_RamConfigPtrType) NULL_PTR,
    .Mcu_NumberOfRamSectors  = 0,
    .Mcu_ClockConfig         = (Mcu_ClockConfigPtrType)&Pwm_Mcu_ClkConfig,
    .Mcu_NumberOfClockConfig = PWM_ARRAYSIZE(Pwm_Mcu_ClkConfig),
};

PwmAppFxn_t                  Pwm_App_FxnTbl[] =
{
    &Pwm_App_MainTest
};

uint32 Pwm_Performance_Result[30];
/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

#if defined CLANG
void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void SchM_Enter_Pwm_PWM_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void SchM_Exit_Pwm_PWM_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();

}
#endif

int main(void)
{
    uint32 i;
	uint32 gTestPassed = E_OK;
    AppUtils_defaultInit();
	AppUtils_sectionInit();
    
    
	/*Enable Domain clocks*/
    Pwm_App_PlatformInit();
	
	/* Enable Interrupts. */
    Pwm_App_InterruptInit();	

    AppUtils_printf(APP_NAME ": Sample Application - STARTS !!!\n\r");

#if (PWM_APP_TEST == STD_ON)
    for (i = 0U; i < PWM_ARRAYSIZE(Pwm_App_FxnTbl); i++)
    {
        Pwm_App_FxnTbl[i]();
    }    
#endif

    GT_1trace(McalAppTrace, GT_INFO,
              " PWM Stack Usage: %d bytes\n\r", AppUtils_getStackUsage());
    if (AppUtils_checkStackAndSectionCorruption() != E_OK)
    {
        gTestPassed = E_NOT_OK;
        GT_0trace(McalAppTrace, GT_ERR, " PWM Stack/section corruption!!!\n\r");
    }

    while(1)
	{
		AppUtils_delay(10000U);
	
	    if(Pwm_App_NotifyCnt >= 1000)
	    {
			AppUtils_printf(APP_NAME ": Chek Isr count 1: %u \n\r", Pwm_App_NotifyCnt);
			Pwm_DisableNotification(PWM_CHANNEL);
	    }
		else
		{
			
		}
	}	
	AppUtils_printf(APP_NAME ": Sample Application - DONE !!!\n\r");
	
    return (0);
}

/* ========================================================================== */
/*                        Application Local Functions                         */
/* ========================================================================== */
static void Pwm_App_MainTest(void)
{
#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
    uint32 startTime = 0U, endTime = 0U, elapsed = 0U;
#endif

    uint32 clkSrc, inputduty;

    uint32 inputfreq, period;
    AppUtils_printf(APP_NAME  ": EPWM being used with Channel # %d!!! \n\r", PWM_CHANNEL);

    AppUtils_TimerInit();

#if (PWM_VERSION_INFO_API == STD_ON)
    Std_VersionInfoType versioninfo;

    /* Version Info Check*/
    Pwm_GetVersionInfo(&versioninfo);
    AppUtils_printf( " \n");
    AppUtils_printf( APP_NAME ": PWM MCAL Version Info\n\r");
    AppUtils_printf( "---------------------\n\r");
    AppUtils_printf( APP_NAME ": Vendor ID           : %d\n\r",
                                                        versioninfo.vendorID);
    AppUtils_printf( APP_NAME ": Module ID           : %d\n\r",
                                                        versioninfo.moduleID);
    AppUtils_printf( APP_NAME ": SW Major Version    : %d\n\r",
                                                versioninfo.sw_major_version);
    AppUtils_printf( APP_NAME ": SW Minor Version    : %d\n\r",
                                                versioninfo.sw_minor_version);
    AppUtils_printf( APP_NAME ": SW Patch Version    : %d\n\r",
                                                versioninfo.sw_patch_version);
    AppUtils_printf( " \n");
#endif

	AppUtils_printf(APP_NAME  ": Variant - Post Build being used !!!\n\r");
    const Pwm_ConfigType *pCfgPtr = &PwmChannelConfigSet;
    Pwm_Init(pCfgPtr);
	AppUtils_delay(1000U);

#if (STD_ON == PWM_SET_DUTY_CYCLE_API)
    /* Duty Cycle 80% */
    AppUtils_printf( " \n");
    AppUtils_printf(APP_NAME  ": Changing the Duty cycle from to 50 to 80 percent \n\r");
    inputduty = 0x6666U;
    Pwm_SetDutyCycle(PWM_CHANNEL_0, inputduty);
    AppUtils_printf(APP_NAME  ": This example waits for 30 seconds Please probe\n\r");
    AppUtils_delay(1000U);
#endif

#if (STD_ON == PWM_SET_DUTY_CYCLE_API)
    /* Duty Cycle 80% */
    AppUtils_printf( " \n");
    AppUtils_printf(APP_NAME  ": Changing the Duty cycle to 100 Percent \n\r");
    inputduty = 0x8000U;
    Pwm_SetDutyCycle(PWM_CHANNEL_0, inputduty);
	inputduty = 0x1625U;
    AppUtils_printf(APP_NAME  ": This example waits for 10 seconds Please probe\n\r");
    AppUtils_delay(1000U);
#endif

#if (STD_ON == PWM_SET_OUTPUT_TO_IDLE_API)
    /* Output to Idle */
    AppUtils_printf( " \n");
    AppUtils_printf(APP_NAME  ": Setting Output to Idle state \n\r");
    Pwm_SetOutputToIdle(PWM_CHANNEL_0);
	inputduty = 0x2500U;
    AppUtils_printf(APP_NAME  ": This example waits for 10 seconds Please probe\n\r");
    AppUtils_delay(1000U);
#endif

#if (STD_ON == PWM_SET_PERIOD_AND_DUTY_API)
    AppUtils_printf( " \n");
    AppUtils_printf(APP_NAME
         ": Changing Frequency from 1kHz to 10KHz and Duty Cycle to 50%%\n\r");

    inputfreq = 10000U;
    /* 50% duty cycle */
    /* Counter Direction set is - UPDOWN Mode to generate a symmetric PWM
     * PRD = (TBCLK/PWM_FREQ) / 2
     * TBCLK = (SYSCLK / (HSPCLKDIV x CLKDIV) , PWM_FREQ = 1 / TPWM
     * SYSCLK - System Clock(PWM_SYS_CLOCK)
     * HSPCLKDIV and CLKDIV are Prescalers to calculate Time base clock
     * Please refer PWM User guide for more details
     * Make sure period changed should use same prescalers used in init */
    clkSrc = pCfgPtr->chCfg[0].instanceClkHz;
    period = (clkSrc / (4U * 1U)); /* Prescalers */
    period = (period / inputfreq) / 2; /* UPDOWN Mode  */

    AppUtils_printf(APP_NAME ": period is set to %d\n\r", period);
    inputduty = 0x4000U;
    Pwm_SetPeriodAndDuty(PWM_CHANNEL_0, period, inputduty);
#endif /*PWM_SET_PERIOD_AND_DUTY_API */

#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
    Pwm_EnableNotification(PWM_CHANNEL, PWM_FALLING_EDGE);

    AppUtils_printf(APP_NAME ": This app again waits for 30 seconds please probe\n\r");

    /* Wait for ISR count */
#if (STD_ON == PWM_SET_DUTY_CYCLE_API) || \
        (STD_ON == PWM_SET_PERIOD_AND_DUTY_API)
    if ((inputduty == PWM_DUTY_0_PERCENT) ||
            (inputduty == PWM_DUTY_100_PERCENT))
    {
        AppUtils_delay(1000);
    }
    else
#endif /*PWM_SET_DUTY_CYCLE_API*/
    {
        while (Pwm_App_NotifyCnt < (inputfreq * 30U)); /*Multiplied with 30 for 30 seconds run */
    }
	
    AppUtils_printf(APP_NAME
              ": Pwm Isr Count: %u \n\r", Pwm_App_NotifyCnt);
			  
#endif /*PWM_NOTIFICATION_SUPPORTED*/

#if (STD_ON == PWM_NOTIFICATION_SUPPORTED)
    /* Disable Notifications */
    Pwm_App_NotifyCnt = 0;
    AppUtils_printf( " \n");
    AppUtils_printf(APP_NAME  ": Disabling Notifications for PWM channel\n\r");
    Pwm_DisableNotification(PWM_CHANNEL);
#endif

#if defined TEST_EHRPWM

#if (STD_ON == PWM_SET_PERIOD_AND_DUTY_API)
    AppUtils_printf( " \n");
    AppUtils_printf(APP_NAME
         ": Changing Frequency from 1kHz to 1.25MHz and Duty Cycle to 40.5%%\n\r");

    inputfreq = 1250000U;
    /* 50% duty cycle */
    /* Counter Direction set is - UPDOWN Mode to generate a symmetric PWM
     * PRD = (TBCLK/PWM_FREQ) / 2
     * TBCLK = (SYSCLK / (HSPCLKDIV x CLKDIV) , PWM_FREQ = 1 / TPWM
     * SYSCLK - System Clock(PWM_SYS_CLOCK)
     * HSPCLKDIV and CLKDIV are Prescalers to calculate Time base clock
     * Please refer PWM User guide for more details
     * Make sure period changed should use same prescalers used in init */
    clkSrc = pCfgPtr->chCfg[0].instanceClkHz;
    period = (clkSrc / (4U * 1U)); /* Prescalers */
    period = (period / inputfreq); /* UPDOWN Mode  */

    AppUtils_printf(APP_NAME ": period is set to %d\n\r", period);
    inputduty = period/2;
    Pwm_SetPeriodAndDuty(PWM_CHANNEL_0, period, inputduty);
#endif /*PWM_SET_PERIOD_AND_DUTY_API */
    AppUtils_printf(APP_NAME ": This app again waits for 30 seconds please probe\n\r");

    AppUtils_delay(30000);
	
	AppUtils_printf("PWM Test Passed for configuration!!\n\r");
    AppUtils_printf("PWM Test Completed !!!\n\r");

#endif /* TEST_EHRPWM */

   AppUtils_printf("PWM Test Completed12 !!!\n\r");
   AppUtils_TimerDeinit();
}

static void Pwm_App_PlatformInit(void)
{
    uint16 mss_uart_tx_pin, mss_uart_rx_pin;

    AppUtils_TimerInit();

#if (STD_OFF == MCU_NO_PLL)
    uint32 sys_clk_freq_vclk_const;

    /* MCU PLL Config */
    Pwm_App_McuConfig.Mcu_PllConfig = McuModuleConfiguration.Mcu_PllConfig;
    Pwm_App_McuConfig.Mcu_PllSourceId = McuModuleConfiguration.Mcu_PllSourceId;
#endif
	Mcu_Init(&McuModuleConfiguration);
	Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14;

    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);
    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
	Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);

    #if (STD_OFF == MCU_NO_PLL)
    if(McuModuleConfiguration.Mcu_PllSourceId == MCU_CLKSRC_DPLL)
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.
		Mcu_PllConfig[MCU_CLKSRC_DPLL].Mcu_PllClk1.MCU_PLL_HSDIV2 / (2e6)) * 30;
    }
    else
    {
        sys_clk_freq_vclk_const = (McuModuleConfiguration.
		Mcu_PllConfig[MCU_CLKSRC_APLL].Mcu_PllClk1.MCU_PLL_HSDIV0 / (2e6)) * 30;
    }
    Enable_Uart();
    #else
    Enable_Uart();
    #endif

    AppUtils_TimerDeinit();
    return;
}

static void Pwm_App_InterruptConfig(uint32 channelId)
{
    Vim_IntCfg intCfg;
    intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_PULSE;   
	
	intCfg.intNum = 146U;
    intCfg.handler = Pwm_Ch9Isr;
    intCfg.priority = 15;
    vimRegisterInterrupt(&intCfg);
}

static void Pwm_App_InterruptDisable(uint32 channelId)
{
    switch(channelId)
    {
        case 0: vimDisableInterrupt(146U);
                break; 
		case 1: vimDisableInterrupt(147U);
                break; 
    }
#ifdef PWMTIMER_QT 
        AppUtils_delay(2U);
#else         
        AppUtils_delay(20U);
#endif                 
    return;

}

static void Pwm_App_InterruptInit(void)
{
	vimInit();
	Pwm_App_InterruptConfig(0);
}

void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        assert(FALSE);
    }
}

void Pwm_Notification_Channel1(void)
{
    Pwm_App_NotifyCnt++;
}

void Pwm_Notification_Channel2(void)
{
    Pwm_App_NotifyCnt++;
}

void Pwm_Notification_Channel3(void)
{
    Pwm_App_NotifyCnt++;
}

void Pwm_Notification_Channel4(void)
{
    Pwm_App_NotifyCnt++;
}

/*EoF*/