/* ======================================================================
 *   Copyright (C) 2023 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     UartEchoDma.c
 *
 *  \brief    UART Echo DMA App example file
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/*LDRA_NOANALYSIS*/
#include "assert.h"
#include "string.h"
#include "Std_Types.h"
#include "Det.h"
#include "Dem.h"
/*LDRA_ANALYSIS*/
#include "Mcu.h"
#include "Mcu_Cfg.h"
#include "Port.h"
#include "app_utils.h"
#include "sys_vim.h"
#include "Cdd_Uart_Irq.h"
#include "CacheP.h"
#include "Cdd_Dma_Cfg.h"
#include "Cdd_Dma.h"
#include "UartEchoDma.h"

/* ========================================================================== */
/*                         Global Variables Declarations                      */
/* ========================================================================== */

CddUartDataBufferType CddUart_Buffer[CDD_UART_APP_BUFFERSIZE] __attribute__((aligned(Mcal_CacheP_CACHELINE_ALIGNMENT)));
CddUartDataBufferType CddUart_ReceiveBuffer[CDD_UART_APP_RECEIVE_BUFFERSIZE] __attribute__((aligned(Mcal_CacheP_CACHELINE_ALIGNMENT)));
volatile uint32 Uart_writeDone = 0, Uart_readDone = 0;

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

/* UART Interrupt Configuration */
void Cdd_Uart_InterruptConfiguration(uint8 interruptNumber)
{
    vimInit();
    Vim_IntCfg interruptCfg;
    interruptCfg.map = VIM_INTTYPE_IRQ;
    interruptCfg.type = VIM_INTTRIGTYPE_PULSE;
    interruptCfg.priority = VIM_PRIORITY_15;
    interruptCfg.intNum = interruptNumber;
    interruptCfg.handler = &Cdd_Uart_ChannelEdmaIsr;
    vimRegisterInterrupt(&interruptCfg);
}

/* Entry point of example */
int main(void)
{
    Apps_switch_to_system_mode();

    Mcu_Init(&McuModuleConfiguration);                                         /* MCU Clock Configuration */
    Port_Init(&PortConfigSet_0);                                               /* Pin MUX Configuration */
    Cdd_Uart_InterruptConfiguration(72U);                                      /* Interrupt Configuration (Registering Interrupt to R5 core, ISR) */

    Cdd_Dma_Init(NULL_PTR);                                                    /* DMA Initialization */

    uint32 count;
    uint8 channel = CddUartConf_CddUartChannelConfiguration_CddUartChannel_0;  /* Channel Index of the corresponding configuration from CDD_UART_Cfg Configurator */
    Cdd_Uart_Init(&CddUartDriver_0);                                           /* UART CDD Initialization */

    /* Creating buffer pointer */
    void *writeBuffer = &CddUart_Buffer, *readBuffer = &CddUart_ReceiveBuffer[0];

    /* Write completion message to the terminal */
    strncpy(writeBuffer, "\rThis is Uart Echo Test in DMA Mode\n\r", CDD_UART_APP_BUFFERSIZE);
    count = strlen(writeBuffer);
    Mcal_CacheP_wb(writeBuffer, count, Mcal_CacheP_TYPE_ALL);
    assert (Cdd_Uart_Write(channel, writeBuffer, count, 0) != E_NOT_OK);
    while(Uart_writeDone == 0);

    /* Echo prompt to the terminal */
    Uart_writeDone = 0;
    strncpy(writeBuffer, "\rPlease input 8 characters:\n\r", CDD_UART_APP_BUFFERSIZE);
    count = strlen(writeBuffer);
    Mcal_CacheP_wb(writeBuffer, count, Mcal_CacheP_TYPE_ALL);
    assert (Cdd_Uart_Write(channel, writeBuffer, count, 0) != E_NOT_OK);
    while(Uart_writeDone == 0) {}                                              /* Wait for write completion */

    /* Read characters */
    count = CDD_UART_APP_RECEIVE_BUFFERSIZE;
    Mcal_CacheP_wbInv(readBuffer, count, Mcal_CacheP_TYPE_ALL);
    assert (Cdd_Uart_Read(channel, readBuffer, count, 0) != E_NOT_OK);
    while(Uart_readDone == 0) {}                                               /* Wait for read completion */

    /* Echo characters read */
    Uart_writeDone = 0;
    Mcal_CacheP_wb(readBuffer, count, Mcal_CacheP_TYPE_ALL);
    assert (Cdd_Uart_Write(channel, readBuffer, count, 0) != E_NOT_OK);
    while(Uart_writeDone == 0) {}                                              /* Wait for write completion */

    /* Write completion message to the terminal */
    strncpy(writeBuffer, "\n\rEcho has been completed.\n\n\rAll tests have passed!\r\n", CDD_UART_APP_BUFFERSIZE);
    count = strlen(writeBuffer);
    Mcal_CacheP_wb(writeBuffer, count, Mcal_CacheP_TYPE_ALL);
    assert (Cdd_Uart_Write(channel, writeBuffer, count, 0) != E_NOT_OK);
    while(Uart_writeDone == 0);                                                /* Wait for write completion */

    Cdd_Uart_DeInit();                                                         /* UART CDD Deinitialization */

    return 0;
}

/* Callback functions */
void CddUart_CallbackReadNotify()
{
    Uart_readDone = 1U;
}
void CddUart_CallbackWriteNotify()
{
    Uart_writeDone = 1U;
}
void CddUart_CallbackErrorNotify() 
{
}

#if (defined CLANG) || (defined DIAB)
void SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
#endif

void SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void SchM_Enter_Cdd_Dma_DMA_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Cdd_Dma_DMA_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void SchM_Enter_Cdd_Uart_UART_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Cdd_Uart_UART_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void  App_disable(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   App_restore(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}
