/** @file CRC.h
*   @brief CRC Driver Header File
*
*   This file contains:
*   - Definitions
*   - Types
*   - Interface Prototypes
*   .
*   which are relevant for the CRC driver.
*/

/*
* Copyright (C) 2022 Texas Instruments Incorporated - TI web adress www.ti.com
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*---------------------------------------------------------------------------------------
* Revision History   :
*---------------------------------------------------------------------------------------
* Version  Date        Author       Defect No          Description
*---------------------------------------------------------------------------------------
*  0.1    31Mar2015    Jitendra   SDOCM00116715      Inital version
*  0.2    16Aug2016    Jitendra   AUTORADAR-555      All ENUMs removed, CRC configure
*                                                    structure modified to support compression
*                                                    on data size.
*  0.3    09Jun2017    Prathyusha AUTORADAR-1041     Changing the default static config values
*  0.4    07Sep2017    Prathyusha AUTORADAR-1221     MISRA 2012 complaince fixes
*  0.5    08Sep2017    Prahlad V  AUTORADAR-1197     Remove unused code
****************************************************************************************
*/


#ifndef CRC_H
#define CRC_H
#include "hal_stdtypes.h"

#ifdef __cplusplus
extern "C" {
#endif

/* CRC General Definitions */

/** @def CRCLEVEL_ACTIVE
*   @brief Alias name for CRC error operation level active
*/
#define CRCLEVEL_ACTIVE 0x00U


/** @def CRC_AUTO
*   @brief Alias name for CRC auto mode
*/
#define CRC_AUTO 0x00000001U


/** @def CRC_SEMI_CPU
*   @brief Alias name for semi cpu mode setting
*/
#define CRC_SEMI_CPU 0x00000002U


/** @def CRC_FULL_CPU
*   @brief Alias name for CRC cpu full mode
*/
#define CRC_FULL_CPU 0x00000003U


/** @def CRC_CH4_TO
*   @brief Alias name for channel4 time out interrupt flag
*/
#define CRC_CH4_TO 0x10000000U

/** @def CRC_CH4_UR
*   @brief Alias name for channel4 underrun interrupt flag
*/
#define CRC_CH4_UR 0x08000000U

/** @def CRC_CH4_OR
*   @brief Alias name for channel4 overrun interrupt flag
*/
#define CRC_CH4_OR 0x04000000U

/** @def CRC_CH4_FAIL
*   @brief Alias name for channel4 crc fail interrupt flag
*/
#define CRC_CH4_FAIL 0x02000000U

/** @def CRC_CH4_CC
*   @brief Alias name for channel4 compression complete interrupt flag
*/
#define CRC_CH4_CC 0x01000000U

/** @def CRC_CH3_TO
*   @brief Alias name for channel3 time out interrupt flag
*/
#define CRC_CH3_TO 0x00100000U

/** @def CRC_CH3_UR
*   @brief Alias name for channel3 underrun interrupt flag
*/
#define CRC_CH3_UR 0x00080000U

/** @def CRC_CH3_OR
*   @brief Alias name for channel3 overrun interrupt flag
*/
#define CRC_CH3_OR 0x00040000U

/** @def CRC_CH3_FAIL
*   @brief Alias name for channel3 crc fail interrupt flag
*/
#define CRC_CH3_FAIL 0x00020000U

/** @def CRC_CH3_CC
*   @brief Alias name for channel3 compression complete interrupt flag
*/
#define CRC_CH3_CC 0x00010000U

/** @def CRC_CH2_TO
*   @brief Alias name for channel2 time out interrupt flag
*/
#define CRC_CH2_TO 0x00001000U

/** @def CRC_CH2_UR
*   @brief Alias name for channel2 underrun interrupt flag
*/
#define CRC_CH2_UR 0x00000800U

/** @def CRC_CH2_OR
*   @brief Alias name for channel2 overrun interrupt flag
*/
#define CRC_CH2_OR 0x00000400U

/** @def CRC_CH2_FAIL
*   @brief Alias name for channel2 crc fail interrupt flag
*/
#define CRC_CH2_FAIL 0x00000200U

/** @def CRC_CH2_CC
*   @brief Alias name for channel2 compression complete interrupt flag
*/
#define CRC_CH2_CC 0x00000100U

/** @def CRC_CH1_TO
*   @brief Alias name for channel1 time out interrupt flag
*/
#define CRC_CH1_TO 0x00000010U

/** @def CRC_CH1_UR
*   @brief Alias name for channel1 underrun interrupt flag
*/
#define CRC_CH1_UR 0x00000008U


/** @def CRC_CH1_OR
*   @brief Alias name for channel1 overrun interrupt flag
*/
#define CRC_CH1_OR 0x00000004U

/** @def CRC_CH1_FAIL
*   @brief Alias name for channel1 crc fail interrupt flag
*/
#define CRC_CH1_FAIL 0x00000002U

/** @def CRC_CH1_CC
*   @brief Alias name for channel1 compression complete interrupt flag
*/
#define CRC_CH1_CC 0x00000001U

/** @def CRC_CH_INT_TO
*   @brief Alias name for channel1 time out interrupt flag
*/
#define CRC_CH_INT_TO 0x00000010U

/** @def CRC_CH_INT_UR
*   @brief Alias name for channel1 underrun interrupt flag
*/
#define CRC_CH_INT_UR 0x00000008U

/** @def CRC_CH_INT_OR
*   @brief Alias name for channel1 overrun interrupt flag
*/
#define CRC_CH_INT_OR 0x00000004U

/** @def CRC_CH_INT_FAIL
*   @brief Alias name for channel1 crc fail interrupt flag
*/
#define CRC_CH_INT_FAIL 0x00000002U

/** @def CRC_CH_INT_CC
*   @brief Alias name for channel1 compression complete interrupt flag
*/
#define CRC_CH_INT_CC 0x00000001U

/** @def CRC_CH1
*   @brief Alias name for channel1
*/
#define CRC_CH1    0x00000000U

/** @def CRC_CH2
*   @brief Alias name for channel2
*/
#define CRC_CH2    0x00000001U

/** @def CRC_CH3
*   @brief Alias name for channel3
*/
#define CRC_CH3    0x00000002U

/** @def CRC_CH4
*   @brief Alias name for channel4
*/
#define CRC_CH4    0x00000003U


/*
 * Signature Data size select
 */
#define CRC_DSIZE_DWORD         (0x0U)  /**< Data size Double Word */
#define CRC_DSIZE_HWORD         (0x1U)  /**< Data size Half Word   */
#define CRC_DSIZE_WORD          (0x2U)  /**< Data size Word        */

/*
 * CRC type select
 */
#define CRC_TYPE_CRC64          (0x0U)  /**< CRC 64             */
#define CRC_TYPE_CRC16          (0x1U)  /**< CRC16 CCITT        */
#define CRC_TYPE_CRC32E         (0x2U)  /**< CRC32 Ethernet     */
#define CRC_TYPE_CRC8V          (0x4U)  /**< CRC8 VDA CAN       */
#define CRC_TYPE_CRC8H          (0x5U)  /**< CRC8 H2F Autosar   */
#define CRC_TYPE_CRC32C         (0x6U)  /**< CRC32 CASTAGNOLI   */
#define CRC_TYPE_CRC32P         (0x7U)  /**< CRC32 E2E Profile4 */


/*
 * CRC Bit swap select
 */
#define CRC_MSBIT_FIRST         (0x0U)  /**< MSB bit first */
#define CRC_LSBIT_FIRST         (0x1U)  /**< LSB bit first */

/*
 * CRC Byte swap Enable
 */
#define CRC_BYTE_SWAP_DIS       (0x0U)  /**< Disable Byte swap */
#define CRC_BYTE_SWAP_ENA       (0x1U)  /**< Enable Byte swap  */

/*
 * CRC channel mode
 */
#define CRC_CHMODE_DATACAPT     (0x0U)  /**< Data capture mode */
#define CRC_CHMODE_AUTO         (0x1U)  /**< Auto mode         */
#define CRC_CHMODE_RESRV        (0x2U)  /**< Reserved mode     */
#define CRC_CHMODE_CPU          (0x3U)  /**< CPU mode          */

/** @struct crcModConfig
*   @brief CRC mode specific parameters
*
*   This type is used to pass crc mode specific parameters
*/
/** @typedef crcModConfig_t
*   @brief CRC Data Type Definition
*/
typedef struct crcModConfig
{
    uint32  crcChSel;       /**< CRC channel-0,1                                */
    uint32  mode;           /**< Mode of operation                              */
    uint64 *src_data_pat;   /**< Pattern data                                   */
    uint32 crcDataLen;      /**< Pattern data length.Number of 64 bit size word */
    uint8  crcType;         /**< CRC Type CRC-16\32\64                          */
    uint8  crcDataSize;     /**< CRC data size                                  */
} crcModConfig_t;

/** @struct crcConfig
*   @brief CRC configuration for different modes
*
*   This type is used to pass crc configuration
*/
/** @typedef crcConfig_t
*   @brief CRC Data Type Definition
*/
typedef struct crcConfig
{
    uint8 crcDataSize;    /**< CRC data size               */
    uint8 crcType;        /**< CRC Type CRC-16\32\64       */
    uint8 crcBitSwap;     /**< Bit Swap MSBit/LSBit first  */
    uint8 crcByteSwap;    /**< Byte Swap is Enable/Disable */
    uint32 crcChanNum;    /**< CRC channel-0,1             */
    uint32 crcChanMode;   /**< Mode of operation           */
    uint32 pcount;        /**< Pattern count               */
    uint32 scount;        /**< Sector count                */
    uint32 wdg_preload;   /**< Watchdog period             */
    uint32 block_preload; /**< Block period                */
} crcConfig_t;

typedef struct crc_config_reg
{
    uint32 CONFIG_CTRL0;
    uint32 CONFIG_CTRL1;
    uint32 CONFIG_CTRL2;
    uint32 CONFIG_INTS;
    uint32 CONFIG_WDTOPLD1;
    uint32 CONFIG_BCTOPLD1;
    uint32 CONFIG_WDTOPLD2;
    uint32 CONFIG_BCTOPLD2;
}crc_config_reg_t;

#define CRC_CTRL0_CONFIGVALUE            0x0000344AU
#define CRC_CTRL1_CONFIGVALUE            0x00000000U
#define CRC_CTRL2_CONFIGVALUE            ((uint32)((uint32)0U << 4U) | (uint32)(CRC_FULL_CPU)  | \
                                         (uint32)((uint32)CRC_FULL_CPU << 8U))
#define CRC_INTS_CONFIGVALUE             (((uint32)0x0000001EU) | ((uint32)0x00001F00U))
#define CRC_WDTOPLD1_CONFIGVALUE         (0x00FFFFFFU)
#define CRC_BCTOPLD1_CONFIGVALUE         (0x00FFFFFFU)
#define CRC_WDTOPLD2_CONFIGVALUE         (0x00FFFFFFU)
#define CRC_BCTOPLD2_CONFIGVALUE         (0x00FFFFFFU)

/* CRC Interface Functions */
uint16 crcCompute(uint8 *inData, uint32 inDataLen);


/**@}*/
#ifdef __cplusplus
}
#endif /*extern "C" */

#endif
