/* ======================================================================
 *   Copyright (c) 2022 Texas Instruments Incorporated
 *
 *   All rights reserved. Property of Texas Instruments Incorporated.
 *   Restricted rights to use, duplicate or disclose this code are
 *   granted through contract.
 *
 *   The program may not be used without the written permission
 *   of Texas Instruments Incorporated or against the terms and conditions
 *   stipulated in the agreement under which this program has been
 *   supplied.
 * ==================================================================== */

/**
 *  \file     WdgApp.c
 *
 *  \brief    This file contains the SPI test example for WdgApp
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

/*LDRA_NOANALYSIS*/
#include "Std_Types.h"
/*LDRA_ANALYSIS*/
#include "Det.h"
#include "Dem.h"
#include "Wdg.h"
#include "Wdg_Irq.h"
#include "SchM_Wdg.h"
#include "EcuM_Cbk.h"
#include "Mcu.h"
#include "Port_Cfg.h"
#include "Port.h"
/*LDRA_NOANALYSIS*/
#include "string.h"
/*LDRA_ANALYSIS*/
#include "trace.h"
#include "hw_types.h"
/* Starterware Includes */
#include "sys_vim.h"
#include "app_utils.h"
#include "soc.h"
#include "sys_common.h"
#include "sys_vim.h"
#include "hw_ctrl_core.h"
#include "WdgApp.h"

#include "Gpt.h"
#include "Gpt_Irq.h"
#include "SchM_Gpt.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define ARRAYSIZE(x)                    (sizeof ((x)) / sizeof (x[0]))
#define TIME_METRICS_ENABLED        STD_OFF
#define MSS_RCM_PARTITION0          (4)
#define SOC_DOMAIN_ID_MAIN          (0)
#define WDG_CTL_DISABLED          (0x5312ACED)
//#define WDGTIMER_QT 1 
#define GPT_TIMER STD_ON
#define GPT_RTI_CHANNEL1            (0U)
#define GPT_COUNTVAL                (0X1E8480) /* 20 ms =2*10^6*/
#define WDT_MAX_SERVICE             (0X1770) /* in ms */
#define DELAY_COUNT                  (500)  /* 500 *20ms = 10 sec delay*/
/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                 Internal Function Declarations                             */
/* ========================================================================== */

static void Wdg_App_wdgTest(const Wdg_App_TestParams *testPrms);

static void Wdg_App_platformInit(void);

void WDT_Isr();

Mcu_ResetType GetResetReason();

#if (GPT_TIMER == STD_ON)
void delay(uint32 count);
#endif

static void SOC_controlModuleUnlockMMR(uint32 domainId, uint32 partition);
static void SOC_controlModuleLockMMR(uint32 domainId, uint32 partition);

static void WDTApp_interruptConfig();

void Gpt_Channel_Notify5(void);
/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

uint32 triggerValue;
uint32 gTestPassed;
uint32 count = 0;
bool WDT_Reset = false;
static uint32 gptcount = 0;
bool Timer_Done = false;
static Mcu_RamSectionConfigType gWdgAppMcuRamConfig[] =
{
};

#if (STD_OFF == MCU_NO_PLL)
extern CONST(Mcu_ConfigType, MCU_PBCFG) McuModuleConfiguration;
#endif

#define WDGAPP_RTI_SYSCLK1_CLKSRCID    (0)

/* Gpt Channel Configuration parameters */
static CONST(Gpt_ChannelConfigType, GPT_PBCFG) GptChannelConfigSet_0_ChannelConfig[1] =
{
    [0] =
    {
	
	
        .ChannelId = (GPT_RTI_CHANNEL5),
        .ChannelMode = (GPT_CH_MODE_CONTINUOUS),  /* Gpt Channel Mode */
        .CounterBlk = GPT_RTI_COUNTER0,
        .GptChannelTickFrequency = 1U,  /* Pre-scalar value to be set */
        .Notification_pt = Gpt_Channel_Notify5,  /* Notification Call Back function */
        .GptChannelClksrcRef = 200U  /* Gpt Clock derived from Mcu in MHz*/
    },
};

/*<GPT_CONFIGURATION>*/
CONST(Gpt_ConfigType, GPT_PBCFG) GptChannelConfigSet_0 =
{
    GptChannelConfigSet_0_ChannelConfig,/*Pointer to Channel Configuration*/
    1/*Number of Channels*/
};

static Mcu_ClockConfigType      Wdg_AppMcuClockConfig[] =
{
     [0] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_SCI0,
         .Mcu_ClockSourceId = MCU_CLKSRC_3,
         .Mcu_ClockDiv = 3,
         .Mcu_InitCfg  = TRUE,
     },
     [1] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI0,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [2] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI1,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [3] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_RTI2,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
     [4] =
     {
         .Mcu_ClockModuleId = MCU_CLKSRC_MODULE_ID_WDT0,
         .Mcu_ClockSourceId = MCU_CLKSRC_2,
         .Mcu_ClockDiv = 0,
         .Mcu_InitCfg  = TRUE,
     },
};

static Mcu_ConfigType           Wdg_AppMcuConfig =
{
    .Mcu_ResetMode           = MCU_PERFORM_RESET_MODE_WARM,
    .Mcu_ConfigRamSection    = &(gWdgAppMcuRamConfig[0]),
    .Mcu_NumberOfRamSectors  = ARRAYSIZE(gWdgAppMcuRamConfig),
    .Mcu_ClockConfig         = &(Wdg_AppMcuClockConfig[0]),
    .Mcu_NumberOfClockConfig = ARRAYSIZE(Wdg_AppMcuClockConfig)
};

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

int main(void)

{
    Wdg_App_TestParams *testPrms;
    Mcu_ResetType mcu_reset_reason = MCU_RESET_UNDEFINED;

    gTestPassed = E_OK;

    AppUtils_defaultInit();
	AppUtils_sectionInit();
    Wdg_App_platformInit();
	AppUtils_TimerInit();
    mcu_reset_reason = GetResetReason();
	AppUtils_delay(4U);
	AppUtils_printf("Reset Reason %d \n\r", mcu_reset_reason);
    if(mcu_reset_reason != MCU_WARM_RESET_WDT0)
    {
        AppUtils_printf("LoopCount %d \n\r", MCU_WARM_RESET_WDT0);
        /* Application has not executed */
        *(uint32*)0x70001020 = 0x00000000;
#ifndef WDGTIMER_QT        
        AppUtils_printf(APP_NAME " Press Enter to start the application\r\n");
        AppUtils_getChar();
#endif         
        AppUtils_printf(APP_NAME ": Sample Application - STARTS !!!\r\n");
        testPrms         = &Wdg_AppTestPrms[0];
		WDTApp_interruptConfig();
        Wdg_App_wdgTest(testPrms);
    }
    else
    {
        AppUtils_printf("Test Passed!! \n\r");
    }

    AppUtils_printf("WDG Test Completed !!!\r\n");

    GT_1trace(McalAppTrace, GT_INFO,
        " WDG Stack Usage: %d bytes\r\n", AppUtils_getStackUsage());
    if (AppUtils_checkStackAndSectionCorruption() != E_OK)
    {
        gTestPassed = E_NOT_OK;
        GT_0trace(McalAppTrace, GT_ERR, " WDG Stack/section corruption!!!\r\n");
    }
	AppUtils_TimerDeinit();
    return (0);
}

#if (STD_ON == TIME_METRICS_ENABLED)
uint32 gServiceInterval = 0;
#endif

static void Wdg_App_wdgTest(const Wdg_App_TestParams *testPrms)
{

    uint32 maxServiceInterval;
    uint32 preloadVal;
	uint32 Gpt_Count = 0;
    Wdg_WindowSize windowSize;
    uint32 Cnt =0;
	uint32 RTI_Clock = 0;
	uint32 preload = 0;
	Gpt_ChannelType         ChannelId = GPT_RTI_CHANNEL5;
#if (STD_ON == TIME_METRICS_ENABLED)
    uint32              startTime;
    uint32              triggerTime;
#endif

#if (WDG_GET_VERSION_INFO_API == STD_ON)
    Std_VersionInfoType versioninfo;
    /* Get and print version */
    Wdg_GetVersionInfo(&versioninfo);
    AppUtils_printf(" \r\n");
    AppUtils_printf("WDG MCAL Version Info\r\n");
    AppUtils_printf("---------------------\r\n");
    AppUtils_printf("Vendor ID           : %d\r\n", versioninfo.vendorID);
    AppUtils_printf("Module ID           : %d\r\n", versioninfo.moduleID);
    AppUtils_printf("SW Major Version    : %d\r\n", versioninfo.sw_major_version);
    AppUtils_printf("SW Minor Version    : %d\r\n", versioninfo.sw_minor_version);
    AppUtils_printf("SW Patch Version    : %d\r\n", versioninfo.sw_patch_version);
    AppUtils_printf(" \r\n");
#endif  /* #if (WDG_CFG_VERSION_INFO_API == STD_ON) */

    AppUtils_printf("Starting WDG test  !!!\r\n");

    AppUtils_printf("Service Count %d\r\n", testPrms->numServiceCount);

    AppUtils_printf("Service Interval %d milliseconds\r\n",
                    testPrms->serviceInterval);

    triggerValue = ((testPrms->numServiceCount) * (testPrms->serviceInterval));
    AppUtils_printf("Trigger Value = %d milliseconds\r\n", triggerValue);

    /* check if the service interval is valid or not */

    if (testPrms->wdgConfig->defaultMode == WDGIF_FAST_MODE)
    {
        preloadVal = testPrms->wdgConfig->fastModeCfg.preloadValue;
		windowSize = testPrms->wdgConfig->fastModeCfg.windowSize;
    }
    else
    {
        preloadVal = testPrms->wdgConfig->slowModeCfg.preloadValue;
		windowSize = testPrms->wdgConfig->slowModeCfg.windowSize;
    }
	
	/* The expiration time of the DWD Down Counter can be determined with following equation:*/
	/* texp= ((RTI_DWDPRLD + 1) x (2^13) )/ RTI_FCLK */
    /* calculate the maximum service interval */
	/* texp in seconds */
    maxServiceInterval = (preloadVal + 1) * 8192;
    maxServiceInterval = maxServiceInterval/(WDG_RTI_FREQUENCY * 1000);
	switch(windowSize)
	{
		case WDG_WINDOW_SIZE_100_PERCENT:
			maxServiceInterval = maxServiceInterval;
			break;
		case WDG_WINDOW_SIZE_50_PERCENT:
			maxServiceInterval = maxServiceInterval/2;
			break;
		case WDG_WINDOW_SIZE_25_PERCENT:
			maxServiceInterval = maxServiceInterval/4;
			break;
		case WDG_WINDOW_SIZE_12_5_PERCENT:
			maxServiceInterval = maxServiceInterval/8;
			break;
		case WDG_WINDOW_SIZE_6_25_PERCENT:
			maxServiceInterval = maxServiceInterval/16;
			break;
		default:
			maxServiceInterval = maxServiceInterval/32;
			break;
	}
    AppUtils_printf("Maximum Service Interval %d milliseconds\r\n",
                    maxServiceInterval);

    if (testPrms->serviceInterval >= maxServiceInterval)
    {
        AppUtils_printf("service interval should not be greater than maximum interval\n\r");
        AppUtils_printf("Test Failed!!\n\r");
        return;
    }
	else if(triggerValue < maxServiceInterval)
	{
		AppUtils_printf("triggerValue should not be less than maximum interval\n\r");
        AppUtils_printf("Test Failed!!\n\r");
        return;
	}

	AppUtils_printf("Initializing channels\n\r");
	RTI_Clock = GptChannelConfigSet_0.ChannelConfig_pt[0].GptChannelClksrcRef;
	preload = GptChannelConfigSet_0.ChannelConfig_pt[0].GptChannelTickFrequency;
	Gpt_Count = (RTI_Clock*1000*testPrms->serviceInterval)/(preload+1);
    Gpt_Init(&GptChannelConfigSet_0);
	Gpt_EnableNotification(ChannelId);

#if (STD_ON == WDG_VARIANT_PRE_COMPILE)
	AppUtils_printf("Precompile variant is being used .. \n\r");
    Wdg_Init((const Wdg_ConfigType *)NULL_PTR);
#else
	AppUtils_printf("Post-Build variant is being used .. \n\r");
    Wdg_Init(testPrms->wdgConfig);
#endif
	

    if (gTestPassed == E_NOT_OK)
    {
        AppUtils_printf("Error in WatchDog Initialization\n\r");
        AppUtils_printf("Test Failed!!\n\r");
        return;
    }
	
	Gpt_StartTimer(ChannelId, Gpt_Count + 100);
#ifdef WDGTIMER_QT
    Wdg_SetTriggerCondition(5);
#else     
    Wdg_SetTriggerCondition(triggerValue - maxServiceInterval);
#endif 

    while(1)
	{
		if(gptcount >= DELAY_COUNT)
		{
			Gpt_DisableNotification(ChannelId);
			Gpt_StopTimer(ChannelId);
		}
    }
}

static void Wdg_App_platformInit(void)
{

    uint16 mss_uart_tx_pin, mss_uart_rx_pin; 
#if (STD_OFF == MCU_NO_PLL)    
    /* MCU PLL Config */
    Wdg_AppMcuConfig.Mcu_PllConfig = McuModuleConfiguration.Mcu_PllConfig;
    Wdg_AppMcuConfig.Mcu_PllSourceId = McuModuleConfiguration.Mcu_PllSourceId;
#endif	
    Mcu_Init(&Wdg_AppMcuConfig);
    Port_Init(&PortConfigSet_0);
    mss_uart_tx_pin = 13;
    mss_uart_rx_pin = 14; 

    /* Set up the pinmux for UART tx */
    Port_SetPinMode(mss_uart_tx_pin, PORT_PIN_MODE_LIN0);

    /* Set up the pinmux for UART rx */
    Port_SetPinMode(mss_uart_rx_pin, PORT_PIN_MODE_LIN0);
    Port_SetPinDirection(mss_uart_rx_pin, PORT_PIN_IN);
    Enable_Uart();

    return;
}

uint8 Intc_status;

void  SchM_Enter_Mcu_MCU_EXCLUSIVE_AREA_0()
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void   SchM_Exit_Mcu_MCU_EXCLUSIVE_AREA_0()
{
     AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void SchM_Enter_Wdg_WDG_EXCLUSIVE_AREA_0(void)
{
    __asm (" CPSID I");
}

void SchM_Exit_Wdg_WDG_EXCLUSIVE_AREA_0(void)
{
    __asm (" CPSIE I");
}

void  SchM_Enter_Port_PORT_EXCLUSIVE_AREA_0()
{
    __asm (" CPSID I");

}

void   SchM_Exit_Port_PORT_EXCLUSIVE_AREA_0()
{

  __asm (" CPSIE I");
}

void SchM_Enter_Gpt_GPT_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Enter_EXCLUSIVE_AREA_0();
}

void SchM_Exit_Gpt_GPT_EXCLUSIVE_AREA_0(void)
{
    AppUtils_SchM_Exit_EXCLUSIVE_AREA_0();
}

void Dem_ReportErrorStatus(Dem_EventIdType     eventId,
                           Dem_EventStatusType eventStatus)
{
    if (DEM_EVENT_STATUS_FAILED == eventStatus)
    {
        gTestPassed = E_NOT_OK;
        AppUtils_printf("WDG Hardware Error!!\r\n");
        GT_assert(McalAppTrace, FALSE);
    }

    return;
}

Mcu_ResetType GetResetReason()
{
	Mcu_ResetType mcu_reset_reason = MCU_RESET_UNDEFINED;
	uint32 Reset_Reason;
	if(Wdg_AppTestPrms[0].wdgConfig->fastModeCfg.reaction != WDG_GENERATE_NMI)
	{
		mcu_reset_reason = Mcu_GetResetReason();
	}
	else
	{
		Reset_Reason = HW_RD_REG32(0x53208010);
		if(Reset_Reason & 0x20)
		{
			mcu_reset_reason = MCU_WARM_RESET_WDT0;
		}
		HW_WR_REG32(0x53208014, 0x7);
	}
	return mcu_reset_reason;
}

static void SOC_controlModuleUnlockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;

    if(partition==CONTROLSS_CTRL_PARTITION0)
    {
        /*Unlock CONTROLSS_CTRL*/
        baseAddr = (uint32) CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK0_UNLOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK1_UNLOCK_VAL);      /* KICK 1 */
    }

    return;
}

static void SOC_controlModuleLockMMR(uint32 domainId, uint32 partition)
{
    uint32            baseAddr;
    volatile uint32  *kickAddr;
    
    
    if(partition==TOP_CTRL_PARTITION0)
    {
        /*Lock TOP_CTRL*/
        baseAddr = (uint32) CSL_TOP_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_TOP_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_TOP_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 1 */
    }
    
    if(partition==CONTROLSS_CTRL_PARTITION0)
    {
        /*Lock CONTROLSS_CTRL*/
        baseAddr = (uint32) CSL_CONTROLSS_CTRL_U_BASE;
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK0);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 0 */
        kickAddr = (volatile uint32 *) (baseAddr + CSL_CONTROLSS_CTRL_LOCK0_KICK1);
        HW_WR_REG32(kickAddr, TEST_KICK_LOCK_VAL);      /* KICK 1 */
    }

    return;
}

#if (GPT_TIMER == STD_ON)
void delay(uint32 count)
{
	uint32 frc;
	HW_WR_REG32(0x52180000, 0x00000000);//GCTRL
	HW_WR_REG32(0x52180008, 0x00000000);//Capture Control
	HW_WR_REG32(0x5218000C, 0x00000000);//COMP CTRL
	HW_WR_REG32(0x52180010, 0x00000000);//FRC0
	HW_WR_REG32(0x52180014, 0x00000000);//UP Counter
	HW_WR_REG32(0x52180018, 0x00000001);//Compare Up Counter
	//HW_WR_REG32(0x52180050, count*1000);//Compare 0
	HW_WR_REG32(0x52180084, 0x00070F0F);//Clear Interrupt
	//HW_WR_REG32(0x52180080, 0x00000001);//Set Interrupt
	HW_WR_REG32(0x52180000, 0x00000001);//GCTRL Start Timer
	while(1)
	{
		frc = HW_RD_REG32(0x52180010);
		if((count*100000)<frc)
		{
			HW_WR_REG32(0x52180000, 0x00000000);//GCTRL Stop Timer
			break;
		}
	}
}
#endif

__attribute__((target("arm")))
void WDT_Isr()
{
	
	vimDisableInterrupt(128U);
	
	HW_WR_REG32(0x52100090, WDG_CTL_DISABLED);
	
	SOC_controlModuleUnlockMMR(SOC_DOMAIN_ID_MAIN, MSS_RCM_PARTITION0);
	
	HW_WR_REG32(0x53208014, 0x7); //MSS_RCM_R5SS0_RST_CAUSE_CLR-Clear bit for rst cause register
	
	HW_WR_REG32(0x53208024, 0x7000707); // MSS_RCM_R5SS0_RST_WFICHECK
	
	HW_WR_REG32(0x53208044, 0x7000707); //MSS_RCM_R5SS1_RST_WFICHECK
	
	HW_WR_REG32(0x5320851C, 0x7); //MSS_RCM_R5SS1_CORE0_LRST_CTRL- Software needs to ensure the state of the Device/IP before configuring
	
	HW_WR_REG32(0x53208518, 0x7);//MSS_RCM_R5SS0_CORE0_LRST_CTRL
	SOC_controlModuleLockMMR(SOC_DOMAIN_ID_MAIN, MSS_RCM_PARTITION0);
}

void Gpt_Channel_Notify5(void)
{
	gptcount++;
	
	Wdg_SetTriggerCondition(triggerValue);
}

static void WDTApp_interruptConfig()
{
	vimInit();
    Vim_IntCfg intCfg;
    intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_PULSE;
	intCfg.intNum = RTI_WDT0_NMI;
	intCfg.handler = WDT_Isr;
	intCfg.priority = VIM_PRIORITY_10;
	vimRegisterInterrupt(&intCfg);
	
	intCfg.map = VIM_INTTYPE_IRQ;
    intCfg.type = VIM_INTTRIGTYPE_PULSE;
	intCfg.intNum = RTI1_INT0;
	intCfg.handler = Gpt_Ch4Isr;
	intCfg.priority = VIM_PRIORITY_11;
	vimRegisterInterrupt(&intCfg);
}

