//---------------------------------------------------------------------------------------------------------------
//  can_local.cc
//
//  CAN Bus functions.
//
//  Called can_local.cc instead of can.cc as can.h is in the TivaWare library.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//
//  Note that the 32 Message Objects that each CAN controller has are numbered 1-32 instead of 0-31.
//
//  As we're sampling the CANH bus to enable the power, we can't hold CANH at 2.5V as we get
//  CAN_BIT0_ERR - try to output a 0 but bus at 1.  So we're going to selectively use the second CAN
//  interface to send data and the first CAN will just receive characters.  May have to have an
//  alternate way to power up the board.
//
//  For now, we'll allocate Message Objects 1 - 16 to the main CAN-1 (S/W CAN-1) and 17-32 are for
//  CAN-2 (S/W CAN-0).  The odd number Message Objects for Tx messages and the even number Message
//  objects are for Rx messages.
//
//  Note that TivaWare only uses the CAN Interface 1 registers, so if somehow Interface 2 is needed
//  it is available.
//
//  *Important* If the CAN bus errors causes the CAN controller to go offline (E_CAN_BOFF_ERR), the
//  bus must be checked periodically to wait for the bus error condition to go away, after that
//  do the following:
//
//    CANEnable(CAN<0,1>_BASE);
//    while(CANStatusGet(CAN<0,1>_BASE) & CAN_STATUS_BUS_OFF) {};
//
//---------------------------------------------------------------------------------------------------------------


#include "can_local.h"
#include "error.h"
#include "uarts.h"
#include "state.h"
#include "init_cpu.h"
#include "events.h"
#include "dispatch.h"

#include "can.h"
#include "hw_memmap.h"
#include "interrupt.h"
#include "hw_can.h"
#include "hw_types.h"
//#include "tm4c123ae6pm.h"

#include <string.h>
#include <stdio.h>



                                        ///////////////////////////
                                        //                       //
                                        //    MESSAGE OBJECTS    //
                                        //                       //
                                        ///////////////////////////


//--------------------
//  MESSAGE OBJECTS  |
//---------------------------------------------------------------------------------------------------------------
//  tCANMsgObject:
//      u32 ui32MsgID
//      u32 ui32MsgIDMask
//      u32 ui32Flags
//      u32 ui32MsgLen
//      u8* pui8MsgData
//---------------------------------------------------------------------------------------------------------------
//  The Message Objects, 1-16 are for main CAN-1 (S/W CAN-1) and 17-32 to CAN-2 (S/W CAN-0).

enum eMessageNmb {MsgObj1=1, MsgObj2=2, MsgObj17=17, MsgObj18=18};

//---------------------------------------------------------------------------------------------------------------

PRIVATE u8 Can1Data1[8];                 // Message #1, Tx
PRIVATE tCANMsgObject Can1MsgObj1;

PRIVATE u8 Can1Data2[8];                 // Message #2, Rx
PRIVATE tCANMsgObject Can1MsgObj2;

//---------------------------------------------------------------------------------------------------------------
#ifdef ENABLE_CAN2

     u8 Can0Data17[8];                // Message #17, Tx
PRIVATE tCANMsgObject Can0MsgObj17;

     u8 Can0Data18[8];                // Message #18, Rx
PRIVATE tCANMsgObject Can0MsgObj18;

#endif // ENABLE_CAN2
//---------------------------------------------------------------------------------------------------------------





                                    ////////////////////////////////
                                    //                            //
                                    //    H/W CAN 1, S/W CAN 1    //
                                    //                            //
                                    ////////////////////////////////



//---------------------
//  SEND CAN MESSAGE  |
//---------------------------------------------------------------------------------------------------------------
void SendCan1Msg(u32 Id, u8* pMsg, u8 Length) {

    memcpy(Can1Data1, pMsg, Length);                   // Make data persistent.

    Can1MsgObj1.ui32MsgID     = Id;
    Can1MsgObj1.ui32MsgIDMask = 0x0;
    Can1MsgObj1.ui32Flags     = MSG_OBJ_TX_INT_ENABLE;
    Can1MsgObj1.ui32MsgLen    = Length;
    Can1MsgObj1.pui8MsgData   = Can1Data1;
    CANMessageSet(CAN1_BASE, MsgObj1, &Can1MsgObj1, MSG_OBJ_TYPE_TX);
}
//---------------------------------------------------------------------------------------------------------------
void SendCan1Msg(u32 Id, u32 Data, u8 Length) {

    static u32 Msg;

    memcpy(&Msg, &Data, Length);
    SendCan1Msg(Id, (u8*)Msg, Length);
}
//---------------------------------------------------------------------------------------------------------------




//-------------------------------
//  SET UP TO RECEIVE MESSAGES  |
//---------------------------------------------------------------------------------------------------------------
void SetUpRxCan1Msg(u32 Id, u32 IdMask ) {

    Can1MsgObj2.ui32MsgID     = Id;
    Can1MsgObj2.ui32MsgIDMask = IdMask;
    Can1MsgObj2.ui32Flags     = MSG_OBJ_RX_INT_ENABLE;
    Can1MsgObj2.ui32MsgLen    = 8;
    Can1MsgObj2.pui8MsgData   = Can1Data2;
    CANMessageSet(CAN1_BASE, MsgObj2, &Can1MsgObj2, MSG_OBJ_TYPE_RX);
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------
//  DEBUG PRINT CAN ERROR  |  *FIXME* Is this called?
//---------------------------------------------------------------------------------------------------------------
void PrtCanError(u32 Stat) {

    if(Stat == 0) {
        sprintf(sBuff, "!! ERROR - Null CAN Status !!\r\n");
        TxDebugStr(sBuff);
        return;
    }
    sBuff[0] = 0;                                       // Clear message buffer.
    strcat(sBuff, "CAN Error/Status: ");

    if(Stat & CAN_STATUS_BUS_OFF) strcat(sBuff, "CAN Bus Off, ");
    if(Stat & CAN_STATUS_EWARN)   strcat(sBuff, "Errors > 96, ");
    if(Stat & CAN_STATUS_EPASS)   strcat(sBuff, "In Error Passive State, ");
    if(Stat & CAN_STATUS_RXOK)    strcat(sBuff, "CAN Rx Reestablished, ");
    if(Stat & CAN_STATUS_TXOK)    strcat(sBuff, "CAN Tx Reestablished, ");

    Stat &= CAN_STATUS_LEC_MASK;                        // Get last error enumerations.
                                                        // (CAN_STATUS_LEC_MSK = CAN_STATUS_LEC_MASK)
    switch(Stat) {
        case CAN_NO_ERROR:
            strcat(sBuff, "No Error");
            break;
        case STUFF_ERROR:
            strcat(sBuff, "Stuff Error");
            break;
        case FORMAT_ERROR:
            strcat(sBuff, "Format Error");
            break;
        case ACK_ERROR:
            strcat(sBuff, "ACK Error");
            break;
        case BIT_1_ERROR:
            strcat(sBuff, "Bit-1 Error");
            break;
        case BIT_0_ERROR:
            strcat(sBuff, "Bit-0 Error");
            break;
        case CRC_ERROR:
            strcat(sBuff, "CRC Error");
            break;
        case NO_EVENT_ERROR:
            strcat(sBuff, "!! No Event !!");
            break;
        default:
            sprintf(sBuff, "can_local.cc line %d", __LINE__);
            Error(INVAL_SWITCH_VALUE, sBuff);
    }
    strcat(sBuff, "\r\n");
    TxDebugStr(sBuff);
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------
//  CAN 1 INTERRUPT HANDLER  |
//---------------------------------------------------------------------------------------------------------------
//  TO DO:  Need to keep status of going off-line to check status bits RXOK, TXOK, to generate
//  E_CAN_RX_REESTABLISHED and E_CAN_TX_REESTABLISHED.
//
//  For now were using odd Message Object numbers for Transmit interrupts and even numbers for
//  Receive interrupts.
//
//  Note that calling CANStatusGet() resets the interrupt if it was a Status interrupt, and
//  CANMessageGet() resets a Message interrupt.
//---------------------------------------------------------------------------------------------------------------
void Can1_Isr(void) {

    u8  Status;
    u32 Which;                                  // Which Message Object interrupted.
    struct sEvent_Msg Msg;

    Msg.SM = SM_MAIN;                           // All messages to Main SM.

    Status = CAN1_STS_R & 0x000000FF;           // Read Error/Status register.
    CAN1_STS_R = 0;                             // Reset it.

    Which = CAN1_INT_R;                         // See which object interrupted.
    Which = ((Which-1) & 0x1F) + 1;             // Mask
    CANIntClear(CAN1_BASE, Which);              // Reset Interrupt.

    if(Which == 0) {
        sprintf(sBuff, "\r\n- No interrupt pending !! Status = 0x%X\r\n", Status);
        TxDebugStr(sBuff);
        return;
    }
    if((Which & 0x0000FFFF) == 0x8000) {
        TxDebugStr("\r\n-Status Interrupt !!\r\n");
        return;
    }

    //----------------------------//
    //  SEE IF AN ERROR OCCURRED  //
    //----------------------------//

    if((Status & 0b11100000) != 0) {
                                                                        // Mask out RxOK and TxOK.
        if(Status & CAN_STS_BOFF)  Msg.Event = E_CAN_BOFF_ERR;
        if(Status & CAN_STS_EWARN) Msg.Event = E_CAN_EWARN_ERR;
        if(Status & CAN_STS_EPASS) Msg.Event = E_CAN_EPASS_ERR;

        // Send event here ***

        sprintf(sBuff, "\r\n- CAN1 Error Event = %s\r\n", Get_Event_Name(Msg.Event));
        TxDebugStr(sBuff);

    } else if((Status & 0b00000111) != 0) {

        if(Status == 1) Msg.Event = E_CAN_STUFF_ERR;
        if(Status == 2) Msg.Event = E_CAN_FORMAT_ERR;
        if(Status == 3) Msg.Event = E_CAN_ACK_ERR;
        if(Status == 4) Msg.Event = E_CAN_BIT1_ERR;
        if(Status == 5) Msg.Event = E_CAN_BIT0_ERR;
        if(Status == 6) Msg.Event = E_CAN_CRC_ERR;

        // Send event here ***

        sprintf(sBuff, "\r\n- CAN1 Error Event = %s\r\n", Get_Event_Name(Msg.Event));
        TxDebugStr(sBuff);

    } else {                                                            //- ELSE WE HAVE A MESSAGE -//

        //---------------------//
        //  WE HAVE A MESSAGE  //
        //---------------------//

        Msg.Event = E_CAN_MSG;

        if(Status == 8)  TxDebugStr("\r\n- CAN1 Tx OK\r\n");
        if(Status == 16) TxDebugStr("\r\n- CAN1 Rx OK\r\n");

        switch(Which) {

            case MsgObj1:

                CANMessageGet(CAN1_BASE, MsgObj1, &Can1MsgObj1, false);
                Msg.Data = (event_data_t)&Can1MsgObj1;                  // CanObjx has the CAN Msg ID.
//              ISR_Send_Message(SM_EVENT_TASK, (void*)&Msg);           // and a pointer to the Msg.

                TxDebugStr("\r\n- CAN1 Tx Message 1 Interrupt\r\n");
                break;

            case MsgObj2:

                CANMessageGet(CAN1_BASE, MsgObj2, &Can1MsgObj2, false);
//              Msg.Data = (event_data_t)&Can1MsgObj2;                  // CanObjx has the CAN Msg ID
//              ISR_Send_Message(SM_EVENT_TASK, (void*)&Msg);           // and a pointer to the Msg.

                TxDebugStr("\r\n- CAN1 Rx Message 2 Interrupt\r\n");
                sprintf(sBuff, "Msg ID = %d, Length = %d, Data = 0x%X\r\n\n",
                        Can1MsgObj2.ui32MsgID,
                        Can1MsgObj2.ui32MsgLen,
                        *(u32*)Can1MsgObj2.pui8MsgData);
                TxDebugStr(sBuff);

                break;

            default:

                sprintf(sBuff, "= %d, can_local.cc line %d", Which, __LINE__);
                Error(INVAL_SWITCH_VALUE, sBuff);
        }
    }
}
//---------------------------------------------------------------------------------------------------------------




                                        ////////////////////////////////
                                        //                            //
                                        //    H/W CAN 2, S/W CAN 0    //
                                        //                            //
                                        ////////////////////////////////


#ifdef ENABLE_CAN2

//------------------
//  LOOPBACK TEST  |
//---------------------------------------------------------------------------------------------------------------
u8 TxData[8] = { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF };
//---------------------------------------------------------------------------------------------------------------

void Can2LoopbackTest(void) {

    // Turn On Local Loopback //

    HWREG(CAN0_BASE + CAN_O_CTL) |= CAN_CTL_TEST;
    HWREG(CAN0_BASE + CAN_O_TST) |= CAN_TST_LBACK;

    // Create Receive Message Object //

    Can0MsgObj18.ui32MsgID     = 0;
    Can0MsgObj18.ui32MsgIDMask = 0;
    Can0MsgObj18.ui32Flags     = MSG_OBJ_RX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER;
    Can0MsgObj18.ui32MsgLen    = 8;
    Can0MsgObj18.pui8MsgData   = Can0Data18;
    CANMessageSet(CAN0_BASE, MsgObj18, &Can0MsgObj18, MSG_OBJ_TYPE_RX);

    // Create Transmit Message //

    SendCan2Msg(100, TxData, 8);
}
//---------------------------------------------------------------------------------------------------------------




//---------------------
//  SEND CAN MESSAGE  |
//---------------------------------------------------------------------------------------------------------------
void SendCan2Msg(u32 ID, u8* pMsg, u8 Length) {

    memcpy(Can0Data17, pMsg, Length);                   // Make data persistent.
/*
TxDebugStr("\r\n");
for(int i=0; i<Length; i++) {
    sprintf(sBuff, " 0x%X", Can0Data17[i]);
    TxDebugStr(sBuff);
}
TxDebugStr("\r\n");
*/
    Can0MsgObj17.ui32MsgID     = ID;
    Can0MsgObj17.ui32MsgIDMask = 0x0;
    Can0MsgObj17.ui32Flags     = MSG_OBJ_TX_INT_ENABLE;
    Can0MsgObj17.ui32MsgLen    = Length;
    Can0MsgObj17.pui8MsgData   = Can0Data17;
    CANMessageSet(CAN0_BASE, MsgObj17, &Can0MsgObj17, MSG_OBJ_TYPE_TX);
}
//---------------------------------------------------------------------------------------------------------------
void SendCan2Msg(u32 Id, u32 Data, u8 Length) {

    static u32 Msg;

    memcpy(&Msg, &Data, Length);
    SendCan2Msg(Id, (u8*)Msg, Length);
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------------------------
//  CONTROL OBD-2 EMULATOR VIA UDS MESSAGE PROTOCOL  |
//---------------------------------------------------------------------------------------------------------------
//  Sends a request for RPM and returns that value.
//  Response will be:
//  0   0x04    # of bytes
//  1   0x41    Response of mode 1
//  2   0x0C    Requested PID
//  3   0x0F    RPM MSB
//  4   0xA0    RPM LSB         4000 RPM in this example.
//  5,6,7       Padding
//---------------------------------------------------------------------------------------------------------------

uint /*RPM*/ GetRpm(void) {
/*
    CanObj1.ui32MsgID     = 0x7DF;                                          // Request for drivetrain data.
    CanObj1.ui32MsgIDMask = 0xFFFFF;
    CanObj1.ui32Flags     = MSG_OBJ_TX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER;
    CanObj1.pui8MsgData   = Can1Msg;
    Can1Msg[0] = 0x02;                  // 2 bytes follow
    Can1Msg[1] = 0x01;                  // Mode 1
    Can1Msg[2] = 0x0C;                  // PID for RPM.
    Can1Msg[3] = 0x00;
    Can1Msg[4] = 0x00;
    Can1Msg[5] = 0x00;
    Can1Msg[6] = 0x00;
    Can1Msg[7] = 0x00;
    CANMessageSet(CAN1_BASE, 0, &CanObj1, MSG_OBJ_TYPE_TX);

    CanObj2.ui32MsgID     = 0x7E8;                                          // Response to request.
    CanObj2.ui32MsgIDMask = 0xFFFFF;
    CanObj2.ui32Flags     = MSG_OBJ_RX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER;
    CanObj2.ui32MsgLen    = 8;
    CanObj2.pui8MsgData   = Can2Msg;
    CANMessageSet(CAN1_BASE, 1, &CanObj2, MSG_OBJ_TYPE_RX);
*/
    return(0);          // *FIXME*  Return the RPM.
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------
//  CAN 0 INTERRUPT HANDLER  |
//---------------------------------------------------------------------------------------------------------------
//  * TO DO *:  Need to keep status of going off-line to check status bits RXOK, TXOK, to generate
//  E_CAN_RX_REESTABLISHED and E_CAN_TX_REESTABLISHED.
//
//  For now were using odd Message Object numbers for Transmit interrupts and even numbers for
//  Receive interrupts.
//
//  Note that calling CANStatusGet() resets the interrupt if it was a Status interrupt, and
//  CANMessageGet() resets a Message interrupt.
//---------------------------------------------------------------------------------------------------------------
void Can0_Isr(void) {

    u8  Status;
    u32 Which;                                  // Which Message Object interrupted.
    struct sEvent_Msg Msg;

//  Msg.SM = SM_MAIN;                           // All messages to Main SM.

    Status = CAN0_STS_R & 0x000000FF;           // Read Error/Status register.
    CAN0_STS_R = 0;                             // Reset it.

    Which = CAN0_INT_R;                         // See which object interrupted.
    Which = ((Which-1) & 0x1F) + 1;             // Mask
    CANIntClear(CAN0_BASE, Which);              // Reset Interrupt.

    if(Which == 0) {
        sprintf(sBuff, "\r\n- CAN0 No interrupt pending !! Status = 0x%X\r\n", Status);
        TxDebugStr(sBuff);
        return;
    }
    if((Which & 0x0000FFFF) == 0x8000) {
        TxDebugStr("\r\n- CAN0 Status Interrupt !!\r\n");
        return;
    }

    //----------------------------//
    //  SEE IF AN ERROR OCCURRED  //
    //----------------------------//

    if((Status & 0b11100000) != 0) {
                                                                        // Mask out RxOK and TxOK.
        if(Status & CAN_STS_BOFF)  Msg.Event = E_CAN_BOFF_ERR;
        if(Status & CAN_STS_EWARN) Msg.Event = E_CAN_EWARN_ERR;
        if(Status & CAN_STS_EPASS) Msg.Event = E_CAN_EPASS_ERR;

        // Send event here ***

        sprintf(sBuff, "\r\n- CAN0 Error Event = %s\r\n", Get_Event_Name(Msg.Event));
        TxDebugStr(sBuff);

    } else if((Status & 0b00000111) != 0) {

        if(Status == 1) Msg.Event = E_CAN_STUFF_ERR;
        if(Status == 2) Msg.Event = E_CAN_FORMAT_ERR;
        if(Status == 3) Msg.Event = E_CAN_ACK_ERR;
        if(Status == 4) Msg.Event = E_CAN_BIT1_ERR;
        if(Status == 5) Msg.Event = E_CAN_BIT0_ERR;
        if(Status == 6) Msg.Event = E_CAN_CRC_ERR;

        // Send event here ***

        sprintf(sBuff, "\r\n- CAN0 Error Event = %s\r\n", Get_Event_Name(Msg.Event));
        TxDebugStr(sBuff);

    } else {

//      Msg.Event = E_CAN_MSG;

        if(Status == 8)   TxDebugStr("\r\n- CAN0 Tx OK\r\n");
        if(Status == 16)  TxDebugStr("\r\n- CAN0 Rx OK\r\n");

        switch(Which) {

            case MsgObj17:                                              //-- TRANSMIT INTERRUPT --//

                CANMessageGet(CAN0_BASE, MsgObj17, &Can0MsgObj17, false);
                Msg.Data = (event_data_t)&Can0MsgObj17;                 // Can0Obj<X> has the CAN Msg ID.
//              ISR_Send_Message(SM_EVENT_TASK, (void*)&Msg);           // and a pointer to the Msg.

                TxDebugStr("\r\n- CAN0 Tx Message 17 Interrupt\r\n");

                break;

            case MsgObj18:                                              //-- RECEIVE INTERRUPT --//

                CANMessageGet(CAN0_BASE, MsgObj18, &Can0MsgObj18, false);
//              Msg.Data = (event_data_t)&Can0MsgObj18;                 // Can0Obj<X> has the CAN Msg ID.
//              ISR_Send_Message(SM_EVENT_TASK, (void*)&Msg);           // and a pointer to the Msg.

                TxDebugStr("\r\n- CAN0 Rx Message 18 Interrupt\r\n");
                sprintf(sBuff, "Msg ID = %d, Length = %d, Data = 0x%0X 0x0X\r\n\n",
                        Can0MsgObj18.ui32MsgID,
                        Can0MsgObj18.ui32MsgLen,
                       *(u32*)Can0MsgObj18.pui8MsgData);
                TxDebugStr(sBuff);

                break;

                //  tCANMsgObject:
                //      u32 ui32MsgID
                //      u32 ui32MsgIDMask
                //      u32 ui32Flags
                //      u32 ui32MsgLen
                //      u8* pui8MsgData

            default:

                sprintf(sBuff, "= %d, can_local.cc line %d", Which, __LINE__);
                Error(INVAL_SWITCH_VALUE, sBuff);
        }
    }
}
//---------------------------------------------------------------------------------------------------------------

#endif // ENABLE_CAN2



                                          //////////////////////
                                          //                  //
                                          //    INITIALIZE    //
                                          //                  //
                                          //////////////////////



//-----------------------
//  INITIALIZE CAN BUS  |
//---------------------------------------------------------------------------------------------------------------
//  To start off we'll just create 2 message buffers - e.g. 2 CAN Objects and their 8 byte buffers.
//  We'll set them up to only capture message #0xC for engine speed and  and 0xD for speedometer (??).
//
//  To read OBD2 RPM: typically use Mode 01, PID 0x0C.  Send message ID: 0x7DF (Std OBD2 query) -
//  02 01 0C 00 00 00 00 00.  The response will be ID: 0x7E8 #Bytes, Mode 0x41, PID 0x0C, RPM, usually
//  2 bytes - my notes say ((Byte 3 * 256) + Byte 2) / 4.
//---------------------------------------------------------------------------------------------------------------
void Init_CAN(void) {

    //--  H/W CAN-1,  S/W CAN-1  --//

    CANInit(CAN1_BASE);
    CANBitRateSet(CAN1_BASE, GetClkSpeed(), 500000);
    CANIntRegister(CAN1_BASE, Can1_Isr);
    CANIntEnable(CAN1_BASE, CAN_INT_MASTER | CAN_INT_ERROR | CAN_INT_STATUS);
    IntEnable(INT_CAN1);
//  CAN1_TST_R |= CAN_TST_SILENT;               * FIXME *                            // We'll just sniff the bus.
    CANEnable(CAN1_BASE);

    Can1MsgObj2.ui32MsgID     = 0x0;                                        // Receive all messages.
    Can1MsgObj2.ui32MsgIDMask = 0x0;
    Can1MsgObj2.ui32Flags     = MSG_OBJ_RX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER;
    Can1MsgObj2.ui32MsgLen    = 8;
    Can1MsgObj2.pui8MsgData   = Can1Data1;
    CANMessageSet(CAN1_BASE, MsgObj1, &Can1MsgObj2, MSG_OBJ_TYPE_RX);
/*
    CanObj2.ui32MsgID     = 0x7E8;                                          // Response to request.
    CanObj2.ui32MsgIDMask = 0xFFFFF;
    CanObj2.ui32Flags     = MSG_OBJ_TX_INT_ENABLE | MSG_OBJ_USE_ID_FILTER;
    CanObj2.pui8MsgData   = Can2Msg;
    CANMessageSet(CAN1_BASE, Msg2, &CanObj2, MSG_OBJ_TYPE_RX);
*/

#ifdef ENABLE_CAN2

    //--  H/W CAN-2,  S/W CAN-0  --//

    CANInit(CAN0_BASE);
    CANBitRateSet(CAN0_BASE, GetClkSpeed(), 500000);
    CANIntRegister(CAN0_BASE, Can0_Isr);
    CANIntEnable(CAN0_BASE, CAN_INT_MASTER | CAN_INT_ERROR | CAN_INT_STATUS);
    IntEnable(INT_CAN0);
    CANEnable(CAN0_BASE);

#endif // ENABLE_CAN2
}
//---------------------------------------------------------------------------------------------------------------

