//---------------------------------------------------------------------------------------------------------------
//  dispatch.cc
//
//  Cooperative non-preemptive task dispatcher with inter-task messaging.  See timer.c for more description
//  of its operation.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------


#include "tm4c123ae6pm.h"
#include "hw_types.h"
#include "hw_memmap.h"
#include "stdint.h"
#include "cpu.h"
#include "gpio.h"
#include "sysctl.h"
#include "interrupt.h"

#include "common.h"
#include "sm_bluetooth.h"
#include "uarts.h"
#include "error.h"
#include "dispatch.h"
#include "tick.h"
#include "timers.h"
#include "events.h"
#include "monitor.h"
#include "eeprom.h"
#include "test.h"
#include "bt_comm.h"

#include <string.h>	// For memcpy();
#include <stdio.h>




//-----------------------------
//  DISPATCHER DOCUMENTATION  |
//----------------------------------------------------------------------------------------------------------
// This dispatcher implements a simple "Run to Completion" type of scheduler (ala DOS, Win95/98, etc).
// As such, tasks cannot Block, if a task is going to take a long time to run it must Return to the
// dispatcher at reasonable intervals so that other tasks have a chance to run; this can be implemented
// by a task keeping local inter-dispatch state to deal with its own "local" dispatching, or by having
// multiple dispatch entries in the dispatch structure.
//
// Tasks do not have separate stacks, ALL code uses the same (Supervisor) stack.  No task is interruptible
// except by actual interrupt routines - which are expected to gracefully save and restore CPU state and to
// run for the shortest time as possible  Thus, this application follows the "Foreground / Background"
// architecture of many embedded systems (as opposed to preemptable multi-tasking operating systems).
// This is why there isn't a "sleep()" type of call, timing is done by not dispatching for a given
// number of "ticks".
//
// The dispatch table (queue) is statically constructed, i.e., tasks cannot be added to or deleted
// from the dispatch table (queue) at runtime, however, dispatching to a given task can be turned on
// and off dynamically.
//
// Each application (task) that is dispatched to can have its own Message format, implemented as a
// structure.  When a task is dispatched to it is passed the number of messages it has for it and
// a pointer to the first message (or NULL if no messages).  If there is more than one message,
// the task can subsequently call Get_Message(uword Task_Enumeration) to get the pointer to the
// next message for it.  Tasks can decide to handle only one message per dispatch by only handling
// the message sent at dispatch, or can decide to handle multiple messages per dispatch via
// Get_Message().  If desired for a task to have multiple Message formats, it must have another
// call point and entry in the Dispatch Table or have a message structure that has pointers in its
// (native) message format that point to other message queues (not programmed for this yet).
//
// Interrupt Service Routines (ISR's) can post messages to tasks by calling ISR_Send_Message(). This
// puts the message into a temporary queue which is then placed into the regular message queue
// synchronously in-between task dispatches.
//
// **********************
// *** BIG OL' CAVEAT ***  --> If a Task's messaging structure includes character pointers, e.g.
// **********************      includes 'char*', care must be taken in the lifetime of that string!
//			       When a message is sent [via 'Send_Message()'] to a given task, the
// structure is physically copied to that task's message array as the passed array will probably
// be an automatic variable who's lifetime ends when the issuing routine exits.  However, currently,
// Send_Message() does *NOT* copy any actual strings, just the pointers!  I believe that the compiler
// creates static constant arrays of strings when a string is declared literally, e.g. "This is a literal
// string", however, if a routine is dynamically creating a string whose pointer is passed in a message to
// another Task, that string *MUST* be kept somewhere whose lifetime is such that it will exist when
// the message'd task finally reads the message and finishes using (or copies) that string (if
// necessary the message'd task could send a message to the sending task once it is read to allow
// the sending task to free its message buffer).
//
// *WARNING*  Interrupts are disabled for a little while in Send_Message() to allow the above to
//            work.  If any interrupt routines cannot stand this latency they must be made high-
//	      priority and the DISABLE_INTERRUPTS and ENABLE_INTERRUPTS code must not mask them
//	      out.
//
// Any task can change any other program's Dispatch Control and Dispatch Time; if not dispatching on
// time "Set_Dispatch_Control(...)" [or "ISR_Set_Dispatch_Ctl(...)" for ISR's] is used, if a timeout
// is being set then "Set_Dispatch_Time(...)" [or "ISR_Dispatch_Time(...)" for ISR's] is used.  Real
// care must be taken if a task changes another's Priority Control and/or Priority/Dispatch Time;
// normally only an "executive" type function or a task's background process would do this.  An
// example of the type of problem one could get into would be a task implementing the equivalent
// of a "sleep()" via MSG_ONLY and another task changes its Dispatch Control or Priority and it
// gets dispatched to earlier or later than it thinks it has (use of an appropriate message could
// allow the graceful handling of this condition if needed).
//
// If the Ticks_Since_Last_Dispatch counter overflows, it is kept to ~(0) (maximum tick count).
//
// Dispatch_Control has the following values:
//
// 1	Off			Don't dispatch to this process at all (DISP_OFF).
// 2	On, Full Priority	The Priority Counter or Dispatch Times ignored, always dispatch (FULL_PRI).
// 3	On, One-Shot Time Only	Use the Dispatch Time to determine when to dispatch (TIME_ONCE) once.
// 4    On, Repeating Time Only Use the Dispatch Time to determine when to dispatch (TIME_CONT) repeatedly.
// 5	On, Message Priority	Use Repeating Time Priority, as above, unless a message is posted in which
//				case dispatch asap (MSG_PRI) & time is reset (so can time out again).  The
//                              invoked routine looks at the # of messages available to determine if it
//                              was a timeout that caused the dispatch (i.e., simultaneous message & timeout
//                              is considered not to have timed out).
// 6	On, Message Only	Don't dispatch until a message is posted (MSG_ONLY).
//
// Be very careful using Full Priority as the CPU will never power-down when any task is in this mode.
//
// When adding or deleting a task from the dispatch scheduler the following items need to be edited:
//
//    The structure:   DT
//    The functions:   Dispatch_Task(), Get_Message(), Send_Message(), and ISR_Send_Message() as appropriate
//                     (only necessary for routines that accept messages).
//    The message structure declaration in dispatch.h and the definition in this file.  If messages will
//    be posted to this task from an ISR then an ISR message queue must be defined.
//
// Programs dispatched to have the following parameters:
//
// -  The number of messages waiting for them,
// -  A pointer to the first message (NULL if no messages),
// -  If in MSG_PRI mode: true if dispatched on Timeout, FALSE otherwise so a MSG_PRI task can can know
//    if a simultaneous Message Available and Timeout happened.
// -  The number of Ticks that have elapsed since its last dispatch.
//
// **********
// ** NOTE **
// **********
//
// When there are no tasks waiting for a timeout the tick timer completely shuts down and no track of time
// is kept in Ticks (the RTC still runs, though).  As such the "Ticks Since Last Dispatch" value is not
// valid - it is assumed that the Buster is in a completely quiescent state.  // *FIXME*  Is this still true?
//---------------------------------------------------------------------------------------------------------------

//  *FIXME*  Split into two tables, one a const table that goes into Flash, the other dynamic in Ram.

//*FIXME* Neither Messaging_Off nor Max_Msg_Q_Cnt nor ISR_Max_Cnt is used.




//------------------------
//  Task Message Queues  |
//----------------------------------------------------------------------------------------------------------

// *FIXME*  Check the Max Messages sizes for reasonableness and that we're using all of these queues.

    //-- SM_EVENT_TASK --//  SM_EVENT_TASK

#define  MAX_SM_MSGS          10
struct   sEvent_Msg           SM_Queue[MAX_SM_MSGS];
#define  MAX_ISR_SM_MSGS      10
struct   sEvent_Msg           SM_ISR_Q[MAX_ISR_SM_MSGS];

    //-- Monitor_Dispatch --//  MONITOR_TASK

#define  MAX_MON_MSGS        1                              //
struct   sDummy_Msg          MON_Queue[MAX_MON_MSGS];        //
#define  MAX_ISR_MON_MSGS    2                              //
struct   sDummy_Msg          MON_ISR_Q[MAX_ISR_MON_MSGS];    //

    //-- Cancel_Routing_Dispatch --//  CANCEL_RTN_TASK

#define  MAX_RTN_MSGS        1                              //
struct   sDummy_Msg          RTN_Queue[MAX_RTN_MSGS];       //
#define  MAX_ISR_RTN_MSGS    2                              //
struct   sDummy_Msg          RTN_ISR_Q[MAX_ISR_RTN_MSGS];   //

    //-- Bluetooth App-Data Dispatch --//  BT_COMM_TASK

#define  MAX_COMM_MSGS      10
struct   sBtComm_Msg        BT_Comm_Q[MAX_COMM_MSGS];
#define  MAX_ISR_COMM_MSGS  1
struct   sBtComm_Msg        BT_Comm_ISR_Q[MAX_ISR_COMM_MSGS];

//----------------------------------------------------------------------------------------------------------




//-----------------------------------------
//  Dispatch Table and related Variables  |
//---------------------------------------------------------------------------------------------------------------
  PRIVATE struct sstDispatch_Tbl {								// "C" = Constant

       void (*pFunct)(u16, void*, ulong); // Pointer to the Task to run.				C
       u16    Dispatch_Control;           // Dispatching Control (e.g., On, Off, etc).
       long   Timeout_Ctr;                // Timeout Countdown Counter.
       long   Timeout_Ticks;              // Number of Ticks to timeout dispatch.  "0" = not doing time dispatch.
      ulong   Ticks_Since_Last_Dispatch;  // Just like it says. Only valid to ~48 days.
       u16    Messaging_Off;              // If true, don't send Task any messages on dispatch.  *FIXME* Used???
       u16    Max_Msg_Q_Cnt;              // Greatest # of messages queued at once.
       u16    Msg_Count;                  // Number of messages ready to be read.
       u16    Msg_Head_Idx;               // Index to msg head, points to next avail msg slot.
       u16    Msg_Tail_Idx;               // Index to msg tail, points to current msg.
       u16    Max_Task_Msgs;              // Max messages for this given task.				C
const  void*  pMsg_Queue;                 // Pointer to the Message Queue.				C
const  char*  Task_Name;                  // String of task name for error messages.			C
const  u16    Msg_Size;                   // Size of a single message.					C
       u16    ISR_Max_Cnt;                // Greatest # of ISR msgs queued at once.
       u16    ISR_Msg_Cnt;                // ISR Message Counts.
       u16    ISR_Msg_Head;               // ISR Message Head Index.
       u16    ISR_Msg_Tail;               // ISR Message Tail Index.
const  u16    Max_ISR_Msgs;               // Size of ISR Message Queue.					C
const  void*  pISR_Msg_Q;                 // Pointer to ISR Message Queue.				C

} DT[] = {

//  *******!!! WARNING !!!*******
//
//  The Task Enumerations in Dispatch.h are the index into the Dispatch Table and must be kept
//  synchronized, e.g, if the order of the tasks in this table are: Timer, Display, Keyboard;
//  then the the enum in dispatch.h must read: "enum eTasks { TIMER, DISPLAY, KEYBOARD };".
/*
//--------------------------------------------------------------------------------------------------------------+
  Function             Dispatch Timeout Time Tick Msg Max Msg Msg  Msg  Max                   Msg Queue         |
  Pointer              Control    Ctr             Off Ctr Ctr Head Tail Msgs                  Pointer           |
                                                    +------------------------ ISR ----------------------+       |
  Function Name        Message Size                 | Max Msg Msg  Msg  Max                   Msg Queue |       |
                                                    | Ctr Ctr Head Tail Msgs                  Pointer   |       |
----------------------------------------------------+---------------------------------------------------+------*/

// TEST_TASK           test.cc
{ Test_Dispatch,       TIME_CONT,  ms500, ms500,   0,  0,  0,  0,   0,  0,                    NULL,
 "Test_Dispatch",      0,                              0,  0,  0,   0,  0,                    NULL          },

 // SM_EVEMT_TASK      events.cc
{ SM_Event_Dispatch,   MSG_ONLY,   0,     0,   0,  0,  0,  0,  0,   0,  MAX_SM_MSGS,          SM_Queue,
 "SM_Event_Dispatch",  sizeof(SM_Queue[0]),            0,  0,  0,   0,  MAX_ISR_SM_MSGS,      SM_ISR_Q      },

 // TIMER_TASK         timers.cc
{ Timer_Dispatch,      DISP_OFF,   0,     0,   0,  0,  0,  0,  0,   0,  0,                    NULL,
 "Timer_Dispatch",     0,                              0,  0,  0,   0,  0,                    NULL          },

 // MONITOR_TASK       monitor.cc
{ Monitor_Dispatch,    MSG_ONLY,   0,     0,   0,  0,  0,  0,  0,   0,  MAX_MON_MSGS,         MON_Queue,
 "Monitor_Dispatch",   0,                              0,  0,  0,   0,  MAX_ISR_MON_MSGS,     MON_ISR_Q     },

 // CANCEL_RTN_TASK    monitor.cc
{ Rtn_Cancel_Dispatch, MSG_ONLY,   0,     0,   0,  0,  0,  0,  0,   0,  MAX_RTN_MSGS,         RTN_Queue,
 "RtnCancelDispatch",  0,                              0,  0,  0,   0,  MAX_ISR_RTN_MSGS,     RTN_ISR_Q     },
/*
 // BT_TASK            sm_bluetooth.cc
{ BT_Dispatch,         MSG_ONLY,   0,     0,   0,  0,  0,  0,  0,   0,  MAX_BT_MSGS,          BT_Queue,
 "BT_Dispatch",        sizeof(BT_Queue[0]),            0,  0,  0,   0,  MAX_ISR_BT_MSGS,      BT_ISR_Q      },
*/
 // BT_COMM_TASK       bt_comm.cc
{ BT_Comm_Dispatch,    MSG_ONLY,   0,     0,   0,  0,  0,  0,  0,   0,  MAX_COMM_MSGS,        BT_Comm_Q,
 "BT_Comm_Dispatch",   sizeof(BT_Comm_Q[0]),           0,  0,  0,   0,  MAX_ISR_COMM_MSGS,    BT_Comm_ISR_Q },

/* BUTTON_TASK
{ Button_Dispatch,     MSG_ONLY,   0,     0,   0,  0,  0,  0,  0,   0,  MAX_BUTTON_MSGS,      Button_Q,
 "Button_Dispatch",    sizeof(Button_Q[0]),            0,  0,  0,   0,  MAX_ISR_BUTTON_MSGS,  Button_ISR_Q  },
*/
};
//---------------------------------------------------------------------------------------------------------------

PRIVATE const uword  Disp_Tbl_Size = sizeof(DT) / sizeof(struct sstDispatch_Tbl);
PRIVATE uword Cur_Task;      		// Static so that we know what task is calling Get_Message().
PRIVATE uword Global_ISR_Msg_Cnt = 0;	// Count of ISR Messages pending.

//---------------------------------------------------------------------------------------------------------------




//---------------
//  DISPATCHER  |
//---------------------------------------------------------------------------------------------------------------
// This is where tasks are actually dispatched from.  Timeout_Flag = true if the dispatch is because of a timeout,
// used by tasks running on MSG_PRI to determine of it was a message or a timeout that caused the dispatch.
// *FIXME*, not necessary as the invoked MSG_PRI task can see that there are no pending messages.
//

PRIVATE void Dispatch_Task(void) {                      // Dispatch to one of the routines.

    u16 Tail_Idx_Tmp, Msg_Cnt_Tmp;

    // KEEP TRACK OF MESSAGE QUEUE USAGE //

    if(DT[Cur_Task].Msg_Count > DT[Cur_Task].Max_Msg_Q_Cnt) {   // *FIXME* Either use this or delete it.
	DT[Cur_Task].Max_Msg_Q_Cnt = DT[Cur_Task].Msg_Count;
    }

    if((DT[Cur_Task].Dispatch_Control == MSG_ONLY || DT[Cur_Task].Dispatch_Control == MSG_PRI)
      && (DT[Cur_Task].Msg_Count && (! DT[Cur_Task].Messaging_Off))) {	  // *FIXME*  Is Messaging_Off used?
                                                                          //          *NO* it's not.
	// DISPATCH WITH MESSAGE //

	Tail_Idx_Tmp = DT[Cur_Task].Msg_Tail_Idx;	// Must increment the Tail_Idx before dispatch.
	Msg_Cnt_Tmp  = DT[Cur_Task].Msg_Count--;	// so subsequent calls to Get_Message() work.
	if(++DT[Cur_Task].Msg_Tail_Idx >= DT[Cur_Task].Max_Task_Msgs) {
	    DT[Cur_Task].Msg_Tail_Idx = 0;		// Wrap index around.
	}

	// Call the Dispatched Function.

	(*DT[Cur_Task].pFunct)( Msg_Cnt_Tmp,
	                       (char*)DT[Cur_Task].pMsg_Queue + (Tail_Idx_Tmp * DT[Cur_Task].Msg_Size),
			        DT[Cur_Task].Ticks_Since_Last_Dispatch );

    } else {			                        // No messages to send.

	// DISPATCH WITH NO MESSAGE //

	(*DT[Cur_Task].pFunct)(0, NULL, DT[Cur_Task].Ticks_Since_Last_Dispatch);

    }
    DT[Cur_Task].Ticks_Since_Last_Dispatch = 0;
}
//---------------------------------------------------------------------------------------------------------------
// Note that this routine is (only) called from the end of initialization in main() and never returns.
// AKA "The Big or Main Program Loop".

/*PRIVATE*/ void Dispatch(void) {

    static uword Task;			// Don't actually need to be static as we never leave.
    static ulong Ticks;			// but might be faster.
    static u16   Msg_Size;

    Ticks = Timer_Foreground();		// Eat up any ticks occurring since init.

    for(;;) {				// Loop forever.

	//--------------------------------------------------------------------------//
	//--  GET NUMBER OF TICKS SINCE LAST CALL AND DECREMENT TIMEOUT COUNTERS  --//
	//--------------------------------------------------------------------------//

	Ticks = Timer_Foreground();
	if(Ticks) {

	    //--------------------------------------------------------//
	    //--  UPDATE "Ticks Since Last Dispatch" FOR ALL TASKS  --//
	    //--------------------------------------------------------//

	    for(Task=0; Task < Disp_Tbl_Size; Task++) {

		// If counter overflow, set it to FFFFFFFF;

		IntMasterDisable();		// Master disable interrupts.
		if((DT[Task].Ticks_Since_Last_Dispatch + Ticks) > DT[Task].Ticks_Since_Last_Dispatch) {
		    DT[Task].Ticks_Since_Last_Dispatch += Ticks;
		} else {
		    DT[Task].Ticks_Since_Last_Dispatch = ~0;
		}
		if(DT[Task].Timeout_Ticks) {			// If this task using time priority,
		    DT[Task].Timeout_Ctr -= Ticks;		// decrement the counter (underflow OK).
		    if(DT[Task].Timeout_Ctr < 0) DT[Task].Timeout_Ctr = 0;
		}
		IntMasterEnable();		// Master enable interrupts.
	    }
	}

	//--------------------------------------------//
	//--  FULL SCAN THROUGH THE DISPATCH TABLE  --//
	//--------------------------------------------//

	for(Cur_Task = 0; Cur_Task < Disp_Tbl_Size; Cur_Task++) {

	    //---------------------------//
	    //--  HANDLE ISR MESSAGES  --//
	    //---------------------------//

	    while(DT[Cur_Task].ISR_Msg_Cnt) {
/*
if(Cur_Task == CPU_COMM_TASK) {
	HB_LED_On();
	ERR_LED_On();
	while(1) {}
}
*/
		IntMasterDisable();		// Master disable interrupts.

		if(DT[Cur_Task].ISR_Msg_Cnt > DT[Cur_Task].ISR_Max_Cnt) {  // Keep track of queue usage.
		    DT[Cur_Task].ISR_Max_Cnt = DT[Cur_Task].ISR_Msg_Cnt;
		}

		// There are messages from ISRs to this task, move them to the regular message queue.

		Msg_Size = DT[Cur_Task].Msg_Size;
		memcpy( (char*)(DT[Cur_Task].pMsg_Queue) + (DT[Cur_Task].Msg_Head_Idx * Msg_Size),
		        (char*)(DT[Cur_Task].pISR_Msg_Q) + (DT[Cur_Task].ISR_Msg_Tail * Msg_Size),
			 Msg_Size );
		if(++DT[Cur_Task].ISR_Msg_Tail >= DT[Cur_Task].Max_ISR_Msgs) {
		    DT[Cur_Task].ISR_Msg_Tail = 0;		// Wrap index around.
		}
		if(++DT[Cur_Task].Msg_Head_Idx >= DT[Cur_Task].Max_Task_Msgs) {
		    DT[Cur_Task].Msg_Head_Idx = 0;		// Wrap index around.
		}
		DT[Cur_Task].Msg_Count++;
		DT[Cur_Task].ISR_Msg_Cnt--;

		Global_ISR_Msg_Cnt--;		// Decrement the global ISR message count.

		IntMasterEnable();		// Master enable interrupts.
	    }

	    //-------------------------------------//
	    //--  DISPATCH IF THE TASK IS READY  --//
	    //-------------------------------------//

	    switch(DT[Cur_Task].Dispatch_Control) {

		case DISP_OFF:					// Don't dispatch.

		    break;

		case FULL_PRI:					// Always dispatch.

		    Dispatch_Task();
		    break;

		case TIME_ONCE:

	            if(DT[Cur_Task].Timeout_Ctr <= 0) {
                        DT[Cur_Task].Dispatch_Control = DISP_OFF;   // set Dispatching off.
                        Dispatch_Task();
	            }
	            break;

		case TIME_CONT:

                    if(DT[Cur_Task].Timeout_Ctr <= 0) {
                        DT[Cur_Task].Timeout_Ctr = DT[Cur_Task].Timeout_Ticks;  // Reset the timeout ctr.
                        Dispatch_Task();
                    }
                    break;

		case MSG_PRI:

                    if((DT[Cur_Task].Msg_Count) && (! DT[Cur_Task].Messaging_Off)) {
                        DT[Cur_Task].Timeout_Ctr = DT[Cur_Task].Timeout_Ticks;  // Reset the timeout ctr.
                        Dispatch_Task();
                    } else if(DT[Cur_Task].Timeout_Ctr <= 0) {
                        Dispatch_Task();
                    }
                    break;

		case MSG_ONLY:					// Only dispatch on Message.

		    if((DT[Cur_Task].Msg_Count) && (! DT[Cur_Task].Messaging_Off)) {
			Dispatch_Task();
		    }
		    break;

	        default:
	            sprintf(sBuff, "dispatch.cc line %d", __LINE__);
	            Error(INVAL_SWITCH_VALUE, sBuff);

	    }
	}   /* for(Cur_Task=0; Cur_Task<Disp_Tbl_Size; Cur_Task++) */


	//--------------------------------------//
	//--  SEE IF IT'S TIME TO POWER DOWN  --//
	//--------------------------------------//

	s32 Soonest = 0x7FFFFFFF;

	IntMasterDisable();		// Master disable interrupts.

//	if(ISR_Mode_Q_Cnt) goto Dont_Powerdown;  G+ ??

	for(Cur_Task=0; Cur_Task<Disp_Tbl_Size; Cur_Task++) {

	    switch(DT[Cur_Task].Dispatch_Control) {

		case FULL_PRI:					// *FIXME* Don't think FULL_PRI is used.

		    goto Dont_Powerdown;

		case TIME_ONCE:
		case TIME_CONT:

		    if(DT[Cur_Task].Timeout_Ctr < Soonest) {
			Soonest = DT[Cur_Task].Timeout_Ctr;
		    }
		    break;

		case MSG_PRI:

		    if(DT[Cur_Task].ISR_Msg_Cnt || DT[Cur_Task].Msg_Count) goto Dont_Powerdown;
		    if(DT[Cur_Task].Timeout_Ctr < Soonest) Soonest = DT[Cur_Task].Timeout_Ctr;
		    break;

		case MSG_ONLY:

		    if(DT[Cur_Task].ISR_Msg_Cnt || DT[Cur_Task].Msg_Count) goto Dont_Powerdown;
		    break;

		case DISP_OFF:

		    break;

	       default:
                   sprintf(sBuff, "dispatch.cc line %d", __LINE__);
                   Error(INVAL_SWITCH_VALUE, sBuff);
	    }
	}

	if(Soonest == 0x7FFFFFFF) {
	    Set_Wakeup_Time(0);		// Be sure wakeup timers are off.
	} else {
	    Set_Wakeup_Time(Soonest);   // Wake up at soonest occurring timeout.
	}

	// Power Down //

	// *FIXME*  Instead of waking up every Tick, set SysTick or another hardware timer to only wake up when
	//	    either a software timer is done or a 1/4 second count is done.

	IntMasterEnable();			// Master enable interrupts.

	while(Get_Wakeup_Time()) {		// We'll wake up every Tick.
	    if(Global_ISR_Msg_Cnt) break;	// Got an event from an ISR.
//	    while( ! CheckEEPromDone() );       // Be sure EEProm not busy.         G+
	    SysCtlSleep();			// Not wakeup time yet.             G+
	}

Dont_Powerdown:

	IntMasterEnable();		        // Master enable interrupts.

    }   /* for(;;) */
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------
//  ACTIVE TASK: READ MESSAGE FROM QUEUE  |
//---------------------------------------------------------------------------------------------------------------
// Note that this routine will return a NULL pointer if there aren't any (more) messages.

PUBLIC void* Get_Message() {

    void* pTmp = NULL;

    if(DT[Cur_Task].Msg_Count) {					// If there are messages...

	pTmp = (char*)DT[Cur_Task].pMsg_Queue + (DT[Cur_Task].Msg_Tail_Idx * DT[Cur_Task].Msg_Size);

	if(++DT[Cur_Task].Msg_Tail_Idx >= DT[Cur_Task].Max_Task_Msgs) {
	    DT[Cur_Task].Msg_Tail_Idx = 0;				// Wrap index around.
	}
	DT[Cur_Task].Msg_Count--;
    }
    return pTmp;
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------
//  SEND A MESSAGE TO A TASK  |
//---------------------------------------------------------------------------------------------------------------
// We copy messages to the local static queue as the sending task message is probably an automatic variable.
// Returns SUCCESS if message sent, else FAILURE if no more space in the task's message queue.

PUBLIC void Send_Message(uword Task_Idx, void* pMsg) {

    if(DT[Task_Idx].Msg_Count == DT[Task_Idx].Max_Task_Msgs) {
        Error(SEND_MESSAGE_OVERFLOW, "Send_Message()");	        // No more msg space.
    }

    memcpy( (char*)DT[Task_Idx].pMsg_Queue + (DT[Task_Idx].Msg_Head_Idx * DT[Task_Idx].Msg_Size),
             pMsg, DT[Task_Idx].Msg_Size );

    // Increment the Head Index and wrap around if off the end.

    if(++DT[Task_Idx].Msg_Head_Idx >= DT[Task_Idx].Max_Task_Msgs) {
	DT[Task_Idx].Msg_Head_Idx = 0;
    }
    DT[Task_Idx].Msg_Count++;
}
//----------------------------------------------------------------------------------------------------------------




//-----------------------------------------------------------
//  SEND A MESSAGE FROM AN ISR TO A TASK, COPY FULL BUFFER  |
//---------------------------------------------------------------------------------------------------------------
// We copy messages to the local static queue as the sending task message is probably an automatic variable.
// Messages queued here get copied to the normal message queue synchronously with the dispatcher scan.
//
// NOTE: This is running in the context of an interrupt routine.

PUBLIC void ISR_Send_Message(uword Task_Idx, void* pMsg) {

//static bool First = true;
/*
if(Task_Idx == CPU_COMM_TASK) {         // *FIXME*  G+
        HB_LED_On();
        ERR_LED_On();
        while(1) {}
}
*/
    IntMasterDisable();		                // Master disable interrupts.
    if(DT[Task_Idx].ISR_Msg_Cnt >= DT[Task_Idx].Max_ISR_Msgs) {
        Error(ISR_MESSAGE_OVERFLOW);            // No more msg space.
    }
    memcpy( (char*)DT[Task_Idx].pISR_Msg_Q + (DT[Task_Idx].ISR_Msg_Head * DT[Task_Idx].Msg_Size),
             pMsg, DT[Task_Idx].Msg_Size );

    // Increment the Head Index and wrap around if off the end.

    if(++DT[Task_Idx].ISR_Msg_Head >= DT[Task_Idx].Max_ISR_Msgs) {
	DT[Task_Idx].ISR_Msg_Head = 0;
    }
    DT[Task_Idx].ISR_Msg_Cnt++;
    Global_ISR_Msg_Cnt++;		// Increment the global ISR message count.
    IntMasterEnable();			// Master enable interrupts.
}
//---------------------------------------------------------------------------------------------------------------





//-------------------------------
//  SET DISPATCH MODE AND TIME  |
//---------------------------------------------------------------------------------------------------------------

PUBLIC void Set_Dispatch_Time(uword Task, ulong Ticks, uword Dispatch_Control) {

    if(Ticks > 0x7FFFFFFF) Ticks = 0x7FFFFFFF;		// Maximum tick count.
    DT[Task].Timeout_Ctr = Ticks;
    DT[Task].Timeout_Ticks = Ticks;
    DT[Task].Dispatch_Control = Dispatch_Control;
}
//---------------------------------------------------------------------------------------------------------------




//----------------------
//  SET DISPATCH MODE  |
//---------------------------------------------------------------------------------------------------------------

PUBLIC void Set_Dispatch_Control(uword Task, uword Dispatch_Control) {

    DT[Task].Dispatch_Control = Dispatch_Control;
}
//---------------------------------------------------------------------------------------------------------------




//---------------------------------------------------------
//  GET A TASK'S CURRENT TIMEOUT VALUES OR DISPATCH MODE  |
//---------------------------------------------------------------------------------------------------------------

PUBLIC  long  /*Ticks*/  Get_Ticks_Remaining(uword Task) {

    return(DT[Task].Timeout_Ctr);
}

PUBLIC  long  /*Ticks*/  Get_Ticks_Expired(uword Task) {

    return(DT[Task].Timeout_Ticks - DT[Task].Timeout_Ctr);
}
//---------------------------------------------------------------------------------------------------------------


