//------------------------------------------------------------------------------------------------------
//  dispatch.h
//
//  Header for dispatch.c,  structure declarations for the different message types.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//------------------------------------------------------------------------------------------------------
//  11/16/05 Modified for interrupt driven buttons.
//  12/12/05 Loop Count Dispatching and DEFAULT_PRI taken out.
//  08/30/10  Started modifying to port to ARM CPU.
//------------------------------------------------------------------------------------------------------

#ifndef DISPATCH_H_
#define DISPATCH_H_


#include "common.h"
#include "tick.h"
#include "events.h"
#include "sm_bt_at_cmnd.h"




//---------------------------------
//  DISPATCHED TASK ENUMERATIONS  |
//------------------------------------------------------------------------------------------------------
//
// ******!!! WARNING !!! ******
//
// The following task enumerations are the index into the dispatch task table and *MUST* be in exactly
// the same order that the tasks have been entered in the initialization of the table.

enum eTasks {
    TEST_TASK,
    SM_EVENT_TASK,
    TIMER_TASK,
    MONITOR_TASK,
    CANCEL_RTN_TASK,
    BT_COMM_TASK,
//  BUTTON_TASK,
};
//------------------------------------------------------------------------------------------------------




//------------------------------
//  DISPATCH CONTROL SETTINGS  |
//------------------------------------------------------------------------------------------------------
// Dispatch Control Settings.
//

enum eDispatch_Ctl { DISP_OFF, FULL_PRI, TIME_ONCE, TIME_CONT, MSG_PRI, MSG_ONLY };

//------------------------------------------------------------------------------------------------------



// *FIXME*  Should these be put into their respective headers instead of here?

//-----------------------
//  MESSAGE STRUCTURES  |
//------------------------------------------------------------------------------------------------------
//  STATE MACHINE EVENT MESSAGES     SM_EVENT_TASK

struct sEvent_Msg {
    sm_t          SM;
    event_t       Event;
    event_data_t  Data;
};
//------------------------------------------------------------------------------------------------------
//  DUMMY MESSAGE TO DISPATCH FROM ISR ROUTINE
//
//  This is a null message, it's just a flag to dispatch from an ISR routine.
struct sDummy_Msg {
    int Dummy;
};
//------------------------------------------------------------------------------------------------------
//  BUTTON EVENTS   BUTTON_TASK

struct sButton_Msg {
    int   Btn_Event;
};
//------------------------------------------------------------------------------------------------------
//  BLUETOOTH APPLICATION MESSAGE   BT_COMM_TASK

struct sBtComm_Msg {
    u8     MsgNmbr;     // Message number.
    u8     MsgLength;   // Payload size.
    char*  pPayload;    // Pointer to message payload.
};
//------------------------------------------------------------------------------------------------------




//--------------------------
//  EXTERNAL DECLARATIONS  |
//------------------------------------------------------------------------------------------------------

void Dispatch(void);

void Send_Message    (uword Task_Idx, void* pMsg);
void ISR_Send_Message(uword Task_Idx, void* pMsg);

void* /*Message Pointer*/ Get_Message();

void Set_Dispatch_Time    (uword Task, ulong Ticks, uword Dispatch_Control);
void ISR_Set_Dispatch_Time(uword Task, ulong Ticks, uword Dispatch_Control);

void Set_Dispatch_Control(uword Task, uword Dispatch_Control);
void ISR_Set_Dispatch_Ctl(uword Task, uword Dispatch_Control);

void Disable_Dispatch_Messages();
void Enable_Dispatch_Messages();

uword /*Number of Messages Flushed*/ Flush_Msg_Queue(uword Task);

long /*Ticks*/ Get_Ticks_Remaining(uword Task);
long /*Ticks*/ Get_Ticks_Expired(uword Task);

//------------------------------------------------------------------------------------------------------



#endif  /* DISPATCH_H_ */
