//---------------------------------------------------------------------------------------------------------------
//  eeprom.cc
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//
//  ****  WAARNING  ****  ALL VARIABLES ASSUMED TO BE 32 BITS  ****
//
//---------------------------------------------------------------------------------------------------------------
//
//  The TM4C123AE6PM's EEPROM is 2KB comprised as 32 blocks of 16) 32-bit words.
//
//  Note that for "float" values, we cast them to u32 when storing them and then cast them back to float when
//  reading them back out.
//
//  TO DO:
//
//      7) In Init_EEProm(), see if there are more globals to be read in.
//      8) Add checksum to file system data ?
//      9) Read in EEProm size instead of relying on EEPROM_SIZE.
//     10) Would be tits if when an EEPROM block has a failed word we mark as defective and copy over to a
//         spare block.  Or, save everything in duplicate, which would make the EEProm half the available space
//         minus the block(s) necessary to implement this.
//
//---------------------------------------------------------------------------------------------------------------



#include "common.h"
#include "uarts.h"
#include "error.h"
#include "eeprom.h"
#include "sm_main.h"
#include "led.h"

#include <stdio.h>
#include <string.h>

#include "sysctl.h"
#include "C:\ti\TivaWare_C_Series_2_1_4_178\driverlib\eeprom.h"


#pragma diag_suppress 179




//----- FORWARD DECLARATIONS -----//

u32    ReadEEProm(u32 Block, u32 Offset);
float fReadEEProm(u32 Block, u32 Offset);



//---------------------
//  EEPROM ADDRESSES  |
//---------------------------------------------------------------------------------------------------------------
// These are 32 bit word addresses, not byte addresses.
//
//  ***************************************************************
//  ** IMPORTANT **  Keep in sync with array in ListEEPromData() **
//  ***************************************************************








//------------- Single Word Data -------------//
/*
const uint CALIB_ADDR         =  0;
const uint JUMP_1_ADDR        =  1;
const uint JUMP_2_ADDR        =  2;
const uint SCAN_1_ADDR        =  3;
const uint SCAN_2_ADDR        =  4;
const uint AVG_DEPTH_ADDR     =  5;
const uint JUMP_INDICATE_ADDR =  6;
const uint LOW_BATT_ADDR      =  7;
const uint CALIB_YEAR_ADDR    =  8;  // e.g. '2024'.
const uint CALIB_MONTH_ADDR   =  9;  // e.g. '5' = May.
const uint LANGUAGE_ADDR      = 10;
const uint CNT_COLOR_ADDR     = 11;
const uint SPEAKER_VOL_ADDR   = 12;
const uint HEADSET_VOL_ADDR   = 13;
const uint START_MODE_ADDR    = 14;
const uint GRAPH_MODE_ADDR    = 15;
const uint ROTATE_ADDR        = 16;
                                                       // **** SEE NOTE BELOW WHEN ADDING HERE **** //

//---^^^ Insert New Single Word Data Here ^^^---//


//-------------- Multi-Word Data --------------//
                                                       // *************************************************** //
const uint PD_BD_REV_ADDR     = ROTATE_ADDR + 1;       // ** WHEN ADDING NEW SINGLE-WORD DATA, UPDATE THIS ** //
                                                       // *************************************************** //

const uint NextMulti1WordAddr = PD_BD_REV_ADDR + PdBdRevSize + 1;

//---^^^ Insert New Multi-Word Data Here ^^^---//


//-------------- Count Storage ----------------//

const uint CIRC_FILL_ADDR    = NextMulti1WordAddr + 1;  // This is the Single Count Set storage.
const uint CNT_BUFF_QUA_ADDR = CIRC_FILL_ADDR     + 1;
const uint BASE_CNT_ADDR     = CNT_BUFF_QUA_ADDR  + 1;
const uint HAS_BASE_CNT_ADDR = BASE_CNT_ADDR      + 1;
const uint YEAR_ADDR         = HAS_BASE_CNT_ADDR  + 1;
const uint MONTH_ADDR        = YEAR_ADDR          + 1;
const uint DATE_ADDR         = MONTH_ADDR         + 1;
const uint HOUR_ADDR         = DATE_ADDR          + 1;
const uint MINUTES_ADDR      = HOUR_ADDR          + 1;
const uint SECONDS_ADDR      = MINUTES_ADDR       + 1;


// All Data From Here To RecallBuffDepth Is Single Count Storage //  *FIXME* Get Rid of RecallBuffDepth.
*/


//-------------------- File System Count Storage --------------------//    *Always the Last Section*
/*
const uint NmbOfFiles_ADDR    = SECONDS_ADDR       + 1;
const uint OldestFileNmb_ADDR = NmbOfFiles_ADDR    + 1;
const uint NextFileNmb_ADDR   = OldestFileNmb_ADDR + 1;
const uint NextFreeAddr_ADDR  = NextFileNmb_ADDR   + 1;
const uint PerFileOldIdx_ADDR = NextFreeAddr_ADDR  + 1;
const uint PerFileNewIdx_ADDR = PerFileOldIdx_ADDR + 1;
const uint UnusedSize_ADDR    = PerFileNewIdx_ADDR + 1;

const uint PerFile_ADDR       = UnusedSize_ADDR    + 1;         // The PerFile[] structure array.
const uint EndOfPerFileADDR   = PerFile_ADDR + sizeof(PerFile); // 1 beyond end of PerFile[].

const uint FileDataStart_ADDR = PerFile_ADDR + sizeof(PerFile); // Must be

const uint CNT_STORAGE_ADDR  = FileDataStart_ADDR;  // *FIXME* Single_Cnt_Store.

//---------------------------------------------------------------------------------------------------------------

const uint FIRST_EE_ADDR        = CALIB_ADDR;           // First, Last non-Count Data.
const uint LAST_NON_CNT_EE_ADDR = CNT_STORAGE_ADDR - 1;	// *KEEP THIS UPDATED*


const uint FileSysSize = 1000;          // *FIXME* Testing G+ DB+ FS+
//const uint FileSysSize = EEPROM_SIZE - FileDataStart_ADDR;    // Total # of counts is FileSysSize * 2.
//---------------------------------------------------------------------------------------------------------------
*/
const uint EEPROM_SIZE          = 512;                 // Size of EEProm in 32 Bit Words.  *FIXME*



//-----------------------------
//  DISPLAY EEPROM ADDRESSES  |
//---------------------------------------------------------------------------------------------------------------

void  ListEEPromData(void) {
/*
    enum eType { Word, MultiWord, MultiWordStr, Float }; // MultiWordStr: Single Chr/Word 0 terminated.

    struct sEE_Adrs {
        const uint   Addr;
        const char*  Name;
        const eType  Type;
    };

    const sEE_Adrs EEPromAddrs[] = {

       CALIB_ADDR,         "Calibration Count    ", Word,        // Single Word Data
       JUMP_1_ADDR,        "Jump 1 Setting       ", Float,
       JUMP_2_ADDR,        "Jump 2 Setting       ", Float,
       SCAN_1_ADDR,        "Scan 1 Setting       ", Float,
       SCAN_2_ADDR,        "Scan 2 Setting       ", Float,
       AVG_DEPTH_ADDR,     "Averaging Depth      ", Word,
       JUMP_INDICATE_ADDR, "Jump Indicate Type   ", Word,
       LOW_BATT_ADDR,      "Low Battery On       ", Word,
       CALIB_YEAR_ADDR,    "Calibration Year     ", Word,
       CALIB_MONTH_ADDR,   "Calibration Month    ", Word,
       LANGUAGE_ADDR,      "Selected Language    ", Word,
       CNT_COLOR_ADDR,     "Cnt Colorized        ", Word,
       SPEAKER_VOL_ADDR,   "Speaker Volume       ", Word,
       HEADSET_VOL_ADDR,   "Headset Volume       ", Word,
       START_MODE_ADDR,    "Start Mode           ", Word,
       GRAPH_MODE_ADDR,    "Graph Mode           ", Word,
       ROTATE_ADDR,        "Display Rotate On/Off", Word,

       PD_BD_REV_ADDR,     "P/D Board Revision   ", MultiWordStr,  // Multi-Word Data

       CIRC_FILL_ADDR,     "\nCount Circ Fill Idx  ", Word,
       CNT_BUFF_QUA_ADDR,  "Count Buffer Qua     ", Word,
       BASE_CNT_ADDR,      "Base Count           ", Word,
       HAS_BASE_CNT_ADDR,  "Base Cnt Valid (bool)", Word,
       YEAR_ADDR,          "Year                 ", Word,
       MONTH_ADDR,         "Month                ", Word,
       DATE_ADDR,          "Day                  ", Word,
       HOUR_ADDR,          "Hour                 ", Word,
       MINUTES_ADDR,       "Minutes              ", Word,
       SECONDS_ADDR,       "Seconds              ", Word,

       NmbOfFiles_ADDR,    "\nNumber of Files      ", Word,
       OldestFileNmb_ADDR, "Oldest File #        ", Word,
       NextFileNmb_ADDR,   "Next File #          ", Word,
       NextFreeAddr_ADDR,  "Next Free Address    ", Word,
       PerFileOldIdx_ADDR, "Per-File Old Index   ", Word,
       PerFileNewIdx_ADDR, "Per-File New Index   ", Word,
       UnusedSize_ADDR,    "Unused Size          ", Word,

       PerFile_ADDR,       "\nStart PerFile Array  ", Word,
       FileDataStart_ADDR, "Start of Cnt Storage ", Word,
    };

    const uint AddrsSize = sizeof(EEPromAddrs) / sizeof(EEPromAddrs[0]);

    u32  Data, Addr;
    s32  Length;
    s8   Str[PdBdRevSize + 1];

    TxDebugStr("\r\n 32 Bit Word Addresses:\r\n");
    TxDebugStr(" Addr\t\t\t       (Hex)\t (Decimal)\r\n");
    TxDebugStr(" ----\t\t\t     ---------\t-----------\r\n");
    for(uint i = 0; i < AddrsSize; i++) {

        Data = ReadEEPromLin(EEPromAddrs[i].Addr);
        Addr = EEPromAddrs[i].Addr;

        switch(EEPromAddrs[i].Type) {

            case Word:
                sprintf(sBuff," %3d   %s %9X   %10u\r\n", EEPromAddrs[i].Addr, EEPromAddrs[i].Name, Data, Data);
                break;

            case Float:
                sprintf(sBuff," %3d   %s  Float: %F\r\n", EEPromAddrs[i].Addr, EEPromAddrs[i].Name,
                                                            fReadEEPromLin(EEPromAddrs[i].Addr));
                break;

            case MultiWordStr:                                // One character per word.
                Addr = EEPromAddrs[i].Addr;
                for(uint k = 0; k < PdBdRevSize + 1; k++) {   // Read in the string.
                    Str[k] = (char)ReadEEPromLin(Addr++);
                }
                if(Str[0] < 0) {
                    sprintf(sBuff, " %3d   %s  String: (EEProm Not Set)\r\n", EEPromAddrs[i].Addr, EEPromAddrs[i].Name);
                } else if(Str[0] == 0) {
                    sprintf(sBuff, " %3d   %s  String: (NULL String)\r\n", EEPromAddrs[i].Addr, EEPromAddrs[i].Name);
                } else {
                    sprintf(sBuff, " %3d   %s  String: %s\r\n", EEPromAddrs[i].Addr, EEPromAddrs[i].Name, Str);
                }
                break;

            case MultiWord:                                           // ** NOT YET TESTED **
                if(i == AddrsSize - 1) {
                    Error(INVAL_LAST_TBL_ENTRY, "ListEEPromData()");  // Last entry in table can't be multiword.
                }
                Length = EEPromAddrs[i+1].Addr - EEPromAddrs[i].Addr; // Number of words.

                sprintf(sBuff," %3d   %s %9X   %10u\r\n", EEPromAddrs[i].Addr, EEPromAddrs[i].Name, Data, Data);
                TxDebugStr(sBuff);
                for(uint j = 0; j < Length-1; j++) {
                    Addr = EEPromAddrs[j+1].Addr;
                    Data = ReadEEPromLin(Addr);
                    sprintf(sBuff, " %3d               %9X   %10u\r\n", Addr, Data, Data);
                    TxDebugStr(sBuff);
                }
                sBuff[0] = '0';         // Everything already printed out.
                break;

            default:
                sprintf(sBuff, "ListEEPromData = %d", EEPromAddrs[i].Type);
                Error(INVAL_SWITCH_VALUE, sBuff);
        }
        TxDebugStr(sBuff);
    }
*/
}
//---------------------------------------------------------------------------------------------------------------




                                        //////////////////////////////
                                        //                          //
                                        //  EEPROM ACCESS ROUTINES  //
                                        //                          //
                                        //////////////////////////////



//--------------------------
//  CHECK EEPROM NOT BUSY  |
//---------------------------------------------------------------------------------------------------------------
//  Before the CPU can power off must check that the EEPROM is not busy.  true = done, not busy.

bool CheckEEPromDone(void) {

    return( EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING ? false : true );
}
//---------------------------------------------------------------------------------------------------------------




//---------------------------
//  CHECK SUCCESS OF WRITE  |
//---------------------------------------------------------------------------------------------------------------
void CheckWriteStatus(void) {

    u32 Status = EEPROM_EESUPP_R;
    if(Status & (EEPROM_EESUPP_PRETRY | EEPROM_EESUPP_ERETRY)) {
	EEPROM_EESUPP_R |= 1;					// Before punting retry the operation
	while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	// Wait till done.
	Status = EEPROM_EESUPP_R;
	if(Status & (EEPROM_EESUPP_PRETRY | EEPROM_EESUPP_ERETRY)) {
	    Error(EEPROM_WRITE_FAILURE, "CheckWriteStatus()");
	}
    }
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------------------
//  CHECK IF COPY BUFFER NEEDS TO BE ERASED  |
//---------------------------------------------------------------------------------------------------------------
void CheckCopyBuff(void) {

    u32 Status = EEPROM_EESUPP_R;
    if(Status & 2) EEPROM_EESUPP_R |= 1;	// If copy buffer full, erase now. We won't wait for completion.
}
//---------------------------------------------------------------------------------------------------------------




//---------------------------------
//  WRITE u32 OR float TO EEPROM  |
//---------------------------------------------------------------------------------------------------------------
void WriteEEProm(u32 Block, u32 Offset, u32 Data) {

    int i;

    for(i = 2; i > 0; i--) {					// Write retry.
	while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	// Be sure not busy.
	EEPROM_EEBLOCK_R  = Block;
	EEPROM_EEOFFSET_R = Offset;
	EEPROM_EERDWR_R   = Data;
	CheckWriteStatus();
	u32 ReadData = ReadEEProm(Block, Offset);		// Check written correctly.
	if(ReadData == Data) break;
    }
    if(i == 0) Error(EEPROM_WRITE_FAILURE, "WriteEEProm(u32)");
    CheckCopyBuff();						// See if Copy Buffer full.
}
//---------------------------------------------------------------------------------------------------------------
void WriteEEProm(u32 Block, u32 Offset, float fData) {

    int i;

    for(i = 2; i > 0; i--) {					// Write retry.
	while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	// Be sure not busy.
	EEPROM_EEBLOCK_R  = Block;
	EEPROM_EEOFFSET_R = Offset;
	u32* pData = reinterpret_cast<u32*>(&fData);
	EEPROM_EERDWR_R = *pData;
	CheckWriteStatus();
	float fReadData = fReadEEProm(Block, Offset);		// Check written correctly.
	if(fReadData == fData) break;
    }
    if(i == 0) {
        Error(EEPROM_WRITE_FAILURE, "WriteEEProm(float)");
    }
    CheckCopyBuff();						// See if Copy Buffer full.
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------------
//  WRITE u32 OR float TO EEPROM USING LINEAR ADDRESS  |
//---------------------------------------------------------------------------------------------------------------
void WriteEEPromLin(u32 Addr, u32 Data) {

    u32 Block = Addr >> 4;
    WriteEEProm(Block, Addr & 0x0000000F, Data);
}
//---------------------------------------------------------------------------------------------------------------
void WriteEEPromLin(u32 Addr, float fData) {

    u32 Block = Addr >> 4;
    WriteEEProm(Block, Addr & 0x0000000F, fData);
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------
//  READ u32 or float FROM EEPROM  |
//---------------------------------------------------------------------------------------------------------------
u32 ReadEEProm(u32 Block, u32 Offset) {

    while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	// Be sure not busy.
    EEPROM_EEBLOCK_R  = Block;
    EEPROM_EEOFFSET_R = Offset;
    return(EEPROM_EERDWR_R);
}
//---------------------------------------------------------------------------------------------------------------
float fReadEEProm(u32 Block, u32 Offset) {

    while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	// Be sure not busy.
    EEPROM_EEBLOCK_R  = Block;
    EEPROM_EEOFFSET_R = Offset;
    u32 Data = EEPROM_EERDWR_R;
    float* pFloat = reinterpret_cast<float*>(&Data);
    return(*pFloat);
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------------------
//  READ u32 OR float TO EEPROM USING LINEAR ADDRESS  |
//---------------------------------------------------------------------------------------------------------------
//  Note that Addr is 32 bit Word addresses, not Bytes.

u32 ReadEEPromLin(u32 Addr) {

    u32 Block = Addr >> 4;
    return(ReadEEProm(Block, Addr & 0x0000000F));
}
//---------------------------------------------------------------------------------------------------------------
float fReadEEPromLin(u32 Addr) {

    u32 Block = Addr >> 4;
    return(fReadEEProm(Block, Addr & 0x0000000F));
}
//---------------------------------------------------------------------------------------------------------------


void ReadInBLThreshVars(sBacklightThresholds* pArray, uint Size) {}       // Read in Data from EEProm.
void ReadInNeedleVars(sNeedleThresholds* pArray, uint Size) {}            //
void ReadInSpeedoVars(sSpeedoBLThresholds* pArray, uint Size) {}          //


                                    ////////////////////////////
                                    //                        //
                                    //  RESET AND INITIALIZE  //
                                    //                        //
                                    ////////////////////////////



//---------------------------------
//  USER RESET TO FACTORY VALUES  |
//---------------------------------------------------------------------------------------------------------------
void Reset2Factory(void) {

}
//---------------------------------------------------------------------------------------------------------------




//---------------------------------------
//  RESET THE EEPROM TO DEFAULT VALUES  |
//---------------------------------------------------------------------------------------------------------------
void Reset_EEProm_Data(void) {

    Reset2Factory();
}
//---------------------------------------------------------------------------------------------------------------




//------------------------
//  ERASE ALL OF EEPROM  |
//---------------------------------------------------------------------------------------------------------------
//  Return of 0 = Success, otherwise is the error code - see Peripheral Driver Library.

uint MassEraseEEProm(void) {

    return(EEPROMMassErase());
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------
//  READ IN MULTIPLE DATA  |  *FIXME*  Untested.
//---------------------------------------------------------------------------------------------------------------
//  Any module can have an sMultiRead array table of variables it needs to pull out of EEProm.  The table
//  also contains default values if the EEProm is blank (or the MagicNumber is incorrect), in which case it
//  will initialize the EEProm with these defaults.
//---------------------------------------------------------------------------------------------------------------
void  ReadInVars(const sMultiRead* pData) {

    uint    Uint;
    float   Float;
    eColors Color;

    if(ReadEEPromLin(MagicAddr) != MagicNumber) {                           //--- BLANK, FILL WITH DEFAULTS ---

        while(pData->pVar != NULL) {

            switch(pData->Type) {

                case COLOR_T:
                    WriteEEPromLin((u32)pData->EEAddr, (u32)pData->ColorDefault);
                    memcpy(pData->pVar, &pData->ColorDefault, sizeof(eColors));
                    break;

                case UINT_T:
                    WriteEEPromLin((u32)pData->EEAddr, (u32)pData->UintDefault);
                    memcpy(pData->pVar, &pData->UintDefault, sizeof(uint));
                    break;

                case FLOAT_T:
                    WriteEEPromLin((u32)pData->EEAddr, pData->FloatDefault);
                    memcpy(pData->pVar, &pData->FloatDefault, sizeof(float));
                    break;

                default:
                    sprintf(sBuff, "eeprom.cc line %d", __LINE__);
                    Error(INVAL_SWITCH_VALUE, sBuff);
            }
            pData++;
        }

    } else {                                                                //--- NOT BLANK, READ IN VALUES ---

        while(pData->pVar != NULL) {
            switch(pData->Type) {

                case UINT_T:
                    Uint = ReadEEPromLin((u32)pData->EEAddr);
                    memcpy(pData->pVar, &Uint, sizeof(uint));
                case COLOR_T:
                    memcpy(pData->pVar, &pData->ColorDefault, sizeof(eColors));
                    break;

                case FLOAT_T:
                    Float = fReadEEPromLin((u32)pData->EEAddr);
                    memcpy(pData->pVar, &Float, sizeof(float));
                    break;

                default:
                    sprintf(sBuff, "eeprom.cc line %d", __LINE__);
                    Error(INVAL_SWITCH_VALUE, sBuff);
            }
            pData++;
        }
    }
}
//---------------------------------------------------------------------------------------------------------------




//----------------------
//  INITIALIZE EEPROM  |
//---------------------------------------------------------------------------------------------------------------
void Init_EEProm(void) {

    while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	        // Be sure not busy.
    u32 Status = EEPROM_EESUPP_R;
    if(Status & (EEPROM_EESUPP_PRETRY | EEPROM_EESUPP_ERETRY)) {
	EEPROM_EESUPP_R |= 1;				        // Before punting retry the operation.
	while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	// Wait till done.
	Status = EEPROM_EESUPP_R;
	if(Status & (EEPROM_EESUPP_PRETRY | EEPROM_EESUPP_ERETRY)) {
	    Error(INIT_EEPROM_FAILURE, "Init_EEProm() error 1");
	}
    }
    SysCtlPeripheralReset(SYSCTL_PERIPH_EEPROM0);	        // Reset again.
    SysCtlDelay(10);
    while(EEPROM_EEDONE_R & EEPROM_EEDONE_WORKING) {}	        // Wait till not busy.
    Status = EEPROM_EESUPP_R;
    if(Status & (EEPROM_EESUPP_PRETRY | EEPROM_EESUPP_ERETRY)) {
        Error(INIT_EEPROM_FAILURE, "Init_EEProm() error 2");
    }
}
//---------------------------------------------------------------------------------------------------------------



