//------------------------------------------------------------------------------------------------------
//  events.cc
//
//  Event Queues.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//------------------------------------------------------------------------------------------------------
//
//  Functions, ISR routines, and State Machines ("SM") can post "Events".  One event_data_t of data
//  can be sent along with the Event, it is a long long so that anything can be passed by casting
//  it to the applicable type, e.g., a pointer or a float can be passed.
//
//  Events are closely tied to State Machines as a State Machine is quiescent until it receives an Event
//  to process.  An event issuer can send an event to a specific state machine or can send it to no one
//  in particular by specifying the "SM_NULL" (non-existent) State Machine; a good example if this are
//  button pressed events that go to different places at different times and the button event generating
//  routine has no idea where it goes.  To accommodate this, functions or State Machines wishing to
//  receive any given event can "Subscribe" to that event, later it can "Unsubscribe" to the event.  It
//  is possible for a given event sent to SM_NULL to get issued to multiple State Machines.  If an
//  event is not specifically sent to a given State Machine and no State Machine or function has
//  Subscribed to that event it is just ignored.
//
//  Only events that are issued to SM_NULL can be subscribed to, e.g., an event sent to a specific
//  State Machine will not get issued to another State Machine that has also subscribed to that event.
//
//  In addition, (Un)Subscribing to events can be done either synchronously or asynchronously.  When
//  done synchronously the actual (Un)Subscription is not done in the context of the calling routine but
//  is queued and later done by a routine dispatched to by the dispatcher.  Asynchronous
//  (Un)Subscriptions are useful where a given State Machine is subscribed to a given event that when
//  it receives it wants to Unsubscribe to it and then pass it on to another function or State Machine,
//  if the Unsubscription was done synchronously then it would again receive the Event that it has
//  passed on.
//
//  ** The Following is Not Currently Implemented **
//  In addition, a function or State Machine can "Push" a Subscription to an event - this will cause all
//  other Subscriptions to that Event to be pushed onto a (virtual) Stack - they will no longer get the
//  Event, only the "Pusher" will get the event until it Unsubscribes to it, at which point the prior
//  function(s) and/or State Machines will again receive that event in future.  This allows a SM to
//  gain control of the event without needing cooperation from the SM currently subscribed to that
//  event.
//
//------------------------------------------------------------------------------------------------------


#include "common.h"
#include "uarts.h"
#include "error.h"
#include "dispatch.h"
#include "init.h"
#include "events.h"
#include "state.h"
#include "timers.h"
#include "eeprom.h"

#include <stdio.h>




                                        ////////////////
                                        //            //
                                        //   EVENTS   //
                                        //            //
                                        ////////////////



//----------------------------------------
//  GET EVENT NAME STRING FOR DEBUGGING  |
//-------------------------------------------------------------------------------------------------------

static const char* Event_Names[] = {

    "E_INITIALIZE      ",   // Sent to all SM's.

    //- Test Events -//

    "E_TEST            ",

    //- Timer Events -//

    "E_TIMEOUT         ",

    //- Brightness Switch Events -//

    "E_BRITE_SW_1_ON   ",
    "E_BRITE_SW_1_OFF  ",
    "E_BRITE_SW_2_ON   ",
    "E_BRITE_SW_2_OFF  ",

    //- CAN Events -//

    "E_CAN_MSG         ",
                            // -- WARNINGS/ERRORS --
    "E_CAN_BOFF_E      ",   // Has gone into a Bus Off state.
    "E_CAN_EWARN_ERR   ",   // Warning, an error counter is > 96
    "E_CAN_EPASS_ERR   ",   // Rx or Tx error count > 127, is in Error Passive state.
                            // -- COMM REESTABLISHED AFTER ERROR --
    "E_CAN_RX_RESUMED",
    "E_CAN_TX_RESUMED",
                            // -- ERRORS NOT CHANGING STATE --
                            // -- Should Not Happen in Silent Mode --
    "E_CAN_STUFF_ERR   ",   // > 5 equal bits where not allowed.
    "E_CAN_FORMAT_ERR  ",
    "E_CAN_ACK_ERR     ",   // No ACK Received.
    "E_CAN_BIT1_ERR    ",   // Tried to Xmit 1 but bus is 0.
    "E_CAN_BIT0_ERR    ",   // Tried to Xmit 0 but bus is 1.
                            // -- Other Warning/Error --
    "E_CAN_CRC_ERR     ",   // CRC Error in Received Message. * WHEN WILL THIS HAPPEN???

    //- Bluetooth Events -//

    "E_BT_CONNECTED    ",   // Connected
    "E_BT_DISCONNECTED ",   // Connection Lost.
                            //
    "E_BT_SEND_AT_CMNDS",   // Send AT command to BT module.
    "E_BT_AT_CMND_ECHO ",   // Send single AT command and echo response to Debug.
    "E_BT_CMND_RESPONSE",   // Response to AT command received.
    "E_BT_CMND_SUCCESS ",   // AT Command(s) succeeded.
    "E_BT_CMND_FAILURE ",   // AT Command(s) failed.
    "E_BT_CMND_TIMEOUT ",   // AT Command no Response.

};
//------------------------------------------------------------------------------------------------------

const char* Get_Event_Name( event_t Event ) {

    if((int)Event >= sizeof(Event_Names)/sizeof(Event_Names[0])) {

#if defined(CONSOLE_TRACE) and defined(DEBUG)
	sprintf(sBuff, "*Invalid Event Number=%d*", Event);
	return(sBuff);
#else
	sprintf(sBuff, "Get_Event_Name(%d)", Event);
	Error(INVALID_EVENT, sBuff);
#endif
    }
    return(Event_Names[Event]);
}
//------------------------------------------------------------------------------------------------------
//  NOT REAL OPERATIONAL CODE
//  This will throw a compile time error: "size of array 'arry' is negative", if the Event_Name[]
//  array is not the same size as the eEvent enumerations.  This is contained inside a struct that never
//  gets instantiated so it never takes up any memory space.

struct NeverGetsInstantiated__ {
    int arry[ sizeof(Event_Names) / sizeof(Event_Names[0]) == LAST_E_Event_ENTRY ? 1 : -1 ];
};
//------------------------------------------------------------------------------------------------------




//----------------------------------
//  SUBSCRIBE/UNSUBSCRIBE GLOBALS  |
//-------------------------------------------------------------------------------------------------------

const int SUBS_SIZE = 20;                       // *FIXME* What's reasonable?

PRIVATE struct {
    bool     Active;
    event_t  Event;
    sm_t     SM;
    int      Stack;
} Subscr[SUBS_SIZE];

//PRIVATE int Subs_Cnt = 0;
//PRIVATE int Max_Subs = 0;

//-------------------------------------------------------------------------------------------------------



void PrintActiveSubscriptions(void) {

    for(int i=0; i<SUBS_SIZE; i++) {
        if(Subscr[i].Active==true) {
            sprintf(sBuff, "\n\rEvent = %d,  SM = %d, Stack = %d", Subscr[i].Event, Subscr[i].SM, Subscr[i].Stack);
            TxDebugStr(sBuff);
        }
    }
}




//--------------------------------------
//  DISPATCH EVENTS TO STATE MACHINES  |
//------------------------------------------------------------------------------------------------------
//  Note that we have to use 'while(Msg_Cnt--)' instead of 'while(pMsg)' as it is possible for a given
//  SM to reissue the Event it just received via subscription to another SM, and then unsubscribe from
//  that Event; if we use 'while(pMsg)' to terminate the loop the event it just sent will get re-issued
//  to itself since the unsubscribe won't happen until after SM_Event_Dispatch is done.

void  SM_Event_Dispatch( u16 Msg_Cnt, void* vMsg, ulong Ticks ) {

    struct sEvent_Msg* pMsg;

    // For all the messages in the queue...

    pMsg = (struct sEvent_Msg*)vMsg;                              // Cast message pointer from "void".
    while( Msg_Cnt-- ) {

        Handle_Event( pMsg->SM, pMsg->Event, pMsg->Data );        // send Event to that State Machine.
	if( Msg_Cnt ) pMsg = (struct sEvent_Msg*) Get_Message();  // Get next message if there is one.
    }
}
//------------------------------------------------------------------------------------------------------




//-----------------
//  QUEUE EVENTS  |
//------------------------------------------------------------------------------------------------------
void  Send_Event( sm_t SM, event_t Event, event_data_t Data ) {

    static struct sEvent_Msg Event_Msg;

    Event_Msg.SM    = SM;
    Event_Msg.Event = Event;
    Event_Msg.Data  = Data;
    Send_Message( SM_EVENT_TASK, &Event_Msg );
}
//------------------------------------------------------------------------------------------------------




//---------------------
//  ISR QUEUE EVENTS  |
//------------------------------------------------------------------------------------------------------
//  NOTE: This code is running in interrupt context.

void  ISR_Send_Event( sm_t SM, event_t Event, event_data_t Data ) {

    static struct sEvent_Msg Event_Msg;

    Event_Msg.SM    = SM;
    Event_Msg.Event = Event;
    Event_Msg.Data  = Data;
    ISR_Send_Message( SM_EVENT_TASK, &Event_Msg );
}
//------------------------------------------------------------------------------------------------------





