//---------------------------------------------------------------------------------------------------------------
//  led.cc
//
//  Control the LEDs
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//
//  To Do
//
//  1)
//
//---------------------------------------------------------------------------------------------------------------

#include "common.h"

#include "led.h"
#include "error.h"
#include "uarts.h"
#include "eeprom.h"

#include "pwm.h"

#include "sysctl.h"
#include "hw_memmap.h"

#include <math.h>
#include <stdio.h>




                                    /////////////////////////////
                                    //                         //
                                    //  CONSTANTS AND GLOBALS  //
                                    //                         //
                                    /////////////////////////////



PRIVATE const uint  PulsesPerPeriod  = 6250;      // 200 Hz
PRIVATE const float fPulsesPerPeriod = 6250.0;

//-- Maximum PWM width percent --//

PRIVATE const float fMaxRedPWM    = 0.1;      // Value at 100 % brightness.
PRIVATE const float fMaxGreenPWM  = 0.1;
PRIVATE const float fMaxBluePWM   = 0.3;
PRIVATE const float fMaxWhitePWM  = 0.1;
PRIVATE const float fMaxSpeedoPWM = 0.1;
PRIVATE const float fMaxNeedlePWM = 0.1;

//-- PWM Register Mnemonics --//

PRIVATE const uint RED_BASE    = PWM1_BASE;
PRIVATE const uint RED_GEN     = PWM_GEN_1;
PRIVATE const uint RED_OUT     = PWM_OUT_2;
PRIVATE const uint RED_BIT     = PWM_OUT_2_BIT;
PRIVATE const uint GREEN_BASE  = PWM0_BASE;
PRIVATE const uint GREEN_GEN   = PWM_GEN_3;
PRIVATE const uint GREEN_OUT   = PWM_OUT_6;
PRIVATE const uint GREEN_BIT   = PWM_OUT_6_BIT;
PRIVATE const uint BLUE_BASE   = PWM0_BASE;
PRIVATE const uint BLUE_GEN    = PWM_GEN_0;
PRIVATE const uint BLUE_OUT    = PWM_OUT_0;
PRIVATE const uint BLUE_BIT    = PWM_OUT_0_BIT;
PRIVATE const uint WHITE_BASE  = PWM0_BASE;
PRIVATE const uint WHITE_GEN   = PWM_GEN_2;
PRIVATE const uint WHITE_OUT   = PWM_OUT_4;
PRIVATE const uint WHITE_BIT   = PWM_OUT_4_BIT;
PRIVATE const uint NEEDLE_BASE = PWM1_BASE;
PRIVATE const uint NEEDLE_GEN  = PWM_GEN_0;
PRIVATE const uint NEEDLE_OUT  = PWM_OUT_0;
PRIVATE const uint NEEDLE_BIT  = PWM_OUT_0_BIT;
PRIVATE const uint SPEEDO_BASE = PWM1_BASE;
PRIVATE const uint SPEEDO_GEN  = PWM_GEN_3;
PRIVATE const uint SPEEDO_OUT  = PWM_OUT_6;
PRIVATE const uint SPEEDO_BIT  = PWM_OUT_6_BIT;


//-- Current Tach Backlight Brightness Percent 0.0 to 1.0 --//

PRIVATE float fTachBrite;


//-- Pre-defined Colors --//

//                         Red  Grn  Blu  Wht
const sColors Black    = { 0.0, 0.0, 0.0, 0.0 };
const sColors Red      = { 1.0, 0.0, 0.0, 0.0 };
const sColors Yellow   = { 1.0, 1.0, 0.0, 0.0 };
const sColors Orange   = { 1.0, 0.7, 0.0, 0.0 };
const sColors Blue     = { 0.0, 0.0, 1.0, 0.0 };
const sColors Purple   = { 1.0, 0.7, 1.0, 0.0 };
const sColors White    = { 0.0, 0.0, 0.0, 1.0 };
const sColors CoolWht  = { 0.5, 0.5, 0.5, 0.5 };
const sColors BriteWht = { 0.6, 0.6, 0.6, 0.7 };

sColors CurrentColor;   // *** FIXME ***  Is CurrentColor, Colors[], eColors - used ??

const sColors Colors[] = { Black,   Red,      Yellow,
                           Orange,  Blue,     Purple,
                           White,   CoolWht,  BriteWht };

//-- RPM THRESHOLDS --//

PRIVATE const uint RpmHysteresis = 100;         //


//-- BRIGHTNES To AMBIENT CURVE --//

// We'll start off with a trinomial X^2*A + X*B + C curve.

const float fAmbCurveX    = 1.0;                // *FIXME* TBD
const uint  AmbHysteresis = 1;                  //    "     "
const uint  AmbDelay      = 1;                  // Seconds to delay changes.
const uint  AmbChangeTime = 1;                  // Seconds to skew from one level to the next.

//------  DATA TO READ IN FROM EEPROM  ------//
//-- ALSO THE DEFAULT DATA IF EEPROM BLANK --//

/* enum eVarType { NULL_T, UINT_T, FLOAT_T, COLOR_T }
enum         eColors { BlackIdx,  RedIdx,   YellowIdx,  OrangeIdx, BlueIdx,
                       PurpleIdx, WhiteIdx, CoolWhtIdx, BriteWhtIdx };

struct sMultiRead {
    u16       EEAddr;               // Address of data in EEProm
    void*     pVar;                 // Pointer to the variable to write to.
    eVarType  Type;                 // Type of the variable.
    uint      UintDefault;          // Default of uint data.
    float     FloatDefault;         // Default of float data.
    eColors   ColorDefault;         // Default Color
}; */
/*
const struct sMultiRead LedData[] = {

    RpmCaution1Addr,    &RpmCaution1,    UINT_T,  6000,  0.0,  BlackIdx,       // RPM Thresholds.
    RpmCaution2Addr,    &RpmCaution2,    UINT_T,  6500,  0.0,  BlackIdx,
    RpmWarnAddr,        &RpmWarn,        UINT_T,  7000,  0.0,  BlackIdx,
    RpmOverAddr,        &RpmOver,        UINT_T,  7500,  0.0,  BlackIdx,
    BlinkThresholdAddr, &BlinkThreshold, UINT_T,  7250,  0.0,  BlackIdx,

    NormalColorAddr,    &NormalColor,    COLOR_T,    0,  0.0,  WhiteIdx,       // RPM Colors.
    Caution1ColorAddr,  &Caution1Color,  COLOR_T,    0,  0.0,  YellowIdx,
    Caution2ColorAddr,  &Caution2Color,  COLOR_T,    0,  0.0,  OrangeIdx,
    RpmWarnColorAddr,   &RpmWarnColor,   COLOR_T,    0,  0.0,  RedIdx,
    RpmOverColorAddr,   &RpmOverColor,   COLOR_T,    0,  0.0,  PurpleIdx,

    NULL,               NULL,            NULL_T,  NULL, NULL,  NullIdx,        // End of Table.
};
*/



//------------------------
//  THRESHOLD SETPOINTS  |
//---------------------------------------------------------------------------------------------------------------

struct sBacklightThresholds /* {
    uint     RPM;
    sColors  Color;
    uint     Blink;
    uint     Brightness;
    bool     InterpolateColor;
    bool     InterpolateBlink;
    bool     InterpolateBrightness;

}*/ BLThresh[] = {

  {    0,  White,    0,  50, false, false, false },
  { 6000,  Yellow,   0,  50, false, false, false },
  { 7000,  Red,      0,  70, false, false, false },
  { 7500,  Red,     50,  70, false, false, false },
  { 7800,  Purple,  80,  80, false, false, false },
  {20000,  Purple,  80,  80, false, false, false },
};
const uint BLThreshSize = sizeof(BLThresh) / sizeof(BLThresh[0]);
u8 CurBLThreshIdx = 0;

//---------------------------------------------------------------------------------------------------------------

struct sNeedleThresholds /* {
    int    RPM;
    uint   Speed;
    uint   Blink;
    uint   Brightness;

}*/ NeedleThreshhold[] = {

  {  -1,  0,  0,  50 },
  {  -1, 80, 50,  50 },
  {   0,  0,  0,   0 },
  {   0,  0,  0,   0 },
  {   0,  0,  0,   0 },
};
const uint NeedleThreshSize = sizeof(NeedleThreshhold) / sizeof(NeedleThreshhold[0]);
u8 CurNeedleThreshIdx = 0;

//---------------------------------------------------------------------------------------------------------------

struct sSpeedoBLThresholds /* {
    uint   Speed;
    uint   Blink;
    uint   Brightness;

}*/ SpeedoBLThresholds[] = {

  {   0,   0,  50 },
  {  80,   0,  70 },
  { 100,  50,  70 },
};
const uint SpeedoBLThreshSize = sizeof(SpeedoBLThresholds) / sizeof(SpeedoBLThresholds[0]);
u8 CurSpeedoThreshIdx = 0;

//---------------------------------------------------------------------------------------------------------------


//      ** FIXME **

void Tmp(void) {
    sprintf(sBuff, "\r\n  BLThresh[0].Color size = %d bytes", sizeof(BLThresh[0].Color));
    TxDebugStr(sBuff);
    sprintf(sBuff, "\r\n     BlThreshold[0] size = %d bytes,     BlThreshold size = %d bytes", sizeof(BLThresh[0]), sizeof(BLThresh));
    TxDebugStr(sBuff);
    sprintf(sBuff, "\r\n NeedleThreshold[0] size = %d bytes, NeedleThreshold size = %d bytes", sizeof(NeedleThreshhold[0]), sizeof(NeedleThreshhold));
    TxDebugStr(sBuff);
    sprintf(sBuff, "\r\n SpeedoThreshold[0] size = %d bytes, SpeedoThreshold size = %d bytes\r\n", sizeof(SpeedoBLThresholds[0]), sizeof(SpeedoBLThresholds));
    TxDebugStr(sBuff);
}




                                        ///////////////////////////
                                        //                       //
                                        //  BACKLIGHT FUNCTIONS  //
                                        //                       //
                                        ///////////////////////////



//------------------------------
//  UPDATE THE TACH BACKLIGHT  |  *** FIXME *** CurBLThreshIdx is global, why pass it as a parameter ??
//---------------------------------------------------------------------------------------------------------------
void UpdateBacklight(u8 Idx) {


}
//---------------------------------------------------------------------------------------------------------------
void UpdateNeedles(u8 Idx) {}
void UpdateSpeedoBacklight(u8 Idx) {}




//---------------------------------------
//  SET TACH BACKLIGHT FOR CURRENT RPM  |
//---------------------------------------------------------------------------------------------------------------
void SetTachBacklight(uint RPM) {

    if(RPM < (BLThresh[CurBLThreshIdx].RPM - RpmHysteresis)) {
        CurBLThreshIdx--;
        UpdateBacklight(CurBLThreshIdx);
    } else if(RPM >= BLThresh[CurBLThreshIdx + 1].RPM) {
        CurBLThreshIdx++;
        UpdateBacklight(CurBLThreshIdx);
    }
}
//---------------------------------------------------------------------------------------------------------------




//------------------------
//  SET NEEDLES FOR ???  | *** FIXME ***
//---------------------------------------------------------------------------------------------------------------
void SetNeedles(uint Var) {

}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------
//  SET SPEEDOMETER BACKLIGHT FOR CURRENT SPEED  |
//---------------------------------------------------------------------------------------------------------------
void SetSpeedoBacklight(uint MPH) {

}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------
//  UPDATE THE TACH BACKLIGHT PWMs  |
//---------------------------------------------------------------------------------------------------------------
void UpdateTachColor(void) {

    uint  Base, Out, Bit;
    float fMaxPwm, fCurPwm;

for(uint i = (uint)RED; i <= (uint)WHITE; i++) {                        // For all colors ...
        switch(i) {
            case RED:               // 0
                Base    = RED_BASE;
                Out     = RED_OUT;
                Bit     = RED_BIT;
//              fCurPwm = fRedPWM;
                fMaxPwm = fMaxRedPWM;
                break;
            case GREEN:             // 1
                Base    = GREEN_BASE;
                Out     = GREEN_OUT;
                Bit     = GREEN_BIT;
//              fCurPwm = fGreenPWM;
                fMaxPwm = fMaxGreenPWM;
                break;
            case BLUE:              // 2
                Base    = BLUE_BASE;
                Out     = BLUE_OUT;
                Bit     = BLUE_BIT;
//              fCurPwm = fBluePWM;
                fMaxPwm = fMaxBluePWM;
                break;
            case WHITE:             // 3
                Base    = WHITE_BASE;
                Out     = WHITE_OUT;
                Bit     = WHITE_BIT;
//              fCurPwm = fWhitePWM;
                fMaxPwm = fMaxWhitePWM;
                break;
            default:
                sprintf(sBuff, "led.cc line %d", __LINE__);
                Error(INVAL_SWITCH_VALUE, sBuff);
        }
        fCurPwm = fTachBrite * fMaxPwm;                                 // Scale PWM.
        if((uint)fCurPwm == 0) {
            PWMOutputState(Base, Bit, false);                           // Turn PWM off.
        } else {                                                        // Else set the period.
            PWMPulseWidthSet(Base, Out, (uint)round(fPulsesPerPeriod * fCurPwm));
            PWMOutputState(Base, Bit, true);
        }
    }
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------
//  SET TACHOMETER BACKLIGHT COLOR  |
//---------------------------------------------------------------------------------------------------------------
//  Each parameter values are 0 - 100%.

void TachBacklightColor(float fRed, float fGreen, float fBlue, float fWhite) {

    CurrentColor.fRed   = fRed;
    CurrentColor.fGreen = fGreen;
    CurrentColor.fBlue  = fBlue;
    CurrentColor.fWhite = fWhite;
    UpdateTachColor();
}
//---------------------------------------------------------------------------------------------------------------
void TachBacklightColor(eColors ColorIdx) {

    CurrentColor.fRed   = Colors[ColorIdx].fRed;
    CurrentColor.fGreen = Colors[ColorIdx].fGreen;
    CurrentColor.fBlue  = Colors[ColorIdx].fBlue;
    CurrentColor.fWhite = Colors[ColorIdx].fWhite;
    UpdateTachColor();
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------------
//  SET TACHOMETER BACKLIGHT BRIGHTNESS  |
//---------------------------------------------------------------------------------------------------------------
//  Perceived % brightness from 0 to 100

void TachBacklightBrightness(uint Brite) {

    fTachBrite = powf((float)Brite/10.0, 2.2) / 158.49;     // Adjust for perceived brightness, make 0.0 - 1.0
    UpdateTachColor();
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------
//  SET SPEEDOMETER BACKLIGHT BRIGHTNESS  |
//---------------------------------------------------------------------------------------------------------------
//  Perceived % brightness from 0 to 100

void SpeedoBacklightBrightness(uint Brite) {

    float fSpeedoBrite;

    fSpeedoBrite = powf((float)Brite/10.0, 2.2) / 158.49;        // Adjust for perceived brightness, make 0.0-1.0
    fSpeedoBrite = fSpeedoBrite * fMaxSpeedoPWM;                 // Scale PWM.
    if((uint)fSpeedoBrite == 0) {
        PWMOutputState(SPEEDO_BASE, SPEEDO_BIT, false);          // Turn PWM off.
    } else {                                                     // Else set the period.
        PWMPulseWidthSet(SPEEDO_BASE, SPEEDO_OUT, (uint)round(fPulsesPerPeriod * fSpeedoBrite));
        PWMOutputState(SPEEDO_BASE, SPEEDO_BIT, true);
    }
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------
//  SET NEEDLE BRIGHTNESS  |
//---------------------------------------------------------------------------------------------------------------
void NeedleBrightness(uint Brite) {

    float fNeedleBrite;

    fNeedleBrite = powf((float)Brite/10.0, 2.2) / 158.49;        // Adjust for perceived brightness, make 0.0-1.0
    fNeedleBrite = fNeedleBrite * fMaxNeedlePWM;                 // Scale PWM.
    if((uint)fNeedleBrite == 0) {
        PWMOutputState(NEEDLE_BASE, NEEDLE_BIT, false);          // Turn PWM off.
    } else {                                                     // Else set the period.
        PWMPulseWidthSet(NEEDLE_BASE, NEEDLE_OUT, (uint)round(fPulsesPerPeriod * fNeedleBrite));
        PWMOutputState(NEEDLE_BASE, NEEDLE_BIT, true);
    }
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------------
//  SET SPECIFIED LED PWM PULSE WIDTH  |
//---------------------------------------------------------------------------------------------------------------
//  PwmPct range 0 - 100.  Called from monitor.cc

void ManualPWM(ePWM Which, uint PwmPct) {

    uint Base, Out, Bit;

    switch(Which) {
        case RED:               // 0
            Base = RED_BASE;
            Out  = RED_OUT;
            Bit  = RED_BIT;
            break;
        case GREEN:             // 1
            Base = GREEN_BASE;
            Out  = GREEN_OUT;
            Bit  = GREEN_BIT;
            break;
        case BLUE:              // 2
            Base = BLUE_BASE;
            Out  = BLUE_OUT;
            Bit  = BLUE_BIT;
            break;
        case WHITE:             // 3
            Base = WHITE_BASE;
            Out  = WHITE_OUT;
            Bit  = WHITE_BIT;
            break;
        case NEEDLES:           // 4
            Base = NEEDLE_BASE;
            Out  = NEEDLE_OUT;
            Bit  = NEEDLE_BIT;
            break;
        case SPEEDO:            // 5
            Base = SPEEDO_BASE;
            Out  = SPEEDO_OUT;
            Bit  = SPEEDO_BIT;
            break;
        default:
            sprintf(sBuff, "led.cc line %d", __LINE__);
            Error(INVAL_SWITCH_VALUE, sBuff);
    }
    if(PwmPct == 0) {
        PWMOutputState(Base, Bit, false);                       // Turn off PWM, or else it stays high.
    } else {
        PWMPulseWidthSet(Base, Out, (uint)roundf((float)PulsesPerPeriod * ((float)PwmPct / 100.0)));
        PWMOutputState(Base, Bit, true);
    }
}
//---------------------------------------------------------------------------------------------------------------






                                            //////////////////////
                                            //                  //
                                            //  INITIALIZATION  //
                                            //                  //
                                            //////////////////////



//-------------------------------
//  INITIALIZE THE LED CONTROL  |
//---------------------------------------------------------------------------------------------------------------
void Init_LEDs(void) {                                                          //
                                                                                //
    //-- INITIALIZE THE PWMs --//                                               //
                                                                                //
    SysCtlPWMClockSet(SYSCTL_PWMDIV_64);                                        // 1250 KHz PWM Clock.
                                                                                //
    PWMGenConfigure(RED_BASE,    RED_GEN,    PWM_GEN_MODE_DOWN   |              // Set each PWM generator.
                                             PWM_GEN_MODE_NO_SYNC );            //
    PWMGenConfigure(GREEN_BASE,  GREEN_GEN,  PWM_GEN_MODE_DOWN   |              //
                                             PWM_GEN_MODE_NO_SYNC );            //
    PWMGenConfigure(BLUE_BASE,   BLUE_GEN,   PWM_GEN_MODE_DOWN   |              //
                                             PWM_GEN_MODE_NO_SYNC );            //
    PWMGenConfigure(WHITE_BASE,  WHITE_GEN,  PWM_GEN_MODE_DOWN   |              //
                                             PWM_GEN_MODE_NO_SYNC );            //
    PWMGenConfigure(NEEDLE_BASE, NEEDLE_GEN, PWM_GEN_MODE_DOWN   |              //
                                             PWM_GEN_MODE_NO_SYNC );            //
    PWMGenConfigure(SPEEDO_BASE, SPEEDO_GEN, PWM_GEN_MODE_DOWN   |              //
                                             PWM_GEN_MODE_NO_SYNC );            //
                                                                                //
    PWMDeadBandDisable(RED_BASE,    RED_GEN   );                                // Deadband disable.
    PWMDeadBandDisable(GREEN_BASE,  GREEN_GEN );                                //
    PWMDeadBandDisable(BLUE_BASE,   BLUE_GEN  );                                //
    PWMDeadBandDisable(WHITE_BASE,  WHITE_GEN );                                //
    PWMDeadBandDisable(NEEDLE_BASE, NEEDLE_GEN);                                //
    PWMDeadBandDisable(SPEEDO_BASE, SPEEDO_GEN);                                //
                                                                                //
    PWMGenPeriodSet(RED_BASE,    RED_GEN,    PulsesPerPeriod);                  // 200Hz output.
    PWMGenPeriodSet(GREEN_BASE,  GREEN_GEN,  PulsesPerPeriod);                  //
    PWMGenPeriodSet(BLUE_BASE,   BLUE_GEN,   PulsesPerPeriod);                  //
    PWMGenPeriodSet(WHITE_BASE,  WHITE_GEN,  PulsesPerPeriod);                  //
    PWMGenPeriodSet(NEEDLE_BASE, NEEDLE_GEN, PulsesPerPeriod);                  //
    PWMGenPeriodSet(SPEEDO_BASE, SPEEDO_GEN, PulsesPerPeriod);                  //
                                                                                //
    PWMPulseWidthSet(RED_BASE,    RED_OUT,    0);                               // 0% duty cycle.
    PWMPulseWidthSet(GREEN_BASE,  GREEN_OUT,  0);                               //
    PWMPulseWidthSet(BLUE_BASE,   BLUE_OUT,   0);                               //
    PWMPulseWidthSet(WHITE_BASE,  WHITE_OUT,  0);                               // *FIXME******************
    PWMPulseWidthSet(NEEDLE_BASE, NEEDLE_OUT, 0);                               //
    PWMPulseWidthSet(SPEEDO_BASE, SPEEDO_OUT, 0);                               //
                                                                                //
    PWMOutputUpdateMode(PWM0_BASE, BLUE_BIT   |                                 //
                                   WHITE_BIT  |                                 //
                                   GREEN_BIT,                                   //
                                   PWM_OUTPUT_MODE_NO_SYNC);                    //
    PWMOutputUpdateMode(PWM1_BASE, NEEDLE_BIT |                                 //
                                   RED_BIT    |                                 //
                                   SPEEDO_BIT,                                  //
                                   PWM_OUTPUT_MODE_NO_SYNC);                    //
                                                                                //
    PWMOutputState(PWM0_BASE, BLUE_BIT   |                                      // Force them to 0;
                              WHITE_BIT  |                                      //
                              GREEN_BIT,   false);                              //
    PWMOutputState(PWM1_BASE, NEEDLE_BIT |                                      //
                              RED_BIT    |                                      //
                              SPEEDO_BIT,  false);                              //
                                                                                //
    PWMGenEnable(RED_BASE,    RED_GEN   );                                      // Turn em on.
    PWMGenEnable(GREEN_BASE,  GREEN_GEN );                                      //
    PWMGenEnable(BLUE_BASE,   BLUE_GEN  );                                      //
    PWMGenEnable(WHITE_BASE,  WHITE_GEN );                                      //
    PWMGenEnable(NEEDLE_BASE, NEEDLE_GEN);                                      //
    PWMGenEnable(SPEEDO_BASE, SPEEDO_GEN);                                      //
                                                                                //
    //-- PULL THE CURRENT VALUES FROM EEPROM --//                               //
                                                                                //
    ReadInBLThreshVars(&BLThresh[0], BLThreshSize);                              // Read in Data from EEProm.
    ReadInNeedleVars(&NeedleThreshhold[0], NeedleThreshSize);                   //
    ReadInSpeedoVars(&SpeedoBLThresholds[0], SpeedoBLThreshSize);               //
                                                                                //
    // *FIXME*  To Do **************                                            // *FIXME*
                                                                                //
    UpdateBacklight(0);                                                         // Set LED's to quiescent values.
    UpdateNeedles(0);                                                           //
    UpdateSpeedoBacklight(0);                                                   //
}                                                                               //
//---------------------------------------------------------------------------------------------------------------


