//---------------------------------------------------------------------------------------------------------------
//  sm_bluetooth.cc	enum: SM_BLUETOOTH
//
//  Communicating via "Serial Port Connection" between here and iOS or Android.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//
//  The DSD Tech HM-10 Bluetooth module operates by first taking "AT" style modem commands that are used to
//  configure it, each AT command elicits a response message when that command is done.  It is configured as
//  a slave device and starts advertising; once a connection to a master device (smart phone) is made
//  *I THINK* it will send "AT+CONN" and it switches to operating as a transparent "serial port", e.g.,
//  all data is transparently sent/received.  When a connection is broken *I THINK* it will send "OK+LOST".
//  When connected the master/slave protocol will need to take this into account.
//
//  The state of those two modes is kept in the global variable BtCommMode, it is set to eBtCommMode: BtAtMode or
//  BtDataMode.
//
//  The monitor is able to send AT commands and print the Result by calling BtAtCmndEcho().
//
//  Once a connection is made, ** I currently assume ** that binary data can be sent.  Therefore the protocol
//  is as follows:  The first byte is the message number starting at 0, the next byte is the length, the rest
//  is the payload, there is no EOM character.
//
//---------------------------------------------------------------------------------------------------------------


#include "common.h"
#include "sm_bt_at_cmnd.h"
#include "sm_bluetooth.h"
#include "error.h"
#include "events.h"
#include "state.h"
#include "timers.h"
#include "tick.h"
#include "uarts.h"
#include "dispatch.h"

#include <stdio.h>
#include <string.h>


eBtCommMode BtCommMode = BtAtMode;          // The module starts in AT mode.

eBtState BtCurrentState = BtIdle;



                                            /////////////////////////
                                            //                     //
                                            //  SUPPORT FUNCTIONS  //
                                            //                     //
                                            /////////////////////////





//---------------------------------------------
//  SET THE AT COMMAND AND EXPECTED RESPONSE  |
//---------------------------------------------------------------------------------------------------------------
void SetAtCmnd(u8 Idx, char* pCmnd, char* pResp) {

    AtCommands[Idx].pAtCommand        = pCmnd;
    AtCommands[Idx].pExpectedResponse = pResp;
}
//---------------------------------------------------------------------------------------------------------------
inline void SetAtCmnd(u8 Idx, const char* pCmnd, const char* pResp) {

    SetAtCmnd(Idx, const_cast<char*>(pCmnd), const_cast<char*>(pResp));
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------------------------------
//  SET UP MODULE AND START ADVERTISING AS A PERIPHERAL  |
//---------------------------------------------------------------------------------------------------------------
void BtInitialize(void) {

    SetAtCmnd( 0, "AT+IMME1",       "OK+Set:1"       );   // Turn on notifications.
    SetAtCmnd( 1, "AT+ROLE0",       "OK+Set:0"       );   // Slave role.
    SetAtCmnd( 2, "AT+NAMEDbAssoc", "OK+Set:DbAssoc" );   // Add my name.
    SetAtCmnd( 3, "AT+NOTI1",       "OK+Set:1"       );   // Enable OK+CONN and OK+LOST on Connect, Disconnect.
    SetAtCmnd( 4, "AT+PIO11",       "OK+Set:1"       );   // PIO-1 low when not connected, high when connected.
    SetAtCmnd( 5, "AT+ADTY0",       "OK+Set:0"       );   // Start Advertising.
    SetAtCmnd( 6, "AT+FLAG0",       "OK+FLAG0"       );   // Now Kick It.

    Send_Event( SM_BT_AT_CMND, E_BT_SEND_AT_CMNDS, 0 ); // Could go back to passing &AtCommands as event Data.
}
//---------------------------------------------------------------------------------------------------------------





                                        ///////////////////////////////
                                        //                           //
                                        //  BLUETOOTH STATE MACHINE  //
                                        //                           //
                                        ///////////////////////////////


namespace nsBtState {

//PRIVATE handle_t TimerHandle; // B

enum StateNmbs { S0_Idle, S1_WaitAdv, S2_Connected, S3_Disconnected };      // States

//-----------------------------
//  UNHANDLED EVENT FUNCTION  |
//---------------------------------------------------------------------------------------------------------------
//  Implements a crude 2 level "hierarchical state machine"

state_nmb_t Unhandled_Event(state_nmb_t CurrentState, event_t Event, event_data_t EventData ) {

    sprintf(sBuff, "sm_bluetooth.cc Unhandled_Event(CurState %d, Event %d, Data %d)",
            CurrentState, Event, EventData);
    Error(UNHANDLED_EVENT_ERR, sBuff);

    switch(Event) {

        default:
	    return(NO_STATE_CHANGE);
    }
//  return(S1_Nada);
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------
//  STATE 0, ADVERTIZING  S0_Idle  |
//---------------------------------------------------------------------------------------------------------------
//  This is not called on initialization, but will be if another state transitions here.

state_nmb_t S0_Idle_Entry(state_nmb_t PriorState) {

    return(NO_STATE_CHANGE);
}
//---------------------------------------------------------------------------------------------------------------
//  Run on reset, State_0_Event() should do whatever initialization this State Machine needs when it receives
//  a Reset event.

state_nmb_t S0_Idle_Event(event_t Event, event_data_t Data) {

    switch(Event) {                             //
	case E_INITIALIZE:                      // Set up the Bluetooth Module and start Advertising.
	                                        //
	    return(S1_WaitAdv);                 // Go into Advertising State.
	                                        //
	case E_BT_SEND_AT_CMNDS:                // Send AT command to BT module.
	                                        //
	    StringAndPrompt("\r\nsm_bluetooth E_BT_SEND_AT_CMNDS");     // *FIXME* What to do here.
	    return(NO_STATE_CHANGE);            //
	                                        //
	case E_BT_AT_CMND_ECHO:                 // Send single AT command and echo response to Debug.
                                                //
	    StringAndPrompt("\r\nsm_bluetooth E_BT_AT_CMND_ECHO");     // *FIXME* What to do here.
	    return(NO_STATE_CHANGE);            //
	                                        //
    }                                           //
    return(UNHANDLED_EVENT);                    //
}                                               //
//---------------------------------------------------------------------------------------------------------------




//-------------------------------------------------------
//  STATE 1, GOING INTO ADVERTISING STATE   S1_WaitAdv  |
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S1_WaitAdv_Entry(state_nmb_t PriorState) {

    BtInitialize();                     // Start Advertising.
    return(NO_STATE_CHANGE);
}
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S1_WaitAdv_Event(event_t Event, event_data_t Data) {        //
                                                                        //
    switch(Event) {                                                     //
                                                                        //
        case E_BT_CMND_SUCCESS:                                         //
                                                                        //
            BtCurrentState = BtAdvertising;                             //
            return(S0_Idle);                                            //
                                                                        //
        case E_BT_CMND_FAILURE:                                         //
                                                                        //
            sprintf(sBuff, "\r\n  -> Bad AT Response: '%s' Index: %d",  //
                    BtInLinBuff, Data);                                 //
            StringAndPrompt(sBuff);                                     // For now just print the error.
            return(S0_Idle);                                            //
                                                                        //
        case E_BT_CMND_TIMEOUT:                                         //
                                                                        //
            sprintf(sBuff, "\r\n  -> No AT Response: Index: %d", Data); //
            StringAndPrompt(sBuff);                                     // For now just print the error.
            return(S0_Idle);                                            //
                                                                        //
    }                                                                   //
     return(UNHANDLED_EVENT);                                           //
}                                                                       //
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------
//  STATE 2, NOW CONNECTED   S2_Connected  |
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S2_Connected_Entry(state_nmb_t PriorState) {

    BtCurrentState = BtConnected;
    return(NO_STATE_CHANGE);
}
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S2_Connected_Event(event_t Event, event_data_t Data) {

    switch(Event) {

        case E_BT_DISCONNECTED:
            return(S3_Disconnected);

//      case E_TIMEOUT:
//	    return(NO_STATE_CHANGE);
    }
    return(UNHANDLED_EVENT);
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------
//  STATE 3, NOW DISCONNECTED   S3_Disconnected  |
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S3_Disconnected_Entry(state_nmb_t PriorState) {

    BtCurrentState = BtDisconnected;
    return(NO_STATE_CHANGE);
}
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S3_Disconnected_Event(event_t Event, event_data_t Data) {

    return(UNHANDLED_EVENT);
}
//---------------------------------------------------------------------------------------------------------------




//----------------
//  STATE TABLE  |
//---------------------------------------------------------------------------------------------------------------

const StateTbl States[] = {
    { "S0_Idle          ", S0_Idle_Entry,         S0_Idle_Event         },
    { "S1_WaitAdv       ", S1_WaitAdv_Entry,      S1_WaitAdv_Event      },
    { "S2_Connected     ", S2_Connected_Entry,    S2_Connected_Event    },
    { "S3_Disconnected  ", S3_Disconnected_Entry, S3_Disconnected_Event },
};
//---------------------------------------------------------------------------------------------------------------

const int StateTbl_Size = sizeof(States) / sizeof(struct StateTbl);


}  /* namespace nsBtState */


