//---------------------------------------------------------------------------------------------------------------
//  sm_bt_at_cmnd.cc     enum SM_BT_AT_CMND
//
//  Communicating via "Serial Port Connection" between here and iOS or Android.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//
//  Send AT command to the module and wait for it's response.
//
//  A function wanting to send 1 or more AT commands to the Bluetooth module sends a E_BT_SEND_AT_CMNDS event
//  to this state machine with a pointer to a structure array of the AT command along with it's expected
//  response, struct sAtCmnd; the array is null terminated. On completion of all of the AT commands being
//  successful it sends a Message back of that event; if the response from one of the commands is not what was
//  expected a Message is sent with the AT command and the not-expected response that it received.  If the
//  expected response string is null, no checking of the response is done.
//
//  For the Monitor, BtAtCmndEcho() is called to send a single AT command where we wait for the response, or
//  a no-response timeout, and display the response or an error message; it sends a E_BT_AT_CMND_ECHO event
//  to the state machine here.
//
//---------------------------------------------------------------------------------------------------------------


#include "common.h"
#include "sm_bt_at_cmnd.h"
#include "error.h"
#include "events.h"
#include "state.h"
#include "timers.h"
#include "tick.h"
#include "uarts.h"
#include "dispatch.h"

#include <stdio.h>
#include <string.h>



sAtCmnd AtCommands[MaxAtCommands];      // Instantiate AT commands buffer.





                                        /////////////////////////
                                        //                     //
                                        //  SUPPORT FUNCTIONS  //
                                        //                     //
                                        /////////////////////////






//-----------------------------------------
//  SEND AT COMMAND AND DISPLAY RESPONSE  |
//---------------------------------------------------------------------------------------------------------------
void BtAtCmndEcho(char* pAtCmnd) {

    SetAtCmnd(0, pAtCmnd, const_cast<char*>(""));
    Send_Event( SM_BT_AT_CMND, E_BT_AT_CMND_ECHO, 0 );
}
//---------------------------------------------------------------------------------------------------------------





                                    //////////////////////////////////////////
                                    //                                      //
                                    //  BLUETOOTH AT COMMAND STATE MACHINE  //
                                    //                                      //
                                    //////////////////////////////////////////


namespace nsBtAtState {



//----------------------------
//  STATE MACHINE VARIABLES  |
//---------------------------------------------------------------------------------------------------------------
enum eBtAtState { S0_BtAtSend, S1_BtAtResponse, S2_BtEchoResponse };   // Bluetooth AT Command State.

PRIVATE handle_t TimerHandle;
PRIVATE uint     AtCmndIdx = 0;         // Index into sAtCmnd array.
//PRIVATE sAtCmnd* pAtCmnd;               // Pointer to start of sAtCmnd array.
//---------------------------------------------------------------------------------------------------------------




//-----------------------------
//  UNHANDLED EVENT FUNCTION  |
//---------------------------------------------------------------------------------------------------------------
//  Implements a crude 2 level "hierarchical state machine"

state_nmb_t Unhandled_Event(state_nmb_t CurrentState, event_t Event, event_data_t EventData ) {

    switch(Event) {

        case E_TIMEOUT:
            return(NO_STATE_CHANGE);

    }
    return(UNHANDLED_EVENT);
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------
//  STATE 0, SEND AT COMMAND  S0_BtAtSend  |
//---------------------------------------------------------------------------------------------------------------
//  This is not called on initialization, but will be if another state transitions here.

state_nmb_t S0_BtAtSend_Entry(state_nmb_t PriorState) {

    return(NO_STATE_CHANGE);
}
//---------------------------------------------------------------------------------------------------------------
//  Run on reset, State_0_Event() should do whatever initialization this State Machine needs when it receives
//  a Reset event.

state_nmb_t S0_BtAtSend_Event(event_t Event, event_data_t Data) {

    switch(Event) {                             //
        case E_INITIALIZE:                      // Set up the Bluetooth Module and start Advertising.
                                                //
             return(NO_STATE_CHANGE);           //
                                                //
        case E_BT_SEND_AT_CMNDS:                // Send 1 or more AT command(s) to the Bluetooth module.
                                                //
            return(S1_BtAtResponse);            // Wait for Response from command.
                                                //
        case E_BT_AT_CMND_ECHO:                 // Send 1 AT command and display response.
                                                //
            return(S2_BtEchoResponse);          // Wait for Response from command.
                                                //
    }                                           //
    return(UNHANDLED_EVENT);                    //
}                                               //
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------------------------
//  STATE 1, WAIT FOR AT COMMAND RESPONSE  S1_BtAtResponse  |
//---------------------------------------------------------------------------------------------------------------
//  We stay here until all AT commands are sent or an error occurred.
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S1_BtAtResponse_Entry(state_nmb_t PriorState) {     //
                                                                //
    TxBtStr(AtCommands[AtCmndIdx].pAtCommand);                   // Send first AT command.
    TxBtStr("\r\n");                                            //
/*
sprintf(sBuff, "\r\n  -> Send: '%s'", AtCommands[AtCmndIdx].pAtCommand); // B
TxDebugStr(sBuff); // B
*/
    TimerHandle=Set_Timer(ms500, ONE_SHOT, SM_BT_AT_CMND, NULL);//
    return(NO_STATE_CHANGE);                                    //
}                                                               //
//---------------------------------------------------------------------------------------------------------------
//  Response is in BtInLinBuff.  Result is one of: E_BT_CMND_SUCCESS, E_BT_CMND_FAILURE, E_BT_CMND_TIMEOUT.
//  On Response Failure or Timeout, the failing AtCmndIdx is the Data in the Event.

state_nmb_t S1_BtAtResponse_Event(event_t Event, event_data_t Data) {

    int           Result;                                       //
    const char*   pExpected;                                    //
                                                                //
    switch(Event) {                                             //
                                                                //
        case E_BT_CMND_RESPONSE:                                //
                                                                //
            Cancel_Timer(TimerHandle);                          //
            pExpected = AtCommands[AtCmndIdx].pExpectedResponse;// See if response is what we expected.
/*
sprintf(sBuff, "\r\n  -> pExpected '%s'", pExpected); // B
TxDebugStr(sBuff);
*/
            if(strlen(pExpected) > 0) {                         // If expected is Null, ignore result.
                Result = strcmp(BtInLinBuff, pExpected);        //
                if(Result != 0) {                               //
                    Send_Event(SM_BLUETOOTH,                    //
                            E_BT_CMND_FAILURE,                  //
                            AtCmndIdx);                         // -- AT Command Failed --
                    return(S0_BtAtSend);                        //
                }                                               //
            }                                                   //
/*
sprintf(sBuff, "\r\n  -> Returned: '%s'\r\n", BtInLinBuff);
TxDebugStr(sBuff); // B
*/
            if(AtCommands[++AtCmndIdx].pAtCommand == 0) {       // See if at end of command array (Idx++ here).
                Send_Event(SM_BLUETOOTH, E_BT_CMND_SUCCESS, 0); //
            } else {                                            //
/*
sprintf(sBuff, "\r\n  -> Send: '%s'\r\n", AtCommands[AtCmndIdx].pAtCommand); // B
TxDebugStr(sBuff); // B
*/
                TxBtStr(AtCommands[AtCmndIdx].pAtCommand);      // --  Send Next AT Command --
                TxBtStr("\r\n");                                //
                TimerHandle=Set_Timer(ms500,                    //
                                      ONE_SHOT,                 //
                                      SM_BT_AT_CMND,            //
                                      NULL);                    //
                return(NO_STATE_CHANGE);                        // -- Stay Here --
            }                                                   //
            return(S0_BtAtSend);                                // -- All Done --
                                                                //
        case E_TIMEOUT:                                         //
                                                                //
            Send_Event(SM_BLUETOOTH,                            //
                       E_BT_CMND_TIMEOUT,                       //
                       AtCmndIdx);                              //
            AtCmndIdx = 0;                                      // Get ready for next message.
            return(S0_BtAtSend);                                //
    }                                                           //
    return(UNHANDLED_EVENT);                                    //
}                                                               //
//---------------------------------------------------------------------------------------------------------------




//---------------------------------------------------------------
//  STATE 2, WAIT FOR 'AT' COMMAND RESPONSE  S1_BtEchoResponse  |
//---------------------------------------------------------------------------------------------------------------
state_nmb_t S2_BtEchoResponse_Entry(state_nmb_t PriorState) {   //
                                                                //
    TxBtStr(AtCommands[0].pAtCommand);                           // Send AT command.
    TxBtStr("\r\n");                                            //
    TimerHandle=Set_Timer(ms500,ONE_SHOT,SM_BT_AT_CMND,NULL);   //
    return(NO_STATE_CHANGE);                                    //
}                                                               //
//---------------------------------------------------------------------------------------------------------------
//  Response is in BtInLinBuff.

state_nmb_t S2_BtEchoResponse_Event(event_t Event, event_data_t Data) {

    switch(Event) {                                             //
                                                                //
        case E_BT_CMND_RESPONSE:                                //
                                                                //
            Cancel_Timer(TimerHandle);                          //
            sprintf(sBuff, "\r\n  ->  %s\r\n", BtInLinBuff);    //
            StringAndPrompt(sBuff);                             //
            return(S0_BtAtSend);                                //
                                                                //
        case E_TIMEOUT:                                         //
                                                                //
            StringAndPrompt("\r\n  ->  * No Response *");       // No Response
            return(S0_BtAtSend);                                //
    }                                                           //
    return(UNHANDLED_EVENT);                                    //
}                                                               //
//---------------------------------------------------------------------------------------------------------------




//----------------
//  STATE TABLE  |
//---------------------------------------------------------------------------------------------------------------

const StateTbl States[] = {

    { "S0_BtAtSend      ",  S0_BtAtSend_Entry,        S0_BtAtSend_Event       }, // S0_BtAtSend
    { "S1_BtAtResponse  ",  S1_BtAtResponse_Entry,    S1_BtAtResponse_Event   }, // S1_BtAtResponse
    { "S2_BtEchoResponse",  S2_BtEchoResponse_Entry,  S2_BtEchoResponse_Event }, // S2_BtEchoResponse
};
//---------------------------------------------------------------------------------------------------------------

const int StateTbl_Size = sizeof(States) / sizeof(struct StateTbl);




} // namespace nsBtAtState
