//---------------------------------------------------------------------------------------------------------------
//  state.cc
//
//  State Machine.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//
//  This State Machine ("SM") runs multiple independent state machines.  In this file is an array structure
//  (StateMachines[]) of (constant) info on each of the state machines: a pointer to that SM's State table, a
//  string of the SM's name (for debugging, might not get used), the number of states in that SM, and a pointer
//  to a function to handle 'Unhandled Events'.
//
//  States respond to Events that are generated by any function, State Machine, ISR, or Timer; each State Machine
//  has it's own queue of events that it processes until the queue is empty - this is implemented in the
//  events.cc module.  Events are queued by calling either ISR_Send_Event() or Send_Event() or by Timeouts for
//  a given State Machine ( which calls Send_Event() ).
//
//  Each State has 2 functions: an Entry Function that is executed when the State is first entered, and an
//  Event Function that process Events that the State receives.  The Event Function returns the next state to
//  transition to, if any (no transition happens when it returns NO_STATE_CHANGE), the Entry Function can also
//  cause transition to a new state immediately.  Note that a State can transition to itself, causing that
//  State's Entry Function to be executed again.
//
//  If a State receives an Event that it does not know how to handle, it returns UNHANDLED_EVENT, this causes
//  that State Machine's 'Unhandled Event Function' to be invoked.  This allows Events that otherwise would have
//  to be handled in all States in a given State Machine to be handled in one place; examples would be Low Battery
//  or Trigger Off type events (kind of a poor man's 2 level hierarchical state machine).
//
//  On initialization, State 0 of each SM receives the E_INITIALIZE event, it should do whatever initialization
//  is needed.  It can either then transition to another state or just wait for the next event.  State 0's
//  Entry Function is not called at reset, but it will run whenever another State transitions to it (or if it
//  transitions to itself).
//
//  Each SM State Functions are in their own Namespace.  Event enumerations are global among all SM's to make
//  debugging easier.
//
//  State Entry Function should return NO_STATE_CHANGE if not transitioning to a new state; a State Entry Function
//  transitioning to itself will cause an infinite loop.
//
//  Function(s) emitting an event typically know what State Machine the event goes to; it is also possible to
//  emit events to no specific SM by specifying the State Machine "SM_NULL".  State Machines can subscribe and
//  unsubscribe to events dynamically.  When an event is emitted, the specified SM will receive the event -
//  unless SM_NULL is specified - and all other SM's that have subscribed to that event will receive it also.
//  An event specifically sent to a given SM that has also subscribed to that event will only receive the event
//  once.
//
//  Thus Events can be thought of as being 'Pushed' to a given SM when that state machine is known, and/or Events
//  can be 'Pulled' by the SM(s) that want them; for example this 'Pull' (or 'Subscribe') method is used to pass
//  Button Events to different SM's at different times: when counts are being taken the events go to the given
//  counting SM, when the user is accessing the Menus the Button Events are going to the applicable Menu SM.
//
//  Another useful technique is if a given state is to be transitioned to by many different states and be able to
//  return to the calling state is by saving the PriorState passed into the state's Entry function and then
//  transitioning back to it; the 'Freeze' state in the Mod-B Run SM is an example of this.
//
//  Analogously, a SM can potentially receive an Event from one of several other SMs and and be able to return its
//  Events to that SM; the issuing SM adds its SM Enumeration as Event Data.  An example of this (from the days
//  of the PMT based Mod-C code) is SM_HV, when it receives an E_START_HV event the Event Data included is the
//  'invoking' SM and SM_HV sends its events back to that invoking SM in the Mod-C code.
//
//  SM Numbers are indexes into the StateMachines[] table and as such range from 0 to StateMachines_Size - 1.
//  There are "special states" that indicate such things as NO_STATE_CHANGE, UNHANDLED_EVENT, etc.  Their numbers
//  run from SPEC_STATES (100 as of this writing) to SPEC_STATES + End_Special_States - 1.
//
//  A valid design idiom is to have a state machine with only one state to perform functions that need to
//  respond to events but aren't "state oriented".  An example is sm_periodic_timers.cc that performs periodic
//  functions (i.e. updating the display of the state of battery charge) that need to respond to things like
//  going into and out of hibernation.  Another is sm_event_functs.cc that contains miscellaneous functions
//  that need to respond to Events.
//
//  Note that if a given SM is using timeouts at all, all E_TIMEOUT events must check that it's their timeout
//  by checking the Event Handle returned as Data for that event is theirs (and also not INVALID_HANDLE).  This
//  is a "cover your butt" as there might be a small race condition where the wrong handle is concerned - when
//  there's time it would be good to test this and see if it's true or not.
//
//---------------------------------------------------------------------------------------------------------------
//
//  TO DO
//
//      1) If we made each state a class we wouldn't have to use module globals for variables needed in
//          both entry and event functions.
//
//---------------------------------------------------------------------------------------------------------------


//#define STATE_LOGGING		// Log all states.  Located in defines.h


#include "common.h"
#include "sm_bluetooth.h"
#include "sm_bt_at_cmnd.h"
#include "uarts.h"
#include "defines.h"
#include "error.h"
#include "events.h"
#include "state.h"
#include "sm_main.h"

#include <stdio.h>



//------------------------
//  STATE MACHINES DATA  |
//---------------------------------------------------------------------------------------------------------------
//  *NOTE*  Make sure every SM has an Unhandled Event function.
//  *NOTE*  Be sure and keep this in sync with eStateMachines in state.h
//  States are initialized in their order in this table.

const struct sStateMachines {
    const StateTbl*    pStateTbl;		// Pointer to a State Machine's State Table.
    const char*        pName;			// State Machine Name for Tracing.
    const int	       NmbStates;		// # of states in this State Machine.
          state_nmb_t (*pUnhandledFunct)(state_nmb_t, event_t, event_data_t);	// Pointer to "Unhandled Event"
										// function.
} StateMachines[] {

    {   nsBtState::States,  "SM_BLUETOOTH ",  nsBtState::StateTbl_Size,    nsBtState::Unhandled_Event   },
    { nsBtAtState::States,  "SM_BT_AT_CMND",  nsBtAtState::StateTbl_Size,  nsBtAtState::Unhandled_Event },
    {      nsMain::States,  "SM_MAIN      ",  nsMain::StateTbl_Size,       nsMain::Unhandled_Event      },
};

// *NOTE* Above must be kept in sync with eStateMachines in state.h

const int StateMachines_Size = sizeof(StateMachines) / sizeof(struct sStateMachines);

state_nmb_t PriorState[StateMachines_Size];
state_nmb_t CurrentState[StateMachines_Size];

//---------------------------------------------------------------------------------------------------------------




//-------------------------------
//  GET A STATE MACHINE'S NAME  |
//---------------------------------------------------------------------------------------------------------------

const char* Get_SM_Name(sm_t SM) {

#if defined (DEBUG)
    if(SM >= StateMachines_Size) {
	switch(SM) {
	    case NO_STATE_CHANGE:
		return("NoStateChange");
	    case UNHANDLED_EVENT:
		return("UnhandledEvent");
	    case NULL_STATE:
		return("NULL_STATE");
	    case SM_NULL:
	        return("SM_NULL");
            default:
                sprintf(sBuff, "= %d, Get_SM_Name()", SM);
                Error(INVAL_SWITCH_VALUE);
	}
	sprintf(sBuff, "*Invalid SM*=%d", SM);
	return(sBuff);
    }
#endif
    return(StateMachines[SM].pName);
}
//---------------------------------------------------------------------------------------------------------------




//------------------
//  STATE LOGGING  |
//---------------------------------------------------------------------------------------------------------------
#ifdef STATE_LOGGING

PRIVATE sSM_Log SM_Log[SM_LOG_SIZE];
PRIVATE int Log_I   = 0;	    // We don't reset these in initialization as we want to keep all of the prior
PRIVATE int Log_Cnt = 0;	    // states (* How do we know on readout where there was a reinitialization? *)

//---------------------------------------------------------------------------------------------------------------

void Log_States(sm_t SM, event_t Event, event_data_t Data, sm_t Next_State = 0) {

    SM_Log[Log_I].SM         = SM;		// If set to -1 indicates Init (rest of data not valid).
    SM_Log[Log_I].Event      = Event;
    SM_Log[Log_I].Data       = Data;
    SM_Log[Log_I].Next_State = Next_State;	// State transitioned to.
    ++Log_I %= SM_LOG_SIZE;			// Circular Buffer.
    if(Log_Cnt < SM_LOG_SIZE) Log_Cnt++;
}

#else // Not STATE_LOGGING

inline void Log_States(sm_t SM, event_t Event, event_data_t Data, sm_t Next_State = 0) {}

#endif // STATE_LOGGING
//---------------------------------------------------------------------------------------------------------------




//-------------------------
//  TRACE STATE MACHINES  |
//---------------------------------------------------------------------------------------------------------------
//  Variables made module global to pass data between Handle_Event() and TraceState().  Set "Trace_1/2/3" below
//  to the SM's you want to trace.

PRIVATE int             SM;
PRIVATE const StateTbl* pStateTbl;
PRIVATE const StateTbl* pCurrState;

enum eStep { T1, T2, T3 };

//  Set "Trace_1/2/3" below to the SM's you want to trace.

#ifdef STATE_TRACE
    PRIVATE const s8 Dont_Trace = -1;

    PRIVATE const s8 Trace_1 = SM_BLUETOOTH;
    PRIVATE const s8 Trace_2 = SM_BT_AT_CMND;
    PRIVATE const s8 Trace_3 = Dont_Trace;
#endif
//---------------------------------------------------------------------------------------------------------------

void TraceState(eStep Trace, int SM, event_t Event, event_data_t EventData, state_nmb_t State) {

#ifdef STATE_TRACE

    char         EventBuff[15];
    char         Buff[200];
    const char*  TransitionName;

    if( ! (SM==Trace_1 or SM==Trace_2 or SM==Trace_3)) return;                   // Don't trace this SM.
    switch(State) {
        case UNHANDLED_EVENT:  TransitionName = "UNHANDLED_EVENT"; break;
        case NO_STATE_CHANGE:  TransitionName = "NO_STATE_CHANGE"; break;
        case NULL_STATE:       TransitionName = "NULL_STATE";      break;
        case SM_NULL:          TransitionName = "SM_NULL";         break;
        default:               TransitionName =  pStateTbl[State].StateName;
    }
    switch(Trace) {
        case T1:

            sprintf(EventBuff, "0x%X", (int)EventData);
            sprintf(Buff, "\r\n-%s-  State: '%15s'  Event: '%s'  Data: '%s'  Returned: '%s'\r\n",
                    Get_SM_Name(SM), pCurrState->StateName, Get_Event_Name(Event), EventBuff, TransitionName);
            break;

        case T2:

            sprintf(Buff, "\r\n-%s-  Unhandled Event Function Returned:  '%s'\r\n",
                    Get_SM_Name(SM), TransitionName);
            break;

        case T3:

            sprintf(Buff, "\r\n-%s-  State: '%s'  From State:  '%s'          Returned: '%s'\r\n",
                    Get_SM_Name(SM), pStateTbl[CurrentState[SM]].StateName,
                    pStateTbl[PriorState[SM]].StateName, TransitionName);
    }
    TxDebugStr(Buff);

#else

#endif
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------
//  HANDLE EVENT, RUN STATE MACHINE  |
//---------------------------------------------------------------------------------------------------------------
//  A State can transition to itself, causing the State Entry function to run again.  The State Entry Function
//  can cause a transition change.
//---------------------------------------------------------------------------------------------------------------
void Handle_Event(sm_t StateMachine, event_t Event, event_data_t EventData) {

    SM = static_cast<int>(StateMachine);
    pStateTbl  = StateMachines[SM].pStateTbl;
    pCurrState = &pStateTbl[CurrentState[SM]];
    state_nmb_t State;


#ifdef TRACE_EVENTS  //------------------------------------------------------------------------------------------
    TxDebugStr("\n\r");                                                                                        //
    TxDebugStr(Get_Event_Name(Event));                                                                         //
#endif                                                                                                         //
                                                                                                               //
#if defined(DEBUG)  // 1                                                                                       //
                                                                                                               //
    if( (SM > LAST_SM and SM < SPEC_STATES) or SM > (End_Special_States - 1)) {                                //
                                                                                                               //
#if defined(CONSOLE_TRACE) and defined(DEBUG)                                                                  //
	printf("Handle_Event() Invalid SM=%d", SM);                                                            //
#endif                                                                                                         //
	sprintf(sBuff, "= %d, Handle_Event()", StateMachine);                                                  //
	Error(INVALID_STATE_MACHINE, sBuff);                                                                   //
    }                                                                                                          //
                                                                                                               //
    if( Event >= LAST_E_Event_ENTRY ) {                                                                        //
#if defined(CONSOLE_TRACE) and defined(DEBUG)                                                                  //
	printf("Handle_Event() Invalid Event=%d", Event);                                                      //
#endif                                                                                                         //
        sprintf(sBuff, "= %d, Handle_Event()", Event);                                                         //
	Error(INVALID_EVENT);                                                                                  //
    }                                                                                                          //
                                                                                                               //
#endif // #if defined(DEBUG) ------------------------------------------------------------------------------------


    ////////////////////////
    //---  RUN STATES  ---//
    ///////////////////////


    State = pCurrState->pEventFunct(Event, EventData); // Send Event to current state, State=returned from state.
    Log_States(SM, Event, EventData, State);	       // Log the event.

//  TRACING_1;
#ifdef STATE_TRACE
    TraceState(T1, SM, Event, EventData, State);
#endif

    if(State == UNHANDLED_EVENT) {			// If it isn't handled, run the Unhandled Event function.
	State = (*StateMachines[SM].pUnhandledFunct)(CurrentState[SM], Event, EventData);
	Log_States(SM, ~0, ~0, State);			// Log the event.

//	TRACING_2;
#ifdef STATE_TRACE
        TraceState(T2, SM, Event, EventData, State);
#endif

	if(State == UNHANDLED_EVENT) {			// The Unhandled Event function can't handle the event.

	    //-- UNHANDLED EVENT --//


#ifdef DEBUG  //-------------------------------------------------------------------------------------------------
  #ifdef CONSOLE_TRACE                                                                                         //
    #ifdef CONSOLE_TO_TTY                                                                                      //
            sprintf(DbMsgBuff, "Unhandled event in SM: '%s', State: '%s', Event: '%s', Event Data: '0x%X'\n",  //
                    StateMachines[SM].pName, pCurrState->StateName, Get_Event_Name(Event), EventData);         //
            TxDebugStr(DbMsgBuff);                                                                             //
    #endif                                                                                                     //
	    printf("Unhandled event in SM: '%s', State: '%s', Event: '%s', Event Data: '0x%X'\n",              //
	            StateMachines[SM].pName, pCurrState->StateName, Get_Event_Name(Event), EventData);         //
  #endif                                                                                                       //
#endif // DEBUG  ------------------------------------------------------------------------------------------------


	    sprintf(sBuff, "SM %s, State %s, Event %s, Data %d",
	            StateMachines[SM].pName, pCurrState->StateName, Get_Event_Name(Event), EventData);
	    Error(UNHANDLED_EVENT_ERR, sBuff);
	    return;
	}
    }

    //---  TRANSITION STATES IN THIS STATE MACHINE UNTIL "NO_STATE_CHANGE"  ---//

    do {                                                        //
	if(State == NO_STATE_CHANGE) {                          //
	    return;                                             // All Done.
	}                                                       //


#if defined(DEBUG) //--------------------------------------------------------------------------------------------
	if(State >= StateMachines[SM].NmbStates) {                                                             //
  #if defined(CONSOLE_TRACE)                                                                                   //
	    printf("Handle_Event(): Invalid state # = %d\n", State);                                           //
  #endif // CONSOLE_TRACE                                                                                      //
	    sprintf(sBuff, "= %d, SM %d", State, SM);                                                          //
	    Error(INVALID_STATE_NUMBER, sBuff);                                                                //
	}                                                                                                      //
#endif // DEBUG  ------------------------------------------------------------------------------------------------


        PriorState[SM]   = CurrentState[SM];			// Save prior state.
	CurrentState[SM] = State;                               //
	pCurrState = &pStateTbl[State];				// Point to this State's entry.
	State = pStateTbl[State].pEntryFunct(PriorState[SM]);	// Execute State Entry function.
	Log_States(SM, ~0, ~0, State);				// Log the event.

//	TRACING_3;
#ifdef STATE_TRACE
        TraceState(T3, SM, Event, EventData, State);
#endif

    } while(State != CurrentState[SM]);                         // Loop for all transitions.
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------
//  INITIALIZE STATE MACHINES  |
//---------------------------------------------------------------------------------------------------------------
//  Every State Machine is sent E_INITIALIZE.

void Init_State_Machines(void) {

#if defined(CONSOLE_TRACE) and defined(DEBUG)
    PrtStartStr("#");
    PrtDone();
#endif // CONSOLE_TRACE and DEBUG

    for(sm_t SM=0; SM<StateMachines_Size; SM++) {
	CurrentState[SM] = 0;				// Init all state machines to State 0.
	Handle_Event(SM, E_INITIALIZE, 0);
        Log_States(~0, 0, 0, 0);			// Flag Initialization in log.
    }
}
//---------------------------------------------------------------------------------------------------------------

