//----------------------------------------------------------------------------------------------------------
//  tick.cc
//
//  System Tick.  Raw counts and base timer functions are here.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//----------------------------------------------------------------------------------------------------------

#include "common.h"
#include "uarts.h"
#include "tick.h"
#include "init_cpu.h"
//#include "state.h"
#include "events.h"
#include "monitor.h"
#include "error.h"
#include "dispatch.h"
#include "defines.h"

#include "sysctl.h"     // TEST to turn on LED
#include "gpio.h"
#include "sysctl.h"
#include "systick.h"
#include "interrupt.h"
#include "timer.h"
#include "hw_memmap.h"
#include "tm4c123ae6pm.h"

#include "switch.h"   //



//------------------------------
//  SYSTICK INTERRUPT ROUTINE  |
//----------------------------------------------------------------------------------------------------------
                                                //
PRIVATE volatile u32 SysTickCnt     = 0;        // Running total number of SysTicks - for debugging.
PRIVATE volatile s32 SleepTime      = 0;        // If > 0 is the number of Ticks to "sleep".
PRIVATE          u32 LastCntVal     = 0;        // To calculate # of ticks since last dispatch.
/*PRIVATE          u32 CountingTimer  = 0;        // We calculate the counting times here to capture counts.
PRIVATE          u32 CountingReload = 0;        // Counting timer reload value.
                                                //
PRIVATE      count_t TickRunningCnt = 0;        //
PRIVATE      count_t LastTickRunningCnt;        //
PRIVATE          int TickCntIdx;                //
                                                //
PRIVATE          u64 RunningSysTicks = 0;       // K9G+
*/
//----------------------------------------------------------------------------------------------------------
                                                        //
void SysTickISR(void) {                                 //
                                                        //
//  count_t Count;                                      // The counter returns u32, count_t is s32.
//  count_t ThisTickCnt;                                //
                                                        //
    SysTickCnt++;                                       // Running total number of SysTicks - for debugging.
    if(SleepTime) {                                     //
	SleepTime--;				        // If we're sleeping, decrement the count.
    } /* else {                                            //
        ISR_Send_Message(uword Task_Idx, void* pMsg);
    } */
}                                                       //
//----------------------------------------------------------------------------------------------------------





                                        ////////////////////////////////
                                        //                            //
                                        //     BLOCKING MS DELAY      //
                                        //                            //
                                        ////////////////////////////////




//----------------------
//  BLOCKING MS DELAY  |
//----------------------------------------------------------------------------------------------------------
//  *NOTE* Only use on startup routines as it is blocking.
//----------------------------------------------------------------------------------------------------------
bool Timeout;
//----------------------------------------------------------------------------------------------------------
void Timer1Isr(void) {

    Timeout = true;
    TimerIntClear(TIMER1_BASE, TIMER_TIMA_TIMEOUT);
}
//----------------------------------------------------------------------------------------------------------
void DelayMs(int DelayMilliSecs) {

    static bool FirstCall = true;
    static u32  MsTimerVal;

    if(FirstCall) {                                     // If first call, set up the timer.
        FirstCall = false;
        TimerClockSourceSet(TIMER1_BASE, TIMER_CLOCK_SYSTEM);
        TimerConfigure(TIMER1_BASE, TIMER_CFG_ONE_SHOT | TIMER_CFG_A_ACT_NONE);
        TimerIntRegister(TIMER1_BASE, TIMER_A, Timer1Isr);
        MsTimerVal = GetClkSpeed()/1000;                // 1ms.
    }
    Timeout = false;
    TimerLoadSet(TIMER1_BASE, TIMER_A, MsTimerVal * DelayMilliSecs);
    TimerIntEnable(TIMER1_BASE, TIMER_TIMA_TIMEOUT);
    TimerEnable(TIMER1_BASE, TIMER_A);
    while( ! Timeout) {}                                // Delay until one-shot down to 0.
    TimerDisable(TIMER1_BASE, TIMER_A);
}
//----------------------------------------------------------------------------------------------------------







					    //////////////////////////////
					    //                          //
					    //     TIMER FUNCTIONS      //
					    //                          //
					    //////////////////////////////



//-------------------------------------------
//  CALL FROM DISPATCH TO DECREMENT TIMERS  |
//----------------------------------------------------------------------------------------------------------
//  Return # of Ticks since last call here.

u32 Timer_Foreground(void) {

    u32 Ticks, CurrentCntVal;

    CurrentCntVal = SysTickCnt;
    if(CurrentCntVal < LastCntVal) {
	Ticks = (~LastCntVal) + CurrentCntVal + 1;	// SysTickCnt has rolled over.
    } else {
	Ticks = CurrentCntVal - LastCntVal;
    }
    LastCntVal = CurrentCntVal;
    return(Ticks);
}
//----------------------------------------------------------------------------------------------------------




//----------------------------
//  SET TIME TO WAKE CPU UP  |
//----------------------------------------------------------------------------------------------------------
//  Called from Dispatch() to set the number of ticks for the CPU to sleep before the next occurring timeout.

void Set_Wakeup_Time(s32 Ticks) {

    SleepTime = Ticks;
}
//----------------------------------------------------------------------------------------------------------




//------------------------------
//  GET WAKEUP TIME REMAINING  |
//----------------------------------------------------------------------------------------------------------
s32 Get_Wakeup_Time() {

    return(SleepTime);
}
//----------------------------------------------------------------------------------------------------------




//----------------------------
//  GET TICKS SINCE RESTART  |
//---------------------------------------------------------------------------------------------------------------
PRIVATE u32 Run_Time_Start;
//---------------------------------------------------------------------------------------------------------------
void Init_Run_Time(void) {

    Run_Time_Start = SysTickCnt;
}
//---------------------------------------------------------------------------------------------------------------
u32 Get_Run_Time(void) {

    return(SysTickCnt - Run_Time_Start);
}
//---------------------------------------------------------------------------------------------------------------




//---------------------------
//  INITIALIZE SYSTEM TICK  |
//---------------------------------------------------------------------------------------------------------------
void Init_SysTick(void) {

    SysTickIntRegister(SysTickISR);
    SysTickPeriodSet(GetClkSpeed()/100);	// Interrupt every 10ms. (convert nS to mS), keep
    SysTickIntEnable();				// in sync with TICK_TIME_MS in tick.h
    SysTickEnable();

    SysTickCnt = 0;
    SleepTime  = 0;				// If > 0 is the number of Ticks to "sleep".
    LastCntVal = 0;				// To calculate # of Ticks since last dispatch.

    Init_Run_Time();
}
//---------------------------------------------------------------------------------------------------------------

