//----------------------------------------------------------------------------------------------------------
//  timers.cc
//
//  Software Timers.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//----------------------------------------------------------------------------------------------------------
//
//  To Do:
//	1)  Track the number of timeouts with invalid handles to see if we need to check for them.
//	2)  Keep track of the highest Timer[] index used, the scans through it only need to go that far.
//	3)  Might be interesting to see the cost in time and code size to keep the timer queue sorted on a
//	    linked list.
//	4)  Is the TimerQueue[] sized right?
//
//----------------------------------------------------------------------------------------------------------


#include "error.h"
#include "common.h"
#include "dispatch.h"
#include "tick.h"
#include "init.h"
#include "events.h"
#include "timers.h"
#include "state.h"

//#include "cs3.h"

#ifdef DEBUG

#include <stdio.h>

  //#include "print.h"
  //#define TIME_TRACE
  //#define BUFF_TIME_TRACE

  #ifdef BUFF_TIME_TRACE
    char sBuff[LineBuffSize];
  #endif

#endif // DEBUG



//------------------
//  TIMER GLOBALS  |
//------------------------------------------------------------------------------------------------------

void Process_Timers(void);

enum eProc_Timers { TIMERS_DISPATCH, TIMERS_CANCEL, TIMERS_RESET };

#define  NMB_TIMERS  10

PRIVATE u16      Active_Timers;
PRIVATE bool     InTimerDispatch;
PRIVATE handle_t Next_Handle = 0;

struct TimerTable {
    bool	Active;				    // true = Timer in use.
    long        Ticks;				    // Timeout value.
    long	Reload;				    // For CONTINUOUS Timers
    handle_t    Handle;				    // The Handle for this timeout.
    bool	Type;				    // CONTINUOUS or ONE_SHOT.
    sm_t	SM;				    // State Machine #.
    void       (*pCallback_Funct)(handle_t, int);   // Callback function if not a SM.
};

PRIVATE TimerTable Timer[NMB_TIMERS];

struct TimerQ {
    long        Ticks;				    // Timeout value.
    handle_t    Handle;				    // The Handle for this timeout.
    bool	Type;				    // CONTINUOUS or ONE_SHOT.
    sm_t	SM;				    // State Machine #.
    void       (*pCallback_Funct)(handle_t, int);   // Callback function if not a SM.
    bool	Cancel;				    // Used to cancel a queued timer.
};

/*PRIVATE*/ int TimerQueueIdx;			    // Points to next available entry in queue.

PRIVATE TimerQ TimerQueue[NMB_TIMERS];

//------------------------------------------------------------------------------------------------------



/*
void Display_Timers(void) {

    printf("Active Timers = %d\n", Active_Timers);
    int i;
    for( i = 0;  i < NMB_TIMERS;  i++ ) {
        if(Timer[i].Active) {
            printf("  T %d, H %d, SM %d, Funct %X\n", Timer[i].Ticks, Timer[i].Handle, Timer[i].SM, Timer[i].pCallback_Funct);
        }
    }
}
*/




//----------------
//  SET A TIMER  |
//------------------------------------------------------------------------------------------------------
//  The calling function will either supply a State Machine (SM) to post an E_TIMEOUT event to on
//  expiration or a callback function to be called; if the callback function is not NULL it is called,
//  if it is NULL then the SM has E_TIMEOUT posted, so BE SURE and get the parameters right.  State
//  Machines & callback functions should keep the returned handle to make sure it's their timer since
//  there is a window where a routine tries to cancel its timer but the timer has already fired,
//  or if a given SM or callback function has multiple timers running.
//
//  The boolean parameter Type can be either CONTINUOUS or ONE_SHOT.
//
//  12/8/11 Added an int parameter to the callback function, which is the excess number of ticks that
//  expired before the function was called, e.g., if the timer was set for 100 ms but it didn't get called
//  until 110ms had transpired then the excess ticks would = 1 (Ticks are 10ms).
//
//  If a Timer Callback Function calls Set_Timer(), that call is queued until the Callback Function
//  returns.
//
//  If Set_Timer() called with Ticks <= 0 it will be dispatched immediately.

void  Set_Timer_Sub( long      Ticks,
		     handle_t  Handle,
		     bool      Type,
		     sm_t      SM,
		     void     (*pCB_Funct)(handle_t, int)) {

    uword  i;

    for( i = 0;  i < NMB_TIMERS;  i++ ) {		// Search for an unused entry.
	if(!Timer[i].Active) {
	    Timer[i].Active = true;
	    Timer[i].Ticks  = Ticks;
	    Timer[i].Reload = Ticks;
	    Timer[i].Handle = Handle;
	    Timer[i].Type   = Type;
	    Timer[i].SM     = SM;
	    Timer[i].pCallback_Funct = pCB_Funct;


#ifdef TIME_TRACE //-------------------------------------------------------------
if(pCB_Funct) {
    printf("*Set T=%ld H=%d SysTicks=%d Type=%s F=0x%X\n",
	    Ticks, Handle, Get_Run_Time(), Type ? "CONT" : "ONCE", pCB_Funct);
} else {
    printf("*Set T=%ld H=%d SysTicks=%d Type=%s SM=%s\n",
	    Ticks, Handle, Get_Run_Time(), Type ? "CONT" : "ONCE", Get_SM_Name(SM));
}
#endif

#ifdef BUFF_TIME_TRACE
if(pCB_Funct) {
    sprintf(sBuff, "*Set T=%ld H=%d SysTicks=%d Type=%s F=0x%X",
	    Ticks, Handle, Get_Run_Time(), Type ? "CONT" : "ONCE", pCB_Funct);
} else {
    sprintf(sBuff, "*Set T=%ld H=%d SysTicks=%d Type=%s SM=%s",
	    Ticks, Handle, Get_Run_Time(), Type ? "CONT" : "ONCE", Get_SM_Name(SM));
}
PrtStartStr(sBuff); PrtDone();
#endif //-------------------------------------------------------------------------


	    if(Ticks <= 0) Process_Timers();	// *FIXME* see if this ever happens.
	    break;
	}
    }
    if( i >= NMB_TIMERS ) Error(NO_MORE_TIMERS, "Set_Timer_Sub()");


#ifdef TIME_TRACE //--------------------------------------------------------------
if(Active_Timers) {
    printf(" Ticks Remaining SetTSub()=%ld\n", Get_Ticks_Remaining(TIMER_TASK));
}
#endif

#ifdef BUFF_TIME_TRACE
if(Active_Timers) {
    sprintf(sBuff, " Ticks Remaining SetTSub()=%ld", Get_Ticks_Remaining(TIMER_TASK));
}
PrtStartStr(sBuff); PrtDone();
#endif //--------------------------------------------------------------------------


    // If just starting or the new timeout less than current timeout ...

    if(( ++Active_Timers == 1 ) || ( Get_Ticks_Remaining(TIMER_TASK) > Ticks )) {
	Set_Dispatch_Time( TIMER_TASK, Ticks, TIME_ONCE );


#ifdef TIME_TRACE //---------------------------------------------------------------
printf(" Set Dispatch SetTSub()=%ld SysTicks=%d\n", Ticks, Get_Run_Time());
#endif

#ifdef BUFF_TIME_TRACE
sprintf(sBuff, " Set Dispatch SetTSub()=%ld SysTicks=%d", Ticks, Get_Run_Time());
PrtStartStr(sBuff); PrtDone();
#endif //--------------------------------------------------------------------------


    }
}
//------------------------------------------------------------------------------------------------------

handle_t Set_Timer(long Ticks,  bool Type,   sm_t  SM,  void (*pCB_Funct)(handle_t, int)) {

    if(InTimerDispatch) {					  // If this is a call from a Timer
	TimerQueue[TimerQueueIdx].Ticks  = Ticks;		  // dispatched function, queue it until
	TimerQueue[TimerQueueIdx].Handle = Next_Handle;		  // all timer callbacks have been
	TimerQueue[TimerQueueIdx].Type   = Type;		  // processed.
	TimerQueue[TimerQueueIdx].SM     = SM;
	TimerQueue[TimerQueueIdx].pCallback_Funct = pCB_Funct;
	TimerQueue[TimerQueueIdx].Cancel = false;
	if(++TimerQueueIdx >= NMB_TIMERS) Error(TIMER_QUEUE_OVERFLOW, "Set_Timer()");
    } else {
	Set_Timer_Sub(Ticks, Next_Handle, Type, SM, pCB_Funct);	  // Else set it now.
    }
    handle_t Handle = Next_Handle;
    if( ++Next_Handle == INVALID_HANDLE ) Next_Handle++;	  // Skip over the INVALID_HANDLE value.
    return(Handle);
}
//------------------------------------------------------------------------------------------------------




//-------------------
//  CANCEL A TIMER  |
//------------------------------------------------------------------------------------------------------
//  If timer is already expired it doesn't matter.  The timer handle of the caller gets set to
//  INVALID_HANDLE so that the code for an E_TIMEOUT for that timer that is already queued won't run.
//------------------------------------------------------------------------------------------------------
void Cancel_Timer( handle_t&  Handle ) {			//
								//
    int i;							//
								//
    if( Handle == INVALID_HANDLE ) return;			// Ignore the INVALID_HANDLE flag.

    for( i = 0;  i < NMB_TIMERS;  i++ ) {			// Search the handle.
	if( Timer[i].Active and 				//
	    Timer[i].Handle == Handle) {			//
								//
								//
#ifdef TIME_TRACE //--------------------------------------------
printf("*Cancel H=%d SysTicks=%d\n", Handle, Get_Run_Time());
#endif

#ifdef BUFF_TIME_TRACE
sprintf(sBuff, "*Cancel H=%d SysTicks=%d", Handle, Get_Run_Time());
PrtStartStr(sBuff); PrtDone();
#endif //-------------------------------------------------------
								//
								//
	    Timer[i].Active = false;				//
	    if(--Active_Timers == 0) {				//
		Set_Dispatch_Control(TIMER_TASK, DISP_OFF);	// No more active timers.
								//
								//
#ifdef TIME_TRACE //--------------------------------------------
printf(" Dispatch Off\n");
#endif

#ifdef BUFF_TIME_TRACE
sprintf(sBuff, "  Dispatch Off");
PrtStartStr(sBuff); PrtDone();
#endif //-------------------------------------------------------
								//
								//
	    }							//
	}							//
    }								//
    Handle = INVALID_HANDLE;					// Set caller's Handle to Invalid.
}								//
//------------------------------------------------------------------------------------------------------
                                                                //
void  Reset_Timer(long Ticks,  handle_t Handle) {               //
                                                                //
    if( Handle == INVALID_HANDLE ) return;                      // Ignore the INVALID_HANDLE flag.
                                                                //
    for( int i = 0;  i < NMB_TIMERS;  i++ ) {                   // Search the handle.
        if( Timer[i].Active and                                 //
            Timer[i].Handle == Handle) {                        //
                                                                //
            Timer[i].Ticks = Ticks;                             //
            Process_Timers();                                   //
            break;                                              //
        }                                                       //
    }                                                           //
}                                                               //
//------------------------------------------------------------------------------------------------------




//------------------------
//  TIMER_TASK DISPATCH  |
//------------------------------------------------------------------------------------------------------
void  Timer_Dispatch( u16 Cnt, void* pMsg, ulong Ticks ) {

    Process_Timers();
}
//------------------------------------------------------------------------------------------------------




//--------------------------
//  HANDLE EXPIRED TIMERS  |
//------------------------------------------------------------------------------------------------------
void  Process_Timers(void) {						//
									//
    const long MAGIC_NUMBER = 2147483600;				// Close to largest signed 32bit
    uword    i;								//
    ulong    Ticks_Expired;						//
    long     Lowest = MAGIC_NUMBER;					//
									//
    InTimerDispatch = true;						// Flag Set_Timer() calls from
									// timer callback functions.
    Ticks_Expired = Get_Ticks_Expired( TIMER_TASK );			// Will dec all active timers.
									//
    // Scan through Timers array //					//
									//
    for(i = 0;  i < NMB_TIMERS and Active_Timers;  i++) {		//
									//
	if(Timer[i].Active) {						// If active,
									//
	    // Deal with Expired Timers //				//
									//
    	    Timer[i].Ticks -= Ticks_Expired;				// Decrement this timer.
	    if(Timer[i].Ticks <= 0) {					//
									//
									//
#ifdef TIME_TRACE //----------------------------------------------------
if(Timer[i].pCallback_Funct) {
    printf("*Expire Dispatch T=%ld TX=%d H=%d SysTicks=%d F=0x%X\n",
	    Timer[i].Ticks, Ticks_Expired, Timer[i].Handle, Get_Run_Time(), Timer[i].pCallback_Funct);
} else {
    printf("*Expire Dispatch T=%ld TX=%d H=%d SysTicks=%d SM=%s\n",
	    Timer[i].Ticks, Ticks_Expired, Timer[i].Handle, Get_Run_Time(), Get_SM_Name(Timer[i].SM));
}
#endif

#ifdef BUFF_TIME_TRACE
if(Timer[i].pCallback_Funct) {
    sprintf(sBuff, "*Expire Dispatch T=%lu TX=%lu H=%d SysTicks=%u F=0x%X",
	    Timer[i].Ticks, Ticks_Expired, Timer[i].Handle, Get_Run_Time(), Timer[i].pCallback_Funct);
} else {
    sprintf(sBuff, "*Expire Dispatch T=%lu TX=%lu H=%d SysTicks=%u SM=%s",
	    Timer[i].Ticks, Ticks_Expired, Timer[i].Handle, Get_Run_Time(), Get_SM_Name(Timer[i].SM));
}
PrtStartStr(sBuff); PrtDone();
#endif //---------------------------------------------------------------
									//
									//
		if(Timer[i].pCallback_Funct) {				// If there is a callback funct,
		    (*Timer[i].pCallback_Funct)( Timer[i].Handle, 	// call it.
					        -Timer[i].Ticks);	//
		} else {						// Else send event to SM.
		    Send_Event(Timer[i].SM, E_TIMEOUT, Timer[i].Handle);//
		}							//
		if(Timer[i].Type == ONE_SHOT) {				//
		    Timer[i].Active = false;				// It has expired.
		    if( --Active_Timers == 0 ) {			//
		        break;						// We be done.
		    }							//
		} else {						//
		    Timer[i].Ticks = Timer[i].Reload;			// Continuous timing, reload.
		}							//
	    }								//
	    if(Timer[i].Ticks > 0 and Timer[i].Active) {		// Check Active as timer could
		Lowest = MIN( Lowest, Timer[i].Ticks );			// have canceled itself.
	    }								//
	}								//
    }									//
									//
    InTimerDispatch = false;						//
									//
    // Deal with Set_Timer() calls in timer callback function(s) //	//
									//
    while(TimerQueueIdx) {						//
	TimerQueueIdx--; 						//
	Set_Timer_Sub( TimerQueue[TimerQueueIdx].Ticks,			//
		       TimerQueue[TimerQueueIdx].Handle,		//
		       TimerQueue[TimerQueueIdx].Type, 			//
		       TimerQueue[TimerQueueIdx].SM, 			//
		       TimerQueue[TimerQueueIdx].pCallback_Funct );	//
	Lowest = MIN( Lowest, TimerQueue[TimerQueueIdx].Ticks );	//
    }									//
									//
    if(Active_Timers and Lowest != MAGIC_NUMBER) {			// If any more tasks remaining,
									//
	Set_Dispatch_Time( TIMER_TASK, Lowest, TIME_ONCE );		// set next expiring timeout.
									//
									//
#ifdef TIME_TRACE //----------------------------------------------------
printf(" Set Dispatch=%ld SysTicks=%d\n", Lowest, Get_Run_Time());
//printf(" Set Dispatch=%ld H= %d SysTicks=%d\n", Lowest, LowestHandle, Get_Run_Time());
#endif

#ifdef BUFF_TIME_TRACE
sprintf(sBuff, " Set Dispatch=%ld H=%d SysTicks=%u", Lowest, LowestHandle, Get_Run_Time());
PrtStartStr(sBuff); PrtDone();
#endif //---------------------------------------------------------------
									//
									//
    } else {								//
									//
	Set_Dispatch_Control(TIMER_TASK, DISP_OFF);			// No more active timers.
									//
									//
#ifdef TIME_TRACE //----------------------------------------------------
printf(" Dispatch Off\n");
#endif

#ifdef BUFF_TIME_TRACE
sprintf(sBuff, "  Dispatch Off");
PrtStartStr(sBuff); PrtDone();
#endif //---------------------------------------------------------------
									//
									//
    }									//
}									//
//------------------------------------------------------------------------------------------------------





