//---------------------------------------------------------------------------------------------------------------
//  uarts.cc
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//  When the Bluetooth slave interface is connected to a master, the ISR receives characters in a circular buffer
//  until a complete message is received at which time it sends a message to BT_COMM_TASK in bt_comm.cc
//  This then calls a function to process this message with the message number, the length, and a pointer to
//  the payload of the message if any.  Since a circular buffer is used if a message wraps that message will
//  be copied to a linear buffer so the accessing function doesn't have to deal with the roll-over.
//
//  The program must look for the 'O' in place of a message number to detect when a disconnect happens, this is
//  obviously a problem should the disconnect occur in the middle of a message.  An alternate would be to disable
//  AT response messages at the start of a connection and monitoring the P1 line for the disconnect.
//---------------------------------------------------------------------------------------------------------------


#include "common.h"
#include "uarts.h"
#include "tick.h"
#include "error.h"
#include "timers.h"
#include "init_cpu.h"
#include "dispatch.h"
#include "defines.h"
#include "events.h"
#include "sm_bluetooth.h"

#include "hw_uart.h"
#include "hw_memmap.h"
#include "uart.h"
#include "interrupt.h"

#include <cstdlib>
#include <string.h>
#include <stdio.h>


char sBuff[sBuffSize];                  // General use serial buffer.


void TxDebugChar(char);
void DebugUartTxChrs(void);


/* PRIVATE */ eInputMode InputMode;


//const u32 UART_INT_SOURCES = UART_INT_RX | UART_INT_TX | UART_INT_OE | UART_INT_FE;




                                           //////////////////////
                                           //                  //
                                           //  BLUETOOTH UART  //
                                           //                  //
                                           //////////////////////


//                                  *** FIXME ***  Text below is wrong. ***
//--------------------
//  MESSAGE BUFFERS  |
//---------------------------------------------------------------------------------------------------------------
//  Message buffering operates in 2 modes: the HM-11 module uses AT commands to set up a Bluetooth connection,
//  in this mode, 'BtAtMode', a linear input buffer is used to receive AT command responses, once the Bluetooth
//  connection is established, the input is switched to a circular buffer.  Like the Debug circular buffer, each
//  message is put into a linear buffer to read out, and the BT software calls **BtMsgReceived()**, at which time
//  if another message is in the circular buffer it is then read into the linear buffer and the BT software
//  is informed another message is ready.
//
//  In both modes a circular output buffer is used.
//---------------------------------------------------------------------------------------------------------------


//PRIVATE bool BtAtRxReady;               // To check for AT response message overflowing buffer.

    //-- Circular Buffers --//

BtInCircBuff_t  BtInBuff;
BtOutCircBuff_t BtOutBuff;

#ifdef DEBUG_BT_TX
    CpuOutTraceBuff_t BtOutTraceBuff;  // Output Tracing Buffer.
#endif

#ifdef DEBUG_BT_RX
    CpuInTraceBuff_t BtInTraceBuff;    // Input Tracing Buffer.
#endif

    //-- Linear Buffer --//

uint BtInLinCnt;
char BtInLinBuff[BtInLinBuffSize];

//---------------------------------------------------------------------------------------------------------------




//---------------------------------------
//  INITIALIZE SERIAL BUFFERS AND MODE  |
//---------------------------------------------------------------------------------------------------------------
//  We initialize here as opposed to relying on the automatic zeroing of variables at startup in case we need to
//  implement a warm reset.

void Init_Bt_Buffs(void) {

//  BtCommMode      = BtAtMode;
//  BtAtRxReady = false;
    BtInLinCnt  = 0;

    BtInBuff.HeadIdx    = 0;
    BtInBuff.TailIdx    = 0;
    BtInBuff.ChrCnt     = 0;
    BtInBuff.MaxChrCnt  = 0;

    BtOutBuff.HeadIdx   = 0;
    BtOutBuff.TailIdx   = 0;
    BtOutBuff.ChrCnt    = 0;
    BtOutBuff.MaxChrCnt = 0;


#ifdef DEBUG_BT_TX                     // Output Tracing Buffer.

    BtOutTraceBuff.HeadIdx   = 0;
    BtOutTraceBuff.TailIdx   = 0;
    BtOutTraceBuff.ChrCnt    = 0;

#endif

#ifdef DEBUG_BT_RX                     // Input Tracing Buffer.

    BtInTraceBuff.HeadIdx    = 0;
    BtInTraceBuff.TailIdx    = 0;
    BtInTraceBuff.ChrCnt     = 0;

#endif
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------
//  TRANSMIT CHARACTERS FROM TX CIRCULAR BUFFER  |
//---------------------------------------------------------------------------------------------------------------
//  Called both from the UART ISR, TxBtChar() and TxBtStr().

void BtUartTxChrs(void) {

    while(( ! (UART1_FR_R & UART_FR_TXFF)) && BtOutBuff.ChrCnt) {       // If space in the Xmit FIFO, stuff
	UART1_DR_R = BtOutBuff.Buff[BtOutBuff.TailIdx++];	        // what characters we have and can fit.
	if(BtOutBuff.TailIdx==BtOutCircBuffSize) BtOutBuff.TailIdx=0;   // Wrap buffer.
	BtOutBuff.ChrCnt--;
    }
}
//---------------------------------------------------------------------------------------------------------------




//-------------------------------------------
//  BUFFER TX DATA TO THE BLUETOOTH MODULE  |
//---------------------------------------------------------------------------------------------------------------
void BuffBtChar(uchar Char) {                                               //
                                                                            //
    BtOutBuff.Buff[BtOutBuff.HeadIdx++] = Char;                             // Put characters into the buffer.
    if(BtOutBuff.HeadIdx >= BtOutCircBuffSize) BtOutBuff.HeadIdx = 0;       // Wrap Buffer.
    if(++BtOutBuff.ChrCnt >= BtOutCircBuffSize) {                           //
        Error(BT_OUT_CIRC_BUFF_OVF, "BuffBtChar()");                        //
    }                                                                       //
                                                                            //
#ifdef DEBUG_BT_TX                                                          // Output Tracing Buffer
                                                                            //
    BtOutTraceBuff.Buff[BtOutTraceBuff.HeadIdx++] = Char;                   // Put characters into trace buffer.
    if(BtOutTraceBuff.HeadIdx>=CpuOutTraceBuffSize) BtOutTraceBuff.HeadIdx=0;  // Wrap Buffer.
    if(++BtOutTraceBuff.ChrCnt >= CpuOutTraceBuffSize) {                    //
        BtOutTraceBuff.ChrCnt = CpuOutTraceBuffSize;                        // Can't be more characters.
    }                                                                       //                                                                            //
#endif                                                                      //
}                                                                           //
//---------------------------------------------------------------------------------------------------------------
void TxBtChar(uchar Char) {						    //
									    //
    UART1_IM_R &= ~(UART_IM_TXIM);					    // Disable Tx Interrupt.
    BuffBtChar(Char);                                                       //
    BtUartTxChrs();							    // If space in FIFO xmit characters.
    UART1_IM_R |= UART_IM_TXIM;					    	    // Re-enable Tx Interrupt.
}									    //
//---------------------------------------------------------------------------------------------------------------
void TxBtStr(uchar* pChar) {						    //
									    //
    UART1_IM_R &= ~(UART_IM_TXIM);					    // Disable Tx Interrupt.
    while(*pChar) {							    // Put characters into the buffer.
        BuffBtChar(*pChar++);                                               //
    }									    //
    BtUartTxChrs();							    // If space in FIFO xmit characters.
    UART1_IM_R |= UART_IM_TXIM;						    // Re-enable Tx Interrupt.
}									    //
//---------------------------------------------------------------------------------------------------------------

void TxBtStr(const char*  pChar) {
    TxBtStr((uchar*)pChar);
}


#ifdef DEBUG_BT_TX

//--------------------------------------------
//  DISPLAY CPU TX BUFFER TO THE DEBUG UART  | For Debugging.
//---------------------------------------------------------------------------------------------------------------
void DispCpuOutTraceBuff(void) {

    int Idx = BtOutTraceBuff.HeadIdx;

    TxDebugStr("\r\n\nDisp-Bd Inter-CPU Output Buff\r\n\n");
    if(BtOutTraceBuff.ChrCnt == BtOutBuffSixe) {
        for(int i = 0; i < BtOutTraceBuff.ChrCnt; i++) {
            sprintf(DbMsgBuff, " %d,", BtOutTraceBuff.Buff[Idx]);
            TxDebugStr(DbMsgBuff);
            if(++Idx == BtOutBuffSixe) Idx = 0;
        }
    } else if(BtOutTraceBuff.ChrCnt > 0) {
        for(int i = 0; i < BtOutTraceBuff.ChrCnt; i++) {
            sprintf(DbMsgBuff, " %d,", BtOutTraceBuff.Buff[i]);
            TxDebugStr(DbMsgBuff);
        }
    }
    TxDebugStr("\r\n");
}
//---------------------------------------------------------------------------------------------------------------
#endif


#ifdef DEBUG_BT_RX

//--------------------------------------------
//  DISPLAY CPU RX BUFFER TO THE DEBUG UART  | For Debugging.
//---------------------------------------------------------------------------------------------------------------
void DispCpuInTraceBuff(void) {

    int Idx = BtInTraceBuff.HeadIdx;

    TxDebugStr("\r\n\nDisp-Bd Inter-CPU Input Buff\r\n\n");
    if(BtInTraceBuff.ChrCnt == BtInBuffSixe) {
        for(int i = 0; i < BtInTraceBuff.ChrCnt; i++) {
            sprintf(DbMsgBuff, " %d,", BtInTraceBuff.Buff[Idx]);
            TxDebugStr(DbMsgBuff);
            if(++Idx == BtInBuffSixe) Idx = 0;
        }
    } else if(BtInTraceBuff.ChrCnt > 0) {
        for(int i = 0; i < BtInTraceBuff.ChrCnt; i++) {
            sprintf(DbMsgBuff, " %d,", BtInTraceBuff.Buff[i]);
            TxDebugStr(DbMsgBuff);
        }
    }
    TxDebugStr("\r\n");
}
//---------------------------------------------------------------------------------------------------------------
#endif




//--------------------------------------------------
//  PUT RECEIVED CHARACTER INTO APPLICABLE BUFFER  |
//---------------------------------------------------------------------------------------------------------------
//  Called from ISR so running in interrupt state.  AT command mode characters go into a linear buffer,
//  in Data Mode the characters go into a circular buffer.  Data Mode
//
//  struct sBtComm_Msg {
//      u8     MsgNmbr;     // Message number.
//      u8     MsgLength;   // Payload size.
//      uchar* pPayload;    // Pointer to message payload.
//  };
//---------------------------------------------------------------------------------------------------------------
                                                                //
void SaveBtChr(u8 Char) {                                       //
                                                                //
    struct sBtComm_Msg Msg;                                     // Data Mode variables.
    static uint MsgCnt = 0, MsgNmbr, PayloadSize;               // MsgCnt is count of entire message including
    static uint StartIdx, EndIdx, SaveIdx;                      //   Message Number and Payload Size bytes.
                                                                //
    if(BtCommMode == BtDataMode) {                              // -- DATA MODE --    BtCurrentState=BtConnected.
                                                                //
        ///////////////////                                     //
        //-- DATA MODE --//                                     //
        ///////////////////                                     //
                                                                //
        SaveIdx = BtInBuff.HeadIdx;                             //
        BtInBuff.Buff[BtInBuff.HeadIdx++] = Char;               // Buffer character.
        if(BtInBuff.HeadIdx >= BtInCircBuffSize) {              //
            BtInBuff.HeadIdx = 0;                               // Wrap buffer.
        }                                                       //
        if(++BtInBuff.ChrCnt >= BtInCircBuffSize) {             //
            Error(BT_IN_CIRC_BUFF_OVF, "SaveBtChr()");          //
        }                                                       //
                                                                //
        //-- Deal with Message Header --//                      //
                                                                //
        if(MsgCnt == 0) {                                       //
            MsgNmbr = Char;                                     // Save the message number.
            Msg.MsgNmbr = MsgNmbr;                              //
            MsgCnt++;                                           //
        } else if(MsgCnt == 1) {                                //
            PayloadSize = Char;                                 // Save the payload size.
            Msg.MsgLength = PayloadSize;                        //
            if(PayloadSize >= BtInLinBuffSize -                 //
                              BtInExtraChrs) {                  //
                Error(BT_INVALID_MSG_SIZE);                     //
            }                                                   //
            if(PayloadSize == 0) {                              //
                Msg.pPayload = NULL;                            // NULL pointer, no payload.
                ISR_Send_Message(BT_COMM_TASK, &Msg);           //
                MsgCnt = 0;                                     //
                return;                                         // ** Early Return() **
            } else {                                            //
                MsgCnt++;                                       //
            }                                                   //
                                                                //
        } else {                                                // There is a payload.
                                                                //
            //-- Deal with Payload --//                         //
                                                                //
            if(MsgCnt == 2) {                                   //
                StartIdx = SaveIdx;                             // Index of start of payload.
            }                                                   //
            if(PayloadSize + 2 - MsgCnt == 0) {                 // If at the end of message,
                EndIdx = SaveIdx;                               //   save end of payload.
                if(StartIdx > EndIdx) {                         // If buffer wrap copy message to linear buffer.
                    uint BtLinIdx = 0;                          // Start of Linear Buffer.
                    for(int i=0; i<PayloadSize; i++) {          //
                        BtInLinBuff[BtLinIdx++] =               //
                            BtInBuff.Buff[StartIdx++];          //
                        if(BtLinIdx >= BtInLinBuffSize) {       //
                            Error(BT_IN_LIN_BUFF_OVF,           //
                                  "Data Comm Mode");            //
                        }                                       //
                        if(StartIdx >= BtInCircBuffSize) {      //
                            StartIdx = 0;                       // Wrap buffer.
                        }                                       //
                    }                                           //
                    Msg.pPayload = &BtInLinBuff[0];             // Payload is in the Linear Buffer.
                } else {                                        //
                    Msg.pPayload = &BtInBuff.Buff[StartIdx];    // Payload does not wrap in circular buffer.
                }                                               //
                MsgCnt = 0;                                     // Ready for next message.
                ISR_Send_Message(BT_COMM_TASK, &Msg);           //
            } else {                                            //
                MsgCnt++;                                       //
            }                                                   //
        }                                                       //
                                                                //
    } else {                                                    // -- AT COMMAND AND ECHO MODE --
                                                                // BtCommMode == BtAtMode
        /////////////////////////                               //
        //-- AT COMMAND MODE --//                               //
        /////////////////////////                               //
                                                                // Since we wait for response no check for OVF.
        if(Char == '\n') return;                                // Ignore Newline characters.
        BtInLinBuff[BtInLinCnt++] = Char;                       //
        if(BtInLinCnt >= BtInLinBuffSize ) {                    // Must have space for terminating 0.
            Error(BT_IN_LIN_BUFF_OVF,"AT Comm Mode");           //
        }                                                       //
        if(Char == '\r') {                                      // End of AT command response.
            BtInLinBuff[BtInLinCnt - 1] = 0;                    // Terminate the string and erase the '\r'.
            ISR_Send_Event(SM_BT_AT_CMND,                       //
                           E_BT_CMND_RESPONSE, 0);              // A message is ready.
            BtInLinCnt = 0;                                     //
        }                                                       //
    }                                                           //
}                                                               //
//---------------------------------------------------------------------------------------------------------------




//-----------------------------
//  BLUETOOTH UART INTERRUPT  |
//---------------------------------------------------------------------------------------------------------------
void Bt_Uart_Interrupt(void) {                              //
                                                            //
    const int Error_Mask = 0x00000F00;                      //
    const int Data_Mask  = 0x000000FF;                      //
                                                            //
    //---------------------------------------------------// //
    //-- HANDLE INCOMING CHARACTERS FROM BLUETOOTH MOD --// //
    //---------------------------------------------------// //
                                                            //
    while( ! (UART1_FR_R & UART_FR_RXFE)) {		    // Until the FIFO is empty, receive characters.
                                                            //
	int Data_And_Errors = UART1_DR_R;                   //
	int Errors = Data_And_Errors & Error_Mask;          //
	u8  Char   = Data_And_Errors & Data_Mask;           //
	                                                    //
	if(Errors) {                                        //
	    UART1_ECR_R = 0;				    // Reset errors, ignore character.	*FIXME*
	} else {                                            //
            SaveBtChr(Char);                                // Store Character.
	}                                                   //
    }                                                       //
                                                            //
    //-------------------------------------------------//   //
    //-- HANDLE OUTGOING CHARACTERS TO BLUETOOTH MOD --//   //
    //-------------------------------------------------//   //
                                                            //
    BtUartTxChrs();                                         //
                                                            //
    UART1_ICR_R |= UART1_MIS_R;				    // Reset the interrupting source(s).
}                                                           //
//---------------------------------------------------------------------------------------------------------------



/*
//----------------------------------------------
//  DISABLE/RE-ENABLE BLUETOOTH COMMUNICATION  |        *FIXME*  Used ???
//---------------------------------------------------------------------------------------------------------------
void DisableBtComm(void) {

    UART1_IM_R = 0;						// Disable UART Interrupts.
}
//---------------------------------------------------------------------------------------------------------------
void ReEnableBtComm(void) {

    volatile int Data;                                          // volatile to eliminate "Data not used"warning.

    Init_Bt_Buffs();						// Be sure buffers are clear.
    while( ! (UART1_FR_R & UART_FR_RXFE)) {Data = UART1_DR_R;}	// Until the FIFO is empty bit-bucket characters.
    UART1_ICR_R |= UART_INT_RX | UART_INT_TX | UART_INT_RT;	// Reset any interrupts.
    UART1_IM_R   = UART_INT_RX | UART_INT_TX | UART_INT_RT;	// Enable UART Interrupts.
}
//---------------------------------------------------------------------------------------------------------------
*/


const uint MaxDataMsgs = 10;            // We'll buffer 10 messages.

struct sDataMsg {
    u8      Idx;
    uint    StartIdx;
    uint    EndIdx;
}  DataMsg[MaxDataMsgs];

u8 DataMsgNextIdx = 0;                  // Index into dataMsg[];


/*
//---------------------------------------------
//  GET THE MESSAGE POINTED TO BY THE HANDLE  |
//---------------------------------------------------------------------------------------------------------------
uchar* BtGetMsg(u8 Handle) {


}
//---------------------------------------------------------------------------------------------------------------
*/



                                             //////////////////
                                             //              //
                                             //  DEBUG UART  //
                                             //              //
                                             //////////////////




//--------------------------
//  DEBUG MESSAGE BUFFERS  |
//---------------------------------------------------------------------------------------------------------------
						//
    DbOutCircBuff_t  DbOutBuff;	    		// Output UART circular buffer.
						//
    char  DbInBuff[DbInBuffSize];		// Linear UART input line buffer.
    int   DbInBuffIdx;			    	// Input index into DbInBuff[].
    char  CopyBuff[DbInBuffSize];		// Copy of DbInBuff for parameter parsing & conversion.
    char  LastBuff[MaxHistory][DbInBuffSize];   // Copy of last command string.
    int   HistIdx;				// Index into command history.
    char  DbMsgBuff[DbMsgBuffSize];		// To formulate print messages.
						//
//---------------------------------------------------------------------------------------------------------------




//------------------------------------
//  INITIALIZE DEBUG SERIAL BUFFERS  |
//---------------------------------------------------------------------------------------------------------------
void Init_Debug_Buffs(void) {

    DbOutBuff.HeadIdx   = 0;
    DbOutBuff.TailIdx   = 0;
    DbOutBuff.ChrCnt    = 0;
    DbOutBuff.MaxChrCnt = 0;

    DbInBuffIdx = 0;
    HistIdx     = 0;
    for(int i=0; i<MaxHistory; i++) {		// Null commands in the history buffer.
	LastBuff[i][0] = 0;
    }
}
//---------------------------------------------------------------------------------------------------------------




//------------------------
//  TRANSMIT DEBUG DATA  |
//---------------------------------------------------------------------------------------------------------------
void TxDebugChar(char Char) {						    //
									    //
    UART4_IM_R &= ~(UART_IM_TXIM);					    // Disable Tx Interrupt.    G+
    DbOutBuff.Buff[DbOutBuff.HeadIdx++] = Char;				    // Put characters into the buffer.
    if(DbOutBuff.HeadIdx >= DbOutBuffSize) DbOutBuff.HeadIdx = 0;	    // Wrap Buffer.
    if(++DbOutBuff.ChrCnt >= DbOutBuffSize) {                               //
        Error(DEBUG_OUT_BUFF_OVF, "TxDebugChar()");	                    //
    }                                                                       //
    DebugUartTxChrs();							    // If space in FIFO xmit characters.
    UART4_IM_R |= UART_IM_TXIM;					            // Re-enable Tx Interrupt.
}									    //
//---------------------------------------------------------------------------------------------------------------
void TxDebugStr(char* pChar) {					            //
									    //
    UART4_IM_R &= ~(UART_IM_TXIM);					    // Disable Tx Interrupt.
    while(*pChar) {							    // Put characters into the buffer.
	DbOutBuff.Buff[DbOutBuff.HeadIdx++] = *pChar++;			    //
	if(DbOutBuff.HeadIdx >= DbOutBuffSize) DbOutBuff.HeadIdx = 0;	    // Wrap Buffer.
	if(++DbOutBuff.ChrCnt >= DbOutBuffSize) {                           //
	    Error(DEBUG_OUT_BUFF_OVF, "TxDebugStr()");                      //
	}                                                                   //
    }									    //
    DebugUartTxChrs();							    // If space in FIFO xmit characters.
    UART4_IM_R |= UART_IM_TXIM;					            // Re-enable Tx Interrupt.
}									    //
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------
//  DISPLAY A STRING AS HEX VALUES  |
//---------------------------------------------------------------------------------------------------------------
void TxDbStrHex(char* pChar) {                                              //
                                                                            //
    while(*pChar) {                                                         //
        sprintf(DbMsgBuff, "%02X ", *pChar++);                              //
        char* pMBuff = DbMsgBuff;                                           //
        while(*pMBuff) {                                                    //
            TxDebugChar(*pMBuff++);                                         //
        }                                                                   //
    }                                                                       //
}                                                                           //
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------
//  SEND STRING FOLLOWED BY PROMPT  |
//---------------------------------------------------------------------------------------------------------------
void StringAndPrompt(const char* pChar) {

    TxDebugStr(pChar);
    TxDebugStr("\r\n\n# ");
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------------
//  GET OUTPUT BUFFER CHARACTER COUNT  |
//---------------------------------------------------------------------------------------------------------------
int GetOutChrCnt(void) {

    return(DbOutBuff.ChrCnt);
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------
//  PROCESS DEBUG INPUT CHARACTERS  |
//---------------------------------------------------------------------------------------------------------------
//  This is called from the ISR so is running in interrupt state.
//
//  Normally the I/O is assumed to be talking to a VT100 terminal emulator such as Tera Term; the input
//  characters are formed into a line (e.g., terminated by a Carriage Return) which is then parsed in the
//  "background", by a NULL message sent to the MONITOR_TASK. When a routine is 'long' running and wants to
//  be interruptible, it first sets the InputMode here to ESCAPE_INPUT and registers a routine to be called
//  when the next character is entered; this is called in the background.
//---------------------------------------------------------------------------------------------------------------
							//
enum  eEscCode   { EscNone, EscRx, EscLB };		// To handle VT100 (VT52) escape sequence.
const u8 EscChr = 27;					//
							//
//---------------------------------------------------------------------------------------------------------------
void ChangeDebugInputMode(eInputMode Mode) {		//
							//
    InputMode = Mode;					//
}							//
//---------------------------------------------------------------------------------------------------------------
							//
void RxCharacter(char Char) {				//
							//
    static u8 EscSeqState      = EscNone;		//
    struct sDummy_Msg* pDumMsg = NULL;			// Dummy message.
							//
    switch(InputMode) {					//
							//
      case ESCAPE_INPUT:				//----------------------------------
							// Cancel running program.
	ISR_Send_Message(CANCEL_RTN_TASK, pDumMsg);	// Call the callback function.
	InputMode = LINE_INPUT;				//
	break;						//
							//
      case LINE_INPUT:					//----------------------------------
							//
	if(DbInBuffIdx >= DbInBuffSize) {		//
	    Error(DEBUG_IN_BUFF_OVF, "RxCharacter()");	//
	}						//
	if(Char == 0) return;				// Ignore nulls.
							//
	if(Char == EscChr) {	    			// Handle VT52 "Cursor Up" Escape sequence
	    EscSeqState = EscRx;			// to repeat last command.
	    TxDebugStr("\r\033[2K# ");		// Clear line and redraw the prompt.
	    return;					//
	}						//
	if(EscSeqState == EscRx) {			//
	    if(Char == '[') {	    			//
		EscSeqState = EscLB;			//
	    } else {					//
		EscSeqState = EscNone;			//
	    }						//
	    return;					//
	}						//
	if(EscSeqState == EscLB) {			//
	    if(Char == 'A') {				//
		if(--HistIdx < 0) HistIdx=MaxHistory-1; // Wrap command history.
		strcpy(DbInBuff, &LastBuff[HistIdx][0]);// Copy last command string to Input Buffer.
		DbInBuffIdx = strlen(DbInBuff);		//
		TxDebugStr(&LastBuff[HistIdx][0]);	// Echo last command.
	    }						//
	    EscSeqState = EscNone;			//
	    return;					//
	}						//							//
	TxDebugChar(Char);				// Echo character.
	if(Char == '\b') {				// If Backspace,
	    DbInBuffIdx--;				// 'Delete' character from buffer.
	    if(DbInBuffIdx < 0) DbInBuffIdx = 0;	//
	    TxDebugChar(' ');				// Erase the character on the screen.
	    TxDebugChar('\b');				//
	} else if(Char == '\r') {			//
	    TxDebugChar('\n');				// Add NL to CR
	    DbInBuff[DbInBuffIdx++] = 0;		// Terminate string
	    DbInBuffIdx = 0;				// Reset for next command.
 	    ISR_Send_Message(MONITOR_TASK, pDumMsg);	// Let the Monitor know a message is ready.
	} else {					//
	    DbInBuff[DbInBuffIdx++] = Char;		// Buffer character.
	}						//
	break;						//
	                                                //
      default:                                          //
          sprintf(sBuff,"=%d, RxCharacter()",InputMode);//
          Error(INVAL_SWITCH_VALUE, sBuff);             //
    }							//
}							//
//---------------------------------------------------------------------------------------------------------------




//---------------------------------------------
//  TRANSMIT CHARACTERS FROM CIRCULAR BUFFER  |
//---------------------------------------------------------------------------------------------------------------
//  Called both from the UART ISR, TxDebugStr() and TxDebugChar().

void DebugUartTxChrs(void) {

    while(( ! (UART4_FR_R & UART_FR_TXFF)) and DbOutBuff.ChrCnt) {      // If space in the Xmit FIFO, stuff
	UART4_DR_R = DbOutBuff.Buff[DbOutBuff.TailIdx++];	        // what characters we have and can fit.
	if(DbOutBuff.TailIdx==DbOutBuffSize) DbOutBuff.TailIdx=0;       // Wrap buffer.
	DbOutBuff.ChrCnt--;
    }

#ifdef DEBUG_UART_FLUSH

    while( ! (UART4_FR_R & UART_FR_TXFE)) {}                        // Wait for UART FIFO empty.
    while(DbOutBuff.ChrCnt) {                                           // If more characters in buffer ...
        UART4_DR_R = DbOutBuff.Buff[DbOutBuff.TailIdx++];           // send one.
        if(DbOutBuff.TailIdx==DbOutBuffSize) DbOutBuff.TailIdx=0;       // Wrap buffer.
        DbOutBuff.ChrCnt--;                                             //
        while( ! (UART4_FR_R & UART_FR_TXFE)) {}                    // Wait for that character flushed.
    }

#endif // DEBUG_UART_FLUSH
}
//---------------------------------------------------------------------------------------------------------------




//-------------------------
//  DEBUG UART INTERRUPT  |
//---------------------------------------------------------------------------------------------------------------

void DebugUartInterrupt(void) {

    const int Error_Mask = 0x00000F00;
    const int Data_Mask  = 0x000000FF;

    //--------------------------------//
    //-- Handle Incoming Characters --//
    //--------------------------------//

    s32 IntStatus = UARTIntStatus(UART4_BASE, true);    // Clear interrupt.
    UARTIntClear(UART4_BASE, IntStatus);

    while( ! (UART4_FR_R & UART_FR_RXFE)) {	// Until the FIFO is empty, receive characters.

        int Data_And_Errors = UART4_DR_R;
        int Errors = Data_And_Errors & Error_Mask;
        u8  Char   = Data_And_Errors & Data_Mask;

        if(Errors) {
            UART4_ECR_R = 0;		        // Reset errors, ignore character.	*FIXME*
        } else {
            RxCharacter(Char);			// Process the Character.
        }
    }

    //--------------------------------//
    //-- Handle Outgoing Characters --//
    //--------------------------------//

    DebugUartTxChrs();                          // Buffer characters.

//  UART4_ICR_R |= UART4_MIS_R;		        // Reset the interrupting source(s).
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------
//  ENABLE / DISABLE DEBUG UART  |
//---------------------------------------------------------------------------------------------------------------
/*
void Disable_Debug_Uart() {

    UART4_IM_R = 0;                                              // Disable the interrupts
    UARTDisable(UART4_BASE);					    // For warm resets.
}
//---------------------------------------------------------------------------------------------------------------

void Enable_Debug_Uart() {

    // Configure UART //

    UARTFIFOEnable(UART4_BASE);					    // Enable FIFOs.
    UARTFIFOLevelSet(UART4_BASE, UART_FIFO_TX1_8, UART_FIFO_RX4_8); // Set RX interrupt level.
    UARTIntRegister(UART4_BASE, DebugUartInterrupt);		    // Register ISR Function.
    //UARTFlowControlSet(UART4_BASE, UART_FLOWCONTROL_TX | UART_FLOWCONTROL_RX);
    UARTEnable(UART4_BASE);					    // This also enables the FIFOs.

    // Read everything out of UART before enabling interrupt.

    for(volatile int i = 0; i < 1000000; i++);
    while(UARTCharsAvail(UART4_BASE)) { UARTCharGet(UART4_BASE); }

    // Now everything is ready to go, enable the interrupts.

    UART4_IM_R = UART_INT_RX | UART_INT_TX | UART_INT_RT;
}
*/
//---------------------------------------------------------------------------------------------------------------


#include "sysctl.h"
#include "pin_map.h"
#include "hw_memmap.h"
#include "hw_gpio.h"
#include "gpio.h"



                                           //////////////////
                                           //              //
                                           //  INITIALIZE  //
                                           //              //
                                           //////////////////


//--------------
//  INIT UART  |
//---------------------------------------------------------------------------------------------------------------
void Init_Uarts(void) {

    // DEBUG UART-4 //

    Init_Debug_Buffs();                                                         // Initialize the buffers.
//  UARTConfigSetExpClk(UART4_BASE, GetClkSpeed(), 9600,
    UARTConfigSetExpClk(UART4_BASE, GetClkSpeed(), 19200,
             ( UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE ));
    UARTFIFOEnable(UART4_BASE);
    UARTIntRegister(UART4_BASE, DebugUartInterrupt);                            // Register Interrupt Handler.
    UARTIntEnable(UART4_BASE, UART_INT_RX | UART_INT_TX | UART_INT_OE | UART_INT_FE | UART_INT_RT );
    IntEnable(INT_UART4);

    // BLUETOOTH UART-1 //

    Init_Bt_Buffs();
    UARTConfigSetExpClk(UART1_BASE, GetClkSpeed(), 9600,
             ( UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE ));
    UARTFIFOEnable(UART1_BASE);
    UARTFlowControlSet(UART1_BASE, UART_FLOWCONTROL_TX | UART_FLOWCONTROL_RX);
    UARTIntRegister(UART1_BASE, Bt_Uart_Interrupt);                             // Register Interrupt Handler.
    UARTIntEnable(UART1_BASE, UART_INT_RX | UART_INT_TX | UART_INT_OE | UART_INT_FE | UART_INT_RT );
    IntEnable(INT_UART1);
}
//---------------------------------------------------------------------------------------------------------------

