//*****************************************************************************
//
// Copyright (c) 2012-2018 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include <time.h>
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "drivers/pinout.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/rom_map.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/uart.h"
#include "driverlib/hibernate.h"
#include "utils/uartstdio.h"

//*****************************************************************************
//
// Define pin to LED mapping.
//
//*****************************************************************************


#define USER_LED1  GPIO_PIN_0
#define USER_LED2  GPIO_PIN_1

uint32_t g_ui32SysClock;


//*****************************************************************************
//
// Configure the UART and its pins.  This must be called before UARTprintf().
//
//*****************************************************************************
void
ConfigureUART(void)
{
    //
    // Enable the GPIO Peripheral used by the UART.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Enable UART0
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);

    //
    // Configure GPIO Pins for UART mode.
    //
    ROM_GPIOPinConfigure(GPIO_PA0_U0RX);
    ROM_GPIOPinConfigure(GPIO_PA1_U0TX);
    ROM_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Initialize the UART for console I/O.
    //
    UARTStdioConfig(0, 115200, g_ui32SysClock);
}

void ConfigureRTC(void)
{
    struct tm sTime;
    // Initialize to 13:38:00 November 24, 2018
    sTime.tm_sec = 0;
    sTime.tm_min = 38;
    sTime.tm_hour = 13;
    sTime.tm_mday = 24;
    sTime.tm_mon = 10; // November
    sTime.tm_year = 2018 - 1900;
    sTime.tm_wday = 6; // Saturday
    sTime.tm_yday = 326;
    sTime.tm_isdst = 0;
    //
    // Need to enable the hibernation peripheral after wake/reset, before using
    // it.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_HIBERNATE);
    //
    // Wait for the Hibernate module to be ready.
    //
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_HIBERNATE))
    {
    }
    //
    // Enable clocking to the Hibernation module.
    //
    HibernateEnableExpClk(g_ui32SysClock);
    //
    // User-implemented delay here to allow crystal to power up and stabilize.
    //
    //
    // Configure the clock source for Hibernation module and enable the RTC
    // feature.
    //
    HibernateClockConfig(HIBERNATE_OSC_LOWDRIVE);
    HibernateRTCEnable();
    HibernateCounterMode(HIBERNATE_COUNTER_24HR);
    HibernateCalendarSet(&sTime);
}

//*****************************************************************************
//
// Main 'C' Language entry point.  Toggle an LED using TivaWare.
//
//*****************************************************************************
int
main(void)
{

    struct tm sCurrentTime;
    //
    // Run from the PLL at 120 MHz.
    //
    g_ui32SysClock = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                       SYSCTL_OSC_MAIN |
                                       SYSCTL_USE_PLL |
                                       SYSCTL_CFG_VCO_480), 120000000);

    //
    // Enable and wait for the port to be ready for access
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPION);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPION))
    {
    }
    
    //
    // Configure the GPIO port for the LED operation.
    //
    GPIOPinTypeGPIOOutput(GPIO_PORTN_BASE, (USER_LED1|USER_LED2));

    ConfigureUART();
    UARTprintf("Calendar routine!\n");
    ConfigureRTC();

        //
    // Loop Forever
    //
    while(1)
    {
        //
        // Turn on the LED
        //
        GPIOPinWrite(GPIO_PORTN_BASE, (USER_LED1|USER_LED2), USER_LED1);

        //
        // Delay for a bit
        //
        SysCtlDelay(g_ui32SysClock/6);

        //
        // Turn on the LED
        //
        GPIOPinWrite(GPIO_PORTN_BASE, (USER_LED1|USER_LED2), USER_LED2);

        //
        // Delay for a bit
        //
        SysCtlDelay(g_ui32SysClock/6);
        HibernateCalendarGet(&sCurrentTime);
        UARTprintf("\r%02d:%02d:%02d %04d-%02d-%02d",
                   sCurrentTime.tm_hour,
                   sCurrentTime.tm_min,
                   sCurrentTime.tm_sec,
                   sCurrentTime.tm_year + 1900,
                   sCurrentTime.tm_mon + 1,
                   sCurrentTime.tm_mday);

    }
}
