//---------------------------------------------------------------------------------------------------------------
//  init_cpu.cc
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------
//  TO DO
//
//	1) Turn off all modules when not needed.
//
//---------------------------------------------------------------------------------------------------------------


#include "common.h"
#include "error.h"
#include "init_cpu.h"
#include "defines.h"

#include <stdint.h>
#include <stdbool.h>

#include "tm4c123ae6pm.h"

#include "sysctl.h"
#include "pin_map.h"
#include "hw_memmap.h"
#include "hw_gpio.h"
#include "gpio.h"


#include "uarts.h"      // *FIXME* Testing

//----------------------------------------
//  INITIALIZE GLOBAL CPU CONFIGURATION  |
//---------------------------------------------------------------------------------------------------------------
//  Actual Clock Speed and Hardware Version
//
PRIVATE const u32 ClockSpeed = 80000000;

u32 GetClkSpeed(void)  { return(ClockSpeed); }

//---------------------------------------------------------------------------------------------------------------

void Init_CPU(void) {


    SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_XTAL_20MHZ | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN);  // Should be 80MHz

//TestUart();

    //---  ENABLE GPIO  ---//

    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);

    //---  ENABLE OTHER PERIPERALS  ---//

    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART4);    // Enable UART4 Debug Port.
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART1);    // Enable UART1 Bluetooth Port.
    SysCtlPeripheralEnable(SYSCTL_PERIPH_EEPROM0);  // Enable EEPROM
    SysCtlPeripheralEnable(SYSCTL_PERIPH_PWM0);     // Enable PWM 0 and 1.
    SysCtlPeripheralEnable(SYSCTL_PERIPH_PWM1);     //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN1);     // Enable CAN 1.
//  SysCtlPeripheralEnable(SYSCTL_PERIPH_TIMER1);   // For DelayMs()

#ifdef ENABLE_CAN2
    SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN0);     // Enable CAN 2.
#endif

    // Keep Alive During Sleep //

    // ***FIXME*** Change to using the new registers.
/*
    SYSCTL_SCGC1_R = SYSCTL_SCGC1_UART0	    	    // UART 0
	           | SYSCTL_SCGC1_UART1 	    // UART 1
	      //   | SYSCTL_SCGC1_SSI1		    // SSI 1
	           | SYSCTL_SCGC1_I2C0;	    	    // I2C 0

    SYSCTL_SCGCUART_R = SYSCTL_SCGCUART_S5;	    // UART 5

    SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_WTIMER0);  // Wide Timer 0 on during Sleep
    SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_WTIMER1);  // Wide Timer 1 on during Sleep

    // Lower Power in Sleep mode //

    SysCtlSleepPowerSet(SYSCTL_FLASH_LOW_POWER      // Flash and Sram low power in Sleep Mode
         //           | SYSCTL_SRAM_STANDBY);
                      | SYSCTL_SRAM_LOW_POWER);
*/



/////////////////////
//   GPIO PORT A   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // A0 In   Can1Rx
    // A1 Out  Can1Tx
    // A2 In   Brite Sw 1
    // A3 In   Brite Sw 2
    // A4
    // A5
    // A6
    // A7

    //---  CAN  ---//

    GPIOPinTypeCAN(  GPIO_PORTA_BASE, PA_Can1Rx_BIT | PA_Can1Tx_BIT );
    GPIOPinConfigure(GPIO_PA0_CAN1RX);
    GPIOPinConfigure(GPIO_PA1_CAN1TX);

    //---  GPIO  ---//

    GPIOPinTypeGPIOInput(GPIO_PORTA_BASE, PA_BriteSw1_BIT);
    GPIOPinTypeGPIOInput(GPIO_PORTA_BASE, PA_BriteSw2_BIT);
    GPIO_PORTA_PDR_R = 0b111100000;                            // Weak pull-down.



/////////////////////
//   GPIO PORT B   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // B0 In   BtRx     UART1
    // B1 Out  BtTx     UART1
    // B2      TP5
    // B3 Out  BtDisconnect
    // B4 In   Can2Rx
    // B5 Out  Can2Tx
    // B6 Out  Pwm2-1   PWM 0-0

    //---  CAN  ---//

#ifdef ENABLE_CAN2

    GPIOPinTypeCAN  (GPIO_PORTB_BASE, PB_Can2Rx_BIT | PB_Can2Tx_BIT );
    GPIOPinConfigure(GPIO_PB4_CAN0RX);
    GPIOPinConfigure(GPIO_PB5_CAN0TX);

#endif

    //---  PWM  ---//

    GPIOPinTypePWM  (GPIO_PORTB_BASE, PB_Pwm2_1_BIT );
    GPIOPinConfigure(GPIO_PB6_M0PWM0);

    //---  Bluetooth UART  ---//

    GPIOPinTypeUART (GPIO_PORTB_BASE, PB_BtRx_BIT | PB_BtTx_BIT);
    GPIOPinConfigure(GPIO_PB0_U1RX);
    GPIOPinConfigure(GPIO_PB1_U1TX);

    //---  GPIO  ---//

    GPIOPinTypeGPIOOutput(GPIO_PORTB_BASE, PB_BtDiscon_BIT);
    GPIO_PORTB_PDR_R = 0b10000100;                              // Weak pull-down.



/////////////////////
//   GPIO PORT C   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // C4 In   DebugRxD   UART4
    // C5 Out  DebugTxD   UART4
    // C6      TP6
    // C7      TP7

    //---  DEBUG UART  ---//

    GPIOPinTypeUART(GPIO_PORTC_BASE, PC_DebugRx_BIT | PC_DebugTx_BIT);
    GPIOPinConfigure(GPIO_PC4_U4RX);
    GPIOPinConfigure(GPIO_PC5_U4TX);

    GPIO_PORTC_PDR_R = 0b11000000;                              // Weak pull-down.




/////////////////////
//   GPIO PORT D   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // D0 Out   Pwm1-2      M0-6
    // D1       TP11
    // D2 In    Dip Sw 1
    // D3 In    Dip Sw 2
    // D4       TP12
    // D5       TP13

    //---  PWM  ---//

    GPIOPinTypePWM(  GPIO_PORTD_BASE, PD_Pwm1_2_BIT );
    GPIOPinConfigure(GPIO_PD0_M0PWM6);

    //---  GPIO  ---//

    GPIOPinTypeGPIOInput(GPIO_PORTD_BASE, PD_DipSw1_BIT | PD_DipSw2_BIT);
    GPIO_PORTD_PDR_R = 0b11110010;                              // Weak pull-down.



/////////////////////
//   GPIO PORT E   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // E0 In    Fault 3
    // E1 In    Fault 2
    // E2 In    Fault 1
    // E3
    // E4 Out   Pwm1-1   M1-2
    // E5       TP14

    //---  PWM  ---//

    GPIOPinTypePWM(  GPIO_PORTE_BASE, PE_Pwm1_1_BIT );
    GPIOPinConfigure(GPIO_PE4_M1PWM2);

    //---  GPIO  ---//

    GPIOPinTypeGPIOInput(GPIO_PORTE_BASE, PE_Fault3_BIT | PE_Fault2_BIT | PE_Fault1_BIT);
    GPIO_PORTE_PDR_R = 0b00101000;                              // Weak pull-down.



/////////////////////
//   GPIO PORT F   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // F0 Out   BtRts    UART1
    // F1 In    BtCts    UART1
    // F2 Out   Pwm3-2   M1-6
    // F3       TP15
    // F4

    //---  PWM  ---//

    GPIOPinTypePWM(  GPIO_PORTF_BASE, PF_Pwm3_2_BIT );
    GPIOPinConfigure(GPIO_PF2_M1PWM6);

    //---  GPIO  ---//

    GPIOPinTypeGPIOInput(GPIO_PORTF_BASE, PE_Fault3_BIT | PE_Fault2_BIT | PE_Fault1_BIT);
    GPIO_PORTF_PDR_R = 0b00011000;                              // Weak pull-down.



/////////////////////
//   GPIO PORT G   //
///////////////////////////////////////////////////////////////////////////////////////////////////////////

    // G0 Out   Heartbeat LED
    // G1 Out   Error LED
    // G2 Out   Pwm3-1          M1-0
    // G3       TP16
    // G4 Out   Pwm2-2          M0-4
    // G5       TP17

    //---  PWM  ---//

    GPIOPinTypePWM(  GPIO_PORTG_BASE, PG_Pwm3_1_BIT );
    GPIOPinConfigure(GPIO_PG2_M1PWM0);
    GPIOPinTypePWM(  GPIO_PORTG_BASE, PG_Pwm2_2_BIT );
    GPIOPinConfigure(GPIO_PG4_M0PWM4);

    //---  GPIO  ---//

    GPIOPinTypeGPIOOutput(GPIO_PORTG_BASE, PG_HB_LED_BIT | PG_ER_LED_BIT);
    GPIO_PORTG_PDR_R = 0b00101000;                              // Weak pull-down.

}
//---------------------------------------------------------------------------------------------------------------


