//---------------------------------------------------------------------------------------------------------------
//  monitor.cc
//
//  Line Oriented Serial Debug Monitor.
//
//  Copyright (c) 2025 Doug Broadwell, all rights reserved.
//---------------------------------------------------------------------------------------------------------------



#include "common.h"
#include "uarts.h"
#include "main.h"
#include "monitor.h"
#include "error.h"
#include "state.h"
#include "events.h"
#include "sm_main.h"
#include "init_cpu.h"
#include "tick.h"
#include "eeprom.h"
#include "switch.h"
#include "timers.h"
#include "can_local.h"
#include "led.h"
#include "sm_bt_at_cmnd.h"
#include "sm_bluetooth.h"

#include "pwm.h"
#include "can.h"
#include "hw_can.h"
#include "hw_types.h"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>




//-------------------------
//  FORWARD DECLARATIONS  |
//---------------------------------------------------------------------------------------------------------------

void DisplayCmnds    (int NmbParams, u8 CmdData);
void ReReadDipSwitch (int NmbParams, u8 CmdData);
void ResetEEPromData (int NmbParams, u8 CmdData);
void EraseAllEEProm  (int NmbParams, u8 CmdData);
void DispEEProm      (int NmbParams, u8 CmdData);
void SetEEProm       (int NmbParams, u8 CmdData);
void SpecialTest     (int NmbParams, u8 CmdData);
void MonSendEvent    (int NmbParams, u8 CmdData);
void ListEvents      (int NmbParams, u8 CmdData);
void ListStateMachs  (int NmbParams, u8 CmdData);
void ListPromAddrData(int NmbParams, u8 CmdData);
void Memory          (int NmbParams, u8 CmdData);
void MemoryAddrs     (int NmbParams, u8 CmdData);
void DisplayStatus   (int NmbParams, u8 CmdData);
void BacklightColor  (int NmbParams, u8 CmdData);
void BacklightBright (int NmbParams, u8 CmdData);
void NeedleBright    (int NmbParams, u8 CmdData);
void SpeedoBright    (int NmbParams, u8 CmdData);
void SetPWM          (int NmbParams, u8 CmdData);
void BtCommand       (int NmbParams, u8 CmdData);
void BtConnect       (int NmbParams, u8 CmdData);
void CanCtl          (int NmbParams, u8 CmdData);
void CanMode         (int NmbParams, u8 CmdData);
#ifdef ENABLE_CAN2
  void CanTest         (int NmbParams, u8 CmdData);
#endif // ENABLE_CAN2

void InvalNmbParams     (void);
void InvalNegOrZeroParam(void);
void IllFormedNumber    (void);
void DisplayDipFunctions(void);
void ProcessDipFunctions(void);
void InvalValue         (void);

bool Validate1or2UintParams(int NmbParams, uint* pFirstParam, uint* pSecondParam);
bool Validate1or2HexParams (int NmbParams, uint* pFirstParam, uint* pSecondParam);
bool Validate3UintParams   (uint* pParam1, uint* pParam2, uint* pParam3);
bool Validate4UintParams   (uint* pParam1, uint* pParam2, uint* pParam3, uint* pParam4);
bool ValidateHexDecParams  (uint* pFirstParam, uint* pSecondParam);
bool ValDecHexDecParam     (int NmbParams, uint* pParam1, uint* pParam2, uint* pParam3);

//---------------------------------------------------------------------------------------------------------------




//-----------------------
//  INITIALIZE MONITOR  |
//---------------------------------------------------------------------------------------------------------------
void Init_Monitor(void) {

    ProcessDipFunctions();
}
//---------------------------------------------------------------------------------------------------------------




//---------------------
//  PARSE INPUT LINE  |
//---------------------------------------------------------------------------------------------------------------
//  Dispatched from RxCharacter() ISR if there is a command in the input buffer to process.  If there is
//  a valid command the applicable command handling routine is called with the number of parameters (if any) and
//  a pointer to the first parameter.
//---------------------------------------------------------------------------------------------------------------
enum eCmdData { cdNone, cdOneVal, cdTwoVal, cdRnD, cdDec, cdFloat, cdHex, cdT1, cdT2, cdT3, cdCopyR, cdRead,
                cdWrite, cdSoftware, cdHardware, cdToUpper, cdColorName, cdCan1, cdCan2 };
//---------------------------------------------------------------------------------------------------------------
					//
struct sCommands {			//
					//
    const char *CmdName;		// Command mnemonic.
    const u8    CmdData;		// Data sent to command.
    void (*pFunct)(int, u8);	        // Pointer to command function.
					//
} Commands[] = {			// **NOTE** Keep in sync with DisplayCmnds();
					//
    "?",  cdNone,     DisplayCmnds,	// Display Commands.
    "H",  cdNone,     DisplayCmnds,	//    "       "
    "??", cdRnD,      DisplayCmnds,	//    "       "
					//
    "AC", cdNone,     BtConnect,        // Broadcast for connection.
    "AL", cdNone,     BtCommand,        // Send AT command, no upper case conversion.
    "AT", cdToUpper,  BtCommand,        // Send AT command, convert to uppercase.
                                        //
    "C1", cdCan1,     CanCtl,           // Transmit message on CAN1 (CAN-1).
#ifdef ENABLE_CAN2                      //
    "C2", cdCan2,     CanCtl,           // Transmit message on CAN0 (CAN-2).
    "CL", cdNone,     CanTest,          // Perform CAN0 (CAN-2) loopback test.
    "CM", cdNone,     CanMode,          // Set Loopback mode On/Off.
#else                                   //
    "CM", cdNone,     CanMode,          // Set Loopback mode On/Off.
#endif                                  //
                                        //
    "DC", cdCopyR,    DisplayStatus,    // Display Copyright.
    "DH", cdHardware, DisplayStatus,    // Display Hardware Revision.
    "DS", cdSoftware, DisplayStatus,    // Display Software Revision.
					//
    "EA", cdNone,     ListPromAddrData, // List all non Stored-Count data.
    "ED", cdDec,      DispEEProm,	// Display EEProm Data in Decimal.
    "EF", cdFloat,    DispEEProm,	// Display EEProm Data in Float.
    "EH", cdHex,      DispEEProm,	// Display EEProm Data in Hex.
    "EM", cdNone,     EraseAllEEProm,   // Erase all of EEProm.
    "ER", cdNone,     ResetEEPromData,  // Reset EEProm to Default Values.
    "ES", cdNone,     SetEEProm,        // Set a value in EEProm.
					//
    "LB", cdNone,     BacklightBright,  // Tach Backlight Brightness 0 - 100%
    "LC", cdNone,     BacklightColor,   // Color: Red %, Green %, Blue %, White %; or Color Name.
    "L?", cdColorName,BacklightColor,   // List Valid Color Names.
    "LN", cdNone,     NeedleBright,     // Needles Brightness.
    "LP", cdNone,     SetPWM,           // Manually set a PWM period.
    "LS", cdNone,     SpeedoBright,     // Speedometer Backlight Brightness.
					//
    "MR", cdRead,     Memory,           // Read Memory.
    "MW", cdWrite,    Memory,           // Write Memory.
    "M?", cdNone,     MemoryAddrs,      // Display memory regions.
                                        //
    "PE", cdNone,     ListEvents,       // Display All Events.
    "PS", cdNone,     ListStateMachs,   // Display State Machines.
                                        //
    "SE", cdNone,     MonSendEvent,     // Send Event to a State Machine.
					//
    "SW", cdNone,     ReReadDipSwitch,	// Re-Read and process the Dip Switch.
                                        //
    "T1", cdT1,       SpecialTest,      //
    "T2", cdT2,       SpecialTest,      //
    "T3", cdT3  ,     SpecialTest,	// To run special tests as needed.  Does nothing if no test to run.
};					//
					//
const int MaxCmndSize = 2;		// Max # of characters in a command.
					//
const char WhiteSpace[] = " ,\t";	// ' ', ',', and tab are "whitespace characters".
					//
//---------------------------------------------------------------------------------------------------------------
							    //
void Monitor_Dispatch(u16 Count, void* pMsg, ulong Ticks) { // Parse the command entered.
							    //
    int   i, Cnt;					    //
    char Tok[MaxCmndSize + 1];				    //
    char* pToken = &Tok[0];				    // Bug-"must be an lvalue" if only 'Tok' used.
    const int CmndArraySize = 				    //
	sizeof(Commands) / sizeof(Commands[0]);	    	    //
							    //
    int CmndSize = strcspn(DbInBuff, WhiteSpace);	    // See how long the command is.
    if(CmndSize > MaxCmndSize) {			    //
	StringAndPrompt("Command Too Long.\a\r");	    //
	return;						    //
    }							    //
    strcpy(CopyBuff, DbInBuff);				    // Save a copy for parameter parsing (since strtok()
    pToken = strtok(DbInBuff, WhiteSpace);		    //   puts a \0 at the end of every token).
    if(pToken == NULL) {				    // No command.
	StringAndPrompt("");				    //
	return;						    //
    }							    //
    for(i=0; i<CmndSize; i++) {				    //
	*(pToken + i) = toupper(*(pToken + i));		    // Make command upper case.
    }							    //
    for(i=0; i<CmndArraySize; i++) {			    // Scan for valid command.
	if(strcmp(pToken, Commands[i].CmdName)==0) break;   //
    }							    //
    if(i >= CmndArraySize) {				    // No valid command found.
	TxDebugStr("Command Not Found.\a\r\n");	            //
	DisplayCmnds(0, 0);				    //
	return;						    //
    }							    //
    strcpy(&LastBuff[HistIdx][0], CopyBuff);		    // Save last command.
    if(++HistIdx >= MaxHistory) HistIdx = 0;		    // Wrap history buffer.
							    //
    for(Cnt=0; strtok(NULL, WhiteSpace) != NULL; Cnt++) {}  // Count number of parameters
    (*Commands[i].pFunct)(Cnt, Commands[i].CmdData);	    // Call the routine handling this command
}							    // with the number of Parameters there are.
//---------------------------------------------------------------------------------------------------------------




//---------------------------------
//  CANCEL LONG RUNNING FUNCTION  |
//---------------------------------------------------------------------------------------------------------------
//  When a long running monitor routine is active it registers a callback function here to cancel it
//  if the user presses a key on the keyboard.

void (*pEscFunct)(void) = NULL;				    // Routine callback function.

//---------------------------------------------------------------------------------------------------------------
void Register_Cancel_Function( void (*pCnclFunct)(void) ) {

    pEscFunct = pCnclFunct;
}
//---------------------------------------------------------------------------------------------------------------
void Rtn_Cancel_Dispatch(u16 Count, void* pMsg, ulong Ticks) {  //
                                                                //
    if(pEscFunct == NULL) {                                     //
        Error(NULL_CALLBACK_ADDR, "Rtn_Cancel_Dispatch()");     //
    }                                                           //
    (*pEscFunct)();					        // Call the cancel routine function.
    pEscFunct = NULL;					        // Mark as unset again.
}                                                               //
//---------------------------------------------------------------------------------------------------------------




//-------------------------------
//  DISPLAY COMMANDS: H, ?, ??  |
//---------------------------------------------------------------------------------------------------------------
void DisplayCmnds(int NmbParameters, u8 CmdData) {

    TxDebugStr("\n\r  ?, ??\tDisplay Commands");

    TxDebugStr("\n\n\r  AC\tAdvertise for connection");
    TxDebugStr("\n\r  AL\tSend AT command without uppercase conversion");
    TxDebugStr("\n\r  AT\tSend AT command, make all of it uppercase");

    TxDebugStr("\n\n\r  C1\tID, u32 Data, Length: Send Message on CAN1 (CAN-1)");
#ifdef ENABLE_CAN2
    TxDebugStr("\n\r  C2\tID, u32 Data, Length: Send Message on CAN0 (CAN-2)");
    TxDebugStr("\n\r  CL\tPerform CAN0 (CAN-2) Loopback test");
    TxDebugStr("\n\r  CM\t1/2 0/1 Set Can-1/2 Loopback On=1, Off=0");
#else
    TxDebugStr("\n\r  CM\tSet Can-1 Loopback On=1, Off=0");
#endif

    TxDebugStr("\n\n\r  DC\tCopyright");
    TxDebugStr("\n\r  DH\tHardware Revision");
    TxDebugStr("\n\r  DS\tSoftware Revision");

    TxDebugStr("\n\n\r  - EEProm Addr are 32 bit Words, not Bytes (Decimal Fmt) -");
    TxDebugStr("\n\r  EA\tList non Stored-Count EEProm Data");
    TxDebugStr("\n\r  ED (start) (end)  Display EEProm Data in Decimal");
    TxDebugStr("\n\r  EF (start) (end)  Display EEProm Data in Float");
    TxDebugStr("\n\r  EH (start) (end)  Display EEProm Data in Hex");
    TxDebugStr("\n\r  EM 1\tErase all of the EEProm; the '1' is mandatory");
    TxDebugStr("\n\r  ER\tReset EEProm to Default Values");
    TxDebugStr("\n\r  ES    Addr  Data   Set a word in EEProm (Decimal Format)");

    TxDebugStr("\n\n\r  LB\tSet Backlight Brightness 0%-100%");
    TxDebugStr("\n\r  LC\tSet Backlight Color: Red%, Green%, Blue%, White%; or by Name");
    TxDebugStr("\n\r  L?\tList Valid Color Names");
    TxDebugStr("\n\r  LN\tSet Needle Brightness 0%-100%");
    TxDebugStr("\n\r  LP\tManually Set PWM On Percent: ePWM 0%-100% (** BE CAREFUL **)");
    TxDebugStr("\n\r  LS\tSet Speedometer Backlight Brightness 0%-100%");

    TxDebugStr("\n\n\r  MR\t0xAddr  (CNT)   Read Ram");
    TxDebugStr("\n\r  MW\t0xAddr 0xData   Write to Ram");
    TxDebugStr("\n\r  M?\tDisplay Memory Regions");

    TxDebugStr("\n\n\r  PE\tList Events");
    TxDebugStr("\n\r  PS\tList State Machines");

    TxDebugStr("\n\n\r  SE\tSend Event to State Machine: SM, Event, Data");
    TxDebugStr("\n\r  SW\tRe-read Dip Switch");

    TxDebugStr("\n\n\r  T1\tRun Special Test (if any)");
    TxDebugStr("\n\r  T2\tRun Special Test (if any)");
    TxDebugStr("\n\r  T3\tRun Special Test (if any)");

/*
    TxDebugStr(  "\n\n\r  Dip Switch:");
    TxDebugStr(  "\n\r   1: Off = K9G         On = K9B Mode.");
    TxDebugStr(  "\n\r   2: Off = Real Counts On = Dummy Count Mode.");
*/
    StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------




                                            //---- A ----//



//-------------------------------------------
//  START ADVERIZING FOR CONNECTION:  "AC"  |
//---------------------------------------------------------------------------------------------------------------
void BtConnect(int NmbParams, u8 CmdData) {

    if(NmbParams != 0) {
        InvalNmbParams();
        return;
    }
//  BtAdvertise();                      ****  FIXME  ****
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------------------------
//  SEND AT COMMAND TO BLUETOOTH MODULE:  "AL" "AT"  |
//---------------------------------------------------------------------------------------------------------------
//  Do not include the trailing "\r\n".

void BtCommand(int NmbParams, u8 CmdData) {

    const uint CommSize = 30;

    char  pCommand[MaxCmndSize+1];                  // Mustn't forget the trailing \0.
    static char  pAtCommand[CommSize];               // The AT command, can't be on the stack.
    char* pAtCmd = pAtCommand;                       // Pointer to AT command.

    if(NmbParams != 1) {
        InvalNmbParams();
        return;
    }
    if(strlen(CopyBuff) >= CommSize + 2) {
        StringAndPrompt("AT Command Too Long");
        return;
    }
    sscanf(CopyBuff, " %s %s", pCommand, pAtCmd);   // Parse the command line.
    if(CmdData == cdToUpper) {
        while(*pAtCmd) {                            // Make sure all characters are upper case.
            *pAtCmd = toupper(*pAtCmd);
            pAtCmd++;
        }
        pAtCmd = pAtCommand;                         // Reset pointer to beginning of command.
    }
    BtAtCmndEcho(pAtCmd);
//  StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------




                                            //---- C ----//



//-----------------------------------
//  SEND CAN MESSAGE:  "C1" ("C2")  |
//---------------------------------------------------------------------------------------------------------------
void ValCanCtlFormat(void) {
    TxDebugStr("\n\r Valid parameter format is: ID: 0-2047, Data: 0-0xFFFFFFFF, Length: 0-8\r\n");
}
//---------------------------------------------------------------------------------------------------------------
void CanCtl(int NmbParams, u8 CmdData) {

    uint ID, Msg, Length;

    if(ValDecHexDecParam(NmbParams, &ID, &Msg, &Length) == FAILURE) {
        ValCanCtlFormat();
        return;
    }
    if(ID > 2047) {
        ValCanCtlFormat();
        InvalValue();
        return;
    }
    if(Length > 8) {
        ValCanCtlFormat();
        InvalValue();
        return;
    }

    switch(CmdData) {

        case cdCan1:
            SendCan1Msg(ID, Msg, (u8)Length);
            break;

#ifdef ENABLE_CAN2
        case cdCan2:
            SendCan2Msg(ID, Msg, (u8)Length);
            break;
#endif // ENABLE_CAN2

        default:
            sprintf(sBuff, "monitor.cc line %d", __LINE__);
            Error(INVAL_SWITCH_VALUE, sBuff);
    }

    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------



#ifdef ENABLE_CAN2
//------------------------------------
//  CAN BUS LOOPBACK TEST(S):  "CL"  |
//---------------------------------------------------------------------------------------------------------------
void CanTest(int NmbParams, u8 CmdData) {

    Can2LoopbackTest();
}
//---------------------------------------------------------------------------------------------------------------
#endif // ENABLE_CAN2



//-------------------------------------
//  CAN BUS SET LOOPBACK MODE ON/OFF  |
//---------------------------------------------------------------------------------------------------------------
void CanMode(int NmbParams, u8 CmdData) {

#ifdef ENABLE_CAN2

    uint WhichCan, OnOff, Base;

    if(Validate1or2UintParams(2, &WhichCan, &OnOff) == FAILURE) {
        return;
    }
    if(WhichCan == 0 or WhichCan > 2) {
        StringAndPrompt("CAN must be 1 or 2");
        return;
    }
    if(OnOff > 1) {
        StringAndPrompt("Must be 0 = Off or 1 = On");
        return;
    }
    Base = WhichCan == 1 ? CAN1_BASE : CAN0_BASE;
    if(OnOff == 0) {
        HWREG(Base + CAN_O_CTL) &= ~CAN_CTL_TEST;       // Turn Off loopback;
        HWREG(Base + CAN_O_TST) &= ~CAN_TST_LBACK;
    } else {
        HWREG(Base + CAN_O_CTL) |= CAN_CTL_TEST;        // Turn On loopback;
        HWREG(Base + CAN_O_TST) |= CAN_TST_LBACK;
    }

#else

    uint OnOff;

    if(Validate1or2UintParams(1, &WhichCan) == FAILURE) {
        return;
    }
    if(OnOff > 1) {
        StringAndPrompt(Must be 0 = Off or 1 = On);
        return;
    }
    if(OnOff == 0) {
        HWREG(CAN1_BASE + CAN_O_CTL) &= ~CAN_CTL_TEST;       // Turn Off loopback;
        HWREG(CAN1_BASE + CAN_O_TST) &= ~CAN_TST_LBACK;
    } else {
        HWREG(CAN1_BASE + CAN_O_CTL) |= CAN_CTL_TEST;        // Turn On loopback;
        HWREG(CAN1_BASE + CAN_O_TST) |= CAN_TST_LBACK;
    }

#endif // ENABLE_CAN2

    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------


                                            //---- D ----//



//---------------------------------------------------
//  DISPLAY COPYRIGHT / REVISIONS:  "DC" "DH" "DS"  |
//---------------------------------------------------------------------------------------------------------------
void DisplayStatus(int NmbParams, u8 CmdData) {

    switch(CmdData) {
        case cdCopyR:
            sprintf(DbMsgBuff, "\r\n %s", Copyright);
            TxDebugStr(DbMsgBuff);
            break;
        case cdSoftware:
            sprintf(DbMsgBuff, "\r\n S/W Rev: r%s  %s", SoftwareVer, SoftDate);
            TxDebugStr(DbMsgBuff);
            break;
        case cdHardware:
            TxDebugStr("\r\nH/W Rev: r1.0");
            break;
        default:
            sprintf(sBuff, "monitor.cc line %d", __LINE__);
            Error(INVAL_SWITCH_VALUE, sBuff);
    }
/*
    sprintf(DbMsgBuff, "\r\n Display Bd H/W Rev:      %s", GetHwVersionStr());
    TxDebugStr(DbMsgBuff);

    uint Switch = Read_Dip_Switch();
    sprintf(DbMsgBuff, "\r\n\n Dip Switch:  1 %s  2 %s  3 %s  4 %s",
        Switch & 1 ? "off" : "ON ", Switch & 2 ? "off" : "ON ");
    TxDebugStr(DbMsgBuff);
    DisplayDipFunctions();
*/
    StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------




                                        //---- E ----//



//-------------------------------------
//  DISPLAY EEPROM ADDRESSES:  ""EA"  |
//---------------------------------------------------------------------------------------------------------------
void ListPromAddrData(int NmbParams, u8 CmdData) {

    ListEEPromData();
    StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------
//  DISPLAY EEPROM DATA:  "ED" "EF" "EH"  |
//---------------------------------------------------------------------------------------------------------------
void DispProm(uint StartAddr, uint EndAddr, u8 CmdData);
//---------------------------------------------------------------------------------------------------------------

void DispEEProm(int NmbParams, u8 CmdData) {

    uint  FirstAddr, LastAddr;

    if(NmbParams == 1 or NmbParams > 2) {
        StringAndPrompt("Either 0 parameters for all used EEProm or Start and End address.\a\r");
        return;
    }
    if(NmbParams == 0) {                                    // Display all used addresses.
        DispProm(0, EEPROM_SIZE - 1, CmdData);
    } else {
        if(Validate1or2UintParams(2, &FirstAddr, &LastAddr) == FAILURE) {
            IllFormedNumber();
            return;
        }
        if(FirstAddr > LastAddr) {
            StringAndPrompt("First Addr > Last Addr.\a\r");
            return;
        }
        if(LastAddr >= EEPROM_SIZE) {
            StringAndPrompt("Address too big.\a\r");
            return;
        }
        DispProm(FirstAddr, LastAddr, CmdData);
    }
    StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------
void DispProm(uint StartAddr, uint EndAddr, u8 CmdData) {

    int LineCnt = 0;
    TxDebugStr ("\r\n");

    for(uint Addr = StartAddr; Addr <= EndAddr; Addr++) {
        switch(CmdData) {
            case cdDec:                         // Display in Decimal;
                sprintf(DbMsgBuff, "%d\t", ReadEEPromLin(Addr));
                break;
            case cdFloat:                       // Display in Float;
                sprintf(DbMsgBuff, "%f\t", fReadEEPromLin(Addr));
                break;
            case cdHex:                         // Display in Hex;
                sprintf(DbMsgBuff, "%08X\t", ReadEEPromLin(Addr));
                break;
            default:
                sprintf(sBuff, "= %d DispProm()", CmdData);
                Error(INVAL_SWITCH_VALUE, sBuff);
        }
        TxDebugStr(DbMsgBuff);
        if(++LineCnt == 8) {
            LineCnt = 0;
            TxDebugStr ("\r\n");
        }
    }
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------
//  ERASE ALL OF THE EEPROM:  "EM"  |
//---------------------------------------------------------------------------------------------------------------
void EraseAllEEProm(int NmbParams, u8 CmdData) {

    uint Parameter, DummyParam;

    if(NmbParams > 1 or NmbParams == 0) {
        InvalNmbParams();
        return;
    }
    if(Validate1or2UintParams(1, &Parameter, &DummyParam) == FAILURE) return;
    if(Parameter != 1) {
        StringAndPrompt("Invalid Parameter");
        return;
    }
    uint Result = MassEraseEEProm();
    if(Result == 0) {
        StringAndPrompt("OK");
    } else {
        sprintf(sBuff, "Error, MassEraseEEProm() returned 0x%X", Result);
        StringAndPrompt(sBuff);
    }
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------
//  RESET EEPROM TO DEFAULT VALUES:  "ER"  |
//---------------------------------------------------------------------------------------------------------------
void ResetEEPromData(int NmbParams, u8 CmdData) {

    StringAndPrompt("OK");
    Reset_EEProm_Data();
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------
//  SET A WORD IN EEPROM:  "ES"  |
//---------------------------------------------------------------------------------------------------------------
void SetEEProm(int NmbParams, u8 CmdData) {

    uint Addr, Data, Was;

    if(NmbParams != 2) {
        StringAndPrompt("Format is 'ES Addr Data'");
        return;
    }
    if(Validate1or2UintParams(2, &Addr, &Data) == FAILURE) {
        IllFormedNumber();
        return;
    }
    if(Addr >= EEPROM_SIZE) {
        StringAndPrompt("Address too big.\a\r");
        return;
    }
    Was = ReadEEPromLin(Addr);
    WriteEEPromLin(Addr, Data);
    sprintf(DbMsgBuff, "Was: %u Now: %u", Was, Data);
    StringAndPrompt(DbMsgBuff);
}
//---------------------------------------------------------------------------------------------------------------




                                            //---- L ----//



//--------------------------------------------------------
//  SET TACHOMETER BACKLIGHT BRIGHTNESS 0 - 100%:  "LB"  |
//---------------------------------------------------------------------------------------------------------------
void BacklightBright(int NmbParams, u8 CmdData) {

//  TachBacklightBrightness(uint Brite);
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------------------------------
//  SET PERCENTAGE OF EACH TACHOMETER BACKLIGHT LED COLOR:  "LC"  |
//---------------------------------------------------------------------------------------------------------------
//  Parameters can be either a string of the name of the desired pre-defined color or 4 integers of 0 - 100%
//  of the colors: Red, Green, Blue, and White.
//---------------------------------------------------------------------------------------------------------------
void ColorString(void);         // Forward Declarations.
void SeparateColors(void);
void ValidColorNames(void);
//---------------------------------------------------------------------------------------------------------------
void BacklightColor(int NmbParams, u8 CmdData) {

    if(CmdData == cdColorName) {
        ValidColorNames();
        StringAndPrompt("OK");
        return;
    } else if(NmbParams != 4 or NmbParams != 1) {
        InvalNmbParams();
        return;
    }
    if(NmbParams == 1) {
        ColorString();
    } else {
        SeparateColors();
    }
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------
void SeparateColors(void) {

    uint Red, Green, Blue, White;

    if(Validate4UintParams(&Red, &Green, &Blue, &White) == FAILURE) {
        IllFormedNumber();
        return;
    }
    if(Red > 100 or Green > 100 or Blue > 100 or White > 100) {
        StringAndPrompt("Values must be in range 0 - 100");
        return;
    }
    TachBacklightColor((float)Red/100.0, (float)Green/100.0, (float)Blue/100.0, (float)White/100.0);
}
//---------------------------------------------------------------------------------------------------------------
void ValidColorNames(void) {

    TxDebugStr("\r\nRed");
    TxDebugStr("\r\nYel      Yellow");
    TxDebugStr("\r\nOrg      Orange");
    TxDebugStr("\r\nBlu      Blue");
    TxDebugStr("\r\nPur      Purple");
    TxDebugStr("\r\nWht      White");
    TxDebugStr("\r\nCoolWht  CoolWhite");
    TxDebugStr("\r\nBriteWht BriteWhite BrightWhite");
}
//---------------------------------------------------------------------------------------------------------------
void ColorString(void) {

    struct sColors {
        const char   *ColorName;        // Color Mnemonic.
        const eColors ColorIdx;         // Color Index.
    } ColorNames[] = {
        "BLK"
        "RED",         RedIdx,
        "YEL",         YellowIdx,
        "YELLOW",      YellowIdx,
        "ORG",         OrangeIdx,
        "ORANGE",      OrangeIdx,
        "BLU",         BlueIdx,
        "BLUE",        BlueIdx,
        "PUR",         PurpleIdx,
        "PURPLE",      PurpleIdx,
        "WHT",         WhiteIdx,
        "WHITE",       WhiteIdx,
        "COOLWHT",     CoolWhtIdx,
        "COOLWHITE",   CoolWhtIdx,
        "BRITEWHT",    BriteWhtIdx,
        "BRITEWHITE",  BriteWhtIdx,
        "BRIGHTWHITE", BriteWhtIdx,
    };

    const uint MaxColorStr = 11;                                //
    const uint ColorArraySize =                                 //
        sizeof(ColorNames) / sizeof(ColorNames[0]);             //
    int   i;                                                    //
    char Tok[MaxColorStr + 1];                                  //
    char* pToken = &Tok[0];                                     // Bug-"must be an lvalue" if only 'Tok' used.
                                                                //
    int ColorSize = strcspn(CopyBuff, WhiteSpace);              // See how long the color name is.
    if(ColorSize > MaxColorStr) {                               //
        StringAndPrompt("Color Name Too Long.\a\r");            //
        return;                                                 //
    }                                                           //
    pToken = strtok(CopyBuff, WhiteSpace);                      // Puts a \0 at the end of every token).
    for(i=0; i<MaxColorStr; i++) {                              //
        *(pToken + i) = toupper(*(pToken + i));                 // Make name upper case.
    }                                                           //
    for(i=0; i<ColorArraySize; i++) {                           // Scan for valid color.
        if(strcmp(pToken, ColorNames[i].ColorName)==0) break;   //
    }                                                           //
    if(i >= ColorArraySize) {                                   // No valid color found.
        TxDebugStr("Color Not Found.\a\r\n\n");                 //
        ValidColorNames();                                      //
        return;                                                 //
    }                                                           //
    TachBacklightColor(ColorNames[i].ColorIdx);                 // Set the color.
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------
//  SET NEEDLE BRIGHTNESS 0 - 100%:  "LN"  |
//---------------------------------------------------------------------------------------------------------------
void NeedleBright(int NmbParams, u8 CmdData) {

    uint Percent, Dummy;

    if(NmbParams != 1) {
        InvalNmbParams();
        return;
    }
    if(Validate1or2UintParams(1, &Percent, &Dummy) == FAILURE) return;
    if(Percent > 100) {
        StringAndPrompt("Values must be in range 0 - 100");
        return;
    }
    NeedleBrightness(Percent);
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------------
//  MANUALLY SET PWM ON-PERIOD 0 - 100%:  "LP"  |
//---------------------------------------------------------------------------------------------------------------
void SetPWM(int NmbParams, u8 CmdData) {

    uint Percent = 10;
    uint Which   =  0;             // RED 0, GREEN 1, BLUE 2, WHITE 3, NEEDLES 4, SPEEDO 5.

//  char Command[MaxCmndSize+1];   // Where the "LP" gets placed.       // *FIXME* Not used.

    if(NmbParams != 2) {
        InvalNmbParams();
        return;
    }
    if(Validate1or2UintParams(2, &Which, &Percent) == FAILURE) return;
    if(Percent > 100) {
        StringAndPrompt("Values must be in range 0-100");
        return;
    }
    if(Which > SPEEDO) {
        StringAndPrompt("Which PWM color must be in ePWM enumeration");
        return;
    }
    ManualPWM((ePWM)Which, Percent);
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------




//---------------------------------------------------------
//  SET SPEEDOMETER BACKLIGHT BRIGHTNESS 0 - 100%:  "LS"  |
//---------------------------------------------------------------------------------------------------------------
void SpeedoBright(int NmbParams, u8 CmdData) {

//  SpeedoBacklightBrightness(uint Brite);
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------




                                            //---- M ----//



//----------------------
//  READ/WRITE MEMORY  |
//---------------------------------------------------------------------------------------------------------------
//  Address and Data in Hex format.

void Memory(int NmbParams, u8 CmdData) {

    uint HexParam1, HexParam2, DecParam;
    char pBuff[20];

    strcpy(pBuff, "");
    switch(CmdData) {

        case cdRead:

            if(NmbParams < 1 or NmbParams > 2) {
                InvalNmbParams();
                break;
            }
            if(Validate1or2HexParams(1, &HexParam1, &HexParam2) == FAILURE) return;
            if(NmbParams == 2) {
                if(ValidateHexDecParams(&HexParam1, &DecParam) == FAILURE) return;
                if(DecParam == 0 or DecParam > 50) {
                    strcpy(pBuff, "Cnt must be > 0 and <= 50");
                    break;
                }
                TxDebugStr("\r\n    Addr         Data\r\n ----------   ----------\r\n");
                for(int i=0; i<DecParam; i++) {
                    sprintf(sBuff, " 0x%08X   0x%08X\r\n", HexParam1, *(volatile uint32_t *)HexParam1);
                    TxDebugStr(sBuff);
                    HexParam1 += 4;
                }
            } else {
                sprintf(pBuff, "\r\n 0x%08X", *(volatile uint32_t *)HexParam1);
            }
            break;

        case cdWrite:

            if(NmbParams != 2) {
                InvalNmbParams();
                break;
            }
            if(Validate1or2HexParams(2, &HexParam1, &HexParam2) == FAILURE) return;
            (*((volatile uint32_t *)HexParam1)) = HexParam2;
            strcpy(pBuff, "OK");
            break;
    }
    StringAndPrompt(pBuff);
}
//---------------------------------------------------------------------------------------------------------------




//---------------------------------
//  DISPLAY MEMORY RANGES:  "M?"  |
//---------------------------------------------------------------------------------------------------------------
void MemoryAddrs(int NmbParams, u8 CmdData) {

    TxDebugStr(" Flash:    0x0000,0000 - 0x000F,FFFF\n\r Ram:      0x0200,0000 - 0x0203,FFFF\n\r");
    TxDebugStr(" Reg:      0x4000,0000 - 0x400F,F51C\n\r EC Reg:   0x4403,0400 - 0x4403,0418\n\r");
    StringAndPrompt(" NVIC Reg: 0xE000,E008 - 0xE000,EF3C");
}
//---------------------------------------------------------------------------------------------------------------




                                            //---- P ----//



//---------------------------
//  LIST ALL EVENTS:  "PE"  |
//---------------------------------------------------------------------------------------------------------------
void ListEvents(int NmbParams, u8 CmdData) {

    TxDebugStr("\r\n");
    for(int i=0; i<LAST_E_Event_ENTRY; i++) {
        sprintf(sBuff, " %2d  %s\r\n", i, Get_Event_Name(i));
        TxDebugStr(sBuff);
    }
    StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------




//-------------------------------
//  LIST STATE MACHINES:  "PS"  |
//---------------------------------------------------------------------------------------------------------------
void ListStateMachs(int NmbParams, u8 CmdData) {

    TxDebugStr("\r\n");
    for(int i=0; i<=LAST_SM; i++) {
        sprintf(sBuff, " %3d  %s\r\n", i, Get_SM_Name(i));
        TxDebugStr(sBuff);
    }
    sprintf(sBuff, " %3d  %s\r\n", NULL_STATE, Get_SM_Name(NULL_STATE));
    TxDebugStr(sBuff);
    sprintf(sBuff, " %3d  %s\r\n", SM_NULL, Get_SM_Name(SM_NULL));
    TxDebugStr(sBuff);
    StringAndPrompt("");
}
//---------------------------------------------------------------------------------------------------------------




                                            //---- S ----//



//-----------------------------------------
//  SEND EVENT TO A STATE MACHINE:  "SE"  |
//---------------------------------------------------------------------------------------------------------------
void MonSendEvent(int NmbParams, u8 CmdData) {

    uint SM, Event, Data;

    if(NmbParams != 3) {
        InvalNmbParams();
        return;
    }
    if(Validate3UintParams(&SM, &Event, &Data) == FAILURE) {
        IllFormedNumber();
        return;
    }
    if((SM > LAST_SM) and (SM != NULL_STATE) and (SM != SM_NULL)) {
        StringAndPrompt("Invalid State Machine #");
        return;
    }
    if(Event >= LAST_E_Event_ENTRY) {
        StringAndPrompt("Invalid Event");
        return;
    }
    Send_Event( (sm_t)SM, (event_t)Event, (event_data_t)Data );
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------------------
//  RE-READ AND PROCESS THE DIP SWITCH:  "SW"  |
//---------------------------------------------------------------------------------------------------------------
void ReReadDipSwitch(int NmbParams, u8 CmdData) {

    ProcessDipFunctions();
    DisplayDipFunctions();
    StringAndPrompt("OK");
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------
//  READ AND PROCESS DIP SWITCH FUNCTIONS  |
//---------------------------------------------------------------------------------------------------------------
void ProcessDipFunctions(void) {
/*      G+
    uint DipSwitch = Read_Dip_Switch();
    K9G_Mode       = DipSwitch & DIPSW_B_G_MODE  ? true  : false;
    DummyCntMode = DipSwitch & DIPSW_DUMMY_CNT ? false : true;
    K9BSE_Mode     = DipSwitch & DIPSW_BSE_MODE  ? false : true;
//  Cnt_Hist_Mode  = (!(DipSwitch & DIPSW_CNT_HIST_MODE)) or K9BSE_Mode ? true : false;
    if(DummyCntMode) {
        DipNotDefaultIndicationOn();
    } else {
        if((! PD_DIP_NON_DEFAULT)) DipNotDefaultIndicationOff();
    }
*/
}
//---------------------------------------------------------------------------------------------------------------




//----------------------------------------------
//  DISPLAY CURRENT DIP SWITCH FUNCTION STATE  |
//---------------------------------------------------------------------------------------------------------------
void DisplayDipFunctions(void) {
/*
    sprintf(DbMsgBuff, "\n\r K9B/G Mode:           %s", K9G_Mode       ? "K9G"   : "K9B");
    TxDebugStr(DbMsgBuff);
    sprintf(DbMsgBuff, "\n\r Count Mode:           %s", DummyCntMode ? "Dummy" : "Real");
    TxDebugStr(DbMsgBuff);
//  sprintf(DbMsgBuff, "\n\r Count History Mode:   %s", Cnt_Hist_Mode  ? "Yes"   : "No");
//  TxDebugStr(DbMsgBuff);
    sprintf(DbMsgBuff, "\n\r K9BSE Mode:           %s", K9BSE_Mode     ? "K9BSE" : "K9B/G");
    TxDebugStr(DbMsgBuff);
*/
}
//---------------------------------------------------------------------------------------------------------------




                                               //---- T ----//



//-------------------------------------------
//  RUN ANY SPECIAL TESTS:  "T1" "T2" "TT"  |
//---------------------------------------------------------------------------------------------------------------
//  If no special tests do nothing.
//---------------------------------------------------------------------------------------------------------------
void DispBrightness(uint Pct) {

    float fPct    = (float)Pct / 10.0;
    float fPwmPct = powf(fPct, 2.2);
    int   PwmPct  = round(fPwmPct / 1.58);      // Normalize back to 0 - 100.
    sprintf(sBuff, "\r\nInput brightness Pct = %d, fPWMPct = %f, PWM value = %d", Pct, fPwmPct, PwmPct);
    TxDebugStr(sBuff);
    printf(sBuff);
}

void SpecialTest(int NmbParams, u8 CmdData) {

    switch(CmdData) {

        case cdT1:      // T1
/*
//          CAN_TST_TX_DOMINANT to drive low
//          CAN_TST_TX_RECESSIVE to drive high

            HWREG(CAN0_BASE + CAN_O_CTL) |= CAN_CTL_TEST;
            HWREG(CAN0_BASE + CAN_O_TST) |= CAN_TST_TX_DOMINANT;        // Drive low.
*/

            u8 Msg[8];

            Msg[0] = 1;
            Msg[1] = 2;
            Msg[2] = 3;
            Msg[3] = 4;
            Msg[4] = 5;
            Msg[5] = 6;
            Msg[6] = 7;
            Msg[7] = 8;

//          SendCan1Msg(100, Msg, 8);
            SendCan2Msg(100, Msg, 8);

            break;

        case cdT2:      // T2
/*
//          sprintf(sBuff, "\r\n- Control = 0x%X", CANStatusGet(CAN0_BASE, CAN_STS_CONTROL));
            sprintf(sBuff, "\r\n- Control = 0x%X", *(volatile uint32_t *)0x40040000);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n- TxReq   = 0x%X", CANStatusGet(CAN0_BASE, CAN_STS_TXREQUEST));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n- NewDat  = 0x%X", CANStatusGet(CAN0_BASE, CAN_STS_NEWDAT));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n- MsgVal  = 0x%X\r\n", CANStatusGet(CAN0_BASE, CAN_STS_MSGVAL));
            TxDebugStr(sBuff);
            */
/*
            sprintf(sBuff, "\r\n - 0 Control = 0x%X", *(volatile uint32_t *)0x40040000);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n - 0 Status  = 0x%X", *(volatile uint32_t *)0x40040004);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n - 0 ErrCnt  = 0x%X", *(volatile uint32_t *)0x40040008);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n - 0 Interr  = 0x%X\r\n", *(volatile uint32_t *)0x40040010);
            TxDebugStr(sBuff);

            sprintf(sBuff, "\r\n - 1 Control = 0x%X", *(volatile uint32_t *)0x40041000);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n - 1 Status  = 0x%X", *(volatile uint32_t *)0x40041004);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n - 1 ErrCnt  = 0x%X", *(volatile uint32_t *)0x40041008);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n - 1 Interr  = 0x%X\r\n", *(volatile uint32_t *)0x40041010);
            TxDebugStr(sBuff);
*/
/*
            sprintf(sBuff, "\r\n  CANCTL0 = 0x%X", HWREG(CAN0_BASE + CAN_O_CTL));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n  CANTST0 = 0x%X", HWREG(CAN0_BASE + CAN_O_TST));
            TxDebugStr(sBuff);
*/
            sprintf(sBuff, "\r\n  GPIOAFSEL = 0x%X", GPIO_PORTB_AFSEL_R);
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n  GPIOPCTL  = 0x%X", GPIO_PORTB_PCTL_R);
            TxDebugStr(sBuff);

            break;

        case cdT3:      // T3

            sprintf(sBuff, "\r\n  CAN Status Reg         = 0x%X", HWREG(CAN0_BASE + CAN_O_STS));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n  CAN Interface Msg Req  = 0x%X", HWREG(CAN0_BASE + CAN_O_IF1CRQ));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n  CAN Interface Msg Ctrl = 0x%X", HWREG(CAN0_BASE + CAN_O_IF1MCTL));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n  CAN TxReq Register 1   = 0x%X", HWREG(CAN0_BASE + CAN_O_TXRQ1));
            TxDebugStr(sBuff);
            sprintf(sBuff, "\r\n  CAN TxReq Register 2   = 0x%X\r\n", HWREG(CAN0_BASE + CAN_O_TXRQ2));
            TxDebugStr(sBuff);
/*
            //          CAN_TST_TX_DOMINANT to drive low
            //          CAN_TST_TX_RECESSIVE to drive high

            HWREG(CAN0_BASE + CAN_O_CTL) |= CAN_CTL_TEST;
            HWREG(CAN0_BASE + CAN_O_TST) |= CAN_TST_TX_RECESSIVE;        // Drive high.
*/
//          Can2LoopbackTest();

//          Tmp();      // **FIXME** Display size of LED structure arrays.
/*
            DispBrightness(0);
            DispBrightness(25);
            DispBrightness(50);
            DispBrightness(75);
            DispBrightness(100);
*/
            break;

        default:
            sprintf(sBuff, "= %d, SpecialTest()", CmdData);
            Error(INVAL_SWITCH_VALUE, sBuff);
    }

//----------------------

    StringAndPrompt(DbMsgBuff);
}
//---------------------------------------------------------------------------------------------------------------






                        ///////////////////////////////////////////
                        //                                       //
                        //  PARAMETER CONVERSION AND VALIDATION  //
                        //                                       //
                        ///////////////////////////////////////////




/*  Used on Power/Detector Board.
//-----------------------------------------------------------------------
//  VALIDATE 1 INTEGER OR FP AND OPTIONALLY A SECOND INTEGER PARAMETER  |
//---------------------------------------------------------------------------------------------------------------
bool Val1FpOrIntAndOpt2ondIntParams(int NmbParams, float* pFirstParam, uint* pSecondParam, eCntType CountType) {

    char Command[MaxCmndSize+1];			// Mustn't forget the trailing \0.

    if(CountType == DISC_COUNTS and NmbParams == 0) return(SUCCESS);
    if(CountType == STAT_COUNTS and (NmbParams > 2 or NmbParams == 0)) {
	InvalNmbParams();
	return(FAILURE);
    } else if(NmbParams > 2) {				// Count Type == DISC_COUNTS.
	InvalNmbParams();
	return(FAILURE);
    }
    if(NmbParams == 1) {
	if(sscanf(CopyBuff, " %s %f", Command, pFirstParam) <2) {
	    IllFormedNumber();
	    return(FAILURE);
	}
    } else {						// NmbParams==2
	if(sscanf(CopyBuff, " %s %f %d", Command, pFirstParam, pSecondParam) < 3) {
	    IllFormedNumber();
	    return(FAILURE);
	}
    }
    if(*pFirstParam <= 0.0) {
	InvalNegOrZeroParam();
	return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------
*/



//------------------------------------------------------
//  VALIDATE DECIMAL, HEXADECIMAL, DECIMAL PARAMETERS  |
//---------------------------------------------------------------------------------------------------------------
bool ValDecHexDecParam(int NmbParams, uint* pParam1, uint* pParam2, uint* pParam3) {

    char Command[MaxCmndSize+1];   // Mustn't forget the trailing \0. We scan the whole line.

    if(NmbParams != 3) {
        InvalNmbParams();
        return(FAILURE);
    }
    if(sscanf(CopyBuff, "%s %u %x %u", Command, pParam1, pParam2, pParam3) < 4) {
        IllFormedNumber();
        return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------------------
//  VALIDATE 1 AND OPTIONALLY 2 UNSIGNED INTEGER PARAMETERS  |
//---------------------------------------------------------------------------------------------------------------
bool Validate1or2UintParams(int NmbParams, uint* pFirstParam, uint* pSecondParam) {

    char Command[MaxCmndSize+1];   // Mustn't forget the trailing \0. We scan the whole line.

    if(NmbParams == 1) {
	if(sscanf(CopyBuff, " %s %u", Command, pFirstParam) < 2) {
	    IllFormedNumber();
	    return(FAILURE);
	}
    } else if(NmbParams == 2) {
	if(sscanf(CopyBuff, " %s %u %u", Command, pFirstParam, pSecondParam) < 3) {
	    IllFormedNumber();
	    return(FAILURE);
	}
    } else {
	InvalNmbParams();
	return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------------------------------
//  VALIDATE 1 AND OPTIONALLY 2 UNSIGNED HEXADECIMAL INTEGER PARAMETERS  | v8.1
//---------------------------------------------------------------------------------------------------------------
bool Validate1or2HexParams(int NmbParams, uint* pFirstParam, uint* pSecondParam) {

    char Command[MaxCmndSize+1];                // Mustn't forget the trailing \0. We scan the whole line.

    if(NmbParams == 1) {
        if(sscanf(CopyBuff, " %s %x", Command, pFirstParam) < 2) {
            IllFormedNumber();
            return(FAILURE);
        }
    } else if(NmbParams == 2) {
        if(sscanf(CopyBuff, " %s %x %x", Command, pFirstParam, pSecondParam) < 3) {
            IllFormedNumber();
            return(FAILURE);
        }
    } else {
        InvalNmbParams();
        return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------------------------------------------------------
//  VALIDATE UNSIGNED HEXADECIMAL INTEGER PARAMETER 1 AND DECIMAL PARAMETER 2  | v8.1
//---------------------------------------------------------------------------------------------------------------
bool ValidateHexDecParams(uint* pFirstParam, uint* pSecondParam) {

    char Command[MaxCmndSize+1];                // Mustn't forget the trailing \0. We scan the whole line.

    if(sscanf(CopyBuff, " %s %x %d", Command, pFirstParam, pSecondParam) < 3) {
        IllFormedNumber();
        return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------




//-------------------------------------------
//  VALIDATE 3 UNSIGNED INTEGER PARAMETERS  |
//---------------------------------------------------------------------------------------------------------------
bool Validate3UintParams(uint* pParam1, uint* pParam2, uint* pParam3) {

    char Command[MaxCmndSize+1];                // Mustn't forget the trailing \0. We scan the whole line.

    if(sscanf(CopyBuff, " %s %u %u %u", Command, pParam1, pParam2, pParam3) < 4) {
        IllFormedNumber();
        return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------




//-------------------------------------------
//  VALIDATE 4 UNSIGNED INTEGER PARAMETERS  |
//---------------------------------------------------------------------------------------------------------------
bool Validate4UintParams(uint* pParam1, uint* pParam2, uint* pParam3, uint* pParam4) {

    char Command[MaxCmndSize+1];                // Mustn't forget the trailing \0. We scan the whole line.

    if(sscanf(CopyBuff, " %s %u %u %u %u", Command, pParam1, pParam2, pParam3, pParam4) < 5) {
        IllFormedNumber();
        return(FAILURE);
    }
    return(SUCCESS);
}
//---------------------------------------------------------------------------------------------------------------




//--------------------------------
//  INVALID NUMER OF PARAMETERS  |
//---------------------------------------------------------------------------------------------------------------
void InvalNmbParams(void) {

    StringAndPrompt("Invalid Number of Parameters.\a\r");
}
//---------------------------------------------------------------------------------------------------------------




//------------------------------
//  INVALID ZERO PARAMETER(S)  |
//---------------------------------------------------------------------------------------------------------------
void InvalNegOrZeroParam(void) {

    StringAndPrompt("Invalid Negative or Zero Parameter(s).\a\r");
}
//---------------------------------------------------------------------------------------------------------------




//----------------------
//  ILL-FORMED NUMBER  |
//---------------------------------------------------------------------------------------------------------------
void IllFormedNumber(void) {

    StringAndPrompt("Ill-Formed Number(s).\a\r");
}
//---------------------------------------------------------------------------------------------------------------




//-----------------------------------------------------
//  ONE OF THE PARAMETERS IS NOT FORMATTED CORRECTLY  |
//---------------------------------------------------------------------------------------------------------------
void InvalValue(void) {

    StringAndPrompt("Invalid Format of One of the Parameters.\a\r");
}
//---------------------------------------------------------------------------------------------------------------



