//*****************************************************************************
//
// pinout.c - Function to configure the device pins on the DK-TM4C129X.
//
// Copyright (c) 2013-2015 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.1.1.71 of the DK-TM4C129X Firmware Package.
//
//*****************************************************************************

#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_gpio.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "driverlib/epi.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/rom.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "drivers/pinout.h"

// FPGA Adressbereich (16 BIT ZUGRIFF)
volatile unsigned short * epiFPGA  = (unsigned short *)0xA0000000;

//*****************************************************************************
//
//! Configures the device pins for the standard usages on the DK-TM4C129X.
//!
//! This function enables the GPIO modules and configures the device pins for
//! the default, standard usages on the DK-TM4C129X.  Applications that require
//! alternate configurations of the device pins can either not call this
//! function and take full responsibility for configuring all the device pins,
//! or can reconfigure the required device pins after calling this function.
//!
//! \return None.
//
//*****************************************************************************
void
PinoutSet(uint32_t SystemClk)
{
	//
    // Enable all the GPIO peripherals.
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOC);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOG);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOH);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOJ);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOK);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOL);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOM);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPION);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOP);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOQ);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOR);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOS);
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOT);

    //
    // Init. EPI0 Bus (HB16) (20 bits + CE0[26], CE1[27], RD[28], WR[29], ALE[30])
    // or
    // Init. EPI0 Bus (General Purpose) (16 bits Data [0..15] + 12 bits Addr [16..27], WR[28], RD[29], FRAME[30], CLK[31])
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_EPI0);

    ROM_GPIOPinConfigure(GPIO_PA6_EPI0S8);
    ROM_GPIOPinConfigure(GPIO_PA7_EPI0S9);
	GPIOPinTypeEPI(GPIO_PORTA_BASE, EPI_PORTA_PINS);

	ROM_GPIOPinConfigure(GPIO_PB2_EPI0S27);
	ROM_GPIOPinConfigure(GPIO_PB3_EPI0S28);
	GPIOPinTypeEPI(GPIO_PORTB_BASE, EPI_PORTB_PINS);

	ROM_GPIOPinConfigure(GPIO_PC4_EPI0S7);
	ROM_GPIOPinConfigure(GPIO_PC5_EPI0S6);
	ROM_GPIOPinConfigure(GPIO_PC6_EPI0S5);
	ROM_GPIOPinConfigure(GPIO_PC7_EPI0S4);
	GPIOPinTypeEPI(GPIO_PORTC_BASE, EPI_PORTC_PINS);

	ROM_GPIOPinConfigure(GPIO_PG0_EPI0S11);
	ROM_GPIOPinConfigure(GPIO_PG1_EPI0S10);
	GPIOPinTypeEPI(GPIO_PORTG_BASE, EPI_PORTG_PINS);

	ROM_GPIOPinConfigure(GPIO_PH0_EPI0S0);
	ROM_GPIOPinConfigure(GPIO_PH1_EPI0S1);
	ROM_GPIOPinConfigure(GPIO_PH2_EPI0S2);
	ROM_GPIOPinConfigure(GPIO_PH3_EPI0S3);
	GPIOPinTypeEPI(GPIO_PORTH_BASE, EPI_PORTH_PINS);

	ROM_GPIOPinConfigure(GPIO_PK5_EPI0S31);
	GPIOPinTypeEPI(GPIO_PORTK_BASE, EPI_PORTK_PINS);

	ROM_GPIOPinConfigure(GPIO_PL0_EPI0S16);
	ROM_GPIOPinConfigure(GPIO_PL1_EPI0S17);
	ROM_GPIOPinConfigure(GPIO_PL2_EPI0S18);
	ROM_GPIOPinConfigure(GPIO_PL3_EPI0S19);
	ROM_GPIOPinConfigure(GPIO_PL4_EPI0S26);
	GPIOPinTypeEPI(GPIO_PORTL_BASE, EPI_PORTL_PINS);

	ROM_GPIOPinConfigure(GPIO_PM0_EPI0S15);
	ROM_GPIOPinConfigure(GPIO_PM1_EPI0S14);
	ROM_GPIOPinConfigure(GPIO_PM2_EPI0S13);
	ROM_GPIOPinConfigure(GPIO_PM3_EPI0S12);
	GPIOPinTypeEPI(GPIO_PORTM_BASE, EPI_PORTM_PINS);

	ROM_GPIOPinConfigure(GPIO_PP2_EPI0S29);
	ROM_GPIOPinConfigure(GPIO_PP3_EPI0S30);
	GPIOPinTypeEPI(GPIO_PORTP_BASE, EPI_PORTP_PINS);

	// General Purpose Init
	ROM_EPIDividerSet(EPI0_BASE, 0x00010001);		// 1/2 (System) = 120MHz * 0.5 = 60MHz
	ROM_EPIModeSet(EPI0_BASE, EPI_MODE_GENERAL);	// General Purpose
	ROM_EPIConfigGPModeSet(EPI0_BASE, EPI_GPMODE_CLKPIN | EPI_GPMODE_ASIZE_12 | EPI_GPMODE_DSIZE_16, 0, 0);
	ROM_EPIAddressMapSet(EPI0_BASE, EPI_ADDR_PER_SIZE_256B | EPI_ADDR_PER_BASE_A);


//		HB16 version
/*
	ROM_EPIDividerSet(EPI0_BASE, 0x00050005);	// Same divider for both CS
	ROM_EPIModeSet(EPI0_BASE, EPI_MODE_HB16);
	ROM_EPIConfigHB16Set(EPI0_BASE,	EPI_HB16_CSCFG_ALE_DUAL_CS | EPI_HB16_CSBAUD | EPI_HB16_WRWAIT_1 | EPI_HB16_RDWAIT_1 | EPI_HB16_BURST_TRAFFIC, 0xFF);
	ROM_EPIConfigHB16CSSet(EPI0_BASE,0,EPI_HB16_ALE_HIGH);
	ROM_EPIConfigHB16CSSet(EPI0_BASE,1,EPI_HB16_ALE_HIGH);

	ROM_EPIAddressMapSet(EPI0_BASE, EPI_ADDR_PER_SIZE_16MB | EPI_ADDR_PER_BASE_A |		// FPGA 16MB @ 0xA0000000
									EPI_ADDR_RAM_SIZE_16MB | EPI_ADDR_RAM_BASE_6 );		// SRAM 16MB @ 0x60000000
*/
    //
    // GPIO Init.
    //
    ROM_GPIOPinTypeGPIOInput	(GPIO_PORTS_BASE, GPIO_PIN_6);			// PIN_FPGA_INIT
    ROM_GPIOPinTypeGPIOInput	(GPIO_PORTT_BASE, GPIO_PIN_1);			// PIN_FPGA_DONE
    ROM_GPIOPinTypeGPIOOutput	(GPIO_PORTT_BASE, GPIO_PIN_0);			// POUT_FPGA_PROG
    POUT_FPGA_PROG	= INDX_FPGA_PROG;
    ROM_GPIOPinTypeGPIOOutput	(GPIO_PORTS_BASE, GPIO_PIN_7);			// POUT_FLASH_CS
	POUT_FLASH_CS	= INDX_FLASH_CS;


    ROM_GPIOPinTypeGPIOOutput	(GPIO_PORTN_BASE, GPIO_PIN_5);			// POUT_LED_R
    POUT_LED_R	= 0;
    ROM_GPIOPinTypeGPIOOutput	(GPIO_PORTQ_BASE, GPIO_PIN_7);			// POUT_LED_G
    POUT_LED_G	= 0;
    ROM_GPIOPinTypeGPIOOutput	(GPIO_PORTQ_BASE, GPIO_PIN_4);			// POUT_LED_B
    POUT_LED_B	= 0;

	ROM_GPIOPinTypeGPIOOutput	(GPIO_PORTJ_BASE, GPIO_PIN_2);			// PWR 1
	POUT_PWR_1		= INDX_PWR_1;

    //
    // PA0-1 are used for UART0.
    //
    ROM_GPIOPinConfigure(GPIO_PA0_U0RX);
    ROM_GPIOPinConfigure(GPIO_PA1_U0TX);
    ROM_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // PA2-5 are used for SSI0 to the second booster pack.
    //
    ROM_GPIOPinConfigure(GPIO_PA2_SSI0CLK);
    ROM_GPIOPinConfigure(GPIO_PA3_SSI0FSS);
    ROM_GPIOPinConfigure(GPIO_PA4_SSI0XDAT0);
    ROM_GPIOPinConfigure(GPIO_PA5_SSI0XDAT1);

    //
    // PG4-5-7 are used for SSI2 to the Serial FLASH (MCS File)
    //
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI2);
//    ROM_SysCtlPeripheralReset(SYSCTL_PERIPH_SSI2);
    ROM_GPIOPinConfigure(GPIO_PG7_SSI2CLK);
    ROM_GPIOPinConfigure(GPIO_PG5_SSI2XDAT0);
    ROM_GPIOPinConfigure(GPIO_PG4_SSI2XDAT1);
    ROM_GPIOPinTypeSSI(GPIO_PORTG_BASE, GPIO_PIN_7 | GPIO_PIN_5 | GPIO_PIN_4);
    ROM_SSIConfigSetExpClk(SSI2_BASE, SystemClk, SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, 2000000, 8);
    ROM_SSIEnable(SSI2_BASE);

    //
    // PB0-1/PD6-7/PL6-7 are used for USB.
    //
    HWREG(GPIO_PORTD_BASE + GPIO_O_LOCK) = GPIO_LOCK_KEY;
    HWREG(GPIO_PORTD_BASE + GPIO_O_CR) = 0xff;
    ROM_GPIOPinConfigure(GPIO_PD6_USB0EPEN);
    ROM_GPIOPinConfigure(GPIO_PD7_USB0PFLT);
    ROM_GPIOPinTypeUSBAnalog(GPIO_PORTB_BASE, GPIO_PIN_0 | GPIO_PIN_1);
    ROM_GPIOPinTypeUSBDigital(GPIO_PORTD_BASE, GPIO_PIN_6 | GPIO_PIN_7);
    ROM_GPIOPinTypeUSBAnalog(GPIO_PORTL_BASE, GPIO_PIN_6 | GPIO_PIN_7);

    //
    // PB6-7 are used for I2C to the TMP100 and the EM connector.
    //
    ROM_GPIOPinConfigure(GPIO_PB6_I2C6SCL);
    ROM_GPIOPinConfigure(GPIO_PB7_I2C6SDA);
    ROM_GPIOPinTypeI2CSCL(GPIO_PORTB_BASE, GPIO_PIN_6);
    ROM_GPIOPinTypeI2C(GPIO_PORTB_BASE, GPIO_PIN_7);

    //
    // PE5/PN3/PP1 are used for the push buttons.
    //
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTE_BASE, GPIO_PIN_5);
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTN_BASE, GPIO_PIN_3);
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTP_BASE, GPIO_PIN_1);

    //
    // PE7/PP7/PT2-3 are used for the touch screen.
    //
    HWREG(GPIO_PORTE_BASE + GPIO_O_LOCK) = GPIO_LOCK_KEY;
    HWREG(GPIO_PORTE_BASE + GPIO_O_CR) = 0xff;
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTE_BASE, GPIO_PIN_7);
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTP_BASE, GPIO_PIN_7);
    ROM_GPIOPinTypeGPIOInput(GPIO_PORTT_BASE, GPIO_PIN_2 | GPIO_PIN_3);
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTE_BASE, GPIO_PIN_7);
    ROM_GPIOPinWrite(GPIO_PORTE_BASE, GPIO_PIN_7, 0);
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTP_BASE, GPIO_PIN_7);
    ROM_GPIOPinWrite(GPIO_PORTP_BASE, GPIO_PIN_7, 0);
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTT_BASE, GPIO_PIN_2 | GPIO_PIN_3);
    ROM_GPIOPinWrite(GPIO_PORTT_BASE, GPIO_PIN_2 | GPIO_PIN_3, 0);

    //
    // PF0/PF4-5/PH4/PQ0-2 are used for the SPI flash (on-board and SD card).
    // PH4 selects the SD card and PQ1 selects the on-board SPI flash.
    //
    ROM_GPIOPinConfigure(GPIO_PF0_SSI3XDAT1);
    ROM_GPIOPinConfigure(GPIO_PF4_SSI3XDAT2);
    ROM_GPIOPinConfigure(GPIO_PF5_SSI3XDAT3);
    ROM_GPIOPinConfigure(GPIO_PQ0_SSI3CLK);
    ROM_GPIOPinConfigure(GPIO_PQ2_SSI3XDAT0);
    ROM_GPIOPinTypeSSI(GPIO_PORTF_BASE, GPIO_PIN_0 | GPIO_PIN_4 | GPIO_PIN_5);
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTH_BASE, GPIO_PIN_4);
    ROM_GPIOPinWrite(GPIO_PORTH_BASE, GPIO_PIN_4, GPIO_PIN_4);
    ROM_GPIOPinTypeSSI(GPIO_PORTQ_BASE, GPIO_PIN_0 | GPIO_PIN_2);
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTQ_BASE, GPIO_PIN_1);
    ROM_GPIOPinWrite(GPIO_PORTQ_BASE, GPIO_PIN_1, GPIO_PIN_1);

    //
    // PF1/PK4/PK6 are used for Ethernet LEDs.
    //
    ROM_GPIOPinConfigure(GPIO_PF1_EN0LED2);
    ROM_GPIOPinConfigure(GPIO_PK4_EN0LED0);
    ROM_GPIOPinConfigure(GPIO_PK6_EN0LED1);
    GPIOPinTypeEthernetLED(GPIO_PORTF_BASE, GPIO_PIN_1);
    GPIOPinTypeEthernetLED(GPIO_PORTK_BASE, GPIO_PIN_4);
    GPIOPinTypeEthernetLED(GPIO_PORTK_BASE, GPIO_PIN_6);

    //
    // PF6-7/PJ6/PS4-5/PR0-7 are used for the LCD.
    //
    ROM_GPIOPinConfigure(GPIO_PF7_LCDDATA02);
    ROM_GPIOPinConfigure(GPIO_PJ6_LCDAC);
    ROM_GPIOPinConfigure(GPIO_PR0_LCDCP);
    ROM_GPIOPinConfigure(GPIO_PR1_LCDFP);
    ROM_GPIOPinConfigure(GPIO_PR2_LCDLP);
    ROM_GPIOPinConfigure(GPIO_PR3_LCDDATA03);
    ROM_GPIOPinConfigure(GPIO_PR4_LCDDATA00);
    ROM_GPIOPinConfigure(GPIO_PR5_LCDDATA01);
    ROM_GPIOPinConfigure(GPIO_PR6_LCDDATA04);
    ROM_GPIOPinConfigure(GPIO_PR7_LCDDATA05);
    ROM_GPIOPinConfigure(GPIO_PS4_LCDDATA06);
    ROM_GPIOPinConfigure(GPIO_PS5_LCDDATA07);
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE, GPIO_PIN_6);
    ROM_GPIOPinWrite(GPIO_PORTF_BASE, GPIO_PIN_6, GPIO_PIN_6);
    GPIOPinTypeLCD(GPIO_PORTF_BASE, GPIO_PIN_7);
    GPIOPinTypeLCD(GPIO_PORTJ_BASE, GPIO_PIN_6);
    GPIOPinTypeLCD(GPIO_PORTR_BASE,
                       (GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2 | GPIO_PIN_3 |
                        GPIO_PIN_4 | GPIO_PIN_5 | GPIO_PIN_6 | GPIO_PIN_7));
    GPIOPinTypeLCD(GPIO_PORTS_BASE, GPIO_PIN_4 | GPIO_PIN_5);
}

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************
