/*
 * Copyright (c) 2015-2018 Texas Instruments Incorporated - http://www.ti.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 *  ======== DFU.c ========
 *  Application entry point to re-enumerate the USB device in DFU mode
 *
 *  Softare:  TivaWare C Series 2.1.1.71b
 *  Hardware: EK-TM4C1294XL LaunchPad
 *
 *  This file uses ROM bindings which are device variant specific. Must
 *  define the following symbols when compiling this file:
 *
 *  TARGET_IS_TM4C129_RA0
 *  PART_TM4C1294NCPDT
 *  TIVAWARE
 */

#include <stdbool.h>                            /* needed by sysctl.h */
#include <stdint.h>

#include <inc/hw_memmap.h>
#include <inc/hw_nvic.h>
#include <inc/hw_types.h>
#include <driverlib/debug.h>
#include <driverlib/rom.h>
#include <driverlib/rom_map.h>
#include <driverlib/sysctl.h>
#include <driverlib/systick.h>
#include <driverlib/usb.h>
#include <usblib/usblib.h>
#include <usblib/device/usbdevice.h>

#include "DFU.h"


/*
 *  ======== DFU_renumerate ========
 */
void DFU_renumerate(void)
{
    uint32_t ui32SysClock;

    /* terminate the usb device controller and detach from the bus */
    USBDCDTerm(0);

    /* disable all interrupts */
    ROM_IntMasterDisable();

    /* disable SysTick and its interrupt */
    ROM_SysTickIntDisable();
    ROM_SysTickDisable();

    /*  Disable all processor interrupts. Instead of disabling them
     *  one at a time, a direct write to NVIC is done to disable all
     *  peripheral interrupts.
     */
    HWREG(NVIC_DIS0) = 0xffffffff;
    HWREG(NVIC_DIS1) = 0xffffffff;
    HWREG(NVIC_DIS2) = 0xffffffff;
    HWREG(NVIC_DIS3) = 0xffffffff;
    HWREG(NVIC_DIS4) = 0xffffffff;

    /* run from the pll at 120 MHz */
    ui32SysClock = MAP_SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ | SYSCTL_OSC_MAIN
            | SYSCTL_USE_PLL | SYSCTL_CFG_VCO_480), 120000000);

    /* enable and reset the usb peripheral */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_USB0);
    ROM_SysCtlPeripheralReset(SYSCTL_PERIPH_USB0);
    ROM_USBClockEnable(USB0_BASE, 8, USB_CLOCK_INTERNAL);

    /* wait for about a second */
    ROM_SysCtlDelay(ui32SysClock / 3);

    /* re-enable interrupts at the NVIC level */
    ROM_IntMasterEnable();

    /* call the usb boot loader */
    ROM_UpdateUSB(0);

    /* should never get here */
    while (true);
}
