//*****************************************************************************
//
// Copyright (c) 2012-2020 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.2.0.295 of the EK-TM4C1294XL Firmware Package.
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "driverlib/sysctl.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/ssi.h"

//*****************************************************************************
//
// Define pin to LED mapping.
//
//*****************************************************************************

//*****************************************************************************
//
//! \addtogroup example_list
//! <h1>Project Zero (project0)</h1>
//!
//! This example demonstrates the use of TivaWare to setup the clocks and
//! toggle GPIO pins to make the LED blink. This is a good place to start
//! understanding your launchpad and the tools that can be used to program it.
//
//*****************************************************************************

#define CMD_WR 0x00
#define CMD_RD 0x80

//*****************************************************************************
//
// The error routine that is called if the driver library encounters an error.
//
//*****************************************************************************
#ifdef DEBUG
void
__error__(char *pcFilename, uint32_t ui32Line)
{
}
#endif

const char string[] = "Now is the time for all good men to come to the aid of their country.\r\n";
unsigned char receiveBuffer[128];
//*****************************************************************************
//
// Main 'C' Language entry point.  Toggle an LED using TivaWare.
//
//*****************************************************************************
int
main(void)
{
    uint32_t ui32SysClock, i, temp;

    //
    // Run from the PLL at 120 MHz.
    // Note: SYSCTL_CFG_VCO_240 is a new setting provided in TivaWare 2.2.x and
    // later to better reflect the actual VCO speed due to SYSCTL#22.
    //
    ui32SysClock = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ |
                                       SYSCTL_OSC_MAIN |
                                       SYSCTL_USE_PLL |
                                       SYSCTL_CFG_VCO_240), 120000000);

    //
    // Enable and wait for the port to be ready for access
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_GPIOA))
    {
    }
    
    //
    // Configure the GPIO port for the Quad SPI operation.
    //
    GPIOPinConfigure(GPIO_PA2_SSI0CLK);
    GPIOPinConfigure(GPIO_PA3_SSI0FSS);
    GPIOPinConfigure(GPIO_PA4_SSI0XDAT0);
    GPIOPinConfigure(GPIO_PA5_SSI0XDAT1);
    GPIOPinConfigure(GPIO_PA6_SSI0XDAT2);
    GPIOPinConfigure(GPIO_PA7_SSI0XDAT3);
    GPIOPinTypeSSI(GPIO_PORTA_BASE, GPIO_PIN_2 | GPIO_PIN_3 | GPIO_PIN_4 |
                                    GPIO_PIN_5 | GPIO_PIN_6 | GPIO_PIN_7);
    // Configure SSI0 as Quad Master at 2M baud
    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);
    while(!SysCtlPeripheralReady(SYSCTL_PERIPH_SSI0))
    {
    }
    SSIConfigSetExpClk(SSI0_BASE, ui32SysClock, SSI_FRF_MOTO_MODE_0,
                       SSI_MODE_MASTER, 2000000, 8);
    SSIEnable(SSI0_BASE);
    //
    // Loop Forever
    //
    while(1)
    {
        // Send write command
        SSIAdvModeSet(SSI0_BASE, SSI_ADV_MODE_QUAD_WRITE);
        SSIAdvFrameHoldEnable(SSI0_BASE);
        SSIAdvDataPutFrameEnd(SSI0_BASE, CMD_WR | sizeof(string));
        for(i = 0; i < (sizeof(string) - 1); i++)
        {
            SSIDataPut(SSI0_BASE, string[i]);
        }
        SSIAdvDataPutFrameEnd(SSI0_BASE, string[i]);

        // Send read command
        SSIAdvModeSet(SSI0_BASE, SSI_ADV_MODE_QUAD_WRITE);
        SSIAdvFrameHoldEnable(SSI0_BASE);
        SSIAdvDataPutFrameEnd(SSI0_BASE, CMD_RD | sizeof(string));
        SSIAdvModeSet(SSI0_BASE, SSI_ADV_MODE_QUAD_READ);
        for(i = 0; i < (sizeof(string) - 1); i++)
        {
            SSIDataPut(SSI0_BASE, 0);
            SSIDataGet(SSI0_BASE, &temp);
            receiveBuffer[i] = (unsigned char)(temp & 0xFF);
        }
        SSIAdvDataPutFrameEnd(SSI0_BASE, string[i]);

    }
}
