/*
 * network_interface.cpp
 *
 * This class serves as a wrapper over Ethernet and Wifi.  Wifi is provided
 * by a third party chip.  Ethernet is provided by a TIVA peripheral through
 * TI NDK.  TI NDK has the annoying requirement that all access to a particular
 * socket must be performed within the same task else the software blows-up/hangs.
 *
 *  Created on: Jun 22, 2016
 *      Author: yli
 */


#include "network_interface.h"
#include "ethernet.h"
#include "message_handler.h"

#include <ti/sysbios/BIOS.h>
#include <xdc/cfg/global.h>
#include <ti/sysbios/knl/Semaphore.h>
#include <ti/sysbios/knl/task.h>

uint8_t NetworkInterface::m_network_rcv_buf[NETWORK_RCV_BUF_SZ];
uint16_t NetworkInterface::m_tcp_port_num = TCP_PORT_NUM;
bool NetworkInterface::m_are_ethernet_sockets_opened = false;

NetworkInterface::NetworkInterface() {}

void NetworkInterface::MaintainLink()
{
    if(m_are_ethernet_sockets_opened == false)
    {
        OpenSockets();
    }
    // monitor Link status and restart DHCP if needed
    Ethernet::MonitorLink();
}

void NetworkInterface::SendReceive()
{
    NetworkData net_data;

    net_data.interface = ETHERNET_INTF;
    net_data.p_data = m_network_rcv_buf;
    if(Ethernet::get_status() == Ethernet::SOCKETS_OPEN)
    {
        // manage new socket requests
        if(Ethernet::TCPListen(m_tcp_port_num) == true)
        {
            int32_t client_handle;
            uint32_t client_addr;
            uint16_t client_port;

            Ethernet::TCPAccept(m_tcp_port_num, &client_addr, &client_port, &client_handle);
        }

        // receive ethernet packets
        net_data.data_size = Ethernet::TCPReceive( &(net_data.port_num), m_network_rcv_buf, sizeof(m_network_rcv_buf),
                &(net_data.client_addr), &(net_data.client_id) );

        if(net_data.data_size > 0)
        {
            MessageHandler::HandleNewMessage(&net_data);
        }

        net_data.data_size = Ethernet::UDPRecv(UDP_PORT_NUM, m_network_rcv_buf, sizeof(m_network_rcv_buf),
                &(net_data.client_addr), &(net_data.remote_port_num) );

        if(net_data.data_size > 0)
        {
            net_data.p_data = m_network_rcv_buf;
            MessageHandler::HandleNewMessage(&net_data);
        }

        // send outgoing data
        Ethernet::SendData();
    }
}

void NetworkInterface::HandleIPAddrAssignedEvent(Interface_t interface)
{
    // an IP address has been assigned. open TCP ports
    m_tcp_port_num = TCP_PORT_NUM;
}

void NetworkInterface::OnEthernetDisconnect(uint32_t dummy_val)
{
    m_are_ethernet_sockets_opened = false;
}

bool NetworkInterface::OpenSockets()
{
    bool result = false;
    bool is_ethernet_active = Ethernet::get_is_link_active();

    // ethernet
    if( (m_are_ethernet_sockets_opened == false) && (is_ethernet_active == true) )
    {
        // if network is not up, attempt to start it
        if(Ethernet::get_status() != Ethernet::SOCKETS_OPEN)
        {
            // NDK has the annoying requirement that all access to a particular socket be made from the same thread.
            // As such this function must be called within the context of the Network thread (task_network_comm) else
            // TI-RTOS blows up and the TIVA hangs.
            result = Ethernet::OpenSockets(UDP_PORT_NUM, m_tcp_port_num, task_ethernet);
            if(result == true)
            {
                m_are_ethernet_sockets_opened = true;
            }
            else
            {
                Task_sleep(1000);
                m_tcp_port_num++;
                if(m_tcp_port_num > MAX_TCP_PORT_NUM)
                {
                    m_tcp_port_num = TCP_PORT_NUM;
                }
            }
        }
        else //socket is already running
        {
            m_are_ethernet_sockets_opened = true;
        }
    }

    return(result);
}

uint8_t NetworkInterface::get_num_ports_avail(uint16_t *p_port_num, Interface_t interface)
{
    uint8_t num_ports_avail = 0;
    if(interface == ETHERNET_INTF)
    {
        num_ports_avail = Ethernet::get_num_ports_avail(p_port_num);
    }
    return(num_ports_avail);
}

void NetworkInterface::Reply(NetworkData *p_net_data)
{
    if(p_net_data->interface == ETHERNET_INTF)
    {
        EthernetSendCfg_t send_cfg;
        send_cfg.bytes_to_send = p_net_data->data_size;
        send_cfg.protocol = ETHERNET_TCP;
        send_cfg.client_idx = p_net_data->client_id;
        send_cfg.port_num = p_net_data->port_num;
        Ethernet::QueueToSend(p_net_data->p_data, &send_cfg);
    }
}

bool NetworkInterface::QueueToSend(NetworkData *p_net_data, Interface_t interface)
{
	bool result = true;
    if(interface == ETHERNET_INTF)
    {
        EthernetSendCfg_t send_cfg;
        send_cfg.bytes_to_send = p_net_data->data_size;
        send_cfg.protocol = p_net_data->protocol;
        send_cfg.client_idx = p_net_data->client_id;
        send_cfg.port_num = p_net_data->port_num;
        send_cfg.client_addr = p_net_data->client_addr;
        send_cfg.remote_port_num = p_net_data->remote_port_num;
        result = Ethernet::QueueToSend(p_net_data->p_data, &send_cfg);
    }
    return(result);
}

// Send data to all TCP clients
bool NetworkInterface::QueueToSend(uint8_t *p_data, uint32_t size, Interface_t interface)
{
	bool result = true;
    if(interface == ETHERNET_INTF)
    {
        EthernetSendCfg_t send_cfg;
        send_cfg.bytes_to_send = size;
        send_cfg.protocol = ETHERNET_TCP;
        send_cfg.client_idx = TCP_ALL;
        send_cfg.port_num = TCP_ALL;
        result = Ethernet::QueueToSend(p_data, &send_cfg);
    }
    return(result);
}

void NetworkInterface::NotifySocketConnected(EthTCPSocketData_t *p_socket_data)
{
    printf("Socket Connected\n");
}

uint32_t NetworkInterface::GetIPAddr()
{
    return(Ethernet::get_ip_addr());
}

