/*
 * Copyright (c) 2015, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */
/*
 *  ======== sockutils ========
 */
#include <xdc/std.h>
#include <xdc/runtime/System.h>

#include <ti/sysbios/knl/Task.h>

#include <ti/ndk/inc/netmain.h>
#include <ti/ndk/inc/_stack.h>

static char *States[] = { "CLOSED","LISTEN","SYNSENT","SYNRCVD",
                          "ESTABLISHED","CLOSEWAIT","FINWAIT1","CLOSING",
                          "LASTACK","FINWAIT2","TIMEWAIT" };

/*
 *  ======== SockUtils_printSockTable ========
 */
void SockUtils_printSockTable(unsigned int protocol)
{
    unsigned char *pBuf;
    int entries;
    int i;
    SOCKPCB *ppcb;
    char str[40];

    /* check argument passed and translate */
    if (protocol == IPPROTO_TCP) {
        protocol = SOCKPROT_TCP;
    }
    else if (protocol == IPPROTO_UDP) {
        protocol = SOCKPROT_UDP;
    }
    else {
        System_printf("SockUtils_printSockTable: Error: invalid protocol\n");
    }

    pBuf = mmBulkAlloc(2048);
    if (!pBuf) {
        System_printf("SockUtils_printSockTable: Error: out of memory\n");
        return;
    }

    /* Use llEnter / llExit since we're calling into the stack */
    llEnter();
    entries = SockGetPcb(protocol, 2048, pBuf);
    llExit();

    System_printf("\nLocal IP         LPort  Foreign IP       FPort  State\n");
    System_printf("---------------  -----  ---------------  -----  -----------\n");

    for (i = 0; i < entries; i++) {
        ppcb = (SOCKPCB *)(pBuf + (i * sizeof(SOCKPCB)));

        NtIPN2Str(ppcb->IPAddrLocal, str);
        System_printf("%-15s  %-5u  ", str, htons(ppcb->PortLocal));

        NtIPN2Str(ppcb->IPAddrForeign, str);
        System_printf("%-15s  %-5u", str, htons(ppcb->PortForeign));

        if (protocol == SOCKPROT_TCP) {
            System_printf("  %s\n", States[ppcb->State]);
        }
        else {
            System_printf("\n");
        }
    }
    System_printf("\n");

    mmBulkFree(pBuf);
}

static volatile int STEVE_printTable = 0;
void STEVE_printSktTsk()
{
    while (1) {
        if (STEVE_printTable) {
                SockUtils_printSockTable(IPPROTO_TCP);
        }
        else {
            Task_sleep(1000);
        }
    }
}
