/*
 * ethernet.cpp
 *
 *  Created on: Jun 15, 2016
 *      Author: yli
 */


#include "ethernet.h"
#include "tiva_emac.h"
#include "tiva_timer.h"
#include <sys/socket.h>
#include "ndk_interface.h"
#include "ndk_test_defines.h"
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Mailbox.h>
#include <ti/sysbios/knl/task.h>
#include <string.h>
#include <stdlib.h>
#include "network_interface.h"

#include "SockUtils.h"

#define MSG_RTC_DUPIP           260

extern Mailbox_Handle mbox_ethernet_send;


/*-----------------------------------------------------------------------------
* Config Tag: CFGTAG_SERVICE
* Copied from netcfg.h.  The header files in nettools are such a rats nest that
* it is just easier to copy the #defines here
*----------------------------------------------------------------------------*/
#define CFGITEM_SERVICE_TELNET          0x0001
#define CFGITEM_SERVICE_HTTP            0x0002
#define CFGITEM_SERVICE_NAT             0x0003
#define CFGITEM_SERVICE_DHCPSERVER      0x0004
#define CFGITEM_SERVICE_DHCPCLIENT      0x0005
#define CFGITEM_SERVICE_DNSSERVER       0x0006
#define CFGITEM_SERVICE_MAX             0x0006


/*-----------------------------------------------------------------------------
* Copied from dhcpif.h.  The header files in nettools are such a rats nest that
* it is just easier to copy the #defines here
*----------------------------------------------------------------------------*/
#define DHCPCODE_IPADD          0x11    /* Client has added an address */
#define DHCPCODE_IPREMOVE       0x12    /* IP address removed and CFG erased */
#define DHCPCODE_IPRENEW        0x13    /* IP renewed, DHCP config space reset */

// This function is called by BIOS NDK when an IP address has been assigned to the TIVA
extern "C" void netIPAddr(uint32_t IPAddr, uint32_t IfIdx, uint32_t fAdd)
{
    if(fAdd > 0)
    {
        Ethernet::HandleIPAddrAssignedEvent(htonl(IPAddr));
    }
}

extern "C" void RTCReportCallback(uint32_t Msg, uint32_t dwParam1, uint32_t dwParam2 )
{
    if(Msg == MSG_RTC_DUPIP)
    {
        Ethernet::set_duplicate_IP_addr_detect_flag();
        PRINTFD("!!!Error duplicate IP detected msg:%u IP=0x%x 0x%x\n", Msg, dwParam1, dwParam2);
    }
}


extern "C" void netStatusReport(uint32_t Item, uint32_t Status, uint32_t Report, void * h)
{
    if(Item == CFGITEM_SERVICE_DHCPCLIENT)
    {
        if( (Report >> 8) && ((Report & 0xFF) == DHCPCODE_IPADD) )
        {
            PRINTFD("Ethernet DHCP IP address added\n");
        }
        else if((Report >> 8) && ((Report & 0xFF) == 0) )
        {
            PRINTFD("Ethernet DHCP Client Started\n");
        }
        else if((Report & 0xFF) == DHCPCODE_IPREMOVE)
        {
            PRINTFD("Ethernet DHCP Client IP Removed\n");
        }
        else
        {
            PRINTFD("Ethernet Status report 0x%x, 0x%x, 0x%x\n", Item, Status, Report);
        }
    }
}


extern "C" void netCloseHook()
{
    PRINTFD("Ethernet Shutdown\n");
}


#define ERROR_FAIL_U8          0xFF

uint32_t Ethernet::m_ip_addr = 0;
UDPContext_t Ethernet::m_udp_sockets[MAX_UDP_SOCKETS] = {0};
TCPHostContext_t Ethernet::m_tcp_host_sockets[MAX_TCP_SOCKETS] = {0};
TCPClientContext_t Ethernet::m_tcp_client_socket = {false, false, 0, 0};
bool Ethernet::m_is_fd_session_initialized = false;
bool Ethernet::m_is_ip_valid = false;
bool Ethernet::m_is_dhcp_ip_assigned = false;
bool Ethernet::m_is_dhcp_started = false;
bool Ethernet::m_is_cable_connected = false;
bool Ethernet::m_is_apipa_enabled = false;
bool Ethernet::m_is_ethernet_initialized = false;
bool Ethernet::m_is_duplicate_ip_detected = false;
bool Ethernet::m_is_new_ip_addr_assigned = false;
uint8_t Ethernet::m_last_client_idx = 0;
uint8_t Ethernet::m_last_socket_idx = 0;
uint8_t Ethernet::m_dhcp_restart_tick_count = 0;
uint32_t Ethernet::m_static_ip_addr = 0;
uint32_t Ethernet::m_new_ip_addr = 0;
uint32_t Ethernet::m_last_dhcp_restart_time = 0;
uint8_t Ethernet::m_apipa_enable_tick_count = 0;
uint32_t Ethernet::m_apipa_enable_start_time = 0;
Ethernet::EthernetSocketStatus_t Ethernet::m_ethernet_socket_status = SOCKETS_CLOSED;

Ethernet::Ethernet() {}

bool Ethernet::OpenSockets(uint16_t udp_port, uint16_t tcp_port, void *task_handle)
{
    bool retval = false;
    bool result = false;

    if(m_is_ip_valid == true)
    {
        // check that sockets are up. Assume that if UDP is down, then TCP is also down
        if(m_ethernet_socket_status != SOCKETS_OPEN)
        {
            if(GetUDPSocketStatus(udp_port) == false)
            {
                result = OpenUDPSocket(udp_port, task_handle);
                if(result != true)
                {
                    m_ethernet_socket_status = UDP_ERROR;
                }
            }
            else
            {
                result = true;
            }

            if(isTCPSocketOpen(tcp_port) == false)
            {
                result = result && OpenTCPSocket(tcp_port, task_handle);
            }
            else
            {
                m_ethernet_socket_status = TCP_ERROR;
            }

            if(result == true)
            {
                PRINTFD("Ethernet Sockets Created. Port Num = %d.\n", tcp_port);
                m_ethernet_socket_status = SOCKETS_OPEN;
                retval = true;
            }
        }
        else
        {
            retval = true;
        }
    }

    return(retval);
}

void Ethernet::HandleIPAddrAssignedEvent(uint32_t ip_addr)
{
    // Event is coming from NDK thread.  To avoid threads stepping on each other
    // just set flags here and let host_comm thread handle socket and other ethernet related work
    m_new_ip_addr = ip_addr;
    m_is_new_ip_addr_assigned = true;
}

void Ethernet::IPAddressUpdate()
{
    if(m_is_new_ip_addr_assigned)
    {
        m_is_new_ip_addr_assigned = false;
        m_ip_addr = m_new_ip_addr;
        m_is_ip_valid = true;

        PRINTFD("Ethernet IP address added: %d.%d.%d.%d\n",
                    m_new_ip_addr>>24, (m_new_ip_addr>>16)&0xFF, (m_new_ip_addr>>8)&0xFF, m_new_ip_addr&0xFF);

        // check if address is coming from DHCP
        if( (m_new_ip_addr & 0xFFFF0000) != APIPA_BASE_ADDR )
        {
            // DHCP address, check if APIPA is active and if so, squash it
            if(m_is_apipa_enabled)
            {
                NetworkInterface::OnEthernetDisconnect(0);
                CloseAllSockets();
                RemoveStaticIP();
                Task_sleep(5);      // allow time for APIPA removal
                m_is_apipa_enabled = false;
            }
            m_is_dhcp_ip_assigned = true;
        }

        NetworkInterface::HandleIPAddrAssignedEvent(ETHERNET_INTF);
    }
}

bool Ethernet::OpenUDPSocket(uint16_t port_num, void * task_handle)
{
    int status;
    struct sockaddr_in localAddr;
    uint8_t arry_idx = 0;
    bool retval = false;

    if(m_is_fd_session_initialized == false)
    {
        m_is_fd_session_initialized = call_fdOpenSession(task_handle);
    }

    if( (isUDPSocketOpen(port_num, &arry_idx) == false) && (m_is_fd_session_initialized) )
    {
        arry_idx = getUDPSocket();
        if (arry_idx < MAX_UDP_SOCKETS)
        {
            localAddr.sin_family = AF_INET;
            localAddr.sin_addr.s_addr = htonl(m_ip_addr);
            localAddr.sin_port = htons(port_num);
            status = call_ndk_bind(m_udp_sockets[arry_idx].handle, (struct sockaddr *)&localAddr, sizeof(localAddr));
            if(status == 0)
            {
                PRINTFD("Ethernet UDP Socket Created. Port Num = %d.\n", port_num);
                m_udp_sockets[arry_idx].port_num = port_num;
                m_udp_sockets[arry_idx].is_open = true;
                retval = true;
            }
            else
            {
                PRINTFD("Error - cannot open Ethernet UDP Socket. Cannot bind to socket. Port Num = %d.\n", port_num);
            }
        }
        else
        {
            if(arry_idx == ERROR_FAIL_U8)
            {
                PRINTFD("Error - NDK cannot create Etherent UDP Socket. Port Num = %d.\n", port_num);
            }
            else
            {
                PRINTFD("Error - cannot open Etherent UDP Socket. No more ports open. Port Num = %d.\n", port_num);
            }
        }
    }
    else
    {
        PRINTFD("Error - Etherent UDP Port Num %d already open.\n", port_num);
    }

    return(retval);
}


bool Ethernet::GetUDPSocketStatus(uint16_t port_num)
{
    uint8_t arry_idx;
    return(isUDPSocketOpen(port_num, &arry_idx));
}


uint16_t Ethernet::UDPSend(uint16_t port_num, uint8_t *p_data, uint16_t bytes_to_send, uint32_t dest_addr, uint16_t dest_port_num)
{
    uint8_t arry_idx;
    uint16_t bytesSent = 0;
    struct sockaddr_in destAddr;

    if(isUDPSocketOpen(port_num, &arry_idx) == true)
    {
        // MAC is big endian.  So convert little endian to big endian here
        destAddr.sin_addr.s_addr = htonl(dest_addr);    // endian conversion for 32-bit value
        destAddr.sin_port = htons(dest_port_num);            // endian conversion for 16-bit value
        destAddr.sin_family = AF_INET;
        bytesSent = call_ndk_sendto(m_udp_sockets[arry_idx].handle, p_data, bytes_to_send, 0,
                (struct sockaddr *)&destAddr, sizeof(destAddr));
    }
    else
    {
        PRINTFD("Ethernet UDP - Sending data on unknown port %d.\n", port_num);
    }

    return(bytesSent);
}


uint16_t Ethernet::UDPRecv(uint16_t port_num, uint8_t *p_data, uint16_t bytes_to_get,
                           uint32_t *p_sender_Addr, uint16_t *p_sender_port)
{
    uint8_t arry_idx;
    uint16_t retval = 0;
    int32_t bytes_rcvd = 0;
    struct sockaddr_in sender_addr;
    socklen_t sender_addr_len;

    if(isUDPSocketOpen(port_num, &arry_idx) == true)
    {
        bytes_rcvd = call_ndk_recvfrom(m_udp_sockets[arry_idx].handle, p_data, bytes_to_get, 0,
                (struct sockaddr *)&sender_addr, &sender_addr_len);

        if(bytes_rcvd > 0)
        {
            // MAC is big endian.  So convert little endian to big endian here
            *p_sender_Addr = htonl(sender_addr.sin_addr.s_addr);
            *p_sender_port = htons(sender_addr.sin_port);
            retval = bytes_rcvd;
        }
    }

    return(retval);
}


bool Ethernet::isUDPSocketOpen(uint16_t port_num, uint8_t *p_index)
{
    bool retval = false;
    uint8_t arry_idx = 0;

    while( (retval == false) && (arry_idx < MAX_UDP_SOCKETS) )
    {
        if(m_udp_sockets[arry_idx].port_num == port_num)
        {
            if(m_udp_sockets[arry_idx].is_open == true)
            {
                retval = true;
                *p_index = arry_idx;
            }
            else
            {
                arry_idx++;
            }
        }
        else
        {
            arry_idx++;
        }
    }

    return(retval);
}

// returns the m_udp_sockets array index if successful,
// MAX_UDP_SOCKETS if no sockets available, and ERROR_FAIL_U8 if NDK error
uint8_t Ethernet::getUDPSocket()
{
    bool found = false;
    uint8_t arry_idx = 0;

    while( (found == false) && (arry_idx < MAX_UDP_SOCKETS) )
    {
        if(m_udp_sockets[arry_idx].is_open == false)
        {
            m_udp_sockets[arry_idx].handle = call_ndk_socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
            if((void *)(m_udp_sockets[arry_idx].handle) == INVALID_SOCKET)
            {
                arry_idx = ERROR_FAIL_U8;
                PRINTFD("Error code = %d\n", call_fdError());
            }
            found = true;

            // set timeout to 1ms
            struct timeval timeout;
            timeout.tv_sec = 0;
            timeout.tv_usec = 1;
            if(call_setsockopt(m_udp_sockets[arry_idx].handle, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout)) < 0)
            {
                arry_idx = ERROR_FAIL_U8;
                PRINTFD("Error - set Etherent UDP rcv timeout fail. code = %d\n", call_fdError());
            }

            timeout.tv_usec = 5;
            if(call_setsockopt(m_udp_sockets[arry_idx].handle, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout)) < 0)
            {
                arry_idx = ERROR_FAIL_U8;
                PRINTFD("Error - set Etherent UDP send timeout fail. code = %d\n", call_fdError());
            }

            struct ip_mreq multicast_req;
            multicast_req.imr_interface.s_addr = htonl(m_ip_addr);
            multicast_req.imr_multiaddr.s_addr = htonl(MULTICAST_ADDR);
            if(call_setsockopt(m_udp_sockets[arry_idx].handle, IPPROTO_IP, IP_ADD_MEMBERSHIP, &multicast_req, sizeof(multicast_req)) < 0)
            {
                arry_idx = ERROR_FAIL_U8;
                PRINTFD("Error - set Etherent UDP multicast req fail. code = %d\n", call_fdError());
            }
        }
        else
        {
            arry_idx++;
        }
    }

    return(arry_idx);
}

// task_handle is the calling function's task handle
bool Ethernet::OpenTCPSocket(uint16_t port_num, void * task_handle)
{
    int status;
    struct sockaddr_in localAddr;
    uint8_t socket_idx = 0;
    bool retval = false;

    if(m_is_fd_session_initialized == false)
    {
        m_is_fd_session_initialized = call_fdOpenSession(task_handle);
    }

    if( (isTCPSocketOpen(port_num) == false) && (m_is_fd_session_initialized) )
    {
    	PRINTFD("Ethernet::OpenTCPSocket(), printing socket table\n\n");
    	SockUtils_printSockTable(IPPROTO_TCP);
    	PRINTFD("\n\n");
        socket_idx = getTCPSocket();
        if (socket_idx < MAX_TCP_SOCKETS)
        {
            localAddr.sin_family = AF_INET;
            localAddr.sin_addr.s_addr = htonl(m_ip_addr);       // convert little endian to big endian
            localAddr.sin_port = htons(port_num);               // convert little endian to big endian
            status = call_ndk_bind(m_tcp_host_sockets[socket_idx].handle, (struct sockaddr *)&localAddr, sizeof(localAddr));
            if(status == 0)
            {
                PRINTFD("Ethernet TCP Socket Created. Port Num = %d.\n", port_num);
    	        PRINTFD("\n\n");
    	        SockUtils_printSockTable(IPPROTO_TCP);

                m_tcp_host_sockets[socket_idx].port_num = port_num;
                m_tcp_host_sockets[socket_idx].is_open = true;
                retval = true;
            }
            else
            {
                int err = call_fdError();
                PRINTFD("Error - cannot open Ethernet TCP Socket. Cannot bind to socket. Port Num = %u. err = 0x%x\n", port_num, err);
            }
        }
        else
        {
            if(socket_idx == ERROR_FAIL_U8)
            {
                PRINTFD("Error - NDK cannot create Etherent TCP Socket. Port Num = %d.\n", port_num);
            }
            else
            {
                PRINTFD("Error - cannot open Etherent TCP Socket. No more ports open. Port Num = %d.\n", port_num);
            }
        }
    }
    else
    {
        PRINTFD("Error - Etherent TCP Port Num %d already open.\n", port_num);
    }

    return(retval);
}


bool Ethernet::GetTCPSocketStatus(uint16_t port_num)
{
    return(isTCPSocketOpen(port_num));
}


bool Ethernet::TCPListen(uint16_t port_num)
{
    uint8_t socket_idx = 0;
    int32_t result;
    bool retval = false;
    if(isTCPSocketOpen(port_num) == true)
    {
        getTCPSocketIdx(port_num, &socket_idx);
        result = call_ndk_listen(m_tcp_host_sockets[socket_idx].handle, 1);
        if(result == 0)
        {
            retval = true;
        }
    }

    return(retval);
}


bool Ethernet::TCPAccept(uint16_t port_num, uint32_t *p_dest_addr, uint16_t *p_dest_port, int32_t *p_client_id)
{
    bool retval = false;
    uint8_t socket_idx = 0;
    uint8_t client_idx = 0;
    struct sockaddr_in clientAddr;
    int32_t addrlen;
    int32_t client_handle;
    if(isTCPSocketOpen(port_num) == true)
    {
        getTCPSocketIdx(port_num, &socket_idx);
        client_idx = getFreeTCPClientSlot(socket_idx);
        if(client_idx < MAX_TCP_CLIENTS)
        {
            client_handle = call_ndk_accept(m_tcp_host_sockets[socket_idx].handle, (struct sockaddr *)&clientAddr, &addrlen);
            if(client_handle >= 0)
            {
                m_tcp_host_sockets[socket_idx].client[client_idx].dest_addr = htonl(clientAddr.sin_addr.s_addr);
                *p_dest_addr = m_tcp_host_sockets[socket_idx].client[client_idx].dest_addr;
                m_tcp_host_sockets[socket_idx].client[client_idx].dest_port = htons(clientAddr.sin_port);
                *p_dest_port = m_tcp_host_sockets[socket_idx].client[client_idx].dest_port;
                m_tcp_host_sockets[socket_idx].client[client_idx].client_handle = client_handle;
                *p_client_id = client_idx;
                m_tcp_host_sockets[socket_idx].client[client_idx].is_connected = true;

                PRINTFD("Ethernet TCP - Client connected. ");
                PRINTFD("Addr = %d.%d.%d.%d\n", clientAddr.sin_addr.s_addr&0xFF, (clientAddr.sin_addr.s_addr>>8)&0xFF,
                        (clientAddr.sin_addr.s_addr>>16)&0xFF, clientAddr.sin_addr.s_addr>>24);
    	PRINTFD("Ethernet::TCPAccept(), printing socket table\n\n");
    	SockUtils_printSockTable(IPPROTO_TCP);

                EthTCPSocketData_t		socket_data;
                socket_data.client_idx = client_idx;
                socket_data.ip_addr = *p_dest_addr;
                socket_data.port_num = clientAddr.sin_port;
                NetworkInterface::NotifySocketConnected(&socket_data);
                retval = true;
            }
        }
        else
        {
            PRINTFD("Error - Ethernet TCP - Max clients reached.\n");
        }
    }

    return(retval);
}


uint8_t Ethernet::get_num_ports_avail(uint16_t *p_port_num)
{
    uint8_t num_ports_avail = 0;
    uint8_t num_connections_avail = 0;
    bool found = false;

    for(uint8_t socket_idx = 0; socket_idx < MAX_TCP_SOCKETS; socket_idx++)
    {
        if(m_tcp_host_sockets[socket_idx].is_open == true)
        {
            found = false;
            p_port_num[socket_idx] = m_tcp_host_sockets[socket_idx].port_num;
            for(uint8_t client_idx = 0; client_idx < MAX_TCP_CLIENTS; client_idx++)
            {
                if(m_tcp_host_sockets[socket_idx].client[client_idx].is_connected == false)
                {
                    num_connections_avail++;
                    found = true;
                }
            }

            if(found)
            {
                num_ports_avail++;
            }
        }
    }

    return(num_ports_avail);
}

bool Ethernet::QueueToSend(uint8_t *p_data, EthernetSendCfg_t *p_send_cfg)
{
	bool result = false;
    EthernetMBox_t  mbox_message;

    if( (m_is_cable_connected) && (m_ethernet_socket_status == SOCKETS_OPEN) )
    {
#if INCLUDE_3F
        mbox_message.data[0] = 0x3F;
        memcpy(mbox_message.data + 1, p_data, p_send_cfg->bytes_to_send);
#else
        memcpy(mbox_message.data, p_data, p_send_cfg->bytes_to_send);
#endif
        memcpy((void *)&(mbox_message.send_cfg), (uint8_t *)p_send_cfg, sizeof(EthernetSendCfg_t));

        if(Mailbox_post(mbox_ethernet_send, &mbox_message, ETHERNET_MBOX_TIMEOUT) == false)
        {
            PRINTFD("Error - Ethernet Send Mailbox Timeout.\n");
        }
        else
        {
        	result = true;
        }
    }
    return result;
}


void Ethernet::SendData()
{
    EthernetMBox_t  mbox_message;
    if(Mailbox_pend(mbox_ethernet_send, &mbox_message, BIOS_NO_WAIT) == true)
    {
        if(mbox_message.send_cfg.protocol == NET_PROT_UDP)
        {
            UDPSend(mbox_message.send_cfg.port_num, mbox_message.data, mbox_message.send_cfg.bytes_to_send,
                    mbox_message.send_cfg.client_addr, mbox_message.send_cfg.remote_port_num);
        }
        else
        {
            TCPSend(mbox_message.send_cfg.port_num, mbox_message.send_cfg.client_idx, mbox_message.data,
                    mbox_message.send_cfg.bytes_to_send);
        }
    }
}

uint16_t Ethernet::TCPSend(uint16_t port_num, uint8_t client_id, uint8_t *p_data, uint16_t bytes_to_send)
{
    uint8_t socket_idx = 0;
    uint8_t client_idx = client_id;
    int32_t bytesSent = 0;
    uint16_t retval = 0;

    if(isTCPSocketOpen(port_num) == true)
    {
        getTCPSocketIdx(port_num, &socket_idx);
        if(isTCPClientConnected(socket_idx, client_idx) == true)
        {
            bytesSent = call_ndk_send(m_tcp_host_sockets[socket_idx].client[client_idx].client_handle, p_data, bytes_to_send, 0);
            if(bytesSent >= 0)
            {
                retval = bytesSent;
            }
            else
            {
                int32_t error_code = call_fdError();
                PRINTFD("Error - Etherent TCP Send. Client Addr=0x%x err code= 0x%x. bytes to send = %u\n",
                        m_tcp_host_sockets[socket_idx].client[client_idx].dest_addr, error_code, bytes_to_send);
            }
        }
    }

    return(retval);
}


// looks through all client connections for a message
uint16_t Ethernet::TCPReceive(uint16_t *p_port_num, uint8_t *p_data, int32_t size, uint32_t *p_client_addr, uint8_t *p_client_id)
{
    uint8_t socket_idx = incr_last_socket_idx();
    uint8_t client_idx = incr_last_client_idx();
    uint8_t num_sockets = 0;
    uint16_t port_num = 0;
    uint32_t bytes_rcvd = 0;

    while( (bytes_rcvd == 0) && (num_sockets < MAX_TCP_SOCKETS) )
    {
        port_num = m_tcp_host_sockets[m_last_socket_idx].port_num;
        if(isTCPSocketOpen(port_num) == true)
        {
            uint8_t num_clients = 0;
            getTCPSocketIdx(port_num, &socket_idx);
            while( (bytes_rcvd == 0) && (num_clients < MAX_TCP_CLIENTS) )
            {
                bytes_rcvd = TCPReceive(port_num, client_idx, p_data, size);
                num_clients++;
                if(bytes_rcvd == 0)
                {
                    client_idx = incr_last_client_idx();
                }
                else
                {
                    *p_client_addr = m_tcp_host_sockets[m_last_socket_idx].client[m_last_client_idx].dest_addr;
                    *p_client_id = m_last_client_idx;
                    *p_port_num =  m_tcp_host_sockets[m_last_socket_idx].port_num;
                }
            }
        }

        num_sockets++;
        if(bytes_rcvd == 0)
        {
            socket_idx = incr_last_socket_idx();
        }
    }

    return(bytes_rcvd);
}


// Receive from a specific client
uint16_t Ethernet::TCPReceive(uint16_t port_num, uint8_t client_id, uint8_t *p_data, int32_t size)
{
    uint8_t socket_idx = 0;
    uint8_t client_idx = client_id;
    int32_t bytesRcvd = 0;
    uint16_t retval = 0;

    if(isTCPSocketOpen(port_num) == true)
    {
        getTCPSocketIdx(port_num, &socket_idx);
        if(isTCPClientConnected(socket_idx, client_idx) == true)
        {
            // using no copy version of receive to reduce RAM usage and to avoid a copy.  Need to call recvncfree to free rcv buffer.
            bytesRcvd = call_ndk_recv(m_tcp_host_sockets[socket_idx].client[client_idx].client_handle, p_data, size, 0);

            if(bytesRcvd > 0)
            {
                retval = bytesRcvd;
            }
            // 0 indicates connection has been closed by the peer
            else if(bytesRcvd == 0)
            {
                DisconnectClient(socket_idx, client_idx);
            }
            else
            {
                int32_t error_code = call_fdError();
                if(error_code == EWOULDBLOCK)
                {
                    // This is OK.  No data to receive
                }
                else
                {
                    // connection is either no longer up or in error - might as well close the connection to free heap mem
                    DisconnectClient(socket_idx, client_idx);
                }
            }
        }
    }

    return(retval);
}


uint8_t Ethernet::getFreeTCPClientSlot(uint8_t socket_idx)
{
    bool found = false;
    uint8_t client_idx = 0;

    if(socket_idx < MAX_TCP_SOCKETS)
    {
        while( (found == false) && (client_idx < MAX_TCP_CLIENTS) )
        {
            if(m_tcp_host_sockets[socket_idx].client[client_idx].is_connected == false)
            {
                found = true;
            }
            else
            {
                client_idx++;
            }
        }
    }

    return(client_idx);
}


bool Ethernet::getTCPSocketIdx(uint16_t port_num, uint8_t *p_socket_idx)
{
    bool found = false;
    uint8_t socket_idx = 0;
    *p_socket_idx = 0;

    while( (found == false) && (socket_idx < MAX_TCP_SOCKETS) )
    {
        if(m_tcp_host_sockets[socket_idx].port_num == port_num)
        {
            found = true;
            *p_socket_idx = socket_idx;
        }
        else
            socket_idx++;
    }

    return(found);
}

bool Ethernet::isTCPSocketOpen(uint16_t port_num)
{
    bool retval = false;
    uint8_t socket_idx = 0;

    if(getTCPSocketIdx(port_num, &socket_idx))
    {
        retval = m_tcp_host_sockets[socket_idx].is_open;
    }

    return(retval);
}

bool Ethernet::isTCPClientConnected(uint8_t socket_idx, uint8_t client_idx)
{
    bool retval = false;

    if( (socket_idx < MAX_TCP_SOCKETS) && (client_idx < MAX_TCP_CLIENTS) )
    {
        retval = m_tcp_host_sockets[socket_idx].client[client_idx].is_connected;
    }

    return(retval);
}

bool Ethernet::isTCPClientConnected(uint8_t socket_idx)
{
    bool found = false;
    uint8_t client_idx = 0;

    while( (!found) && (client_idx < MAX_TCP_CLIENTS) )
    {
        found = m_tcp_host_sockets[socket_idx].client[client_idx].is_connected;
        client_idx++;
    }

    return(found);
}

void Ethernet::DisconnectClient(uint8_t socket_idx, uint8_t client_idx)
{
    if(call_ndk_close(m_tcp_host_sockets[socket_idx].client[client_idx].client_handle) != 0)
    {
        PRINTFD("Error - Ethernet TCP client disconnect fail. code = %d\n", call_fdError());
    }
    else
    {
        m_tcp_host_sockets[socket_idx].client[client_idx].is_connected = false;
        m_tcp_host_sockets[socket_idx].client[client_idx].client_handle = 0;
        m_tcp_host_sockets[socket_idx].client[client_idx].dest_addr = 0;
        m_tcp_host_sockets[socket_idx].client[client_idx].dest_port = 0;
        PRINTFD("Etherent TCP Client Disconnected. Addr = 0x%x\n",
                m_tcp_host_sockets[socket_idx].client[client_idx].dest_addr);
    }
}

bool Ethernet::CreateTCPSocket(int &handle)
{
    bool retval = true;
    handle = call_ndk_socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(handle == (int)INVALID_SOCKET)
    {
        retval = false;
        PRINTFD("!!! Err - CreateTCPSocket failed Error code = %d\n", call_fdError());
    }
    return(retval);
}

bool Ethernet::SetTCPSocketOpts(int handle)
{
    bool retval = true;
    int32_t optval = 0;

    // set socket to non-blocking
    if(call_setsockopt(handle, SOL_SOCKET, SO_BLOCKING, &optval, sizeof(optval)) != 0)

    {
        retval = false;
        PRINTFD("Error - set Ethernet TCP to non-blocking fail. code = %d\n", call_fdError());
    }

    if(retval)
    {
        optval = 60;
        // set keep-alive
        if (call_setsockopt(handle, SOL_SOCKET, SO_KEEPALIVE, &optval, sizeof(optval)) < 0)
        {
            retval = false;
            PRINTFD("Error - set Etherent TCP keep alive fail. code = %d\n", call_fdError());
        }
    }

    struct timeval timeout;
    // set timeout to 500ms
    if(retval)
    {
        timeout.tv_sec = 0;
        timeout.tv_usec = 500;
        if(call_setsockopt(handle, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout)) < 0)
        {
            retval = false;
            PRINTFD("Error - set Etherent TCP rcv timeout fail. code = %d\n", call_fdError());
        }
    }

    if(retval)
    {
        timeout.tv_sec = 1;
        timeout.tv_usec = 0;
        if(call_setsockopt(handle, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout)) < 0)
        {
            retval = false;
            PRINTFD("Error - set Etherent TCP send timeout fail. code = %d\n", call_fdError());
        }
    }

    if(retval)
    {
        struct linger linger_opt;
        linger_opt.l_linger = 1;
        linger_opt.l_onoff = 0;
        if(call_setsockopt(handle, SOL_SOCKET, SO_LINGER, &linger_opt, sizeof(linger_opt)) < 0)
        {
            retval = false;
            PRINTFD("Error - set Etherent TCP send timeout fail. code = %d\n", call_fdError());
        }
    }

    return(retval);
}

// returns the m_tcp_host_sockets array index if successful,
// MAX_TCP_CLIENTS if no sockets available, and ERROR_FAIL_U8 if NDK error
uint8_t Ethernet::getTCPSocket()
{
    bool found = false;
    uint8_t socket_idx = 0;

    while( (found == false) && (socket_idx < MAX_TCP_SOCKETS) )
    {
        if(m_tcp_host_sockets[socket_idx].is_open == false)
        {
            m_tcp_host_sockets[socket_idx].handle = call_ndk_socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
            if((void *)(m_tcp_host_sockets[socket_idx].handle) == INVALID_SOCKET)
            {
                socket_idx = ERROR_FAIL_U8;
                PRINTFD("Error code = %d\n", call_fdError());
            }
            else
                SetTCPSocketOpts(m_tcp_host_sockets[socket_idx].handle);
            found = true;

        }
        else
        {
            socket_idx++;
        }
    }

    return(socket_idx);
}


uint8_t Ethernet::incr_last_client_idx()
{
    m_last_client_idx++;
    if(m_last_client_idx >= MAX_TCP_CLIENTS)
    {
        m_last_client_idx = 0;
    }

    return(m_last_client_idx);
}


uint8_t Ethernet::incr_last_socket_idx()
{
    m_last_socket_idx++;
    if(m_last_socket_idx >= MAX_TCP_SOCKETS)
    {
        m_last_socket_idx = 0;
    }

    return(m_last_socket_idx);
}


void Ethernet::print_ip_addr()
{
    PRINTFD("Ethernet IP is: %d.%d.%d.%d\n",
            m_ip_addr>>24, (m_ip_addr>>16)&0xFF, (m_ip_addr>>8)&0xFF, m_ip_addr&0xFF);
}

void Ethernet::ChangeStaticIPAddr()
{
    m_static_ip_addr = ((m_static_ip_addr + (rand()&0xFFFF)) & APIPA_SUBNET_REV_MSK) + APIPA_BASE_ADDR;
}

void Ethernet::Init()
{
    if(m_is_ethernet_initialized == false)
    {
        AddRTCHook(RTCReportCallback);

        uint8_t *p_mac_address = TIVAEMAC::m_mac_address;
        m_static_ip_addr = APIPA_BASE_ADDR + (p_mac_address[4] << 8) + p_mac_address[5];

        m_is_ethernet_initialized = true;
    }
}

void Ethernet::MonitorLink()
{
    Init();
    IPAddressUpdate();

    if( (m_is_duplicate_ip_detected == true) && (m_is_apipa_enabled == true) && (m_is_ip_valid == true) )
    {
        // Duplicate IP address: remove fixed IP and increment IP address
        m_is_duplicate_ip_detected = false;
        m_is_apipa_enabled = false;
        m_is_ip_valid = false;
        CloseAllSockets();
        RemoveStaticIP();
        ChangeStaticIPAddr();
        NetworkInterface::OnEthernetDisconnect(0);
    }
    else if( (m_is_cable_connected == false) || (m_is_ip_valid == false) || (m_is_apipa_enabled == true) )
    {
        // link is down, attempt restart periodically
        bool is_cable_now_connected = TIVAEMAC::isLinkActive();

        // check if cable has been reconnected
        if(m_is_cable_connected != is_cable_now_connected)
        {
            // If cable is reconnected, try DHCP for 30s then switch to APIPA if DHCP does not succeed
            m_is_cable_connected = is_cable_now_connected;
            m_is_dhcp_started = false;

            if(m_is_cable_connected == true)
            {
				//m_dhcp_restart_tick_count = ETHERNET_NUM_TICKS_PER_RESTART;     // start in DHCP
                m_dhcp_restart_tick_count = 0;     // start in DHCP
#if DEBUG_SLOW_ENET
				m_apipa_enable_tick_count = 0;
#else
				m_apipa_enable_tick_count = 0;                                  // clear timer to APIPA, switch to APIPA when APIPA times out
#endif
				//PRINTFD("Ethernet cable detected\n");
            }
        }

        // cable connection detected
        if(m_is_cable_connected == true)
        {
            bool is_time_expired = IsDHCPRestartTimerExpired();

            // DHCP gives up after 2 minutes.  Restart periodically.
            if( (m_is_cable_connected == true) && (m_is_dhcp_ip_assigned == false) && is_time_expired)
            {
                bool result;

                // starting dhcp the first time and restarting dhcp needs to be handled differently
                if(m_is_dhcp_started == false)
                {
                    result = StartNDKDHCP();
                    if(result)
                        m_is_dhcp_started = true;
                }
                else
                    result = RestartNDKDHCP();

                if(result)
                {
                    PRINTFD("Starting DHCP\n");
                    m_dhcp_restart_tick_count = 0;
                    m_last_dhcp_restart_time = TivaTimer::get_count();
                }
                else
                {
                    PRINTFD("!!! Error - Start DHCP Failed\n");
                }
            }

            is_time_expired = false;
            if( (m_is_cable_connected == true) && (m_is_ip_valid == false) && is_time_expired && (m_is_apipa_enabled == false))
            {
                // disable DHCP if APIPA is started, DHCP will restart when DHCP timer expires.
                DisableNDKDHCP();
                bool result = SpecifyIPNetwork(m_static_ip_addr, APIPA_SUBNET_MSK);
                if(result)
                {
                    PRINTFD("DHCP Timed out APIPA started\n");
                    m_is_apipa_enabled = true;
                    m_apipa_enable_tick_count = 0;
                    m_apipa_enable_start_time = TivaTimer::get_count();
                    m_dhcp_restart_tick_count = ETHERNET_NUM_TICKS_PER_RESTART;  //forces dhcp to restart
                }
                else
                {
                    PRINTFD("!!! Error - APIPA Failed\n");
                }
            }
        }
    }
    else
    {
        // link is up, check if link goes down
        m_is_cable_connected = TIVAEMAC::isLinkActive();
        if(m_is_cable_connected == false)
        {
            m_is_ip_valid = false;
            m_ip_addr = 0;
            CloseAllSockets();
            PRINTFD("Ethernet cable unplugged\n");
            m_is_dhcp_ip_assigned = false;
            m_is_apipa_enabled = false;
            DisableNDKDHCP();
            NetworkInterface::OnEthernetDisconnect(0);
        }
    }
}


void Ethernet::CloseAllSockets()
{
    uint8_t socket_idx;
    int status = 0xFF;
    bool result = true;

    for(socket_idx = 0; socket_idx < MAX_UDP_SOCKETS; socket_idx++)
    {
        if(m_udp_sockets[socket_idx].is_open == true)
        {
            m_udp_sockets[socket_idx].is_open = false;
            status = call_ndk_close(m_udp_sockets[socket_idx].handle);
            if(status != 0)
            {
                result = false;
                PRINTFD("!!!Error - Attempt to clode Ethernet UDP port failed\n");
            }
            else
            {
                result = (result && true);
            }
        }
    }

    for(socket_idx = 0; socket_idx < MAX_TCP_SOCKETS; socket_idx++)
    {
        for(uint8_t client_idx = 0; client_idx < MAX_TCP_CLIENTS; client_idx++)
        {
			m_tcp_host_sockets[socket_idx].client[client_idx].is_connected = false;
        }

        if(m_tcp_host_sockets[socket_idx].is_open == true)
        {
            m_tcp_host_sockets[socket_idx].is_open = false;
            status = call_ndk_close(m_tcp_host_sockets[socket_idx].handle);
            if(status != 0)
            {
                result = false;
                PRINTFD("!!!Error - Attempt to close Ethernet TCP port failed\n");
            }
            else
            {
                result = (result && true);
            }
        }
    }

    if(result == true)
    {
        PRINTFD("Ethernet TCP ports closed\n");
    }

    m_ethernet_socket_status = SOCKETS_CLOSED;
}

bool Ethernet::IsDHCPRestartTimerExpired()
{
    bool retval = false;
    uint32_t curr_time = TivaTimer::get_count();
    uint32_t delta_time = TivaTimer::CalcDeltaTime(curr_time, m_last_dhcp_restart_time);
    if(delta_time >= ETHERNET_TICK_PERIOD)
    {
        m_dhcp_restart_tick_count++;
        m_last_dhcp_restart_time = curr_time - (delta_time - ETHERNET_TICK_PERIOD);
    }

    if(m_dhcp_restart_tick_count >= ETHERNET_NUM_TICKS_PER_RESTART)
    {
        retval = true;
    }

    return(retval);
}

bool Ethernet::IsAPIPAStartTimerExpired()
{
    bool retval = false;
    uint32_t curr_time = TivaTimer::get_count();
    uint32_t delta_time = TivaTimer::CalcDeltaTime(curr_time, m_apipa_enable_start_time);
    if(delta_time >= ETHERNET_TICK_PERIOD)
    {
        m_apipa_enable_tick_count++;
        m_apipa_enable_start_time = curr_time - (delta_time - ETHERNET_TICK_PERIOD);
    }
    if(m_apipa_enable_tick_count >= ETHERNET_TIMEOUT_TO_APIPA)
    {
        retval = true;
    }
    return(retval);
}

bool Ethernet::OpenSocket(uint16_t tcp_port, void *task_handle)
{
    bool retval = false;
    bool result = false;

    // can't do anything if we don't have an IP address yet
    if(m_is_ip_valid == true)
    {
        // check if sockets are already.
        if(isTCPSocketOpen(tcp_port) == false)
        {
            result = OpenTCPSocket(tcp_port, task_handle);

            if(result == true)
            {
                PRINTFD("Ethernet Socket Created. Port Num = %d.\n", tcp_port);
                retval = true;
            }
        }
        else
        {
            PRINTFD("Ethernet Socket Already open. Port Num = %d.\n", tcp_port);
            retval = true;
        }
    }

    return(retval);
}

bool Ethernet::isTcpSocketConnected(uint16_t tcp_port)
{
    bool retval = false;
    uint8_t socket_idx = 0;

    // can't do anything if we don't have an IP address yet
    if(m_is_ip_valid == true)
    {
        // check if any sockets are already connected.
        if(getTCPSocketIdx(tcp_port, &socket_idx) == true)
        {
            retval = isTCPClientConnected(socket_idx);
        }
    }

    return(retval);
}

bool Ethernet::TcpConnect(uint16_t tcp_port, uint32_t remote_host_addr)
{
    bool retval = false;
    int result = 0;
    struct sockaddr_in remote_addr;

    if( (m_tcp_client_socket.is_open == true) && (m_tcp_client_socket.is_connected == false) )
    {
        remote_addr.sin_addr.s_addr = htonl(remote_host_addr);   // convert to big endian format
        remote_addr.sin_port = htons(tcp_port);
        remote_addr.sin_family = AF_INET;

        result = call_ndk_connect(m_tcp_client_socket.handle, (struct sockaddr *)&remote_addr, sizeof(remote_addr));

        if(result == 0)
        {
            PRINTFD("Ethernet Socket connected. Port Num = %d.\n", tcp_port);
            m_tcp_client_socket.is_connected = true;
            retval = true;
        }
        else
        {
            int err = call_fdError();
            if(err == EINPROGRESS)
            {
                PRINTFD("Ethernet Socket connectting. Port Num = %d.\n", tcp_port);
                m_tcp_client_socket.is_connected = true;
                retval = true;
            }
            else
            {
                PRINTFD("!!!error Ethernet Socket connect failed. Port Num = %d. Err=%d\n", tcp_port, call_fdError());
            }
        }
    }

    return(retval);
}

bool Ethernet::GetHostIPByName(char *p_h_name, uint32_t *p_ip_addr)
{
    bool result = NDK_GetHostIPByName(p_h_name, p_ip_addr);

    return(result);
}


bool Ethernet::OpenTCPSocketClient(uint16_t port_num)
{
    int handle, status;
    struct sockaddr_in localAddr;
    bool retval = false;

    if(m_tcp_client_socket.is_open == false)
    {
        // First create socket and set ssocket options
        bool result = CreateTCPSocket(handle);
        if(result)
        {
            m_tcp_client_socket.handle = handle;
            result = SetTCPSocketOpts(handle);
        }

        // Now bind the socket to ethernet IP address
        if (result)
        {
            localAddr.sin_family = AF_INET;
            localAddr.sin_addr.s_addr = htonl(m_ip_addr);       // convert little endian to big endian
            localAddr.sin_port = htons(port_num);               // convert little endian to big endian
            status = call_ndk_bind(m_tcp_client_socket.handle, (struct sockaddr *)&localAddr, sizeof(localAddr));
            if(status == 0)
            {
                PRINTFD("Ethernet TCP Socket Created. Port Num = %d.\n", port_num);
                m_tcp_client_socket.port_num = port_num;
                m_tcp_client_socket.is_open = true;
                retval = true;
            }
            else
            {
                int err = call_fdError();
                PRINTFD("Error - cannot open Ethernet TCP Socket. Cannot bind to socket. Port Num = %d. err=0x%x\n",
                        port_num, err);
            }
        }
    }
    else
    {
        PRINTFD("!!!Err - TCP socket client already open.\n");
    }

    return(retval);
}


uint16_t Ethernet::TCPClientSocketSend(uint8_t *p_data, uint16_t bytes_to_send)
{
    int bytes_sent = call_ndk_send(m_tcp_client_socket.handle, (void *)p_data, bytes_to_send, 0);
    if(bytes_sent < 0)
    {
        int err = call_fdError();
        PRINTFD("!!! TCP send Error - %d\n", err);
        bytes_sent = 0;
    }
    return(bytes_sent);
}

uint16_t Ethernet::TCPClientReceive(uint8_t *p_data, uint16_t size)
{
    int bytes_rcvd = 0;
    if(m_tcp_client_socket.is_connected)
    {
        bytes_rcvd = call_ndk_recv(m_tcp_client_socket.handle, p_data, size, 0);
        if(bytes_rcvd < 0)
        {
            int err = call_fdError();
            PRINTFD("!!! TCP RCV Error - %d\n", err);
            bytes_rcvd = 0;
        }
    }
    else
    {
        PRINTFD("!!! err - TCPClientReceive - socket not connected\n");
    }

    return((uint16_t)bytes_rcvd);
}

void Ethernet::DisconnectFromTCPHost()
{
    // if socket is created, memory will need to be deallocated.
    if(m_tcp_client_socket.handle != 0)
    {
        call_ndk_close(m_tcp_client_socket.handle);
        m_tcp_client_socket.handle = 0;
    }

    // there is really no close socket NDK API. Set it to false as a new socket would be required for the next connection.
    m_tcp_client_socket.is_connected = false;
    m_tcp_client_socket.is_open = false;
}

bool Ethernet::OpenFDSession()
{
    bool retval = false;
    int result = call_fdOpenSession(call_TaskSelf());
    if(result != 0)
        retval = true;
    else
    {
        PRINTFD("!!! OpenFDSession failed\n");
    }
    return(retval);
}

void Ethernet::CloseFDSession()
{
    call_fdCloseSession(call_TaskSelf());
}
