/*
 * ethernet.h
 *
 *  Created on: Jun 20, 2016
 *      Author: yli
 */

#ifndef INCLUDE_ETHERNET_H_
#define INCLUDE_ETHERNET_H_


#include <stdint.h>
#include "network_data.h"
#include "ndk_test_defines.h"

#define INCLUDE_3F                      0

#define MAX_TCP_SOCKETS                 1
#define MAX_TCP_CLIENTS                 MAX_CLIENTS
#define MAX_UDP_SOCKETS                 1
#define TCP_ALL                         0xFF

#define MAX_CLIENTS                     16
#define MULTICAST_ADDR                  0xEF000064
#define BROADCAST_ADDR                  0xFFFFFFFF
#define NET_PROT_UDP                    1
#define NET_PROT_TCP                    2

#define MAC_ADDRESS_SZ                  6

// mailbox timeout in ms
#define ETHERNET_MBOX_TIMEOUT           200

#define ETHERNET_UDP                    NET_PROT_UDP
#define ETHERNET_TCP                    NET_PROT_TCP

// DHCP is configured to give up after 2 minutes.  Therefore, we need to restart periodically
// 2.5 minutes per DHCP restart
#define ETHERNET_TICK_PERIOD            (TIVA_TIMER3_TIME_1s * 5)
#define ETHERNET_NUM_TICKS_PER_RESTART  1

#if DEBUG_SLOW_ENET
    // After DHCP fails, use APIPA.
    #define ETHERNET_TIMEOUT_TO_APIPA       10
#else
    #define ETHERNET_TIMEOUT_TO_APIPA       6
#endif

// APIPA addresses are in range of 169.254.0.0 to 169.254.255.255
#define APIPA_BASE_ADDR                 0xA9FE0000
#define APIPA_SUBNET_MSK                0xFFFF0000
#define APIPA_SUBNET_REV_MSK            (~APIPA_SUBNET_MSK)

#pragma pack(push, 1)
typedef struct{
    uint8_t protocol;
    uint8_t client_idx;
    uint16_t port_num;
    uint16_t remote_port_num;
    uint16_t bytes_to_send;
    uint32_t client_addr;
}EthernetSendCfg_t;

typedef struct{
    EthernetSendCfg_t send_cfg;
    uint8_t data[MAX_MSG_SIZE - sizeof(EthernetSendCfg_t)];
}EthernetMBox_t;

typedef struct{
    bool            is_connected;
    uint16_t        dest_port;
    uint32_t        dest_addr;
    int32_t         client_handle;
    int32_t         hBuffer;
}ClientStatus_t;

typedef struct{
    bool            is_open;
    uint16_t        port_num;
    int32_t         handle;
    ClientStatus_t  client[MAX_TCP_CLIENTS];    // status of each client to this host TCP socket
}TCPHostContext_t;

typedef struct{
    bool            is_open;
    bool            is_connected;
    uint16_t        port_num;
    int32_t         handle;
}TCPClientContext_t;

typedef struct{
    bool            is_open;
    uint16_t        port_num;
    int32_t         handle;
}UDPContext_t;

typedef struct{
	uint32_t 		ip_addr;
    uint16_t        port_num;
	uint8_t			client_idx;
}EthTCPSocketData_t;

#pragma pack(pop)

class Ethernet
{
public:
    typedef enum{
        SOCKETS_OPEN,
        SOCKETS_CLOSED,
        UDP_ERROR,
        TCP_ERROR
    }EthernetSocketStatus_t;

    static bool OpenSockets(uint16_t udp_port, uint16_t tcp_port, void *task_handle);
    static bool OpenSocket(uint16_t tcp_port, void *task_handle);
    static EthernetSocketStatus_t get_status() {return(m_ethernet_socket_status);}
    static void HandleIPAddrAssignedEvent(uint32_t ip_addr);
    static bool get_is_link_active() {return(m_is_cable_connected);}
    static uint32_t get_ip_addr() {return(m_ip_addr);}
    static void print_ip_addr();
    static bool OpenUDPSocket(uint16_t port_num, void * task_handle);
    static bool GetUDPSocketStatus(uint16_t port_num);
    static uint16_t UDPRecv(uint16_t port_num, uint8_t *p_data, uint16_t bytes_to_get, uint32_t *p_sender_Addr, uint16_t *p_sender_port);
    static bool OpenTCPSocket(uint16_t port_num, void * task_handle);
    static bool OpenCloudTCPSocket(uint16_t port_num, void * new_task_handle, void * old_task_handle);
    static bool GetTCPSocketStatus(uint16_t port_num);
    static bool TCPListen(uint16_t port_num);
    static bool TCPAccept(uint16_t port_num, uint32_t *p_dest_addr, uint16_t *p_dest_port, int32_t *p_socket_handle);
    static uint8_t get_num_ports_avail(uint16_t *p_port_num);
    static void SendData();
    static bool QueueToSend(uint8_t *p_data, EthernetSendCfg_t *p_send_cfg);
    static void set_duplicate_IP_addr_detect_flag() {m_is_duplicate_ip_detected = true;}
    static bool isTcpSocketConnected(uint16_t tcp_port);
    static bool OpenTCPSocketClient(uint16_t port_num);

    // Establishes a client TCP connection to a remote host
    static bool TcpConnect(uint16_t tcp_port, uint32_t remote_host_addr);

    static bool GetHostIPByName(char *p_h_name, uint32_t *p_ip_addr);
    static uint16_t TCPClientSocketSend(uint8_t *p_data, uint16_t bytes_to_send);
    static void DisconnectFromTCPHost();
    static bool OpenFDSession();
    static void CloseFDSession();

    static uint16_t TCPClientReceive(uint8_t *p_data, uint16_t size);

    // returns with data from a specific client.
    // if return value is non-zero, p_data will point to data in ethernet receive buffer.
    static uint16_t TCPReceive(uint16_t port_num, uint8_t client_id, uint8_t *p_data, int32_t size);

    // searches all TCP connections and returns with the first message found.  returns 0 if no data in any TCP connection
    // if return value is non-zero, p_data will point to data in ethernet receive buffer.
    static uint16_t TCPReceive(uint16_t *p_port_num, uint8_t *p_data, int32_t size, uint32_t *p_client_addr, uint8_t *p_client_id);

    static void MonitorLink();
    static void CloseAllSockets();

private:
    Ethernet();
    static bool isUDPSocketOpen(uint16_t port_num, uint8_t *p_index);
    static bool isTCPSocketOpen(uint16_t port_num);
    static bool CreateTCPSocket(int &p_handle);
    static bool SetTCPSocketOpts(int handle);
    static bool getTCPSocketIdx(uint16_t port_num, uint8_t *p_socket_idx);
    static uint8_t getFreeTCPClientSlot(uint8_t socket_idx);
    static void DisconnectClient(uint8_t socket_idx, uint8_t client_idx);
    static uint8_t getUDPSocket();
    static uint8_t getTCPSocket();
    static uint8_t incr_last_client_idx();
    static uint8_t incr_last_socket_idx();
    static uint16_t UDPSend(uint16_t port_num, uint8_t *p_data, uint16_t bytes_to_send, uint32_t dest_addr, uint16_t dest_port_num);
    static uint16_t TCPSend(uint16_t port_num, uint8_t client_id, uint8_t *p_data, uint16_t bytes_to_send);
    static bool IsDHCPRestartTimerExpired();
    static bool IsAPIPAStartTimerExpired();
    static void ChangeStaticIPAddr();
    static void Init();
    static void IPAddressUpdate();
    static bool isTCPClientConnected(uint8_t socket_idx, uint8_t client_idx);
    static bool isTCPClientConnected(uint8_t socket_idx);

    static UDPContext_t m_udp_sockets[MAX_UDP_SOCKETS];
    static TCPHostContext_t m_tcp_host_sockets[MAX_TCP_SOCKETS];
    static TCPClientContext_t m_tcp_client_socket;
    static bool m_is_fd_session_initialized;
    static bool m_is_ip_valid;
    static bool m_is_dhcp_ip_assigned;
    static bool m_is_dhcp_started;
    static bool m_is_new_ip_addr_assigned;

    // Link establishment occurs when Ethernet cable is connecting Ethernet port to another device's Ethernet port.
    // This variable indicates if such a connection is made.
    static bool m_is_cable_connected;

    static bool m_is_apipa_enabled;
    static bool m_is_ethernet_initialized;
    static bool m_is_duplicate_ip_detected;
    static uint8_t m_last_client_idx;
    static uint8_t m_last_socket_idx;
    static uint8_t m_dhcp_restart_tick_count;
    static uint32_t m_static_ip_addr;
    static uint32_t m_new_ip_addr;
    static uint32_t m_last_dhcp_restart_time;
    static uint8_t m_apipa_enable_tick_count;
    static uint32_t m_apipa_enable_start_time;
    static uint32_t m_ip_addr;

    // This is the status of the host comm socket
    static EthernetSocketStatus_t m_ethernet_socket_status;

};

#endif /* INCLUDE_ETHERNET_H_ */
