/*
 * ndk_interface.c
 *
 *  This module parforms a translation of NDK APIs from C to C++
 *
 *  Created on: Jun 21, 2016
 *      Author: yli
 */


#include <stdbool.h>
#include <string.h>
#include "ndk_interface.h"
#include "ndk_test_defines.h"
#include "SockUtils.h"

#define HANDLE int32_t
#define uint uint32_t
#define _extern extern
typedef uint8_t          UINT8;
typedef int32_t          INT32;
typedef uint16_t         UINT16;
typedef uint32_t         UINT32;
typedef uint32_t         IPN;             /* IP Address in NETWORK format */
#include <ti/ndk/inc/nettools/nettools.h>
#include <ti/ndk/inc/nettools/inc/dhcpif.h>
#include <ti/ndk/inc/nettools/inc/httpif.h>
#include <ti/ndk/inc/nettools/inc/natif.h>
#include <ti/ndk/inc/nettools/inc/telnetif.h>
#include <ti/ndk/inc/nettools/inc/configif.h>
#include <ti/ndk/inc/nettools/netcfg.h>
#include <ti/ndk/inc/nettools/inc/dnsif.h>
#include <ti/ndk/inc/serrno.h>
#include <ti/ndk/inc/os/osif.h>

extern uint32_t RTCAddHook( void (*pfn)( uint32_t, uint32_t, uint32_t ) );
extern int fdOpenSession( void *hTask );
extern void ti_ndk_config_Global_serviceReport(uint Item, uint Status, uint Report, HANDLE h);

#define NDK_CFG_ENTRY_FOUND         1
#define NDK_SUCCESS                 0

int call_ndk_accept(int socket_handle, struct sockaddr *addr, socklen_t *addrlen)
{
    return(accept(socket_handle, addr, addrlen));
}

int call_ndk_bind(int socket_handle, struct sockaddr *p_addr, socklen_t addrlen)
{
    return(bind(socket_handle, p_addr, addrlen));
}

int call_ndk_connect(int socket_handle, struct sockaddr *p_addr, socklen_t addrlen)
{
    return(connect(socket_handle, p_addr, addrlen));
}

int call_ndk_listen(int socket_handle, int maxcon)
{
    return(listen(socket_handle, maxcon));
}

int call_ndk_recv(int tcp_handle, void *p_data, int size, int flags)
{
    return(recv(tcp_handle, p_data, size, flags));
}

int call_ndk_recvfrom(int socket_handle, void *p_data, int size, int flags, struct sockaddr *p_addr, socklen_t *p_addrlen)
{
    return(recvfrom(socket_handle, p_data, size, flags, p_addr, p_addrlen));
}

int call_ndk_recvnc(int tcp_handle, void **p_data, int flags, void **phBuffer)
{
    return(NDK_recvnc((SOCKET)tcp_handle, p_data, flags, phBuffer));
}

void call_ndk_recvncfree(void * hBuffer)
{
    NDK_recvncfree(hBuffer);
}

int call_ndk_close(int tcp_handle)
{
    return(close(tcp_handle));
}

int call_setsockopt(int socket_handle, int level, int optname, const void *optval, socklen_t optlen)
{
    return(setsockopt(socket_handle, level, optname, optval, optlen));
}

int call_ndk_socket(int domain, int type, int protocol)
{
    return(socket(domain, type, protocol));
}

int call_ndk_send(int socket_handle, const void *p_data, int size, int flags)
{
    return(send(socket_handle, p_data, size, flags));
}

ssize_t call_ndk_sendto(int socket_handle, const void *p_data, size_t len, int flags,
        const struct sockaddr *to, socklen_t tolen)
{
    return(sendto(socket_handle, p_data, len, flags, to, tolen));
}

int call_fdError()
{
    int retval = fdError();
    return(retval);
}

int call_fdOpenSession( void *hTask )
{
    return(fdOpenSession(hTask));
}

void call_fdCloseSession( void *hTask )
{
    fdCloseSession(hTask);
}

void *call_TaskSelf()
{
    return((void *)TaskSelf());
}

void AddRTCHook(void (*pfn)( uint32_t, uint32_t, uint32_t ))
{
    if ( RTCAddHook( pfn ) == 0 )
    {
        PRINTFD("!!!Error: could not add NDK RTC hook\n");
    }
}

uint32_t swap_32(uint32_t in_val)
{
    uint32_t retval = (in_val >> 24) + ((in_val & 0xFF0000) >> 8) + ((in_val & 0xFF00) << 8) + (in_val << 24);
    return(retval);
}

bool SpecifyIPNetwork(uint32_t ip_addr, uint32_t subnet_msk)
{
    bool retval = true;
    int hCfg = 0;
    CI_IPNET net_cfg;

    // get the default configuration
    hCfg = CfgGetDefault();
    if(hCfg == 0)
    {
        retval = false;
        PRINTFD("Error - in func SpecifyIPNetwork, CfgGetDefault failed\n");
    }
    else
    {
        memset((uint8_t *)&net_cfg, 0, sizeof(net_cfg));
        net_cfg.IPAddr = swap_32(ip_addr);
        net_cfg.IPMask = swap_32(subnet_msk);
        net_cfg.NetType = 0;
        net_cfg.hBind = 0;
        strncpy(net_cfg.Domain, "fss.net", sizeof(net_cfg.Domain));
        retval = CfgAddEntry(hCfg, CFGTAG_IPNET, CFGITEM_SERVICE_TELNET, 0, sizeof(CI_IPNET), (UINT8 *)&net_cfg, 0);
    }

    return(retval);
}

bool RemoveStaticIP()
{
    bool retval = true;
    int hCfg = 0;
    int entry;
    int status;

    // get the default configuration
    hCfg = CfgGetDefault();
    if(hCfg == 0)
    {
        retval = false;
        PRINTFD("Error - in func RemoveStaticIP, CfgGetDefault failed\n");
    }
    else
    {
        status = CfgGetEntry(hCfg, CFGTAG_IPNET, CFGITEM_SERVICE_TELNET, 1, &entry);
        if(status == 0)
        {
            retval = false;
            PRINTFD("Error - in func RemoveStaticIP, couldn't get Static IP entry\n");
        }
        else
        {
            status = CfgRemoveEntry(hCfg, entry);
            if(status < 0)
            {
                PRINTFD("Error - in func RemoveStaticIP, couldn't remove Static IP entry\n");
                retval = false;
            }
            else
            {
                PRINTFD("Static IP successfully removed\n");
            }
        }
    }

    return(retval);
}

bool DisableNDKDHCP()
{
    int hCfg = 0;
    int entry;
    int status;
    bool retval = true;

    // get the default configuration
    hCfg = CfgGetDefault();
    if(hCfg == 0)
    {
        retval = false;
        PRINTFD("Error - in func DisableNDKDHCP, CfgGetDefault failed\n");
    }

    if(retval == true)
    {
        // get the DHCP client configuration
        status = CfgGetEntry(hCfg, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT, 1, &entry);

        if(status == NDK_CFG_ENTRY_FOUND)
        {
            status = CfgRemoveEntry(hCfg, entry);
            if(status != NDK_SUCCESS)
            {
                PRINTFD("!!!Error - DisableNDKDHCP, remove DHCP client entry failed\n");
                retval = false;
            }
            else
            {
                PRINTFD("DisableNDKDHCP success\n");
    	PRINTFD("\n\n");
    	SockUtils_printSockTable(IPPROTO_TCP);
            }
        }
        else if(status == NDK_SUCCESS)
        {
            PRINTFD("DisableNDKDHCP: no DHCP client entry found\n");
        }
        else
        {
            retval = false;
            PRINTFD("!!!Error - DisableNDKDHCP: find DHCP client entry error\n");
        }

    }

    return(retval);
}

bool RestartNDKDHCP()
{
    bool result = true;

    result = DisableNDKDHCP();

    // get the default configuration
    if(result == true)
    {
        result = StartNDKDHCP();
    }

    return(result);
}

// this section of code is taken from TI forum - http://e2e.ti.com/support/embedded/tirtos/f/355/t/439972
bool StartNDKDHCP()
{
    int hCfg = 0;
    int dhcp_entry;
    int status;
    bool result = true;

    // get the default configuration
    if(result == true)
    {
        hCfg = CfgGetDefault();
        if(hCfg == 0)
        {
            result = false;
            PRINTFD("Error - in func StartNDKDHCP, CfgGetDefault failed\n");
        }
    }

    // check if DHCP config already exists and needs to be removed
    if(result)
    {
        status = CfgGetEntry(hCfg, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT, 1, &dhcp_entry);
        if(status == NDK_CFG_ENTRY_FOUND)
        {
            PRINTFD("StartNDKDHCP: DHCP entry already exists. attempting to remove\n");
            status = CfgRemoveEntry(hCfg, dhcp_entry);
            if(status != NDK_SUCCESS)
            {
                PRINTFD("!!! Error - StartNDKDHCP: remove DHCP entry failed\n");
                result = false;
           }
        }
    }

    if(result == true)
    {
        CI_SERVICE_DHCPC dhcpc;

        UINT8 DHCP_OPTIONS[] =
                {
                DHCPOPT_SUBNET_MASK,
                };

        /* Specify DHCP Service on IF specified by "IfIdx" */
        memset(&dhcpc, 0, sizeof(dhcpc));
        dhcpc.cisargs.Mode   = 1;
        dhcpc.cisargs.IfIdx  = 1;
        dhcpc.cisargs.pCbSrv = &ti_ndk_config_Global_serviceReport;
        dhcpc.param.pOptions = DHCP_OPTIONS;
        dhcpc.param.len = 1;

        status = CfgAddEntry(hCfg, CFGTAG_SERVICE, CFGITEM_SERVICE_DHCPCLIENT, 0, sizeof(dhcpc), (UINT8 *)&dhcpc, 0);
        if(status != 1)
        {
            result = false;
        }
    }

    return(result);
}

bool NDK_GetHostIPByName(char *p_h_name, uint32_t *p_ip_addr)
{
    bool retval = false;
    uint8_t buffer[NDK_DNSBUFSIZE];
    HOSTENT *p_query_result = (HOSTENT *)buffer;
    *p_ip_addr = 0;

    int status = DNSGetHostByName(p_h_name, buffer, NDK_DNSBUFSIZE);
    if(status == NOERROR)
    {
        *p_ip_addr = htonl(p_query_result->h_addr[0]);
        PRINTFD("%d addresse(s) found. Length = %d, type=%d, first addr=%d.%d.%d.%d\n",
                p_query_result->h_addrcnt, p_query_result->h_length, p_query_result->h_addrtype,
                *p_ip_addr >> 24, (*p_ip_addr >> 16) & 0xFF,
                (*p_ip_addr >> 8) & 0xFF, *p_ip_addr & 0xFF);
        retval = true;
    }
    else
    {
        PRINTFD("!!! Err - NDK_GetHostIPByName failed. Err = %d, Name = %s\n", fdError(), p_h_name);
    }

    return(retval);
}

