

#include <stdio.h>
#include <string.h>

#include <inc_user/app/PFCMonitorNew.h>
#include "inc_user/osal/rtos_task.h"

#include "main.h"
#include "inc_user/hal/hal_gpio.h"
#include "inc_user/hal/hal_uart.h"
#include "inc_user/hal/hal_i2c.h"
#include "inc_user/hal/hal_spi.h"
#include "inc_user/hal/hal_adc.h"
#include "inc_user/hal/hal_eeprom.h"
#include "inc_user/hal/hal.h"

static _Bool initBoard(void);
static _Bool initPfcUartWrapper(void);
static _Bool initUserUartWrapper(void);
static _Bool initAccessoryUartWrapper(void);
static void systemAbort(char *message);

// ====================================================================================
// Main Function
// ====================================================================================
/**
 * @brief Main entry point of the application.
 * Initializes the system clock, peripherals, tasks, and starts the RTOS scheduler.
 */
int main(void)
{
    g_ui32SysClock = configSysClock(SYSTEM_MAIN_CLOCK);

    // Enable the Floating Point Unit
    FPUEnable();
    FPULazyStackingEnable();


    if (initBoard() == __FAIL)
    {
        // The initBoard function will call appSysAbort internally,
        // so the system will halt there. This is a fallback.
        boardInitFailIndicator(g_ui32SysClock);
    }


    g_DeviceConfig = DeviceConfiguration();

    // Initialize all application tasks

    // Enable communication interrupts now that tasks are ready
    enableOSInterrupt(BB_COMMUNICATION);
    enableOSInterrupt(PBCOMMUNICATION);
    enableOSInterrupt(Ay_COMMUNICATION);
//
//    // Start the TI-RTOS scheduler. This call does not return.
    startAppBIOS();
    // Should never be reached

    return 0;
}

// ====================================================================================
// Initialization Functions
// ====================================================================================

/**
 * @brief Initializes all board peripherals in a structured sequence.
 * Uses a table of initialization functions for maintainability and error reporting.
 * @return __PASS on success, __FAIL on failure.
 */
static _Bool initBoard(void)
{
    // Table of peripheral initialization functions.
    // This makes the sequence clear and easy to modify.
    const PeripheralInit_t initTable[] = {
        {initInternalEeprom,    "Internal EEPROM"},
        {initPortA,             "Port A"},
        {initPortB,             "Port B"},
        {initPortC,             "Port C"},
        {initPortD,             "Port D"},
        {initPortE,             "Port E"},
        {initPortF,             "Port F"},
        {initPortG,             "Port G"},
        {initPortH,             "Port H"},
        {initPortK,             "Port K"},
        {initPortL,             "Port L"},
        {initPortM,             "Port M"},
        {initPortN,             "Port N"},
        {initPortP,             "Port P"},
        {initPortQ,             "Port Q"},
        {dacInit,               "DAC"},
        {initInternalAdc,       "Internal ADC"},
        

    const size_t numModules = sizeof(initTable) / sizeof(initTable[0]);

    size_t i = 0;
    for (i = 0; i < numModules; ++i)
    {
        if (initTable[i].initFunction() == __FAIL)
        {
            char errorMsg[64];
            snprintf(errorMsg, sizeof(errorMsg), "%s initialization failed", initTable[i].moduleName);
            systemAbort(errorMsg); // This function will not return
            //TODO: Need to think if it is needed
            return __FAIL; // Should be unreachable
        }
    }

    return __PASS;
}

/**
 * @brief Handles a fatal system error.
 * This function is called when an unrecoverable error occurs. It puts the
 * hardware into a safe state, disables interrupts, and halts the system.
 * @param message A descriptive error message to be logged.
 */
void systemAbort(char *message)
{
    boardInitFailIndicator(g_ui32SysClock);

    System_printf("FATAL ERROR: %s\n", message);
    appSysAbort(message); // Halts the system
}

/**
 * @brief Provides safe access to the static device info.
 */

/**
 * @brief Provides safe access to the global device status.
 */
void getDeviceStatus(deviceStatus_t* status)
{
    if (status != NULL)
    {
        // Use a critical section to ensure the read is atomic
        uint32_t key = Hwi_disable();
        memcpy(status, &g_deviceStatus, sizeof(deviceStatus_t));
        Hwi_restore(key);
    }
}

/**
 * @brief Atomically updates a field in the global device status, compares it to
 * the previous state, and requests a host update if a change occurred.
 */

    // Compare the entire current status struct with the previous one.
    if (memcmp(&g_deviceStatus, &s_previousDeviceStatus, sizeof(deviceStatus_t)) != 0)
    {
        // A change was detected. Request the host communication task to send an update.
       
        requestHostResponse(RESPONSE_DEVICE_STATUS);

        // Update the shadow copy to the new current state.
        memcpy(&s_previousDeviceStatus, &g_deviceStatus, sizeof(deviceStatus_t));
    }

    // Exit the critical section.
    Hwi_restore(key);
}

void resetEstop(void)
{
    clearEstop(HIGH);
    appTaskSleepMs(100);
    clearEstop(LOW);
}

static _Bool initPfcUartWrapper(void)
{
    // Use the baud rate defined in the respective driver/header
    return ini(PB_BAUDRATE);
}

/**
 * @brief Wrapper for User/BB UART init to match PeripheralInit_t signature.
 */
static _Bool initUserUartWrapper(void)
{
    // Use the baud rate defined in the respective driver/header
    return initCommUser(BB_BAUDRATE); // Assuming BB_BAUDRATE is available
}

/**
 * @brief Wrapper for Accessory UART init to match PeripheralInit_t signature.
 */
static _Bool initAcUartWrapper(void)
{
    // Use the baud rate defined in the respective driver/header
    return initAcCom(AC_BAUDRATE); // Assuming AC_BAUDRATE is available
}
