#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "inc/hw_memmap.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/ssi.h"
#include "driverlib/sysctl.h"
#include "driverlib/uart.h"
#include "utils/uartstdio.h"

#define NUM_SSI_DATA 15

/*SSI init function*/
//*****************************************************************************
//
// This function sets up UART0 to be used for a console to display information
// as the example is running.
//
//*****************************************************************************
void
InitConsole(void)
{
    //
    // Enable GPIO port A which is used for UART0 pins.
    // TODO: change this to whichever GPIO port you are using.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    //
    // Configure the pin muxing for UART0 functions on port A0 and A1.
    // This step is not necessary if your part does not support pin muxing.
    // TODO: change this to select the port/pin you are using.
    //
    GPIOPinConfigure(GPIO_PA0_U0RX);
    GPIOPinConfigure(GPIO_PA1_U0TX);

    //
    // Enable UART0 so that we can configure the clock.
    //
    SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);

    //
    // Use the internal 16MHz oscillator as the UART clock source.
    //
    UARTClockSourceSet(UART0_BASE, UART_CLOCK_PIOSC);

    //
    // Select the alternate (UART) function for these pins.
    // TODO: change this to select the port/pin you are using.
    //
    GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

    //
    // Initialize the UART for console I/O.
    //
    UARTStdioConfig(0, 115200, 16000000);
}

void ssi_init(void)
{
	uint32_t ui32SysClkFreq;
	uint32_t temp;

    ui32SysClkFreq = SysCtlClockFreqSet((SYSCTL_XTAL_25MHZ | SYSCTL_OSC_MAIN | SYSCTL_USE_PLL | SYSCTL_CFG_VCO_480), 120000000);

    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI0);
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

    GPIOPinConfigure(GPIO_PA2_SSI0CLK);
    GPIOPinConfigure(GPIO_PA3_SSI0FSS);
    GPIOPinConfigure(GPIO_PA4_SSI0XDAT0);
    GPIOPinConfigure(GPIO_PA5_SSI0XDAT1);
    GPIOPinTypeSSI(GPIO_PORTA_BASE,GPIO_PIN_5|GPIO_PIN_4|GPIO_PIN_3|GPIO_PIN_2);

    SSIConfigSetExpClk(SSI0_BASE, ui32SysClkFreq, SSI_FRF_MOTO_MODE_0, SSI_MODE_MASTER, 10000, 8);
    SSIEnable(SSI0_BASE);

    SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI2);  //Enable ssi2
    SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOD); //Enable PORT B GPIO to be used with ssi2 data and frame signals

    GPIOPinConfigure(GPIO_PD3_SSI2CLK);
    GPIOPinConfigure(GPIO_PD2_SSI2FSS);
    GPIOPinConfigure(GPIO_PD1_SSI2XDAT0);
    GPIOPinConfigure(GPIO_PD0_SSI2XDAT1);
    GPIOPinTypeSSI(GPIO_PORTD_BASE,GPIO_PIN_0|GPIO_PIN_1|GPIO_PIN_2|GPIO_PIN_3);

    SSIConfigSetExpClk(SSI2_BASE, ui32SysClkFreq, SSI_FRF_MOTO_MODE_0, SSI_MODE_SLAVE, 10000, 8);
    SSIEnable(SSI2_BASE);
}


/*function to read the data from ssi0 in ssi2 and write it back to ssi0 and compare both*/
void main(void)
{
    uint32_t ui32Index;
    uint32_t ui32Data;
    char spi_data_received_back[NUM_SSI_DATA];
    uint32_t spi_received_char;
    const char spi_test_command[NUM_SSI_DATA] = {"This is a test\0"};
    bool pass_or_fail;

    InitConsole();
    ssi_init();

    memset(spi_data_received_back, '\0', NUM_SSI_DATA);
    for(ui32Index = 0; ui32Index < NUM_SSI_DATA; ui32Index++)
    {
		ui32Data = spi_test_command[ui32Index];
		SSIDataPut(SSI0_BASE, ui32Data);
        // SSIDataGet waits until a character is received
		SSIDataGet(SSI0_BASE, &spi_received_char); // Dummy read to empty master rx buffer
		SSIDataGet(SSI2_BASE, &spi_received_char); // Get data from slave rx buffer
		SSIDataPut(SSI2_BASE, spi_received_char);  // send back to master
		spi_data_received_back[ui32Index] = (char)spi_received_char;
    }
    if(strcmp(spi_test_command, spi_data_received_back) == 0)
    {
        pass_or_fail = 1;
        UARTprintf("\n\rPASSED\n\r");
    }
    else
    {
        pass_or_fail = 0;
        UARTprintf("\n\rFAILED\n\r");
    }

    return;
}
