/** @file sci.h
*   @brief SCI Driver Definition File
*   @date 24.Feb.2014
*   @version 03.08.01
*   
*/

/* (c) Texas Instruments 2009-2014, All rights reserved. */


#ifndef __SCI_H__
#define __SCI_H__

#include "reg_sci.h"

#ifdef __cplusplus
extern "C" {
#endif

/* USER CODE BEGIN (0) */
/* USER CODE END */

/** @enum sciIntFlags
*   @brief Interrupt Flag Definitions
*
*   Used with sciEnableNotification, sciDisableNotification
*/
enum sciIntFlags
{
    SCI_FE_INT    = 0x04000000U,  /* framing error */
    SCI_OE_INT    = 0x02000000U,  /* overrun error */
    SCI_PE_INT    = 0x01000000U,  /* parity error */
    SCI_RX_INT    = 0x00000200U,  /* receive buffer ready */
    SCI_TX_INT    = 0x00000100U,  /* transmit buffer ready */
    SCI_WAKE_INT  = 0x00000002U,  /* wakeup */
    SCI_BREAK_INT = 0x00000001U   /* break detect */
};

/** @def SCI_IDLE
*   @brief Alias name for the SCI IDLE Flag
*
*   This is an alias name for the SCI IDLE Flag.
*
*/
#define SCI_IDLE 0x00000004U

/** @struct sciBase
*   @brief SCI Register Definition
*
*   This structure is used to access the SCI module registers.
*/
/** @typedef sciBASE_t
*   @brief SCI Register Frame Type Definition
*
*   This type is used to access the SCI Registers.
*/

enum sciPinSelect
{
    PIN_SCI_TX = 4U,
    PIN_SCI_RX = 2U
};


/* Configuration registers */
typedef struct sci_config_reg
{
    uint32_t CONFIG_GCR0;
    uint32_t CONFIG_GCR1;
    uint32_t CONFIG_SETINT;
    uint32_t CONFIG_SETINTLVL;
    uint32_t CONFIG_FORMAT;
    uint32_t CONFIG_BRS;
    uint32_t CONFIG_PIO0;
    uint32_t CONFIG_PIO1;
    uint32_t CONFIG_PIO6;
    uint32_t CONFIG_PIO7;	
	uint32_t CONFIG_PIO8;
} sci_config_reg_t;


/* Configuration registers initial value for SCI*/
#define SCI_GCR0_CONFIGVALUE       0x00000001U   
#define SCI_GCR1_CONFIGVALUE       ((uint32_t)((uint32_t)1U << 5U) \
                                   |(uint32_t)((uint32_t)(2U-1U) << 4U) \
                                   |(uint32_t)((uint32_t)0U << 3U) \
                                   |(uint32_t)((uint32_t)0U << 2U) \
                                   |(uint32_t)((uint32_t)1U << 1U) \
                                   |(uint32_t)((uint32_t)0U << 2U) \
                                   |(uint32_t)(0x03000080U))
								   
#define SCI_SETINTLVL_CONFIGVALUE  ((uint32_t)((uint32_t)0U << 26U) \
                                   |(uint32_t)((uint32_t)0U << 25U) \
                                   |(uint32_t)((uint32_t)0U << 24U) \
                                   |(uint32_t)((uint32_t)0U << 9U) \
                                   |(uint32_t)((uint32_t)0U << 8U) \
                                   |(uint32_t)((uint32_t)0U << 1U) \
                                   |(uint32_t)((uint32_t)0U << 0U))
                                   
#define SCI_SETINT_CONFIGVALUE     ((uint32_t)((uint32_t)0U << 26U) \
                                   |(uint32_t)((uint32_t)0U << 25U) \
                                   |(uint32_t)((uint32_t)0U << 24U) \
                                   |(uint32_t)((uint32_t)0U << 9U) \
                                   |(uint32_t)((uint32_t)0U << 1U) \
                                   |(uint32_t)((uint32_t)0U << 0U))
                                   
#define SCI_FORMAT_CONFIGVALUE     (8U - 1U)
#define SCI_BRS_CONFIGVALUE        (585U)
#define SCI_PIO0_CONFIGVALUE       ((uint32_t)((uint32_t)1U << 2U ) | (uint32_t)((uint32_t)1U << 1U))
#define SCI_PIO1_CONFIGVALUE       ((uint32_t)((uint32_t)1U << 2U ) | (uint32_t)((uint32_t)0U << 1U))
#define SCI_PIO6_CONFIGVALUE       ((uint32_t)((uint32_t)0U << 2U ) | (uint32_t)((uint32_t)0U << 1U))
#define SCI_PIO7_CONFIGVALUE       ((uint32_t)((uint32_t)0U << 2U ) | (uint32_t)((uint32_t)0U << 1U))
#define SCI_PIO8_CONFIGVALUE       ((uint32_t)((uint32_t)1U << 2U ) | (uint32_t)((uint32_t)1U << 1U))
 



/** 
 *  @defgroup SCI SCI
 *  @brief Serial Communication Interface Module.
 *  
 *  The SCI module is a universal asynchronous receiver-transmitter that implements the standard nonreturn
 *  to zero format. The SCI can be used to communicate, for example, through an RS-232 port or over a K-line.
 *
 *	Related Files
 *   - reg_sci.h
 *   - sci.h
 *   - sci.c
 *  @addtogroup SCI
 *  @{
 */
 
/* SCI Interface Functions */
void sciInit(void);
void sciSetFunctional(sciBASE_t *sci, uint32_t port);
void sciSetBaudrate(sciBASE_t *sci, uint32_t baud);
uint32_t  sciIsTxReady(sciBASE_t *sci);
void sciSendByte(sciBASE_t *sci, uint8_t byte);
void sciSend(sciBASE_t *sci, uint32_t length, uint8_t * data);
uint32_t  sciIsRxReady(sciBASE_t *sci);
uint32_t  sciIsIdleDetected(sciBASE_t *sci);
uint32_t  sciRxError(sciBASE_t *sci);
uint32_t  sciReceiveByte(sciBASE_t *sci);
void sciReceive(sciBASE_t *sci, uint32_t length, uint8_t * data);
void sciEnableNotification(sciBASE_t *sci, uint32_t flags);
void sciDisableNotification(sciBASE_t *sci, uint32_t flags);
void sciEnableLoopback(sciBASE_t *sci, loopBackType_t Loopbacktype);
void sciDisableLoopback(sciBASE_t *sci);
void sciGetConfigValue(sci_config_reg_t *config_reg, config_value_type_t type);
/** @fn void sciNotification(sciBASE_t *sci, uint32_t flags)
*   @brief Interrupt callback
*   @param[in] sci   - sci module base address
*   @param[in] flags - copy of error interrupt flags
*
* This is a callback that is provided by the application and is called upon
* an interrupt.  The parameter passed to the callback is a copy of the 
* interrupt flag register.
*/
void sciNotification(sciBASE_t *sci, uint32_t flags);

/* USER CODE BEGIN (1) */
/* USER CODE END */
/**@}*/
#ifdef __cplusplus
}
#endif

#endif
