/*
 *  Copyright (C) 2022-23 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES ARE DISCLAIMED.
 */

#include <stdlib.h>
#include "ti_drivers_config.h"
#include "ti_board_config.h"
#include <kernel/dpl/HwiP.h>
#include <kernel/dpl/SystemP.h>
#include <drivers/rti.h>

/* External function prototypes (implemented elsewhere) */
void mcan_external_read_write_main(void *args);
void periodicTx(void);  /* This function handles the periodic CAN transmission */

/* Global RTI Hwi object for periodic transmissions */
static HwiP_Object gRtiPeriodicHwi;

/* RTI Periodic ISR wrapper: calls periodicTx() every 1 second */
void RTIPeriodicISR(void *arg)
{
    periodicTx();
}


/* Setup RTI for periodic transmission.
 * This registers an RTI ISR that fires every 1 second.
 */
void setupRTIPeriodicTx(void)
{
    HwiP_Params rtiParams;
    HwiP_Params_init(&rtiParams);
    rtiParams.intNum   = CONFIG_RTI0;      // RTI interrupt number defined in config
    rtiParams.callback = RTIPeriodicISR;   // ISR that calls periodicTx()
    rtiParams.args     = NULL;
    int32_t status = HwiP_construct(&gRtiPeriodicHwi, &rtiParams);
    if (status != SystemP_SUCCESS)
    {
        while (1);  // Halt on error
    }
    /* Enable the RTI interrupt for the designated timer block */
    RTI_intEnable(CONFIG_RTI0_BASE_ADDR, RTI_TMR_CNT_BLK_INDEX_0);
}

int main(void)
{
    /* Initialize system and board */
    System_init();
    Board_init();

    /* Enable the RTI counter for the timer block */
    RTI_counterEnable(CONFIG_RTI0_BASE_ADDR, RTI_TMR_CNT_BLK_INDEX_0);

    /* Set up the periodic RTI interrupt for continuous 1-sec transmissions */
    setupRTIPeriodicTx();

    /* Main loop: continuously process handshake messages.
     * The handshake processing function (mcan_external_read_write_main) handles incoming
     * CAN messages and sends appropriate responses via MCAN interrupts.
     */
    while (true)
    {
        mcan_external_read_write_main(NULL);
    }

    /* Cleanup (typically never reached in an embedded infinite loop) */
    Board_deinit();
    System_deinit();
    
    return 0;
}
