/*
 *  Copyright (C) 2022-23 Texas Instruments Incorporated
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES ARE DISCLAIMED.
 */

/*
 * This example demonstrates the use of the MCAN peripheral to implement a handshake
 * protocol across three states using J1939-formatted CAN IDs, and a continuous periodic
 * transmission every 1 second. The handshake state machine runs in the main loop and
 * responds to incoming CAN messages. Independently, an RTI timer interrupt fires every
 * 1 second to continuously transmit all three CAN requests (cycling through states 0, 1,
 * and 2) irrespective of handshake events.
 */

#include <stdio.h>
#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <kernel/dpl/DebugP.h>
#include <kernel/dpl/AddrTranslateP.h>
#include <kernel/dpl/SemaphoreP.h>
#include <kernel/dpl/HwiP.h>
#include <drivers/mcan.h>
#include "drivers/mcan/v0/hw_mcanss.h"
#include "ti_drivers_config.h"
#include "ti_drivers_open_close.h"
#include <drivers/rti.h>

/*----------------------------------------------------------------------------
 * Macro: Build a J1939 CAN Identifier from PGN, Priority, and Source Address.
 *----------------------------------------------------------------------------
 */
#define J1939_ID_FROM_PGN(pri, pgn, sa)  (((pri) << 26) | ((pgn) << 8) | ((sa) & 0xFF))

/*----------------------------------------------------------------------------
 * Configuration Defines
 *----------------------------------------------------------------------------
 */
#define APP_MCAN_BASE_ADDR          (CONFIG_MCAN0_BASE_ADDR)
#define APP_MCAN_INTR_NUM           (CONFIG_MCAN0_INTR)

#define APP_MCAN_STD_ID_FILTER_CNT  (0U)
#define APP_MCAN_EXT_ID_FILTER_CNT  (3U)
#define APP_MCAN_TX_BUFF_CNT        (1U)
#define APP_MCAN_TX_FIFO_CNT        (0U)
#define APP_MCAN_TX_EVENT_FIFO_CNT  (0U)
#define APP_MCAN_FIFO_0_CNT         (1U)
#define APP_MCAN_FIFO_1_CNT         (1U)

#define MCAN_EXT_ID_MASK            (0x1FFFFFFFU)
#define MCAN_EXT_ID_SHIFT           (0U)
#define MCAN_DATA_SIZE_8BYTES       (8U)

/*----------------------------------------------------------------------------
 * Global Variables
 *----------------------------------------------------------------------------
 */
static SemaphoreP_Object gMcanTxDoneSem;
static SemaphoreP_Object gMcanRxDoneSem;
static HwiP_Object       gMcanHwiObject;         // For handshake MCAN interrupts
static HwiP_Object       gRtiPeriodicHwi;         // For periodic 1-sec transmission
static uint32_t          gMcanBaseAddr;

MCAN_TxBufElement        txMsg;
MCAN_RxBufElement        rxMsg;

/* Predefined TX data for each state */
static const uint8_t txDataState0[MCAN_DATA_SIZE_8BYTES] = { 0x0A, 0x92, 0x3F, 0x67, 0xD4, 0xC5, 0x88, 0xB7 };
static const uint8_t txDataState1[MCAN_DATA_SIZE_8BYTES] = { 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88 };
static const uint8_t txDataState2[MCAN_DATA_SIZE_8BYTES] = { 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF, 0x12, 0x34 };

/*----------------------------------------------------------------------------
 * Function Prototypes
 *----------------------------------------------------------------------------
 */
static void App_mcanIntrISR(void *arg);
static void App_mcanConfig(Bool enableExternalLpbk);
static void App_mcanInitMsgRamConfigParams(MCAN_MsgRAMConfigParams *msgRAMConfigParams);
static void App_mcanEnableIntr(void);
static void App_mcanInitExtFilterElemParams(MCAN_ExtMsgIDFilterElement *extFiltElem, uint32_t bufNum);
static void App_mcanConfigTxMsgForState(uint32_t state, MCAN_TxBufElement *txMsg);
static int32_t App_getHandshakeState(uint32_t rxId);

/* Handshake processing function (runs in main loop) */
void mcan_external_read_write_main(void *args);

/* Periodic transmission function (called from RTI ISR) */
void periodicTx(void);

/* RTI Periodic ISR wrapper */
static void RTIPeriodicISR(void *arg);

/*----------------------------------------------------------------------------
 * Helper: Determine handshake state from received CAN ID.
 *----------------------------------------------------------------------------
 */
static int32_t App_getHandshakeState(uint32_t rxId)
{
    if (rxId == J1939_ID_FROM_PGN(3, 65281, 243))
        return 0;
    else if (rxId == J1939_ID_FROM_PGN(5, 65282, 243))
        return 1;
    else if (rxId == J1939_ID_FROM_PGN(5, 65283, 243))
        return 2;
    else
        return -1;
}

/*----------------------------------------------------------------------------
 * Periodic Transmission Function
 * Cycles through the three states and transmits a CAN message every 1 second.
 *----------------------------------------------------------------------------
 */
void periodicTx(void)
{
    static uint32_t periodicState = 0;
    int32_t status;
    const uint32_t txBufNum = 0;

    /* Configure TX message for current state */
    App_mcanConfigTxMsgForState(periodicState, &txMsg);
    MCAN_writeMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, txBufNum, &txMsg);
    status = MCAN_txBufAddReq(gMcanBaseAddr, txBufNum);
    if (status == MCAN_STATUS_SUCCESS)
    {
        DebugP_log("Periodic TX: State %d transmitted successfully.\r\n", periodicState);
    }
    else
    {
        DebugP_logError("Periodic TX: State %d transmission failed.\r\n", periodicState);
    }
    periodicState = (periodicState + 1) % 3;
}

/*----------------------------------------------------------------------------
 * RTI Periodic ISR
 * Called by the RTI timer every 1 second.
 *----------------------------------------------------------------------------
 */
void RTIPeriodicISR(void *arg)
{
    periodicTx();
}

/*----------------------------------------------------------------------------
 * Setup RTI for Periodic Transmission
 *----------------------------------------------------------------------------
 */
static void setupRTIPeriodicTx(void)
{
    HwiP_Params rtiParams;
    HwiP_Params_init(&rtiParams);
    rtiParams.intNum   = CONFIG_RTI0;      // RTI interrupt number from configuration
    rtiParams.callback = RTIPeriodicISR;   // ISR that triggers periodicTx()
    rtiParams.args     = NULL;
    int32_t status = HwiP_construct(&gRtiPeriodicHwi, &rtiParams);
    DebugP_assert(status == SystemP_SUCCESS);
    /* Enable the RTI interrupt for the designated timer block */
    RTI_intEnable(CONFIG_RTI0_BASE_ADDR, RTI_TMR_CNT_BLK_INDEX_0);
}

/*----------------------------------------------------------------------------
 * Handshake Processing Function (Responder Mode)
 * Waits for incoming CAN messages and responds when a recognized handshake
 * message is received. Meanwhile, the periodic RTI interrupt continuously transmits
 * all three CAN requests.
 *----------------------------------------------------------------------------
 */
void mcan_external_read_write_main(void *args)
{
    int32_t status = SystemP_SUCCESS;
    HwiP_Params hwiPrms;
    MCAN_RxNewDataStatus newDataStatus;
    MCAN_ErrCntStatus errCounter;
    const uint32_t txBufNum = 0U;
    const uint32_t rxFifoNum = MCAN_RX_FIFO_NUM_0;
    uint32_t rxBufNum, bitPos;
    int32_t rxState;

    /* Open system drivers */
    Drivers_open();
    ///Board_driversOpen();

    /* Create semaphores for TX and RX notifications */
    status = SemaphoreP_constructBinary(&gMcanTxDoneSem, 0);
    DebugP_assert(status == SystemP_SUCCESS);
    status = SemaphoreP_constructBinary(&gMcanRxDoneSem, 0);
    DebugP_assert(status == SystemP_SUCCESS);

    /* Register MCAN ISR for handshake reception */
    HwiP_Params_init(&hwiPrms);
    hwiPrms.intNum   = APP_MCAN_INTR_NUM;
    hwiPrms.callback = App_mcanIntrISR;
    status = HwiP_construct(&gMcanHwiObject, &hwiPrms);
    DebugP_assert(status == SystemP_SUCCESS);

    /* Get local MCAN base address and configure MCAN module */
    gMcanBaseAddr = (uint32_t)AddrTranslateP_getLocalAddr(APP_MCAN_BASE_ADDR);
    App_mcanConfig(TRUE);
    App_mcanEnableIntr();

    /* Setup RTI periodic interrupt for continuous transmission */
    setupRTIPeriodicTx();

    /* Continuous handshake loop: process incoming CAN messages */
    while (true)
    {
        SemaphoreP_pend(&gMcanRxDoneSem, SystemP_WAIT_FOREVER);
        MCAN_getNewDataStatus(gMcanBaseAddr, &newDataStatus);
        MCAN_clearNewDataStatus(gMcanBaseAddr, &newDataStatus);
        MCAN_getErrCounters(gMcanBaseAddr, &errCounter);
        DebugP_assert((0U == errCounter.recErrCnt) && (0U == errCounter.canErrLogCnt));

        for (rxBufNum = 0U; rxBufNum < APP_MCAN_EXT_ID_FILTER_CNT; rxBufNum++)
        {
            bitPos = (1U << rxBufNum);
            if (bitPos & newDataStatus.statusLow)
            {
                MCAN_readMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, rxBufNum, rxFifoNum, &rxMsg);
                uint32_t rxId = (rxMsg.id >> MCAN_EXT_ID_SHIFT) & MCAN_EXT_ID_MASK;
                rxState = App_getHandshakeState(rxId);
                if (rxState >= 0)
                {
                    /* Configure and send handshake response */
                    App_mcanConfigTxMsgForState(rxState, &txMsg);
                    status = MCAN_txBufTransIntrEnable(gMcanBaseAddr, txBufNum, TRUE);
                    DebugP_assert(status == CSL_PASS);
                    MCAN_writeMsgRam(gMcanBaseAddr, MCAN_MEM_TYPE_BUF, txBufNum, &txMsg);
                    status = MCAN_txBufAddReq(gMcanBaseAddr, txBufNum);
                    DebugP_assert(status == CSL_PASS);
                    SemaphoreP_pend(&gMcanTxDoneSem, SystemP_WAIT_FOREVER);
                    break;  // Process one handshake message at a time
                }
                else
                {
                    DebugP_logError("Unrecognized RX ID: 0x%X\r\n", rxId);
                }
            }
        }
    }

    /* Cleanup resources (unreachable in an infinite loop) */
    HwiP_destruct(&gMcanHwiObject);
    SemaphoreP_destruct(&gMcanTxDoneSem);
    SemaphoreP_destruct(&gMcanRxDoneSem);
    MCAN_reset(gMcanBaseAddr);
    ///Board_driversClose();
    Drivers_close();
}

/*----------------------------------------------------------------------------
 * MCAN Module Configuration
 *----------------------------------------------------------------------------
 */
static void App_mcanConfig(Bool enableExternalLpbk)
{
    MCAN_ExtMsgIDFilterElement extFiltElem[APP_MCAN_EXT_ID_FILTER_CNT] = {0U};
    MCAN_InitParams initParams = {0U};
    MCAN_ConfigParams configParams = {0U};
    MCAN_MsgRAMConfigParams msgRAMConfigParams = {0U};
    MCAN_BitTimingParams bitTimes = {0U};
    uint32_t i;

    /* Initialize operational parameters */
    MCAN_initOperModeParams(&initParams);
    initParams.fdMode    = FALSE;
    initParams.brsEnable = FALSE;

    /* Initialize global filter configuration */
    MCAN_initGlobalFilterConfigParams(&configParams);

    /* Set nominal and data bit timing (default: nominal 1Mbps, data 5Mbps) */
    MCAN_initSetBitTimeParams(&bitTimes);

    /* Configure the Message RAM */
    App_mcanInitMsgRamConfigParams(&msgRAMConfigParams);

    /* Initialize extended filter elements for states 0, 1 and 2 */
    for (i = 0U; i < APP_MCAN_EXT_ID_FILTER_CNT; i++)
    {
        App_mcanInitExtFilterElemParams(&extFiltElem[i], i);
    }
    
    /* Wait until Message RAM initialization is complete */
    while (FALSE == MCAN_isMemInitDone(gMcanBaseAddr)) {}

    /* Enter software initialization mode */
    MCAN_setOpMode(gMcanBaseAddr, MCAN_OPERATION_MODE_SW_INIT);
    while (MCAN_OPERATION_MODE_SW_INIT != MCAN_getOpMode(gMcanBaseAddr)) {}

    /* Initialize and configure the MCAN module */
    MCAN_init(gMcanBaseAddr, &initParams);
    MCAN_config(gMcanBaseAddr, &configParams);
    MCAN_setBitTime(gMcanBaseAddr, &bitTimes);
    MCAN_msgRAMConfig(gMcanBaseAddr, &msgRAMConfigParams);

    /* Set the extended ID mask */
    MCAN_setExtIDAndMask(gMcanBaseAddr, MCAN_EXT_ID_MASK);

    /* Add extended message ID filters */
    for (i = 0U; i < APP_MCAN_EXT_ID_FILTER_CNT; i++)
    {
        MCAN_addExtMsgIDFilter(gMcanBaseAddr, i, &extFiltElem[i]);
    }
    
    /* Enable external loopback mode if requested */
    if (TRUE == enableExternalLpbk)
    {
        MCAN_lpbkModeEnable(gMcanBaseAddr, MCAN_LPBK_MODE_EXTERNAL, FALSE);
    }
    
    /* Exit software initialization mode and enter normal operation */
    MCAN_setOpMode(gMcanBaseAddr, MCAN_OPERATION_MODE_NORMAL);
    while (MCAN_OPERATION_MODE_NORMAL != MCAN_getOpMode(gMcanBaseAddr)) {}

    return;
}

/*----------------------------------------------------------------------------
 * Enable MCAN Interrupts
 *----------------------------------------------------------------------------
 */
static void App_mcanEnableIntr(void)
{
    MCAN_enableIntr(gMcanBaseAddr, MCAN_INTR_MASK_ALL, (uint32_t)TRUE);
    MCAN_enableIntr(gMcanBaseAddr, MCAN_INTR_SRC_RES_ADDR_ACCESS, (uint32_t)FALSE);
    MCAN_selectIntrLine(gMcanBaseAddr, MCAN_INTR_MASK_ALL, MCAN_INTR_LINE_NUM_0);
    MCAN_enableIntrLine(gMcanBaseAddr, MCAN_INTR_LINE_NUM_0, (uint32_t)TRUE);
    return;
}

/*----------------------------------------------------------------------------
 * Initialize Message RAM Configuration Parameters
 *----------------------------------------------------------------------------
 */
static void App_mcanInitMsgRamConfigParams(MCAN_MsgRAMConfigParams *msgRAMConfigParams)
{
    int32_t status;
    MCAN_initMsgRamConfigParams(msgRAMConfigParams);

    msgRAMConfigParams->lss = 0;  /* No standard ID filters used */
    msgRAMConfigParams->lse = APP_MCAN_EXT_ID_FILTER_CNT;
    msgRAMConfigParams->txBufCnt = APP_MCAN_TX_BUFF_CNT;
    msgRAMConfigParams->txFIFOCnt = APP_MCAN_TX_FIFO_CNT;
    msgRAMConfigParams->txBufMode = MCAN_TX_MEM_TYPE_BUF;
    msgRAMConfigParams->txEventFIFOCnt = APP_MCAN_TX_EVENT_FIFO_CNT;
    msgRAMConfigParams->rxFIFO0Cnt = APP_MCAN_FIFO_0_CNT;
    msgRAMConfigParams->rxFIFO1Cnt = APP_MCAN_FIFO_1_CNT;
    msgRAMConfigParams->rxFIFO0OpMode = MCAN_RX_FIFO_OPERATION_MODE_BLOCKING;
    msgRAMConfigParams->rxFIFO1OpMode = MCAN_RX_FIFO_OPERATION_MODE_BLOCKING;

    status = MCAN_calcMsgRamParamsStartAddr(msgRAMConfigParams);
    DebugP_assert(status == CSL_PASS);
    return;
}

/*----------------------------------------------------------------------------
 * Configure Extended Filter Elements Based on Buffer Index
 *----------------------------------------------------------------------------
 */
static void App_mcanInitExtFilterElemParams(MCAN_ExtMsgIDFilterElement *extFiltElem, uint32_t bufNum)
{
    switch (bufNum)
    {
        case 0:
            extFiltElem->efid1 = J1939_ID_FROM_PGN(3, 65281, 243);
            break;
        case 1:
            extFiltElem->efid1 = J1939_ID_FROM_PGN(5, 65282, 243);
            break;
        case 2:
            extFiltElem->efid1 = J1939_ID_FROM_PGN(5, 65283, 243);
            break;
        default:
            DebugP_logError("Unsupported filter buffer %d.\r\n", bufNum);
            DebugP_assert(FALSE);
            break;
    }
    extFiltElem->efid2 = bufNum;   /* Assign dedicated RX buffer number */
    extFiltElem->efec  = MCAN_EXT_FILT_ELEM_BUFFER;
    extFiltElem->eft   = MCAN_EXT_FILT_TYPE_CLASSIC;
    return;
}

/*----------------------------------------------------------------------------
 * Configure TX Message Based on Handshake State
 *----------------------------------------------------------------------------
 */
static void App_mcanConfigTxMsgForState(uint32_t state, MCAN_TxBufElement *txMsg)
{
    /* Example sensor values; replace with actual sensor readings */
    uint16_t voltage = 7680;       /* Example: 7680V */
    uint16_t chargeCurrent = 120;  /* Example: 120A */
    uint16_t dischargeCurrent = 100; /* Example: 100A */
    uint16_t maxCellVoltage = 3600;  /* Example: 3.6V * 1000 */

    uint8_t SOC = 85;
    uint8_t SOH = 90;
    uint16_t minCellVoltage = 3250;
    uint8_t minCellModuleNo = 1;
    uint8_t minCellModulePosition = 2;
    uint8_t maxCellModuleNo = 3;
    uint8_t maxCellModulePosition = 4;

    uint16_t cycleCount = 5;
    uint8_t maxTemp = 45;
    uint8_t minTemp = 20;
    uint8_t alarm = 0x01;
    uint8_t fault = 0x00;
    uint8_t maxTempModuleNo = 1;
    uint8_t minTempModuleNo = 3;

    uint8_t txData[MCAN_DATA_SIZE_8BYTES] = {0};
    uint32_t expectedId = 0;

    /* Macro to pack a 16-bit value into two bytes */
    #define SET_16BIT_VALUE(arr, idx, val)  \
        do {                                \
            arr[idx] = (val >> 8) & 0xFF;     \
            arr[idx + 1] = val & 0xFF;        \
        } while (0)

    MCAN_initTxBufElement(txMsg);
    txMsg->dlc = MCAN_DATA_SIZE_8BYTES;
    txMsg->fdf = FALSE;  /* Use standard CAN frame format */
    txMsg->xtd = TRUE;   /* Enable extended ID */

    switch (state)
    {
        case 0:
            expectedId = J1939_ID_FROM_PGN(3, 65281, 243);
            SET_16BIT_VALUE(txData, 0, voltage);
            SET_16BIT_VALUE(txData, 2, chargeCurrent);
            SET_16BIT_VALUE(txData, 4, dischargeCurrent);
            SET_16BIT_VALUE(txData, 6, maxCellVoltage);
            break;
        case 1:
            expectedId = J1939_ID_FROM_PGN(5, 65282, 243);
            txData[0] = SOC;
            txData[1] = SOH;
            SET_16BIT_VALUE(txData, 2, minCellVoltage);
            txData[4] = minCellModuleNo;
            txData[5] = minCellModulePosition;
            txData[6] = maxCellModuleNo;
            txData[7] = maxCellModulePosition;
            break;
        case 2:
            expectedId = J1939_ID_FROM_PGN(5, 65283, 243);
            SET_16BIT_VALUE(txData, 0, cycleCount);
            txData[2] = maxTemp;
            txData[3] = minTemp;
            txData[4] = alarm;
            txData[5] = fault;
            txData[6] = maxTempModuleNo;
            txData[7] = minTempModuleNo;
            break;
        default:
            DebugP_logError("Invalid state %d for TX configuration!\r\n", state);
            DebugP_assert(FALSE);
            return;
    }

    txMsg->id = expectedId;
    memcpy(txMsg->data, txData, MCAN_DATA_SIZE_8BYTES);
}

/*----------------------------------------------------------------------------
 * (Optional) Compare RX Message Against Expected Values
 *----------------------------------------------------------------------------
 */
static bool App_mcanCompareRxMsgForState(uint32_t state, MCAN_RxBufElement *rxMsg)
{
    const uint8_t *expectedData = NULL;
    uint32_t expectedId = 0;

    switch (state)
    {
        case 0:
            expectedId = J1939_ID_FROM_PGN(3, 65281, 243);
            expectedData = txDataState0;
            break;
        case 1:
            expectedId = J1939_ID_FROM_PGN(5, 65282, 243);
            expectedData = txDataState1;
            break;
        case 2:
            expectedId = J1939_ID_FROM_PGN(5, 65283, 243);
            expectedData = txDataState2;
            break;
        default:
            DebugP_logError("Invalid state %d for RX comparison!\r\n", state);
            DebugP_assert(FALSE);
            return false;
    }
    uint32_t rxId = (rxMsg->id >> MCAN_EXT_ID_SHIFT) & MCAN_EXT_ID_MASK;
    if (rxId != expectedId || memcmp(rxMsg->data, expectedData, MCAN_DATA_SIZE_8BYTES) != 0)
    {
        DebugP_logError("State %d: RX message mismatch. Expected ID: 0x%X, Received: 0x%X\r\n",
                        state, expectedId, rxId);
        return false;
    }
    DebugP_log("State %d: RX message matched expected values.\r\n", state);
    return true;
}


/*
 * MCAN Interrupt Service Routine (Handshake RX)
 *----------------------------------------------------------------------------
 */
static void App_mcanIntrISR(void *arg)
{
    uint32_t intrStatus = MCAN_getIntrStatus(gMcanBaseAddr);
    MCAN_clearIntrStatus(gMcanBaseAddr, intrStatus);

    if (intrStatus & MCAN_INTR_SRC_TRANS_COMPLETE)
    {
        SemaphoreP_post(&gMcanTxDoneSem);
    }
    if (intrStatus & MCAN_INTR_SRC_DEDICATED_RX_BUFF_MSG)
    {
        SemaphoreP_post(&gMcanRxDoneSem);
    }
    return;
}

/*
 * End of File
 *
 * To run this example, ensure your project is configured with the required drivers
 * and board files. Call mcan_external_read_write_main() from your application
 * entry point. The RTI timer should be configured (via SysConfig or other means)
 * to fire every 1 second. Both handshake responses (via MCAN RX interrupts) and
 * continuous periodic transmissions (via the RTI periodic ISR) will occur.
 */
