/*
 * Copyright (c) 2021, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ti_msp_dl_config.h"

/**
 * CAN Tx Buffer Operation Variables
 */
uint8_t CANTxCount = 0;
uint8_t CANTxBufferPutIndex = 0;
DL_MCAN_TxBufElement txMsg0;
uint32_t CANTxBufferPendingRequest = 0;
uint8_t CANTxBufferPutIndexCache[0xFF] = {0};

uint32_t CANTxBufferIE = 0;
volatile uint8_t CANTxBufferDone = 0;

/**
 * CAN Rx FIFO/Buffer Operation Variables
 */
uint8_t CANRxCount = 0;
uint8_t CANRxFIFOGetIndex = 0;
DL_MCAN_RxBufElement rxMsg[MCAN0_INST_MCAN_FIFO_0_NUM*2];
uint16_t rxMsgID[MCAN0_INST_MCAN_FIFO_0_NUM*2] = {0};
DL_MCAN_RxFIFOStatus rxFS;

volatile uint8_t CANRxDone = 0;

/**
 * CAN Interrupt
 */
volatile uint32_t gInterruptLine1Status;

volatile bool gButtonFlag;

int main(void)
{
    SYSCFG_DL_init();

    /* Parameters Initialization */
    rxFS.num = DL_MCAN_RX_FIFO_NUM_0;
    rxFS.fillLvl = 0;

    /* Initialize txMsg0 to transmit. */
    /* Identifier Value. */
    txMsg0.id = ((uint32_t)(CANTxCount)) << 18U;
    /* Transmit data frame. */
    txMsg0.rtr = 0U;
    /* 11-bit standard identifier. */
    txMsg0.xtd = 0U;
    /* ESI bit in CAN FD format depends only on error passive flag. */
    txMsg0.esi = 0U;
    /* Transmitting 8 bytes. */
    txMsg0.dlc = 8U;
    /* CAN FD frames transmitted with bit rate switching. */
    txMsg0.brs = 0U;
    /* Frame transmitted in CAN FD format. */
    txMsg0.fdf = 0U;
    /* Store Tx events. */
    txMsg0.efc = 0U;
    /* Message Marker. */
    txMsg0.mm = 0xAAU;
    /* Data bytes. */
    for(int i = 0; i<txMsg0.dlc; i++) {
        txMsg0.data[i] = i;
    }

    DL_MCAN_enableIntr(MCAN0_INST, DL_MCAN_INTERRUPT_RF0N, 1U);
    DL_MCAN_enableIntr(MCAN0_INST, DL_MCAN_INTERRUPT_TC, 1U);

    NVIC_EnableIRQ(MCAN0_INST_INT_IRQN);
    NVIC_EnableIRQ(GPIO_SWITCHES_INT_IRQN);

    while (DL_MCAN_OPERATION_MODE_NORMAL != DL_MCAN_getOpMode(MCAN0_INST))
        ;

    /******************************************************************
     * CAN Interrupt Send Test
     *
     * Send one package first and the send other package in interrupt
     */

    txMsg0.id = ((uint32_t)(CANTxCount)) << 18U;
    txMsg0.data[0] = CANTxCount;

    /* Find the empty Tx buffer to start transfer. */
    CANTxBufferPutIndex = 0;
    CANTxBufferPendingRequest = DL_MCAN_getTxBufReqPend(MCAN0_INST);
    while(CANTxBufferPendingRequest & (1 << CANTxBufferPutIndex)) {
        CANTxBufferPutIndex++;
        if(CANTxBufferPutIndex >= MCAN0_INST_MCAN_TX_BUFF_SIZE) {
            delay_cycles(32);   //Delay 1ms waiting for Tx finish
            CANTxBufferPutIndex = 0;
            CANTxBufferPendingRequest = DL_MCAN_getTxBufReqPend(MCAN0_INST);
        }
    }

    CANTxBufferPutIndexCache[CANTxCount] = CANTxBufferPutIndex;   //Test code

    /* Enable the corresponding Tx Buffer's Interrupt */
    DL_MCAN_TXBufTransIntrEnable(MCAN0_INST, CANTxBufferPutIndex, 1U);

    /* Write Tx Message to the Message RAM. */
    DL_MCAN_writeMsgRam(MCAN0_INST, DL_MCAN_MEM_TYPE_BUF, CANTxBufferPutIndex, &txMsg0);
    /* Add request for transmission. */
    DL_MCAN_TXBufAddReq(MCAN0_INST, CANTxBufferPutIndex);

    DL_GPIO_setPins(GPIOB, Debug_IO0_PIN);  //Test code
    DL_GPIO_clearPins(GPIOB, Debug_IO0_PIN);//Test code

    CANTxCount++;

    /* Stop sending when 0xFF messages have been sent. */
//    if(CANTxCount == 0xFF) {
//        break;
//    }

    /* Waiting for CAN Tx message send finish. */
    while(!CANTxBufferDone){
        __asm("NOP");
    }

    /******************************************************************
     * CAN Interrupt Receive Test
     */
    rxFS.num = DL_MCAN_RX_FIFO_NUM_0;
    rxFS.fillLvl = 0;
//    /* Waiting for New CAN Message Stored into FIFO 0 */
//    while( !(DL_MCAN_getIntrStatus(MCAN0_INST)&DL_MCAN_INTERRUPT_RF0N) );
//    /* Waiting for FIFO 0 Fill Level Not 0 */
//    while ((rxFS.fillLvl) == 0) {
//        DL_MCAN_getRxFIFOStatus(MCAN0_INST, &rxFS);
//    }
//    DL_MCAN_readMsgRam(MCAN0_INST, DL_MCAN_MEM_TYPE_FIFO, 0U, rxFS.num, &rxMsg[CANRxCount]);
//    DL_MCAN_writeRxFIFOAck(MCAN0_INST, rxFS.num, rxFS.getIdx);
//    /* Clear Receive FIFO 0 New Message Interrupt Flag */
//    DL_MCAN_clearIntrStatus(MCAN0_INST, DL_MCAN_INTERRUPT_RF0N, DL_MCAN_INTR_SRC_MCAN_LINE_1);
//
//    rxMsgID[CANRxCount] = (rxMsg[CANRxCount].id >> 18) & 0x07FF;
//
//    DL_GPIO_setPins(GPIOB, Debug_IO0_PIN);  //Test code
//    DL_GPIO_clearPins(GPIOB, Debug_IO0_PIN);//Test code
//
//    CANRxCount++;

    /* Stop receiving when 20 messages have been sent. */
    while(!CANRxDone) {
        __asm("NOP");
    }

    /* Infinite loop here. */
    while(1) {
        __asm("NOP");
    }
}

void MCAN0_INST_IRQHandler(void)
{
    static uint8_t NoCANTxBufferInterruptEnable = 0;
    switch (DL_MCAN_getPendingInterrupt(MCAN0_INST)) {
        case DL_MCAN_IIDX_LINE1:
            /* Check MCAN interrupts fired during TX/RX of CAN package */
            gInterruptLine1Status |= DL_MCAN_getIntrStatus(MCAN0_INST);

            /* New message received by Rx FIFO 0 (Filter matched) */
            if(gInterruptLine1Status&DL_MCAN_INTERRUPT_RF0N) {
                do {
                    DL_MCAN_getRxFIFOStatus(MCAN0_INST, &rxFS);
                } while((rxFS.fillLvl) == 0);

                DL_MCAN_readMsgRam(MCAN0_INST, DL_MCAN_MEM_TYPE_FIFO, 0U, rxFS.num, &rxMsg[CANRxCount]);
                DL_MCAN_writeRxFIFOAck(MCAN0_INST, rxFS.num, rxFS.getIdx);

                rxMsgID[CANRxCount] = (rxMsg[CANRxCount].id >> 18) & 0x07FF;

                DL_GPIO_setPins(GPIOB, Debug_IO0_PIN);  //Test code
                DL_GPIO_clearPins(GPIOB, Debug_IO0_PIN);//Test code

                CANRxCount++;

                /* Stop receiving when 20 messages have been sent. */
                if(CANRxCount == MCAN0_INST_MCAN_FIFO_0_NUM * 2) {
                    CANRxDone = 1;
                }
            }

            if(gInterruptLine1Status&DL_MCAN_INTERRUPT_TC) {
                txMsg0.id = ((uint32_t)(CANTxCount)) << 18U;
                txMsg0.data[0] = CANTxCount;

                CANTxBufferPutIndex = 0;
                CANTxBufferIE = MCAN0_INST->MCANSS.MCAN.MCAN_TXBTIE;
                NoCANTxBufferInterruptEnable = 0;
                while( !(CANTxBufferIE & (1<<CANTxBufferPutIndex)) ) {
                    CANTxBufferPutIndex++;

                    if(CANTxBufferPutIndex == 31) {
                        NoCANTxBufferInterruptEnable = 1;
                        break;
                    }
                }

                if(NoCANTxBufferInterruptEnable == 0) {
                    /* Write Tx Message to the Message RAM. */
                    DL_MCAN_writeMsgRam(MCAN0_INST, DL_MCAN_MEM_TYPE_BUF, CANTxBufferPutIndex, &txMsg0);
                    /* Add request for transmission. */
                    DL_MCAN_TXBufAddReq(MCAN0_INST, CANTxBufferPutIndex);

                    DL_GPIO_setPins(GPIOB, Debug_IO0_PIN);  //Test code
                    DL_GPIO_clearPins(GPIOB, Debug_IO0_PIN);//Test code

                    CANTxCount++;

                    /* Stop sending when 0xFF messages have been sent. */
                    if(CANTxCount == 0xFF) {
                        /* Disable the corresponding Tx Buffer's Interrupt */
                        DL_MCAN_TXBufTransIntrEnable(MCAN0_INST, CANTxBufferPutIndex, 0U);

                        CANTxBufferDone = 1;
                    }
                }
            }

            DL_MCAN_clearIntrStatus(MCAN0_INST, gInterruptLine1Status, DL_MCAN_INTR_SRC_MCAN_LINE_1);
            break;
        default:
            break;
    }
}

void GROUP1_IRQHandler(void)
{
    switch (DL_Interrupt_getPendingGroup(DL_INTERRUPT_GROUP_1)) {
        case GPIO_SWITCHES_INT_IIDX:
            switch (DL_GPIO_getPendingInterrupt(GPIO_SWITCHES_PORT)) {
                case DL_GPIO_IIDX_DIO21:
                    gButtonFlag = true;
                    break;
                default:
                    break;
            }
            break;
        default:
            break;
    }
}

void NMI_Handler(void)
{
    __BKPT(0);
}
void HardFault_Handler(void)
{
    __BKPT(0);
}


/**
 * When DL_MCAN_StdMsgIDFilterElement's sft = 2 or
 *      DL_MCAN_ExtMsgIDFilterElement's eft = 2.
 * We are using SFID1/EFID1 as filter and SFID2/EFID2 as mask.
 * All mask bit are 1: only ID == filter can be received and put into RxFIFO0.
 * All mask bit are 0: All message IDs match.
 */
